#include <stdio.h>

#include "arena.h"

#define Byte w_rot_ih_eByte 	/* - ,    ... */
#   include PNG_HEADER_FILE_FOR_INCLUSION_IN_ARENA
#undef Byte

#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
#endif
#include "error.h"
#include "dither.h"
#include "image.h"
#include "main.h"
#include "png_arena_i.h"
#include "png_arena.h"
#include "util.h"


#define PNG_ERROR(png_ptr, message)	Arena_png_error(png_ptr, message)


#ifdef ARENA_USE_OWN_INTERLACE_HANDLING_INSTEAD_OF_LIBPNG
/* Just to avoid recalculating this */
int arena_png_adam7_vstep[7]  = {8, 8, 8, 4, 4, 2, 2};
int arena_png_adam7_vinit[7]  = {0, 0, 4, 0, 2, 0, 1};
#endif


#ifdef ARENA_DEBUG
static char *png_colour_type[] = {"grayscale",
				  "undefined type",
				  "RGB",
				  "colourmap",
				  "grayscale+alpha",
				  "undefined type",
				  "RGB+alpha"};
#endif


void Arena_png_error(png_structp png_ptr, char *message)
{
 Arena_PrintError("Arena libpng error: %s\n", message);
#if 1
 longjmp(png_ptr->jmpbuf, 1);
# else
 Exit(-1);
#endif
}


void Arena_png_warning(png_structp png_ptr, char *message)
{
 if (png_ptr)
   Arena_PrintError("Arena libpng warning: %s\n", message);
  else
   return;
 /* End if */
}


/* png_rw_ptr */
static void arena_png_read_data(png_structp png_ptr, png_bytep data,
				png_size_t length)
{
 Block *bp = (Block*)png_get_io_ptr(png_ptr);

 if (bp->size > bp->next)
   {
    if (bp->next + length > bp->size) length = bp->size - bp->next;

    memcpy(data, bp->buffer + bp->next, length);
    bp->next += length;
    return;
   }
  else
   {
    PNG_ERROR(png_ptr, "Read Error");
   }
}


static void arena_png_info_callback(png_structp the_png_ptr,
				    png_infop the_png_info)
{
 Image* theimage;
 png_bytep png_image = NULL;
 int bit_depth, colour_type;
 double image_gamma;

 bit_depth = the_png_info->bit_depth;
 colour_type = the_png_info->color_type;

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
 if (IMAGE_TRACE)
   Arena_TracePrint("arena_png_info_callback",
		    "\n\tprocessing PNG image:"
		    " %ld x %ld, %d-bit%s, %s, %sinterlaced\n",
		    the_png_info->width, the_png_info->height,
		    bit_depth, (bit_depth > 1) ? "s" : "",
		    ((colour_type > 6) ?
		     png_colour_type[1] : png_colour_type[colour_type]),
		    the_png_info->interlace_type ? "" : "non-");
#endif

 /* tell libpng to handle the gamma conversion */
 if (the_png_info->valid & PNG_INFO_gAMA)
   image_gamma = the_png_info->gamma;
  else
   image_gamma = (double)0.45; /* FIXME: OR: 1/Gamma ? */

 /* Only gamma correct if >1.1% difference */
 if (abs((Gamma * image_gamma) -1) > 0.011)
   png_set_gamma(the_png_ptr, Gamma, image_gamma);

 if (the_png_info->valid & PNG_INFO_bKGD)
   png_set_background(the_png_ptr, &(the_png_ptr->background),
		      PNG_BACKGROUND_GAMMA_FILE, 1, image_gamma);

 /* tell libpng to convert 16 bits per channel to 8 */
 if (the_png_info->bit_depth == 16) png_set_strip_16(the_png_ptr); 

 /* expand to RGB or 8 bit grayscale (with alpha if present) */
 png_set_expand(the_png_ptr);

 /* atomatically handle interlace expansion (use ``blocky image'' mode) */
 png_set_interlace_handling(the_png_ptr);

 /* Call to update the info_ptr area with the new settings */
 png_read_update_info(the_png_ptr, the_png_info);

 theimage = (Image*)png_get_progressive_ptr(the_png_ptr);
 if (theimage != NULL)
   {
    theimage->npixels = 0;
    theimage->width = the_png_info->width;
    theimage->height = the_png_info->height;

    if ((png_image = (png_bytep)Arena_MAlloc((long)(the_png_info->channels) *
					     (long)(the_png_info->height) *
					     (long)(the_png_info->width),
					     False))
	 == NULL)
      PNG_ERROR(the_png_ptr, "couldn't alloc space for PNG image");

    ((Arena_PNG_struct*)(theimage->aux))->png_image = png_image;
    ((Arena_PNG_struct*)(theimage->aux))->image_status = ARENA_PNG_IMAGE_OLD;
   }

 return;
}


static void arena_png_row_callback(png_structp the_png_ptr,
				   png_bytep new_row,
				   png_uint_32 row_num,
				   int pass)
{
 Image* theimage;

#ifdef ARENA_DEBUG
 if (IMAGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint("arena_png_row_callback",
		    " pass = %i,   row_num = %lu.\n", pass, row_num);
#endif

 theimage = (Image*)png_get_progressive_ptr(the_png_ptr);
 if (theimage == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("arena_png_row_callback",
		       " the PNG image is NULL?\n");
#endif
    return;
   }
 /* End if */

 if (theimage->aux == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("arena_png_row_callback",
		       "\n\tthe PNG image "POINTER_FORMAT
		       " hasn't been initialized?\n",
		       theimage);
#endif
    return;
   }
 /* End if */

 if (((Arena_PNG_struct*)(theimage->aux))->png_image == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("arena_png_row_callback",
		       "\n\tthe PNG image "POINTER_FORMAT
		       " hasn't been initialized properly?\n",
		       theimage);
#endif
    return;
   }
 /* End if */

 if (new_row != NULL)
   {
    long width;
    png_bytep old_row;
    png_infop the_info_ptr;
    Arena_PNG_struct* the_Arena_PNG_struct;

    the_Arena_PNG_struct = (Arena_PNG_struct*)(theimage->aux);
    the_info_ptr = the_Arena_PNG_struct->info_ptr;

    width = (long)(the_info_ptr->channels) * (long)(the_info_ptr->width);
    old_row = the_Arena_PNG_struct->png_image + row_num * width;

    png_progressive_combine_row(the_png_ptr, old_row, new_row);

    the_Arena_PNG_struct->image_status = ARENA_PNG_IMAGE_CHANGED;
   }
 /* End if */

 return;
}


static void arena_png_end_callback(png_structp the_png_ptr,
				   png_infop the_png_info)
{
 Image* theimage;

 /* Just mark the image as finished */
 if ((theimage = (Image*)png_get_progressive_ptr(the_png_ptr)) != NULL)
   {
    ((Arena_PNG_struct*)(theimage->aux))->image_status = ARENA_PNG_IMAGE_FINISHED;
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE  && VERBOSE_TRACE)
      Arena_TracePrint("arena_png_end_callback",
		       "\n\tthe PNG image "POINTER_FORMAT" finished.\n",
		       theimage);
#endif
   }
}


/*
 * Read image into displays 1,2 or 4 bit deep
 */
ImageData* ReadPNGimage_1_2_4(png_struct* png_ptr, png_info* info_ptr,
			      png_byte* png_image)
{
 png_byte* pp;
 ImageData* data;
 unsigned char *dp;
 int ypos;
 int ppb, bpl;
 int len = info_ptr->width;
 int colour_type = info_ptr->color_type;
 int alpha;
 int newbyte;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadPNGimage_1_2_4";
#endif


 ppb = 8/depth; /* pixels per byte */
 bpl = len/ppb + (len%ppb ? 1 : 0); /* bytes per line */

 if ((data = NewImageData((bpl * info_ptr->height), 0)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " no space left for image data.\n");
#endif
    return NULL;
   }

 /* Remove alpha channel from type,  but remember it */
 alpha = colour_type & PNG_COLOR_MASK_ALPHA;
 colour_type &= ~PNG_COLOR_MASK_ALPHA;

 pp = png_image;
 dp = data->image;
 newbyte = 1;
 for (ypos = 0; ypos < png_ptr->height; ypos++)
   {
    int col = ypos & 0x0F;
    int shift = 0;
    int xpos;

    for (xpos = 0; xpos < len; ++xpos)
      {
       int row = xpos & 0x0F;
       int gr;
       int cr = 0, cg = 0, cb = 0, cgr = 0, a = 0;
       int isgrey = 0;

       if (newbyte)
	 {
	  *dp = 0;
	  newbyte = 0;
	 }

/*
   (From gif.c)
   Some quick notes to remember how shift is calculated:

   0 1 2 3 4 5 6 7         xpos
   0 1 2 3 4 5 6 7         xpos % 8
   7 6 5 4 3 2 1 0    7 - (xpos % 8)

   1 0 1 0 1 0 1 0   (7 - (xpos % 8)) % ppb           pixels per byte = 2
   4 0 4 0 4 0 4 0  ((7 - (xpos % 8)) % ppb) * depth  depth = 4

   3 2 1 0 3 2 1                                      pixels per byte = 4
   6 4 2 0 6 4 2                                      depth = 2

*/
       shift = (((xpos % 8) % ppb) * depth);

       if (colour_type == PNG_COLOR_TYPE_GRAY)
	 {
	  cr = cg = cb = cgr = (*pp++);
	  isgrey = 1;
	 }
        else
	 if (colour_type == PNG_COLOR_TYPE_RGB)
	   {
	    int drb, dgb, dbr;

	    cr = (*pp++);
	    cg = (*pp++);
	    cb = (*pp++);
	    cgr = (0.59*cr) + (0.20*cg) + (0.11*cb);

	    drb = abs(cr - cb);
	    dgb = abs(cg - cb);
	    dbr = abs(cb - cr);
	    if (drb < 20 && dgb < 20 && dbr < 20) isgrey = 1;
	   }
	  else
	   {
	    PNG_ERROR(png_ptr,
		      "Unknown PNG colour type seen (ReadPNGimage_1_2_4)");
	   }

       /* Process alpha channel if present */
       if (alpha) a = (*pp++);

       if (alpha && a != 0xff)
	 {
	  int tr, tg, tb;
	  int drb, dgb, dbr;

	  /* From www.c - sorry! */
	  if (depth == 1)
	    {
	     tr = 255; tg = 255; tb = 255;
	    }
	   else
	    {
	     tr = 220; tg = 209; tb = 186;
	    }

	  /* Add picture pixel to background pixel */
	  cr = (a/255.0)*cr + ((255.0 - a)/255.0) * tr;
	  cg = (a/255.0)*cg + ((255.0 - a)/255.0) * tg;
	  cb = (a/255.0)*cb + ((255.0 - a)/255.0) * tb;

	  /* Recalculate grayness */
	  cgr = (0.59 * cr) + (0.20 * cg) + (0.11 * cb);
	  isgrey = 0;
	  drb = abs(cr - cb);
	  dgb = abs(cg - cb);
	  dbr = abs(cb - cr);
	  if (drb < 20 && dgb < 20 && dbr < 20) isgrey = 1;
	 }

       if (alpha && !a)
	 *dp |= transparent << shift;
        else
	 if (imaging == COLOUR232)
	   {
	    if (isgrey)
	      {
	       int gr = cgr & 0xF0;

	       if (cgr - gr > Magic16[(row << 4) + col]) gr += 16;

	       gr = min(gr, 0xF0);

	       *dp |= greymap[gr >> 4] << shift;
	      }
	     else
	      {
	       int r = cr & 0xC0;
	       int g = cg & 0xE0;
	       int b = cb & 0xC0;
	       int v = (row << 4) + col;

	       if (cr - r > Magic64[v]) r += 64;
	       if (cg - g > Magic32[v]) g += 32;
	       if (cb - b > Magic64[v]) b += 64;

	       /* clamp error to keep colour in range 0 to 255 */

	       r = min(r, 255) & 0xC0;
	       g = min(g, 255) & 0xE0;
	       b = min(b, 255) & 0xC0;

	       *dp |= stdcmap[(r >> 6) | (g >> 3) | (b >> 1)] << shift;
	      }
	   }
	  else
	   if (imaging == MONO)
	     {
	      if (cgr < Magic256[(row << 4) + col])
		*dp |= greymap[0] << shift;
	       else
		*dp |= greymap[15] << shift;
	     }
	    else /* Not COLOUR232 or MONO */
	     {
	      gr = cgr & 0xF0;
	      if (cgr - gr > Magic16[(row << 4) + col]) gr += 16;

	      gr = min(gr, 0xF0);
	      *dp |= greymap[gr >> 4] << shift;
	     }

       if ((shift + depth) == 8)
	 {
	  dp++;
	  newbyte = 1;
	 }

      } /* xpos */

    if (shift)
      {
       dp++;   /* make sure we start on a new byte for the next line */
       newbyte = 1;
      }
   } /* ypos */

 return data;
}


/*
 * Read image into displays 24 bit deep
 */
ImageData* ReadPNGimage_24(png_struct* png_ptr, png_info* info_ptr,
			   png_byte* png_image)
{
 png_byte* pp;
 ImageData* data;
 unsigned char *dp;
 unsigned long int ulp;
 int ypos;
 int len = info_ptr->width;
 int colour_type = info_ptr->color_type;
 int alpha;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadPNGimage_24";
#endif


 if ((data = NewImageData((4 * len * info_ptr->height), 0)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " no space left for iamge data.\n");
#endif
    return NULL;
   }

 /* Remove alpha channel from type,  but remember it */
 alpha = colour_type & PNG_COLOR_MASK_ALPHA;
 colour_type &= ~PNG_COLOR_MASK_ALPHA;

 pp = png_image;
 dp = data->image;
 for (ypos = 0; ypos < png_ptr->height; ypos++)
   {
    int xpos;

    for (xpos = 0; xpos < len; ++xpos)
      {
       int cr = 0, cg = 0, cb = 0, cgr = 0, a = 0;

       if (colour_type == PNG_COLOR_TYPE_GRAY)
	 {
	  cr = cg = cb = cgr = (*pp++);
	 }
        else
	 if (colour_type == PNG_COLOR_TYPE_RGB)
	   {
            cr = (*pp++);
            cg = (*pp++);
            cb = (*pp++);
            cgr = (0.59 * cr) + (0.20 * cg) + (0.11 * cb);
          }
	 else
	   {
            PNG_ERROR(png_ptr,
		      "Unknown PNG colour type seen (ReadPNGimage_24)");
	   }

       /* Process alpha channel if present */
       if (alpha) a = (*pp++);

       /* Assume COLOUR888 */
       if (alpha && a != 0xff)
	 {
	  int tr, tg, tb;

	  /* Get tile colour without gamma - from www.c */
	  if (tileData)
	    {
	     int x = xpos % tileWidth;
	     int y = ypos % tileHeight;
	     int i = y * tileWidth + x;
	     unsigned char *s = tileData + 4 * i;
	     tb = s[(RPixelShift) ? 1 : 3];
	     tg = s[2];
	     tr = s[(RPixelShift) ? 3 : 1];
	    }
	   else
	    {
	     tb = (transparent       ) & 0xFF;
	     tg = (transparent  >>  8) & 0xFF;
	     tr = (transparent  >> 16) & 0xFF;
	    }

	  /* Add picture pixel to background pixel */
	  cr = (a/255.0) * cr + ((255.0-a)/255.0) * tr;
	  cg = (a/255.0) * cg + ((255.0-a)/255.0) * tg;
	  cb = (a/255.0) * cb + ((255.0-a)/255.0) * tb;
	 }

       GetPNGcolour(cr, cg, cb, &ulp);
       *dp++ = (ulp      ) & 0xFF;   /* LSB first! */
       *dp++ = (ulp >>  8) & 0xFF;
       *dp++ = (ulp >> 16) & 0xFF;
       *dp++ = '\0';
      } /* xpos */
   } /* ypos */      

 return data;
}


/*
 * Read image into displays 16 bit deep
 */
ImageData* ReadPNGimage_16(png_struct* png_ptr, png_info* info_ptr,
			   png_byte* png_image)
{
 png_byte* pp;
 ImageData* data;
 unsigned char *dp;
 unsigned long int ulp;
 int ypos;
 int len = info_ptr->width;
 int colour_type = info_ptr->color_type;
 int alpha;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadPNGimage_16";
#endif


 if ((data = NewImageData((2 * len * info_ptr->height), 0)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " no space left for image data.\n");
#endif
    return NULL;
   }

 /* Remove alpha channel from type,  but remember it */
 alpha = colour_type & PNG_COLOR_MASK_ALPHA;
 colour_type &= ~PNG_COLOR_MASK_ALPHA;

 pp = png_image;
 dp = data->image;

 for (ypos = 0; ypos < png_ptr->height; ypos++)
   {
    int xpos;
    int cr, cg, cb, cgr, a;

    for (xpos = 0; xpos < len; ++xpos)
      {
       cr = cg = cb = cgr = a = 0;

       if (colour_type == PNG_COLOR_TYPE_GRAY)
	 {
	  cr = cg = cb = cgr = (*pp++);
	 }
        else
	 if (colour_type == PNG_COLOR_TYPE_RGB)
	   {
            cr = (*pp++);
            cg = (*pp++);
            cb = (*pp++);
            cgr = (0.59 * cr) + (0.20 * cg) + (0.11 * cb);
	   }
	  else
	   {
            PNG_ERROR(png_ptr,
		      "Unknown PNG colour type seen (ReadPNGimage_16)");
	   }

       /* Process alpha channel if present */
       if (alpha) a = (*pp++);

       if (alpha && a != 0xff)
	 {
	  int tr, tg, tb;

	  /* Get tile colour without gamma - from www.c */
	  if (tileData)
	    {
	     int x = xpos % tileWidth;
	     int y = ypos % tileHeight;
	     int i = y * tileWidth + x;
	     unsigned char *s = (tileData + 2 * i);
	     unsigned short val;

	     val  = ((unsigned short)(s[0])     );   /* LSB first! */
	     val |= ((unsigned short)(s[1]) << 8);

	     tr= (((val & visual->red_mask  ) >> RPixelShift) << 8) / (RPixelMask+1);
	     tg= (((val & visual->green_mask) >> GPixelShift) << 8) / (GPixelMask+1);
	     tb= (((val & visual->blue_mask ) >> BPixelShift) << 8) / (BPixelMask+1);
	    }
	   else
	    {
	     tr = (((transparent & visual->red_mask  ) >> RPixelShift) << 8) / (RPixelMask+1);
	     tg = (((transparent & visual->green_mask) >> GPixelShift) << 8) / (GPixelMask+1);
	     tb = (((transparent & visual->blue_mask ) >> BPixelShift) << 8) / (BPixelMask+1);
	    }

	  /* Add picture pixel to background pixel */
	  cr = (a/255.0) * cr + ((255.0-a)/255.0) * tr;
	  cg = (a/255.0) * cg + ((255.0-a)/255.0) * tg;
	  cb = (a/255.0) * cb + ((255.0-a)/255.0) * tb;
	 }

       GetPNGcolour(cr, cg, cb, &ulp);
       *dp++ = (unsigned char)((ulp     ) & 0xFF);   /* LSB first! */
       *dp++ = (unsigned char)((ulp >> 8) & 0xFF);
      } /* xpos */
   } /* ypos */      

 return data;
}


/*
 * Read image into displays, not 1,2,4,16 or 24 bit deep
 */
ImageData* ReadPNGimage(png_struct* png_ptr, png_info* info_ptr,
			png_byte* png_image)
{
 png_byte* pp;
 ImageData* data;
 unsigned char *dp;
 int ypos;
 int len = info_ptr->width;
 int colour_type = info_ptr->color_type;
 int alpha;
#ifdef ARENA_DEBUG
 char Iam[] = "ReadPNGimage";
#endif


 if ((data = NewImageData((len * info_ptr->height), 0)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " no space left for image data.\n");
#endif
    return NULL;
   }

 /* Remove alpha channel from type,  but remember it */
 alpha = colour_type & PNG_COLOR_MASK_ALPHA;
 colour_type &= ~PNG_COLOR_MASK_ALPHA;

 pp = png_image;
 dp = data->image;
 for (ypos = 0; ypos < png_ptr->height; ypos++)
   {
    int col = ypos & 0x0F;
    int xpos;

    for (xpos = 0; xpos < len; ++xpos)
      {
       int row = xpos & 0x0F;
       int cr = 0, cg = 0, cb = 0, cgr = 0, a;
       int gr;
       int isgrey = 0;

       if (colour_type == PNG_COLOR_TYPE_GRAY)
	 {
	  cr = cg = cb = cgr = Voltage2Brightness(*pp++);
	  isgrey = 1;
	 }
        else
	 if (colour_type == PNG_COLOR_TYPE_RGB)
	   {
            int drb, dgb, dbr;

            cr = Voltage2Brightness(*pp++);
            cg = Voltage2Brightness(*pp++);
            cb = Voltage2Brightness(*pp++);
            cgr = (0.59 * cr) + (0.20 * cg) + (0.11 * cb);

            drb = abs(cr - cb);
            dgb = abs(cg - cb);
            dbr = abs(cb - cr);
            if (drb < 20 && dgb < 20 && dbr < 20) isgrey = 1;
	   }
	  else
	   {
            PNG_ERROR(png_ptr, "Unknown PNG colour type seen (ReadPNGimage)");
	   }


       /* Process alpha channel if present */
       if (alpha)
	 {
	  a = (*pp++);

	  if (a)
	    {
             dp = Transparent(dp, xpos, ypos);
	     continue;
            }
	   else
	    if (a != 0xff)
	      {
	       int tcolour, tr,tg,tb;
	       int drb, dgb, dbr;

	       /* Get tile colour without gamma - from www.c */
	       if (tileData)
		 {
		  unsigned int x = xpos % tileWidth;
                  unsigned int y = ypos % tileHeight;
                  unsigned int i = y * tileWidth + x;
                  tcolour = tileData[i];
		 }
	        else
		 tcolour = transparent;

	       tr = papercols[tcolour].red;
	       tg = papercols[tcolour].green;
	       tb = papercols[tcolour].blue;

	       /* Add picture pixel to background pixel */
	       cr = (a/255.0) * cr + ((255.0-a)/255.0) * tr;
	       cg = (a/255.0) * cg + ((255.0-a)/255.0) * tg;
	       cb = (a/255.0) * cb + ((255.0-a)/255.0) * tb;

	       /* Recalculate grayness */
	       cgr = (0.59 * cr) + (0.20 * cg) + (0.11 * cb);
	       isgrey = 0;
	       drb = abs(cr - cb);
	       dgb = abs(cg - cb);
	       dbr = abs(cb - cr);
	       if (drb < 20 && dgb < 20 && dbr < 20) isgrey = 1;
	      }
	 }

       if (imaging == COLOUR232)
	 {
	  if (isgrey)
	    {
	     int gr = cgr & 0xF0;

	     if (cgr - gr > Magic16[(row << 4) + col]) gr += 16;

	     gr = min(gr, 0xF0);

	     *dp++ = greymap[gr >> 4];
	    }
	   else
	    {
             int r = cr & 0xC0;
	     int g = cg & 0xE0;
	     int b = cb & 0xC0;
	     int v = (row << 4) + col;

	     if (cr - r > Magic64[v]) r += 64;
	     if (cg - g > Magic32[v]) g += 32;
	     if (cb - b > Magic64[v]) b += 64;

	     /* clamp error to keep colour in range 0 to 255 */

	     r = min(r, 255) & 0xC0;
	     g = min(g, 255) & 0xE0;
	     b = min(b, 255) & 0xC0;

	     *dp++ = stdcmap[(r >> 6) | (g >> 3) | (b >> 1)];
	    }
	 }
        else
	 if (imaging == MONO)
	   {
	    if (cgr < Magic256[(row << 4) + col])
	      *dp++ = greymap[0];
	     else
	      *dp++ = greymap[15];
	   }
	  else /* Not COLOUR232 or MONO */
           {
	    gr = cgr & 0xF0;
	    if (cgr - gr > Magic16[(row << 4) + col]) gr += 16;

	    gr = min(gr, 0xF0);
	    *dp++ = greymap[gr >> 4];
	   }

      } /* xpos */
   } /* ypos */      

 return data;
}


ImageXyoke* LoadPNGimage(Block* bp, unsigned int depth)
{
 png_structp png_ptr;
 png_infop info_ptr;
 png_bytep row_pointer;
 png_bytep png_image = NULL;
 int number_passes, pass, row, bit_depth, image_width;
 ImageData* theImageData = NULL;
 ImageXyoke* theImageXcouple = NULL;
 int colour_type, orig_colour_type;
 double image_gamma;
#ifdef ARENA_DEBUG
 char Iam[] = "LoadPNGimage";
#endif

 png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
 if (png_ptr == NULL) return NULL;

 info_ptr = png_create_info_struct(png_ptr);
 if (info_ptr == NULL)
   {
    png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    return NULL;
   }
 /* End if */

 if (setjmp(png_ptr->jmpbuf))
   {
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    FreeImageData(theImageData);
    Free(png_image);
    return NULL;
   }
 /* End if */

 /* No need to init PNG I/O - only ever do a read */
 png_set_read_fn(png_ptr, bp, &arena_png_read_data);
 png_read_info(png_ptr, info_ptr);
 bit_depth = info_ptr->bit_depth;
 orig_colour_type = colour_type = info_ptr->color_type;


#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
 if (IMAGE_TRACE)
   Arena_TracePrint(Iam,
		    " load PNG image:"
		    " %ld x %ld, %d-bit%s, %s, %sinterlaced\n",
		    info_ptr->width, info_ptr->height,
		    bit_depth, (bit_depth > 1) ? "s" : "",
		    ((colour_type > 6) ?
		     png_colour_type[1] : png_colour_type[colour_type]),
		    info_ptr->interlace_type ? "" : "non-");
#endif


 /* expand to RGB or 8 bit grayscale (with alpha if present) */
 png_set_expand(png_ptr);

 /* tell libpng to handle the gamma conversion */
 if (info_ptr->valid & PNG_INFO_gAMA)
   image_gamma = info_ptr->gamma;
  else
   image_gamma = (double)0.45; /* FIXME: OR: 1/Gamma ? */

 /* Only gamma correct if >1.1% difference */
 /* if (abs((Gamma * image_gamma) -1) > 0.011) */
 png_set_gamma(png_ptr, Gamma, image_gamma);

 /* tell libpng to convert 16 bits per _channel_ to 8 */
 if (bit_depth == 16) png_set_strip_16(png_ptr); 

 /* read all the image into the rows allocated above */
 /* (Automatically handles interlacing) */
 number_passes = png_set_interlace_handling(png_ptr);

 /* Optional call to update the info_ptr area with the new settings */
 png_read_update_info(png_ptr, info_ptr);
 bit_depth   = info_ptr->bit_depth;
 colour_type = info_ptr->color_type;
 image_width = info_ptr->width * info_ptr->channels;

 png_image = Arena_MAlloc(info_ptr->height * image_width, False);
 if (png_image == NULL)
   PNG_ERROR (png_ptr, "couldn't alloc space for PNG image");


 /* optional call to update palette with transformations,
    except it doesn't do any! */
 png_start_read_image(png_ptr);

 for (pass = 0; pass < number_passes; pass++)
   {
   /*
    if ((depth == 24)||(depth == 16))
      row_pointer = (png_byte*)ximage;
     else --patch dsr 24-bits display 8-Dec-95
    */
    row_pointer = png_image;

    for (row = 0; row < info_ptr->height; row++)
      {
       png_read_row(png_ptr, NULL, row_pointer);
       row_pointer += image_width;
      }
   }

 /* Now convert PNG image to X one */
 switch (depth)
   {
    case 0:
      break;
    case 1:
    case 2:
    case 4:
      theImageData = ReadPNGimage_1_2_4(png_ptr, info_ptr, png_image);
      break;
    case 16:
      theImageData = ReadPNGimage_16(png_ptr, info_ptr, png_image);
      break;
    case 24:
      theImageData = ReadPNGimage_24(png_ptr, info_ptr, png_image);
      break;
    default:
      theImageData = ReadPNGimage(png_ptr, info_ptr, png_image);
      break;
   }
 /* End switch */


 /* Finish */
 png_read_end(png_ptr, info_ptr);
 png_read_destroy(png_ptr, info_ptr, (png_info*)0);

 {
  unsigned int width = info_ptr->width, height = info_ptr->height;

  Free(png_ptr);   /* howcome 23/10/95: applied patch from Dave Beckett */
  Free(info_ptr);  /* howcome 23/10/95: applied patch from Dave Beckett */

  Free(png_image);

  theImageXcouple = processImage_data2image(theImageData,
					    width, height, depth);
  theImageData->image = theImageData->mask = NULL;
  FreeImageData(theImageData);
 }
 return theImageXcouple;
}


Bool InitProgressivePNGimage(Image* image)
{
 png_structp png_ptr; 
 png_infop info_ptr;
 Arena_PNG_struct* aux = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "InitProgressivePNGimage";
#endif


 if (image == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image is NULL.\n");
#endif
    return False;
   }

 png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
				  NULL, NULL, NULL);
 if (png_ptr == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tfailed to create png_ptr for the image "
		       POINTER_FORMAT".\n",
		       image);
#endif
    return False;
   }

 info_ptr = png_create_info_struct(png_ptr);
 if (info_ptr == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tfailed to create info_ptr for the image "
		       POINTER_FORMAT".\n",
		       image);
#endif
    png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    /* Free(png_ptr); */
    return False;
   }
 /* End if */

 if (setjmp(png_ptr->jmpbuf))
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tsetjmp has returned nonzero value"
		       "(image = " POINTER_FORMAT").\n",
		       image);
#endif
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    return False;
   }
 /* End if */

 if ((aux = (Arena_PNG_struct*)Arena_CAlloc(1, sizeof(Arena_PNG_struct),
					    False))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " failed to calloc for the image "POINTER_FORMAT".\n",
		       image);
#endif
    return False;
   }
 /* End if */

 aux->png_ptr      = png_ptr;
 aux->info_ptr     = info_ptr;
 aux->png_image    = NULL;
 aux->image_status = ARENA_PNG_IMAGE_EMPTY;

 image->aux = (void*)aux;

 if (png_get_progressive_ptr(png_ptr))
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tprogressive PNG image "POINTER_FORMAT
		       " has already been initialized?\n",
		       image);
#endif
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tinitializing progressive PNG image "
		       POINTER_FORMAT".\n",
		       image);
#endif    
    png_set_progressive_read_fn(png_ptr, (void*)image,
				arena_png_info_callback,
				arena_png_row_callback,
				arena_png_end_callback);
   }

 return True;
}


ImageXyoke* ProcessProgressivePNGimage(Image* image,
				       Block* bp,
				       unsigned int depth,
				       unsigned long length)
{
 png_structp png_ptr; 
 png_infop info_ptr;
 png_bytep png_image;
 ImageData* theImageData = NULL;
 ImageXyoke* theImageXcouple = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ProcessProgressivePNGimage";
#endif


 if (image == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " the PNG image is NULL.\n");
#endif
    return NULL;
   }
 /* End if */

 if (image->aux == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe PNG image "POINTER_FORMAT
		       " has not been initialized.\n",
		       image);
#endif
    return NULL;
   }
 /* End if */

#ifdef ARENA_DEBUG
 if (IMAGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    "\n\tthe PNG image "POINTER_FORMAT
		    " has aux="POINTER_FORMAT".\n",
		    image, image->aux);
#endif

 png_ptr = ((Arena_PNG_struct*)(image->aux))->png_ptr;

 if (png_ptr == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe PNG image "POINTER_FORMAT
		       " has not been initialized properly.\n",
		       image);
#endif
    return NULL;
   }
 /* End if */

 if (png_get_progressive_ptr(png_ptr) != image)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe PNG image "POINTER_FORMAT
		       " structure is invalid.\n",
		       image);
#endif
    return NULL;
   }
 /* End if */

 info_ptr  = ((Arena_PNG_struct*)(image->aux))->info_ptr;

 if (setjmp(png_ptr->jmpbuf))
  {
#ifdef ARENA_DEBUG
   if (IMAGE_TRACE)
     Arena_TracePrint(Iam,
		      "\n\tsetjmp has returned nonzero value"
		      " (image = " POINTER_FORMAT"),\n"
		      "\tmarking the image as finished"
		      " (to ignore the rest of data).\n",
		      image);
#endif
#if 1
   ((Arena_PNG_struct*)(image->aux))->image_status = ARENA_PNG_IMAGE_FINISHED;
# else
   png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
#  ifdef ARENA_DEBUG
   if (IMAGE_TRACE && VERBOSE_TRACE)
     if (image->aux)
       Arena_TracePrint(Iam,
			"\n\tfreeing aux field of the PNG image "
			POINTER_FORMAT".\n",
			image);
#  endif
   Free(image->aux);
   return NULL;
#endif
  }
 else
  {
   png_bytep data = NULL;

   if ((data = (png_bytep)Arena_CAlloc(length, sizeof(char), False)) == NULL)
     {
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam,
			 "\n\tfailed to allocate memory for the data buffer"
			 " for the PNG image "POINTER_FORMAT".\n",
			 image);
#endif
      return NULL;
     }
   /* End if */

   if (bp->size > bp->next)
     {
      if (bp->next + length > bp->size) length = bp->size - bp->next;

      memcpy(data, bp->buffer + bp->next, length);
      bp->next += length;
     }
   /* End if */

   png_process_data(png_ptr, info_ptr, data, length);

   Free(data);
  }
 /* End if */


 png_image = ((Arena_PNG_struct*)(image->aux))->png_image;

 if ((info_ptr == NULL) || (png_image == NULL))
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe PNG image "POINTER_FORMAT
		       " hasn't got enough data to set info_ptr yet.\n",
		       image);
#endif
    return NULL;
   }
 /* End if */


#ifdef ARENA_DEBUG
 if (image->aux == NULL)
   {
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tError: the PNG image "POINTER_FORMAT
		       " aux field is NULL (has been prematurely freed).\n",
		       image);
    return NULL;
   }
#endif

 switch (((Arena_PNG_struct*)(image->aux))->image_status)
   {
    case ARENA_PNG_IMAGE_EMPTY:
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam,
			 " the PNG image "POINTER_FORMAT" is empty?\n",
			 image);
#endif
      return NULL;
      break;
    case ARENA_PNG_IMAGE_OLD:
      return NULL;
      break;
    case ARENA_PNG_IMAGE_CHANGED:
      ((Arena_PNG_struct*)(image->aux))->image_status = ARENA_PNG_IMAGE_OLD;
    case ARENA_PNG_IMAGE_FINISHED:
      /* Now convert PNG image to X one */
      switch (depth)
	{
	 case 0:
	   break;
	 case 1:
	 case 2:
	 case 4:
	   theImageData = ReadPNGimage_1_2_4(png_ptr, info_ptr, png_image);
	   break;
	 case 16:
	   theImageData = ReadPNGimage_16(png_ptr, info_ptr, png_image);
	   break;
	 case 24:
	   theImageData = ReadPNGimage_24(png_ptr, info_ptr, png_image);
	   break;
	 default:
	   theImageData = ReadPNGimage(png_ptr, info_ptr, png_image);
	   break;
	}
      /* End switch */
      break;
    default:
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam,
			 "\n\tunexpected status of the PNG image "
			 POINTER_FORMAT".\n",
			 image);
#endif
      return NULL;
      break;
   }
 /* End switch */


 if (((Arena_PNG_struct*)(image->aux))->image_status
     == ARENA_PNG_IMAGE_FINISHED)
   {
   /* Finish */

#ifdef ARENA_DEBUG
    if (IMAGE_TRACE  && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " finishing the "POINTER_FORMAT" PNG image.\n", image);
#endif

    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

    Free(png_ptr);   /* howcome 23/10/95: applied patch from Dave Beckett */
    Free(info_ptr);  /* howcome 23/10/95: applied patch from Dave Beckett */

    Free(png_image);

#ifdef ARENA_DEBUG
    if (IMAGE_TRACE  && VERBOSE_TRACE)
      if (image->aux)
	Arena_TracePrint(Iam,
			 "\n\tfreeing aux field of the PNG image "
			 POINTER_FORMAT".\n",
			 image);
#endif
    Free(image->aux);
   }
 /* End if */

 theImageXcouple = processImage_data2image(theImageData,
					   image->width, image->height,
					   depth);
 theImageData->image = theImageData->mask = NULL;
 FreeImageData(theImageData);

 return theImageXcouple;
}
