/* display the scrollbar and handle button events */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/keysym.h>
#include <stdio.h>

#include "arena.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
#endif
#include "defs.h"
#include "display.h"
#include "html.h"
#include "main.h"
#include "scrollbar.h"
#include "scrollbar_i.h"
#include "status.h"
#include "toolbar.h"
#include "x11.h"

#include "HTUtils.h"	/* WWW general purpose macros */
#include "HTList.h"
#include "HTAccess.h"


Window ScrollXBarWin = None;
Window ScrollYBarWin = None;
GC     ScrollBarGC   = NULL;


static int px, py, ox, oy;
static Bool scrolling = False, sliding = False, vertical = False,
              uparrow = False,  downarrow = False,
            leftarrow = False, rightarrow = False;


static XRectangle ScrollXBar = { 0, 0, 0, 0 };
static XRectangle ScrollYBar = { 0, 0, 0, 0 };

static XRectangle ScrollBarSlider = { 20, 20, 60, 60 };


Bool UpdateScrollBarGeometry(void)
{
 extern XRectangle ScrollXBar, ScrollYBar;

 if (UpdateToolBarGeometry())
   {
    XRectangle rect = GetToolBarGeometry();

    ScrollXBar.x = rect.x;
    ScrollYBar.y = rect.y + rect.height;
    ScrollXBar.height = ScrollYBar.width = sbar_width;
    ScrollYBar.x = win_width  - ScrollYBar.width;
    ScrollXBar.y = win_height - ScrollXBar.height;
    ScrollXBar.width  = win_width  - ScrollYBar.width  - ScrollXBar.x;
    ScrollYBar.height = win_height - ScrollXBar.height - ScrollYBar.y;

    return True;
   }
 else
   return False;
}


XRectangle GetScrollXBarGeometry(void)
{
 return ScrollXBar;
}


XRectangle GetScrollYBarGeometry(void)
{
 return ScrollYBar;
}


/*
 * Initialize scrollbars X stuff: create subwindows, graphic context.
 */
Bool ScrollBarInit(Window theWindow)
{
 if (theWindow != None && (ScrollXBarWin == None || ScrollYBarWin == None))
   {
    if (UpdateScrollBarGeometry())
      {
       if (ScrollXBarWin == None)
	 {
	  ScrollXBarWin = CreateSubWindow(display, theWindow,
					  ScrollXBar.x,     ScrollXBar.y,
					  ScrollXBar.width, ScrollXBar.height,
					  0, windowShadow);
	 }

       if (ScrollYBarWin == None)
	 {
	  ScrollYBarWin = CreateSubWindow(display, theWindow,
					  ScrollYBar.x,     ScrollYBar.y,
					  ScrollYBar.width, ScrollYBar.height,
					  0, windowShadow);
	 }
      }

    if (ScrollXBarWin != None && ScrollBarGC == NULL)
      ScrollBarGC = XCreateGC(display, ScrollXBarWin, 0, NULL);
   }

 return (ScrollXBarWin != None && ScrollYBarWin != None && (ScrollBarGC));
}


void SetScrollXBarPosition(int indent, int buffer_width)
{
/* Here w is scrollbar runner area width */
 long w = ScrollXBar.width - 2 * (ScrollXBar.height - 1);


 if (buffer_width > ScrollXBar.width)
   {
    ScrollBarSlider.width = (w * ScrollXBar.width)/buffer_width;
    if (ScrollBarSlider.width < 6) ScrollBarSlider.width = 6;

    /* And now w is pixel width slider can move */
    w -= ScrollBarSlider.width;

    ScrollBarSlider.x = (ScrollXBar.height - 1) +
                        (w * indent)/(buffer_width - ScrollXBar.width);
   }
  else
   {
    ScrollBarSlider.x     = ScrollXBar.height - 1;
    ScrollBarSlider.width = w;
   }
}


void SetScrollYBarPosition(long offset, long buffer_height)
{
/* Here h is scrollbar runner area height */
 long h = ScrollYBar.height - 2 * (ScrollYBar.width - 1);


 if (buffer_height > ScrollYBar.height)
   {
    ScrollBarSlider.height = ((double)h * ScrollYBar.height)/buffer_height;
    if (ScrollBarSlider.height < 6) ScrollBarSlider.height = 6;

    /* And now h is pixel height slider can move */
    h -= ScrollBarSlider.height;

    ScrollBarSlider.y = (ScrollYBar.width - 1) +
                      ((double)h * offset)/(buffer_height - ScrollYBar.height);
   }
  else
   {
    ScrollBarSlider.y      = ScrollYBar.width - 1;
    ScrollBarSlider.height = h;
   }
}


int ScrollXButtonDown(int x, int y)
{
 px = x;
 py = y;
 sliding = False;
 RepeatButtonDown = False;
 scrolling = True;

 if (x < ScrollXBar.height - 1)
   {
    leftarrow = True;
    RepeatButtonDown = True;
    MoveLeftLine(CurrentDoc);
    return SCROLLBAR;
   }

 if (x < ScrollBarSlider.x)
   {
    MoveLeftPage(CurrentDoc);
   }

 if (ScrollBarSlider.x <= x && x < ScrollBarSlider.x + ScrollBarSlider.width)
   {
    sliding = True;
    vertical = False;
    ox = x - ScrollBarSlider.x;
    return SCROLLBAR;
   }

 if (x > ScrollXBar.width - ScrollXBar.height + 1)
   {
    rightarrow = True;
    RepeatButtonDown = True;
    MoveRightLine(CurrentDoc);
    return SCROLLBAR;
   }

 if (x >= ScrollBarSlider.x + ScrollBarSlider.width)
   {
    MoveRightPage(CurrentDoc);
   }

 return VOID;
}


int ScrollYButtonDown(int x, int y)
{
 px = x;
 py = y;
 sliding = False;
 RepeatButtonDown = False;
 scrolling = True;

 if (y < ScrollYBar.width - 1)
   {
    uparrow = True;
    RepeatButtonDown = True;
    MoveUpLine(CurrentDoc);
    return SCROLLBAR;
   }

 if (y < ScrollBarSlider.y)
   {
    MoveUpPage(CurrentDoc);
   }

 if (y >= ScrollBarSlider.y && y < ScrollBarSlider.y + ScrollBarSlider.height)
   {
    sliding = True;
    vertical = True;
    oy = y - ScrollBarSlider.y;
    return SCROLLBAR;
   }

 if (y > ScrollYBar.height - ScrollYBar.width + 1)
   {
    downarrow = True;
    RepeatButtonDown = True;
    MoveDownLine(CurrentDoc);
    return SCROLLBAR;
   }

 if (y >= ScrollBarSlider.y + ScrollBarSlider.height)
   {
    MoveDownPage(CurrentDoc);
   }

 return VOID;
}


void ScrollButtonUp(int x, int y)
{
 px = 0;
 py = 0;

 if (sliding) ScrollButtonDrag(x, y);

 scrolling = sliding = downarrow = uparrow = leftarrow = rightarrow = False;
}


/*
 * Update/move slider to reflect new position.
 */
void MoveXSlider(int indent, int buffer_width)
{
 int sx1;


 sx1 = ScrollBarSlider.x;
 SetScrollXBarPosition(indent, buffer_width);

 if (ScrollBarSlider.x != sx1)
   {
    XSetForeground(display, ScrollBarGC, windowShadow);
    XFillRectangle(display, ScrollXBarWin, ScrollBarGC,
		   sx1, 2, ScrollBarSlider.width, ScrollXBar.height - 4);

    DisplayXSlider();
   }
}


/* update/move slider to reflect new position */
void MoveYSlider(long offset, long buffer_height)
{
 int sy1;


 sy1 = ScrollBarSlider.y;
 SetScrollYBarPosition(offset, buffer_height);

 if (ScrollBarSlider.y != sy1)
   {
    XSetForeground(display, ScrollBarGC, windowShadow);
    XFillRectangle(display, ScrollYBarWin, ScrollBarGC,
		   2, sy1, ScrollYBar.width - 4, ScrollBarSlider.height);

    DisplayYSlider();
   }
}


void ScrollButtonXDrag(int x, int y)
{
 int x1, w1;


 px = x;
 py = y;

 x1 = ScrollXBar.height - 1;

 if (x - ox < x1)
   {
    if (x > x1)
      ox = x - x1;
     else
      {
       ox = 0;
       x = x1;
      }
   }

 x1 = ScrollXBar.width - (ScrollXBar.height - 1);

 if (x - ox + ScrollBarSlider.width > x1)
   {
    if (x < x1)
      ox = x - x1 + ScrollBarSlider.width;
     else
      {
       ox = ScrollBarSlider.width;
       x = x1;
      }
   }

 if (x - ox > ScrollBarSlider.x)
   {
    x1 = ScrollBarSlider.x;
    w1 = x - ox - x1;
   }
  else
   {
    x1 = x - ox + ScrollBarSlider.width;
    w1 = ScrollBarSlider.x + ScrollBarSlider.width - x1;
   }

 if (w1 > 0)
   {
    XSetForeground(display, ScrollBarGC, windowShadow);
    XFillRectangle(display, ScrollXBarWin, ScrollBarGC,
		   x1, 2, w1, ScrollXBar.height - 4);
   }

 if (ScrollBarSlider.x != x - ox)
   {
    ScrollBarSlider.x = x - ox;
    DisplayXSlider();
    SlideXDisplay(CurrentDoc,
		  ScrollBarSlider.x - ScrollXBar.height + 1,
		  ScrollXBar.width - 2 * (ScrollXBar.height - 1));
   }
}


void ScrollButtonYDrag(int x, int y)
{
 int y1, h1;


 px = x;
 py = y;

 y1 = ScrollYBar.width - 1;

 if (y - oy < y1)
   {
    if (y > y1)
      oy = y - y1;
     else
      {
       oy = 0;
       y = y1;
      }
   }

 y1 = ScrollYBar.height - (ScrollYBar.width - 1);

 if (y - oy + ScrollBarSlider.height > y1)
   {
    if (y < y1)
      oy = y - y1 + ScrollBarSlider.height;
     else
      {
       oy = ScrollBarSlider.height;
       y = y1;
      }
   }

 if (y - oy > ScrollBarSlider.y)
    {
     y1 = ScrollBarSlider.y;
     h1 = y - oy - y1;
    }
  else
   {
    y1 = y - oy + ScrollBarSlider.height;
    h1 = ScrollBarSlider.y + ScrollBarSlider.height - y1;
   }

 if (h1 > 0)
   {
    XSetForeground(display, ScrollBarGC, windowShadow);
    XFillRectangle(display, ScrollYBarWin, ScrollBarGC,
		   2, y1, ScrollYBar.width - 4, h1);
   }

 if (ScrollBarSlider.y != y - oy)
   {
    ScrollBarSlider.y = y - oy;
    DisplayYSlider();
    SlideYDisplay(CurrentDoc,
		  ScrollBarSlider.y - ScrollYBar.width + 1,
		  ScrollYBar.height - 2 * (ScrollYBar.width - 1));
   }
}


void ScrollButtonDrag(int x, int y)
{
 if (sliding)
   {
#ifdef ARENA_DEBUG	/* QingLong.23-01-97 */
    if (SCROLL_TRACE)
      Arena_TracePrint("ScrollButtonDrag", " drag %+d%+d.\n", x, y);
#endif

    if (vertical)
      ScrollButtonYDrag(x, y);
     else
      ScrollButtonXDrag(x, y);
   }
}


Bool AtStart(long offset)
{
 long h = (long)buf_height - ScrollYBar.height;

 return (h <= 0 || offset <= 0);
}


Bool AtEnd(long offset)
{
 long h = (long)buf_height - ScrollYBar.height;

 return (h <= 0 || offset >= h);
}


Bool AtLeft(int indent)
{
 int w = (int)buf_width - ScrollXBar.width;

 return (indent <= 0 || w <= 0);
}


Bool AtRight(int indent)
{
 int w = (int)buf_width - ScrollXBar.width;

 return (w <= 0 || indent >= w);
}


void DisplayLeftArrow()
{
 XPoint points[3];


 points[0].x = ScrollXBar.height - 2;
 points[0].y = 2;
 points[1].x = ScrollXBar.height - 2;
 points[1].y = ScrollXBar.height - 3;
 points[2].x = 2;
 points[2].y = ScrollXBar.height/2 + 1;

 XSetForeground(display, ScrollBarGC, windowColour);
 XFillPolygon(display, ScrollXBarWin, ScrollBarGC,
	      points, 3, Convex, CoordModeOrigin);

 XSetForeground(display, ScrollBarGC, windowBottomShadow);
 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x -= 1;
 points[0].y += 1;
 points[1].x -= 1;
 points[1].y -= 1;
 points[2].x += 1;

 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x = ScrollXBar.height - 2;
 points[0].y = 2;
 points[1].x = 2;
 points[1].y = ScrollXBar.height/2;

 XSetForeground(display, ScrollBarGC, windowTopShadow);
 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 2, CoordModeOrigin);

 points[0].x -= 1;
 points[0].y += 1;
 points[1].x += 3;
 points[1].y -= 1;
 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 2, CoordModeOrigin);
}


void DisplayRightArrow()
{
 XPoint points[3];


 points[0].x = ScrollXBar.width - 3;
 points[0].y = ScrollXBar.height/2;
 points[1].x = ScrollXBar.width - (ScrollXBar.height - 1);
 points[1].y = 2;
 points[2].x = ScrollXBar.width - (ScrollXBar.height - 1);
 points[2].y = ScrollXBar.height - 3;

 XSetForeground(display, ScrollBarGC, windowColour);
 XFillPolygon(display, ScrollXBarWin, ScrollBarGC,
	      points, 3, Convex, CoordModeOrigin);

 XSetForeground(display, ScrollBarGC, windowTopShadow);
 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x -= 1;
 points[0].y -= 1;
 points[1].x += 1;
 points[1].y += 1;
 points[2].x += 1;
 points[2].y -= 1;

 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x = ScrollXBar.width - 4;
 points[0].y = ScrollXBar.height/2 + 1;
 points[1].x = ScrollXBar.width - (ScrollXBar.height - 2);
 points[1].y = ScrollXBar.height - 4;

 XSetForeground(display, ScrollBarGC, windowBottomShadow);
 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 2, CoordModeOrigin);

 points[0].x -= 1;
 points[0].y += 1;
 points[1].x += 1;
 points[1].y -= 1;

 XDrawLines(display, ScrollXBarWin, ScrollBarGC, points, 2, CoordModeOrigin);
}


void DisplayDownArrow()
{
 XPoint points[3];


 points[0].x = ScrollYBar.width/2;
 points[0].y = ScrollYBar.height - 3;
 points[1].x = 2;
 points[1].y = ScrollYBar.height - (ScrollYBar.width - 1);
 points[2].x = ScrollYBar.width - 3;
 points[2].y = ScrollYBar.height - (ScrollYBar.width - 1);

 XSetForeground(display, ScrollBarGC, windowColour);
 XFillPolygon(display, ScrollYBarWin, ScrollBarGC,
	      points, 3, Convex, CoordModeOrigin);

 XSetForeground(display, ScrollBarGC, windowTopShadow);
 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x -= 1;
 points[0].y -= 1;
 points[1].x += 1;
 points[1].y += 1;
 points[2].x -= 1;
 points[2].y += 1;

 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x = ScrollYBar.width/2 + 1;
 points[0].y = ScrollYBar.height - 4;
 points[1].x = ScrollYBar.width - 4;
 points[1].y = ScrollYBar.height - (ScrollYBar.width - 2);

 XSetForeground(display, ScrollBarGC, windowBottomShadow);
 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 2, CoordModeOrigin);

 points[0].x += 1;
 points[0].y -= 1;
 points[1].x -= 1;
 points[1].y += 1;

 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 2, CoordModeOrigin);
}


void DisplayUpArrow()
{
 XPoint points[3];


 points[0].x = 2;
 points[0].y = ScrollYBar.width - 2;
 points[1].x = ScrollYBar.width - 3;
 points[1].y = ScrollYBar.width - 2;
 points[2].x = ScrollYBar.width/2 + 1;
 points[2].y = 2;

 XSetForeground(display, ScrollBarGC, windowColour);
 XFillPolygon(display, ScrollYBarWin, ScrollBarGC,
	      points, 3, Convex, CoordModeOrigin);

 XSetForeground(display, ScrollBarGC, windowBottomShadow);
 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x += 1;
 points[0].y -= 1;
 points[1].x -= 1;
 points[1].y -= 1;
 points[2].y += 1;

 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 3, CoordModeOrigin);

 points[0].x = 2;
 points[0].y = ScrollYBar.width - 2;
 points[1].x = ScrollYBar.width/2;
 points[1].y = 2;

 XSetForeground(display, ScrollBarGC, windowTopShadow);
 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 2, CoordModeOrigin);

 points[0].x += 1;
 points[0].y -= 1;
 points[1].x -= 1;
 points[1].y += 3;
 XDrawLines(display, ScrollYBarWin, ScrollBarGC, points, 2, CoordModeOrigin);
}


void DrawSlider(GC theGC, Window theWindow,
		int x, int y, unsigned int w, unsigned int h)
{
 if (sbar_width)
   {
    XSetForeground(display, theGC, windowColour);
    XFillRectangle(display, theWindow, theGC, x + 2, y + 2, w - 4, h - 4);
    DrawOutSet(theWindow, theGC, x, y, w, h);
   }
}

 
void DisplayXSlider(void)
{
 DrawSlider(ScrollBarGC, ScrollXBarWin,
	    ScrollBarSlider.x,     2,
	    ScrollBarSlider.width, ScrollXBar.height - 4);
}


void DisplayYSlider()
{
 DrawSlider(ScrollBarGC, ScrollYBarWin,
	    2,                    ScrollBarSlider.y,
	    ScrollYBar.width - 4, ScrollBarSlider.height);
}


void DisplayScrollBar(void)
{
 if (sbar_width)
   {
   /* Widths between 1 and 4 may overrun the windows by a pixel or two.
    * This probably isn't worth worrying about.
    */

    /* draw horizontal scrollbar background */
    XClearWindow(display, ScrollXBarWin);
    DrawInSet(ScrollXBarWin, ScrollBarGC,
	      0, 0, ScrollXBar.width, ScrollXBar.height);

    /* draw vertical scrollbar background */
    XClearWindow(display, ScrollYBarWin);
    DrawInSet(ScrollYBarWin, ScrollBarGC,
	      0, 0, ScrollYBar.width, ScrollYBar.height);

    /* draw the sliders */
    DisplayXSlider();
    DisplayYSlider();

    /* draw the arrow buttons */
    DisplayUpArrow();
    DisplayDownArrow();
    DisplayLeftArrow();
    DisplayRightArrow();
   }
}
