/* Bluefish HTML Editor
 * init.c --> all configurational options are parsed here
 *
 * Copyright (C) 1998-1999 Olivier Sessink and Chris Mazuc
 * rcfile code Copyright (C) 1999 MJ Ray and Hylke van der Schaaf
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <locale.h>
#include <libintl.h>
#define _(STRING) gettext(STRING)

#include "config.h"
#include "bluefish.h"
#include "stringlist.h"
#include "debug.h"

#define DIR_MODE (S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)	/* same as 0755 */
#define FILE_MODE (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)	/* same as 0644 */

typedef struct
  {
    void *pointer;
    unsigned char type;
    gchar *identifier;
  }
Tconfig_list_item;

/* global variables for init.c */
GList *config_rc;
gint main_window_h, main_window_w;
gchar *cfg_browser_cline;
gchar *cfg_weblint_cline;
gint v_html_tb, v_custom_tb, v_main_tb;
gchar *cfg_editor_font;
gint cfg_editor_tabwidth;
gchar *cfg_tab_pos, *cfg_tab_font;
GList *positionlist;
gchar *cfg_thumbnailstring;
gchar *cfg_thumbnailtype;
gint full_p, full_li, lowercase_tags;

gint chk_dir (gchar * name);
gint chk_file (gchar * name, gchar * content);
gint chk_link (gchar * src_name, gchar * dst_name);


/*
 * Function: chk_dir
 * Arguments:
 *      name - name of directory
 * Return value:
 *      0 - if error
 *      1 - if successful
 * Small description:
 *      check directory and if not exist create it
 */
gint
chk_dir (gchar * name)
{
  struct stat filestat;

  if ((stat (name, &filestat) == -1) && (errno == ENOENT))
    {
      DEBUG_MSG ("chk_dir, %s does not exist, creating\n", name);
      if (mkdir (name, DIR_MODE) != 0)
	{
	  DEBUG_ERR ("chk_dir, unable to create directory %s!\n",
		     name);
	  return 0;
	};
    };
  return 1;
};

/*
 * Function: chk_link
 * Arguments:
 *      src_name - name for tarfget link
 *      dst_name - name original file
 * Return value:
 *      0 - if error
 *      1 - if successful
 * Description:
 *      create symbolic link from dst_name to src_name
 */
gint
chk_link (gchar * src_name, gchar * dst_name)
{
  struct stat filestat;

  if ((stat (src_name, &filestat) == -1) && (errno == ENOENT))
    {
      DEBUG_MSG ("chk_link, %s does not exist, creating\n",
		 src_name);
      if ((stat (dst_name, &filestat) == -1) && (errno == ENOENT))
	{
	  DEBUG_MSG ("chk_link, source file %s does not exist, error returning\n",
		     dst_name);
	  return 1;
	};
      if (symlink (dst_name, src_name) != 0)
	{
	  DEBUG_ERR ("unable to create symlink from %s to %s!\n",
		     dst_name, src_name);
	  return 0;
	};
    };
  return 1;
};


/*
 * Function: chk_file
 * Arguments:
 *      name - file name
 *      content - content of file
 * Return value:
 *      0 - if error
 *      1 - if successful
 * Description:
 *      if file 'name' don`t exists, then create it and write to 'content'
 */
gint
chk_file (gchar * name, gchar * content)
{
  struct stat filestat;
  FILE *file;

  if ((stat (name, &filestat) == -1) && (errno == ENOENT))
    {
      DEBUG_MSG ("chk_file, %s does not exist, creating\n",
		 name);
      file = fopen (name, "w");
      if (file == NULL)
	{
	  DEBUG_ERR ("chk_file, unable to create file %s!",
		     name);
	  return 0;
	}
      else
	{
	  if (fputs (content, file) == EOF)
	    {
	      DEBUG_ERR ("chk_file, unable to wrote to %s!",
			 name);
	      return 0;
	    }
	  if (fclose (file) == EOF)
	    {
	      DEBUG_ERR ("chk_file, unable to close %s!",
			 name);
	      return 0;
	    };
	};
    };
  return 1;
};


/*
 * Function: check_directories
 * Arguments:
 *      none
 * Return value:
 *      always 1
 * Description:
 *      Create necessary directories/links/files
 */
gint
check_directories ()
{
  gchar *name, *name2, *homedir;

  DEBUG_MSG ("check_directories, starting\n");
  name = g_malloc (PATH_MAX + 1);
  name2 = g_malloc (PATH_MAX + 1);
  CHECK_MEMORY (name == NULL ||
		  name2 == NULL, "check_directories:name || name2");
  
  /* another g_get_home_dir () */
  homedir = g_get_home_dir ();
  DEBUG_MSG ("check_directories, homedir=%s\n", homedir);
  chk_dir (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/", PATH_MAX));
  chk_dir (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/projects/", PATH_MAX));
  chk_dir (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/menu", PATH_MAX));
  chk_link (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/php3_functions", PATH_MAX),
	    strncat (strncpy (name2, BF_LIB_DIR, PATH_MAX), "php3_functions", PATH_MAX));
  chk_link (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/ssi_functions", PATH_MAX),
	    strncat (strncpy (name2, BF_LIB_DIR, PATH_MAX), "ssi_functions", PATH_MAX));
  chk_link (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/rxml_functions", PATH_MAX),
	    strncat (strncpy (name2, BF_LIB_DIR, PATH_MAX), "rxml_functions", PATH_MAX));
  chk_file (strncat (strncpy (name, homedir, PATH_MAX), "/.bluefish/rcfile", PATH_MAX),
	    "browser: netscape -remote OpenUrl\\(%s\\)\nfont: fixed\nweblint: weblint\n\nnotebook_tab_position: top\n");
  g_free (name);
  g_free (name2);
/*   g_free (homedir); */
  return 1;
}

/*this should add 1 empty entry to the configuration list */
GList *
make_config_list_item (GList * config_list, void *pointer_to_var, unsigned char type_of_var, gchar * name_of_var)
{
  Tconfig_list_item *config_list_item;
  if (!pointer_to_var)
    {
      DEBUG_MSG ("make_config_list_item, pointer to var = NULL !\n");
      return config_list;
    }
  config_list_item = g_malloc (sizeof (Tconfig_list_item));
  config_list_item->pointer = pointer_to_var;
  config_list_item->type = type_of_var;
  config_list_item->identifier = name_of_var;
  return (GList *) g_list_append (config_list, config_list_item);
}


gint
save_config_file (GList * config_list, gchar * filename)
{
  gchar *tmpstring = NULL, *tmpstring2;
  GList *rclist, *tmplist, *tmplist2;
  Tconfig_list_item *tmpitem;

  DEBUG_MSG ("save_config_file, started\n");

  rclist = NULL;

/* We must first make a list with 1 string per item. */
  tmplist = g_list_first (config_list);
  while (tmplist != NULL)
    {
      DEBUG_MSG ("save_config_file, tmpitem at %p\n", tmplist->data);
      tmpitem = tmplist->data;
      DEBUG_MSG ("save_config_file, datatype %c\n", tmpitem->type);
      switch (tmpitem->type)
	{
	case 'i':
	  DEBUG_MSG ("save_config_file, converting \"%p\"\n", tmpitem);
	  DEBUG_MSG ("save_config_file, converting \"%s %i\"\n", tmpitem->identifier, *(int *) (void *) tmpitem->pointer);

	  tmpstring = g_strdup_printf ("%s %i", tmpitem->identifier, *(int *) (void *) tmpitem->pointer);
	  CHECK_MEMORY (tmpstring == NULL, "save_config_file:tmpstring:1");

	  DEBUG_MSG ("save_config_file, adding %s\n", tmpstring);

	  rclist = g_list_append (rclist, tmpstring);
	  break;
	case 's':
	  DEBUG_MSG ("save_config_file, converting \"%p\"\n", tmpitem);
	  DEBUG_MSG ("save_config_file, converting \"%s %s\"\n", tmpitem->identifier, (gchar *) * (void **) tmpitem->pointer);

	  tmpstring = g_strdup_printf ("%s %s", tmpitem->identifier, (gchar *) * (void **) tmpitem->pointer);
	  CHECK_MEMORY (tmpstring == NULL, "save_config_file:tmpstring:2");

	  DEBUG_MSG ("save_config_file, adding %s\n", tmpstring);

	  rclist = g_list_append (rclist, tmpstring);
	  break;
	case 'l':
	  DEBUG_MSG ("save_config_file, tmpitem(%p), &tmpitem=%p\n", tmpitem, &tmpitem);
	  tmplist2 = g_list_first ((GList *) * (void **) tmpitem->pointer);
	  DEBUG_MSG ("save_config_file, the tmplist2(%p)\n", tmplist2);
	  while (tmplist2 != NULL)
	    {
	      tmpstring2 = (char *) tmplist2->data;
	      DEBUG_MSG ("save_config_file, tmpstring2(%p)=%s\n", tmpstring2, tmpstring2);
	      tmpstring = g_strdup_printf ("%s %s", tmpitem->identifier, tmpstring2);
	      CHECK_MEMORY (tmpstring == NULL, "save_config_file:tmpstring:3");
	      DEBUG_MSG ("save_config_file, tmpstring(%p)=%s\n", tmpstring, tmpstring);

	      rclist = g_list_append (rclist, tmpstring);
	      tmplist2 = g_list_next (tmplist2);
	    }
	  break;
	default:
	  break;
	}
      tmplist = g_list_next (tmplist);

    }

  put_stringlist (filename, rclist);
  free_stringlist (rclist);
  return 1;
}




gint
parse_config_file (GList * config_list, gchar * filename)
{
  gchar *tmpstring = NULL, *tmpstring2;
  GList *rclist, *tmplist;
  gint count1, count2;
  Tconfig_list_item *tmpitem;

  DEBUG_MSG ("parse_config_file, started\n");

  rclist = NULL;
  rclist = get_stringlist (filename, rclist);

  /* empty all variables that have type GList ('l') */
  tmplist = g_list_first (config_list);
  while (tmplist != NULL)
    {
      tmpitem = (Tconfig_list_item *) tmplist->data;
      if (tmpitem->type == 'l')
	{
	  free_stringlist ((GList *) * (void **) tmpitem->pointer);
	  (GList *) * (void **) tmpitem->pointer = NULL;
	}
      tmplist = g_list_next (tmplist);
    }
  DEBUG_MSG ("parse_config_file, all the type 'l' have been emptied\n");
  DEBUG_MSG ("parse_config_file, length rclist=%d\n", g_list_length (rclist));
/* And now for parsing every line in the config file, first check if there is a valid identifier at the start. */
  for (count1 = g_list_length (rclist) - 1; count1 >= 0; count1--)
    {
      DEBUG_MSG ("parse_config_file, loop started\n");
      tmpstring = g_list_nth_data (rclist, count1);

      if (tmpstring != NULL)
	{
	  DEBUG_MSG ("parse_config_file, tmpstring=%s\n", tmpstring);
	  g_strchug (tmpstring);

	  for (count2 = g_list_length (config_list) - 1; count2 >= 0; count2--)
	    {
	      tmpitem = g_list_nth_data (config_list, count2);

	      if (g_strncasecmp (tmpitem->identifier, tmpstring, strlen (tmpitem->identifier)) == 0)
		{
		  /* we have found the correct identifier */
		  DEBUG_MSG ("parse_config_file, identifier=%s, string=%s\n", tmpitem->identifier, tmpstring);
		  /* move pointer past the identifier */
		  tmpstring += strlen (tmpitem->identifier);
		  trunc_on_char (tmpstring, '\n');
		  g_strstrip (tmpstring);

		  switch (tmpitem->type)
		    {
		    case 'i':
		      *(int *) (void *) tmpitem->pointer = atoi (tmpstring);

		      break;
		    case 's':
		      (char *) *(void **) tmpitem->pointer = realloc ((char *) *(void **) tmpitem->pointer, strlen (tmpstring) + 1);
		      strcpy ((char *) *(void **) tmpitem->pointer, tmpstring);
		      break;
		    case 'l':
		      tmpstring2 = g_strdup (tmpstring);
		      CHECK_MEMORY (tmpstring2 == NULL, "parse_config_file:tmpstring2");
		      (GList *) * (void **) tmpitem->pointer = g_list_append ((GList *) * (void **) tmpitem->pointer, tmpstring2);
		      DEBUG_MSG ("parse_config_file, *(void **)tmpitem->pointer=%p\n", *(void **) tmpitem->pointer);
		      break;
		    default:
		      break;
		    }
		  count2 = 0;
		}
	    }
	}
    }
  free_stringlist (rclist);
  return 1;
}

/*
 * Function:  rcfile_init
 * Arguments:   none
 * Return value:
 *      TRUE
 * Description:
 *      Initialise variables that are read from the rcfile
 */
gboolean
rcfile_init ()
{
  DEBUG_MSG ("rcfile_init started\n");
  cfg_browser_cline = g_strdup ("netscape -remote OpenUrl\\(%s\\)");
  cfg_editor_font = g_strdup ("fixed,*");
  cfg_tab_font = g_strdup ("fixed,*");
  cfg_tab_pos = g_strdup ("bottom");
  cfg_weblint_cline = g_strdup ("weblint");
  cfg_thumbnailstring = g_strdup ("_thumbnail");
  cfg_thumbnailtype = g_strdup ("jpg");
  CHECK_MEMORY (cfg_browser_cline == NULL
 	    || cfg_editor_font == NULL
 	    || cfg_tab_font == NULL
 	    || cfg_tab_pos == NULL
 	    || cfg_weblint_cline == NULL
 	    || cfg_thumbnailstring == NULL
 	    || cfg_thumbnailtype == NULL
     , "rcfile_init:cfg_*");
  cfg_editor_tabwidth = 8;
  v_html_tb = 1;
  v_main_tb = 1;
  v_custom_tb = 0;
  lowercase_tags = 0;
  main_window_h = 400;
  main_window_w = 600;
  full_p = 0;
  full_li = 0;
  return TRUE;
};

/*
 * Function: parse_config_file
 * Arguments:
 *      void
 * Return value:
 *      void
 * Description:
 *      Parsing config file, initialize some internal variable
 */
void
parse_config_files (void)
{
  gchar *filename;

  DEBUG_MSG ("parse_config_files, started\n");
  DEBUG_MSG ("parse_config_files, g_get_home_dir=%s\n", g_get_home_dir());
/* 
 * read the data into the three lists: fonts, targets, urls and colors 
 * You need to add 1 entry to initialize the list before calling get_stringlist()
 */

  rcfile_init ();


/*Make the config_rc list ready for filling with data */

  config_rc = NULL;
  config_rc = make_config_list_item (config_rc, &cfg_browser_cline, 's', "browser_command:");
  config_rc = make_config_list_item (config_rc, &cfg_editor_font, 's', "editor_font:");
  config_rc = make_config_list_item (config_rc, &cfg_editor_tabwidth, 'i', "editor_tab_width:");
  config_rc = make_config_list_item (config_rc, &cfg_tab_font, 's', "notebook_tab_font:");
  config_rc = make_config_list_item (config_rc, &cfg_tab_pos, 's', "notebook_tab_position:");
  config_rc = make_config_list_item (config_rc, &cfg_weblint_cline, 's', "weblint_command:");
  config_rc = make_config_list_item (config_rc, &v_html_tb, 'i', "view_html_toolbar:");
  config_rc = make_config_list_item (config_rc, &v_custom_tb, 'i', "view_custom_toolbar:");
  config_rc = make_config_list_item (config_rc, &v_main_tb, 'i', "view_main_toolbar:");
  config_rc = make_config_list_item (config_rc, &main_window_h, 'i', "main_window_height:");
  config_rc = make_config_list_item (config_rc, &main_window_w, 'i', "main_window_width:");
  config_rc = make_config_list_item (config_rc, &cfg_thumbnailstring, 's', "thumbnail_string:");
  config_rc = make_config_list_item (config_rc, &cfg_thumbnailtype, 's', "thumbnail_type:");
  config_rc = make_config_list_item (config_rc, &full_p, 'i', "closing_paragraph_tag:");
  config_rc = make_config_list_item (config_rc, &full_li, 'i', "closing_list_item_tag:");
  config_rc = make_config_list_item (config_rc, &lowercase_tags, 'i', "lowercase_tags:");

/*Fill the config_rc list with data */

  /* g_get_homedir() is broken on some systems, so we use the old method */
  filename = g_strconcat (getenv ("HOME"), "/.bluefish/rcfile", NULL);
  CHECK_MEMORY (filename == NULL, "parse_config_file:filename");
  parse_config_file (config_rc, filename);
  g_free (filename);

/* initialise positionlist */
  positionlist = g_list_append (positionlist, "top");
  positionlist = g_list_append (positionlist, "middle");
  positionlist = g_list_append (positionlist, "bottom");
  positionlist = g_list_append (positionlist, "right");
  positionlist = g_list_append (positionlist, "left");
  positionlist = g_list_append (positionlist, "texttop");
  positionlist = g_list_append (positionlist, "baseline");
  positionlist = g_list_append (positionlist, "absmiddle");
  positionlist = g_list_append (positionlist, "absbottom");
};

/*
 * Function: save_config_files
 * Arguments:
 *      void
 * Return value:
 *      void
 * Desciption:
 *      Save configuration in user config file
 */
void
save_config_files (void)
{
  gchar *filename;

  DEBUG_MSG ("save_config_files, started\n");

  /* g_get_home_dir() is still broken !!!!! don't change this into the glib functions!!! */
  filename = g_strconcat (getenv ("HOME"), "/.bluefish/rcfile", NULL);
  CHECK_MEMORY (filename == NULL, "save_config_file:filename");
  save_config_file (config_rc, filename);
  g_free (filename);
};
