/* 
   WebDAV / HTTP authoring client routines,
   Copyright (C) 1998, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef HTTPDAV_H
#define HTTPDAV_H

/* For fetch_list */
#include "protocol.h"

extern char http_error[];

/* Set by http_options to true if server is a WebDAV class 1 server */
extern bool http_webdav_server;
/* Set to false to bypass http_init OPTIONS check */
extern bool http_init_checks;
/* Set to disable 100-continue support. */
extern bool http_disable_expect;
/* Set to limit the number of requests per connection */
extern bool http_conn_limit;

#define REQSIZ 2048

typedef enum {
    http_body_buffer,
    http_body_file,
    http_body_none
} http_body_t;

/* The transfer encoding types */
enum http_te_t {
    /* TODO: Support other types */
    http_te_none,
    http_te_chunked,
    http_te_unknown
};

/* This is called with each block of the response body, if len > 0.
 * For the last block, it is called with len==0 */
typedef void (*http_request_body_read_t) (
    void *userdata,
    const char *buffer,
    const size_t len );

/* This is called with each of the headers in the response */
typedef void (*http_request_hdrs_read_t) (
    const char *name,
    const char *value );

/* HTTP request information */
typedef struct {
    /* Fill in these with http_req_init */
    const char *method;
    char *uri;
    /* Fill in these yourself */
    char headers[REQSIZ];
    /* Set these if you want to send a request body */
    http_body_t body;
    FILE *body_file;
    const char *body_buffer;
    /* Set this is you want to read the response headers */
    http_request_hdrs_read_t hdrs_callback;
    /* Set this if you want to read the response body */
    http_request_body_read_t body_callback;
    /* Response entity-body transfer-encoding */
    enum http_te_t resp_te;
    /* Response entity-body content-length */
    int resp_length;
    /* Bytes left to read */
    int resp_left;
    int resp_chunk_left;
    /* If non-NULL, this will be set to the total size of the
     * response body */
    size_t *resp_body_size;
    /* If non-NULL, passed as first parameter to body_callback */
    void *body_callback_userdata;
    /* We fill in these for you */
    size_t body_size;
    /* You use this for the status response code */
    int status;
    int class; /* == status/100 */
} http_req_t;

/* Public Interface:
 * Call:
 *    http_init( "/", "www.myhttpserver.com", 80, "myusername", "mypassword" );
 * then any of the method functions, in any order
 *    http_put, http_delete, dav_move, dav_rmdir, dav_mkcol.
 * lastly:
 *    http_finish( );
 */

extern const char *http_quotes;
extern const char *http_whitespace;

int http_init( const char *remote_root,
	       struct proto_host_t *server, struct proto_host_t *proxy );
int http_finish( void );

int http_head( const char *directory );
int http_options( const char *directory );

int http_put( const char *local, const char *remote, const bool ascii );
int http_delete( const char *filename );
int http_get( const char *local, const char *remote, const int remotesize,
	      const bool ascii );

int dav_move( const char *from, const char *to );
int dav_rmdir( const char *dirname );
int dav_mkcol( const char *dirname );

int dav_mkref( const char *refres, const char *target );
int dav_chref( const char *refres, const char *target );
int dav_rmref( const char *refres );

#ifdef HAVE_LIBEXPAT
int dav_fetch( const char *dirname, struct proto_file_t **files );
#endif

/* The request mechanism */
int http_request( http_req_t *req );
void http_request_init( http_req_t *req, const char *method, const char *uri );
void http_request_end( http_req_t *req );

#endif /* HTTPDAV_H */
