/* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: sites.c,v 1.34.2.18 1999/09/07 19:15:42 joe Exp $
*/

/* This is the core functionality of sitecopy, performing updates
 * and checking files etc. */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <dirent.h>
#include <fnmatch.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <time.h>
#include <utime.h>

#ifndef HAVE_SNPRINTF
#include <snprintf.h>
#endif /* !HAVE_SNPRINTF */

#include <basename.h>

#include "common.h"
#include "dirname.h"
#include "frontend.h"
#include "protocol.h"
#include "socket.h"
#include "sites.h"

#include "ftp.h"

/* The protocol drivers */
const struct proto_driver ftp_driver = {
    ftp_init,
    ftp_finish,
    ftp_move,
    ftp_put,
    ftp_get,
    ftp_delete,
    ftp_chmod,
    ftp_mkdir,
    ftp_rmdir,
    NULL, /* create link */
    NULL, /* change link target */
    NULL, /* delete link */
    ftp_fetch,
    "ftp", /* service name */
    21, /* service number */
    "FTP",
    ftp_error
};

#ifdef USE_DAV
#include "httpdav.h"

const struct proto_driver dav_driver = {
    http_init, /* Connect */
    http_finish, /* Disconnect */
    dav_move, /* File move */
    http_put, /* File upload - HTTP PUT */
    http_get, /* File download */
    http_delete, /* File delete */
    NULL, /* File chmod... this could be done using either
	   *   a) A special live property
	   *   b) The ACL extensions: draft-ietf-webdav-acl-??.txt
	   */
    dav_mkcol, /* Dir create */
    dav_rmdir, /* Dir remove, same as file remove */
    dav_mkref, /* Create link */
    dav_chref, /* Change link */
    dav_rmref, /* Remove link */
    /* fetch listing */
#ifdef HAVE_LIBEXPAT
    dav_fetch, 
#else
    NULL, 
#endif
    "http", /* Service name */
    80, /* server number */
    "WebDAV", /* User-visible protocol name */
    http_error
};
#endif

/* This is the maximum number of directories which can be held in the queue 
 * at one time - each entry is only a pointer, so the size is not too
 * much of a problem.  */
#define MAXDIRS 500

/* This is the string used in the storage files to indicate the
 * line is a directory rather than a file */
#define DIRWORD "dir"
/* And this is used for links */
#define LINKWORD "link"

/* Shorthand for protocol driver methods */
#define CALL( a ) (*the_site->driver->a)

/* This holds ALL the sites defined in the rcfile */
struct site_t *all_sites;
 
bool fe_prompting;
bool site_keepgoing;

static int proto_init( struct site_t *the_site );

/* Prototypes */
static void site_checkmoved( struct site_t *any_site );

static void site_destroyfile( struct site_file_t *file );
static int site_synch_create_directories( struct site_t *the_site );
static int site_synch_files( struct site_t *the_site );
static int site_synch_remove_directories( struct site_t *the_site );

static int site_update_create_directories( struct site_t *the_site, bool onlymarked );
static int site_update_delete_directories( struct site_t *the_site, bool onlymarked );
static int site_update_delete_files( struct site_t *the_site, bool onlymarked );
static int site_update_files( struct site_t *the_site, bool onlymarked );
static int site_update_links( struct site_t *the_site, bool onlymarked );

static void site_fetch_walk( struct site_t *the_site,
		      struct proto_file_t *files );

static int site_readlocalfiles( struct site_t * );
static int site_readremotefiles( struct site_t * );

static void site_flatlist_items( FILE *f, struct site_t *the_site, 
			  const enum file_diff diff, const char *name );

/* Assigns the _local and _remote filenames to the site file */
static void site_assignnames( struct site_file_t *the_file, struct site_t *the_site );

/* Functions for manipulating the doubly linked files list */
static void file_delete( struct site_t *site, struct site_file_t *item );
static struct site_file_t *file_prepend( struct site_t *site );
static struct site_file_t *file_append( struct site_t *site );
static struct site_file_t *file_create( void );

/* Whether a file is excluded from the site or not */
static bool file_isexcluded( const char *filename, const char *fullpath, 
		      struct site_t *site );
/* Whether a file is ASCII text or binary */
static bool file_isascii( char *filename, struct site_t *site );

/* Creates and initializes a file.
 * Returns NULL if out-of-memory */
inline struct site_file_t *file_create( void ) {
    struct site_file_t *file;
    file = malloc( sizeof(struct site_file_t) );
    if( file!=NULL ) {
	/* Initialize */
	memset( file, 0, sizeof(struct site_file_t) );
    }
    return file;
}

/* Creates a file prepended on to the beginning of the site files list.
 * Returns NULL if the file could not be malloc'ed. */
struct site_file_t *file_prepend( struct site_t *site ) {
    struct site_file_t *file;
    file = file_create();
    if( file == NULL ) return NULL;
    if( site->files == NULL ) {
	/* Empty list */
	site->files = file;
	site->files_tail = file;
    } else {
	/* Non-empty list... update pointers */
	site->files->prev = file;
	file->next = site->files;
	site->files = file;
    }
    return file;
}

/* Deletes the given file from the given site */
void file_delete( struct site_t *site, struct site_file_t *item ) {
    if( item->prev ) {
	/* Not first in list */
	item->prev->next = item->next;
    } else {
	/* Not last in list */
	site->files = item->next;
    }
    if( item->next ) {
	/* Not last in list */
	item->next->prev = item->prev;
    } else {
	/* Last in list */
	site->files_tail = item->prev;
    }
    /* Safety */
    item->prev = NULL;
    item->next = NULL;
}

/* Creates a file added on to the end of the site files list.
 * Returns NULL if the file could not be malloc'ed. */
struct site_file_t *file_append( struct site_t *site ) {
    struct site_file_t *file;
    file = file_create();
    if( file == NULL ) return NULL;
    if( site->files_tail == NULL ) {
	/* Empty list */
	site->files = file;
	site->files_tail = file;
    } else {
	/* Non-empty list... update pointers */
	site->files_tail->next = file;
	file->prev = site->files_tail;
	site->files_tail = file;
    }
    return file;
}

/* Returns whether the given filename is excluded from the
 * given site
 */
bool file_isexcluded( const char *filename, const char *fullpath, 
		      struct site_t *site ) {
    struct exclude_t *excl;

    DEBUG( DEBUG_FILES, "\nChecking excludes:\n" );
    for( excl=site->excludes; excl != NULL; excl=excl->next ) {
	DEBUG( DEBUG_FILES, "%s ", excl->pattern );
	if( excl->haspath ) {
	    if( fnmatch( excl->pattern, fullpath, FNM_PATHNAME ) == 0 )
		break;
	} else {
	    if( fnmatch( excl->pattern, filename, 0 ) == 0 )
		break;
	}
    }
    if( excl != NULL ) { /* excluded */
	DEBUG( DEBUG_FILES, "- matched\n" );
	return true;
    } else {
	DEBUG( DEBUG_FILES, "... not matched.\n" );
	return false;
    }
}

bool file_isascii( char *filename, struct site_t *site ) {
    int n;

    DEBUG( DEBUG_FILES, "\nChecking ASCII list:\n" );
    for( n=0; n != site->numascii; n++ ) {
	DEBUG( DEBUG_FILES, "%s ", site->ascii[n] );
	if( fnmatch( site->ascii[n], filename, 0 ) == 0 )
	    break;
    }
    if( n < site->numascii ) { 
	DEBUG( DEBUG_FILES, "- matched\n" );
	return true;
    } else {
	DEBUG( DEBUG_FILES, "... not matched.\n" );
	return false;
    }
}


struct site_t *site_find( const char *sitename ) {
    struct site_t *current;

    for( current = all_sites; current!=NULL; current=current->next ) {
	if( strcmp( current->name, sitename ) == 0 ) {
	    /* We found it */
	    return current;
	}
    }

    return NULL;
}

static int site_synch_create_directories( struct site_t *the_site ) {
    struct site_file_t *current;
    int ret;
    
    ret = 0;
    
    for( current=the_site->files; current!=NULL; current=current->next ) {
	if( current->dir && current->diff==file_deleted ) {
	    fe_synching( current );
	    if( mkdir( current->full_local, 0755 ) == 0 ) {
		fe_synched( current, true, NULL );
	    } else {
		ret = 1;
		fe_synched( current, false, strerror(errno) );
	    }
	}
    }
    return ret;
}

static int site_synch_files( struct site_t *the_site ) {
    struct site_file_t *current;
    struct utimbuf times;
    int ret;

    ret = 0;

    for( current = the_site->files; current!=NULL; current=current->next ) {
	if( current->dir ) continue;
	switch( current->diff ) {
	case file_changed:
	case file_deleted:
	    fe_synching( current );
	    if( CALL(file_download)( current->full_local,current->full_remote,
				     current->remotesize, current->isascii )
		!= PROTO_OK ) {
		fe_synched( current, false, the_site->driver->last_error );
		ret = 2;
	    } else {
		/* Change the modtime of the local file so it doesn't look
		 * like it's changed already */
		times.actime = current->remotetime;
		times.modtime = current->remotetime;
		if( utime( current->full_local, &times ) < 0 ) {
		    fe_synched( current, false, strerror(errno) );
		    ret = 2;
		} else {
		    fe_synched( current, true, NULL );
		}
	    }
	    break;
	case file_new:
	    fe_synching( current );
	    if( unlink( current->full_local ) != 0 ) {
		fe_synched( current, false, strerror(errno) );
		ret = 2;
	    } else {
		fe_synched( current, true, NULL );
	    }
	    break;
	case file_moved:
	    fe_synching( current );
	    if( rename( current->full_local, 
			current->old->full_local ) == 0 ) {
		fe_synched( current, true, NULL );
	    } else {
		fe_synched( current, false, strerror(errno) );
		ret = 2;
	    }
	default:
	    break;	    
	}

    }

    return ret;

}

static int site_synch_remove_directories( struct site_t *the_site ) {
    struct site_file_t *current;
    int ret;

    ret = 0;

    for( current=the_site->files_tail; current!=NULL; current=current->prev ) {
	if( current->dir && (current->diff==file_new) ) {
	    fe_synching( current );
	    if( rmdir( current->full_local ) == -1 ) {
		fe_synched( current, false, strerror(errno) );
		ret = 3;
	    } else {
		fe_synched( current, true, NULL );
	    }
	}
    }

    return ret;
}

/* Resyncs the LOCAL site with the REMOTE site.
 * This is site_update backwards, and is essentially the same in structure,
 * except with the logic reversed.
 */
int site_synch( struct site_t *the_site ) {
    int ret;

    ret = proto_init( the_site );
    if( ret != SITE_OK )
	return ret;

    ret = site_synch_create_directories( the_site );
    if( ret == 0 ) {
	ret = site_synch_files( the_site );
	if( ret == 0 ) {
	    ret = site_synch_remove_directories( the_site );
	}
    }
    
    CALL(finish)( );

    if( ret == 0 ) {
	return SITE_OK;
    } else {
	return SITE_ERRORS;
    }

}

static int site_update_create_directories( struct site_t *the_site, bool onlymarked) {
    struct site_file_t *current;
    int ret = 0;

    for( current=the_site->files; current!=NULL; current=current->next ) {
	if( onlymarked && !current->marked ) continue;
	if( current->dir && (current->diff == file_new ) ) {
	    /* New dir! */
	    if( fe_prompting ) if( !fe_can_update( current ) ) continue;
	    fe_updating( current );
	    if( CALL(dir_create)( current->full_remote ) != PROTO_OK ) {
		fe_updated( current, false, the_site->driver->last_error );
		ret = 1;
	    } else {
		fe_updated( current, true, NULL );
		current->updated = true;
	    }
	}
    }

    return ret;
}

static int site_update_delete_files( struct site_t *the_site, bool onlymarked ) {
    struct site_file_t *current;
    int ret = 0;

    for( current=the_site->files; current!=NULL; current=current->next ) {
	if( onlymarked && !current->marked ) continue;
	/* Skip directories and links, and only do deleted files on
	 * this pass */
	if( (current->diff == file_deleted ) &&
	    !( current->dir || current->link ) ) {
	    if( fe_prompting ) if( !fe_can_update( current ) ) break;
	    fe_updating( current );
	    if( CALL(file_delete)( current->full_remote ) != PROTO_OK ) {
		fe_updated( current, false, the_site->driver->last_error  );
		ret = 1;
	    } else {
		/* Successful update - file was deleted */
		fe_updated( current, true, NULL );
		current->updated = true;
	    }
	}
	    
    }
    
    return ret;

}

/* Does everything but file deletes */
static int site_update_files( struct site_t *the_site, bool onlymarked ) {
    struct site_file_t *current;
    char *driver_error = the_site->driver->last_error;
    int ret = 0;

    DEBUG( DEBUG_SITES, "update_files entry.\n" );

    for( current=the_site->files; current!=NULL; current=current->next ) {
	DEBUG( DEBUG_SITES, "loop start.\n" );
	if( current->dir || current->link || 
	    (onlymarked && !current->marked) ) continue;
	switch( current->diff ) {
	case file_new: /* File is new, upload it */
	    DEBUG( DEBUG_SITES, "file_new.\n" );
	    if( fe_prompting ) if( !fe_can_update( current ) ) break;
	    fe_updating( current );
	    if( CALL(file_upload)( current->full_local, current->full_remote, current->isascii ) != PROTO_OK ) {
		DEBUG( DEBUG_SITES, "file_upload failed.\n" );
		fe_updated( current, false, driver_error );
		ret = 1;
	    } else { 
		/* Successful upload... now we need to chmod it? */
		DEBUG( DEBUG_SITES, "file_upload success.\n" );
		if( (the_site->perms == sitep_all) ||
		    ( (current->mode & S_IXUSR) &&
		      (the_site->perms == sitep_exec) ) ) {
		    /* We need to chmod the file ... */
		    if( CALL(file_chmod)( current->full_remote, current->mode ) != PROTO_OK ) {
			fe_updated( current, false, driver_error );
			ret = 1;
		    } else {
			fe_updated( current, true, NULL );
			current->updated = true;
		    }
		} else {
		    fe_updated( current, true, NULL );
		    current->updated = true;
		}
	    }
	    break;
	    
	case file_changed: /* File has changed, upload it */
	    DEBUG( DEBUG_SITES, "file_changed.\n" );
	    if( fe_prompting ) if( !fe_can_update( current ) ) continue;
	    if( the_site->nooverwrite ) {
		/* Must delete remote file before uploading new copy.
		 * FIXME: Icky hack to convince the FE we are about to
		 * delete the file */
		current->diff = file_deleted;
		fe_updating( current );
		if( CALL(file_delete)( current->full_remote ) != PROTO_OK ) {
		    fe_updated( current, false, driver_error );
		    ret = 1;
		    current->diff = file_changed;
		    /* Don't upload it! */
		    break;
		} else {
		    fe_updated( current, true, NULL );
		    current->diff = file_changed;
		}
	    }
	    fe_updating( current );
	    /* Now, upload it */
	    if( CALL(file_upload)( current->full_local, current->full_remote, current->isascii ) != PROTO_OK ) {
		fe_updated( current, false, driver_error );
		ret = 1;
	    } else {
		/* Successful upload... now we need to chmod it? */
		if( (the_site->perms == sitep_all) ||
		    ( (current->mode & S_IXUSR) &&
		      (the_site->perms == sitep_exec) ) ) {
		    /* We need to chmod the file ... */
		    if( CALL(file_chmod)( current->full_remote, current->mode ) != PROTO_OK ) {
			fe_updated( current, false, driver_error );
			ret = 1;
		    } else {
			fe_updated( current, true, NULL );
			current->updated = true;
		    }
		} else {
		    fe_updated( current, true, NULL );
		    current->updated = true;
		}
	    }
	    break;
		
	case file_moved:
	    /* The file has been moved */
	    DEBUG( DEBUG_SITES, "file_moved.\n" );
	    if( fe_prompting ) {
		if( !fe_can_update( current ) ) break;
	    } else {
		fe_updating( current );
	    }
	    if( CALL(file_move)( current->old->full_remote, 
				current->full_remote ) != PROTO_OK ) {
		ret = 1;
		fe_updated( current, false, driver_error );
	    } else {
		/* Successful update - file was moved */
		current->updated = true;
		fe_updated( current, true, NULL );
	    }
	    break;
	    
	default: /* Ignore everything else */
	    DEBUG( DEBUG_SITES, "file ignored.\n" );
	    break;
	}
	DEBUG( DEBUG_SITES, "loop end.\n" );
    }
    DEBUG( DEBUG_SITES, "update_files exit.\n" );

    return ret;
    
}

static int site_update_delete_directories( struct site_t *the_site, bool onlymarked ){
    struct site_file_t *current;
    int ret = 0;

    /* This one must iterate through the list BACKWARDS, so
     * directories are deleted bottom up */
    for( current=the_site->files_tail; current!=NULL; current=current->prev ) {
	if( onlymarked && !current->marked ) continue;
	if( current->dir && (current->diff == file_deleted ) ) {
	    if( fe_prompting ) if( !fe_can_update( current ) ) continue;
	    fe_updating( current );
	    if( CALL(dir_remove)( current->full_remote ) != PROTO_OK ) {
		ret = 1;
		fe_updated( current, false, the_site->driver->last_error );
	    } else {
		/* Successful delete */
		current->updated = true;
		fe_updated( current, true, NULL );
	    }
	}
    }
    return ret;
}

static int site_update_links( struct site_t *the_site, bool onlymarked ) {
    struct site_file_t *current;
    int ret = 0;

    for( current=the_site->files; current!=NULL; current=current->next ) {
	if( onlymarked && !current->marked ) continue;
	if( current->link ) {
	    switch( current->diff ) {
	    case file_new:
		fe_updating( current );
		if( CALL(link_create)( current->full_remote,
				      current->locallink ) != PROTO_OK ) {
		    fe_updated( current, false, the_site->driver->last_error );
		    ret = 1;
		} else {
		    fe_updated( current, true, NULL );
		    current->updated = true;
		}
		break;
	    case file_changed:
		fe_updating( current );
		if( CALL(link_change)( current->full_remote,
				      current->locallink ) != PROTO_OK ) {
		    fe_updated( current, false, the_site->driver->last_error );
		    ret = 1;
		} else {
		    fe_updated( current, true, NULL );
		    current->updated = true;
		}
		break;
	    case file_deleted:
		fe_updating( current );
		if( CALL(link_delete)( current->full_remote ) !=
		    PROTO_OK ) {
		    fe_updated( current, false, the_site->driver->last_error );
		    ret = 1;
		} else {
		    fe_updated( current, true, NULL );
		    current->updated = true;
		}
	    default:
		break;
	    }
	}
    }
    return ret;

}

static int proto_init( struct site_t *the_site ) {
    /* TODO: Move the proto_host's into site_t */
    struct proto_host_t server, proxy, *proxy_pass;
    int ret;

    ftp_use_passive = the_site->ftp_pasv_mode;
#ifdef USE_DAV
    http_disable_expect = the_site->http_no_expect;
    http_conn_limit = the_site->http_limit;
#endif

    server.hostname = the_site->server;
    server.port = the_site->port;
    server.username = the_site->username;
    server.password = the_site->password;

    proxy.hostname = the_site->proxy_server;
    proxy.port = the_site->proxy_port;
    proxy.username = the_site->username;
    proxy.password = the_site->password;

    if( the_site->proxy_server == NULL ) {
	proxy_pass = NULL;
    } else {
	proxy_pass = &proxy;
    }

    ret = CALL(init)( the_site->remote_root, &server, proxy_pass );
    switch( ret ) {
    case PROTO_LOOKUP:
	ret = SITE_LOOKUP;
	break;
    case PROTO_CONNECT:
	ret = SITE_CONNECT;
	break;
    case PROTO_AUTH:
	ret = SITE_AUTH;
	break;
    default:
	ret = SITE_OK;
	break;
    }
    return ret;
}


/* Updates the remote site with the local copy.
 * This is the core functionality of sitecopy.
 * All we do is compare the local files list with the remote files
 * list, and upload any changed or new files.
 * Directories are treated differently - any new ones must be created BEFORE
 * any files are uploaded, and any old ones must be deleted AFTER any files
 * are deleted, since most FTP servers will not allow you to DELE a 
 * non-empty directory.
 */
int site_update( struct site_t *the_site, bool onlymarked )  {
    int ret = 0;
    
    ret = proto_init( the_site );
    if( ret != SITE_OK )
	return ret;

    ret = site_update_create_directories( the_site, onlymarked );
    if( ret == 0 || site_keepgoing ) {
	/* Delete files first, since they might be quota-limited
	 * on the remote site */
	if( !the_site->nodelete ) {
	    ret += site_update_delete_files( the_site, onlymarked );
	}
	if( ret == 0 || site_keepgoing ) {
	    ret += site_update_files( the_site, onlymarked );
	    if( (ret == 0 || site_keepgoing) && 
		(the_site->symlinks == sitesym_maintain ) ) {
		ret += site_update_links( the_site, onlymarked );
	    }
	    if( (ret == 0 || site_keepgoing) && !the_site->nodelete ) {
		ret += site_update_delete_directories( the_site, onlymarked );
	    }
	}
    }

    CALL(finish)( );

    if( ret == 0 ) {
	/* Site updated successfully.
	 * Mark the site as 'unchanged' so, in a GUI FE, they
	 * don't try and update it again. Hopefully.
	 */
	the_site->is_different = false;
	return SITE_OK;
    } else {
	/* Update not totally successfull */
	return SITE_ERRORS;
    }

}

/* This populates the additional names in the site_file_t struct given,
 * using information in the given site */
void site_assignnames( struct site_file_t *the_file, struct site_t *the_site ){

    the_file->full_remote = malloc( strlen(the_site->remote_root) + 
				    strlen(the_file->directory) +
				    strlen(the_file->filename) + 1 );
    the_file->full_local = malloc( strlen(the_site->local_root) + 
				   strlen(the_file->directory) +
				   strlen(the_file->filename) + 1 );
    the_file->rel_local = malloc( strlen(the_file->directory) +
				  strlen(the_file->filename) + 2 );
    the_file->rel_remote = malloc( strlen(the_file->directory) +
				   strlen(the_file->filename) + 2 );
    
    strcpy( the_file->full_remote, the_site->remote_root );
    strcat( the_file->full_remote, the_file->directory );
    strcat( the_file->full_remote, the_file->filename );

    strcpy( the_file->full_local, the_site->local_root );
    strcat( the_file->full_local, the_file->directory );
    strcat( the_file->full_local, the_file->filename );

    strcpy( the_file->rel_local, "/" );
    strcat( the_file->rel_local, the_file->directory );
    strcat( the_file->rel_local, the_file->filename );

    strcpy( the_file->rel_remote, "/" );
    strcat( the_file->rel_remote, the_file->directory );
    strcat( the_file->rel_remote, the_file->filename );

}

/* This reads off the remote files and the local files */
int site_readfiles( struct site_t *the_site ) {
    if( site_readremotefiles( the_site ) < 0 )
	return -1;
    if( site_readlocalfiles( the_site ) )
	return -2;
    /* Check for moved files - compare new files with deleted ones */
    if( the_site->checkmoved ) {
	site_checkmoved( the_site );
    }
    /* Summing up... */
    if( (the_site->numchanged +
	 the_site->nummoved + 
	 the_site->numnew +
	 (the_site->nodelete?0:the_site->numdeleted) ) > 0 ) {
	the_site->is_different = true;
    } else {
	the_site->is_different = false;
    }
    return 0;
}

int site_readremotefiles( struct site_t *the_site ) {
    FILE *fp;
    char buf[BUFSIZ], /* for the line */
	tmp[BUFSIZ], /* for the current field */
	*pos, /* for the current position within buf */
	*point, /* for the curpos within tmp */
	*this_file = NULL;
    struct site_file_t *current;
    size_t dir_length;
    int state;
    /* Get the remote files from the storage file */
    DEBUG( DEBUG_FILES, "Reading info file: %s\n", the_site->infofile );
    fp = fopen( the_site->infofile, "r" );
    if( fp == NULL ) {
	/* This is not an error condition, since the info file WILL NOT 
	 * exist until the site has been created, which is okay, just means
	 * there are 'no' files on the remote site */
	return 1;
    }
    /* The file exists, so read it.
     * Format: one item / line, tab-separated fields.
     * First field is filename of item.
     * Second field is 'dir' for directory items, or mtime for files
     * Third field (files only) is size of file */
    current = NULL;
    while( fgets( buf, BUFSIZ, fp) != NULL ) {
	/* Create a new file item and lob it on the end of the linked 
	 * list */
	current = file_append( the_site );
	/* Make sure we have an end-of-buffer */
	buf[BUFSIZ-1] = '\0';
	this_file = NULL;
	/* Now parse the line. Simple DFA, states are:
	 *  0: Reading filename, field 1
	 *  1: Reading date/time stamp or DIRWORD, field 2
	 *  2: Reading file size, field 3 (if file)
	 *  3: Junk state - consume-all-crud-till-end-of-line.
	 * 
	 * We read the current field into tmp char by char.
	 * point is the current point within tmp. */
	state = 0;
	point = tmp;
	for( pos = buf; *pos!='\0'; pos++ ) {
	    if( *pos < 0 ) state = 5;
	    switch( state ) {
	    case 0:
		if( *pos == '\t' ) {
		    /* End of the filename */
		    *point = '\0';
		    this_file = strdup( tmp );
		    point = tmp;
		    state = 1;
		} else {
		    /* Still in the filename */
		    *(point++) = *pos;
		}
		break;
	    case 1:
		if( *pos == '\t' || *pos == '\n' ) {
		    /* End of the second field */
		    *point = '\0';
		    if( strlen( tmp ) > 0 ) {
			if( strcmp( tmp, DIRWORD ) == 0 ) {
			    /* It's a directory! */
			    current->dir = true;
			    state = 3; /* that's all we need */
			} else if( strcmp( tmp, LINKWORD ) == 0 ) {
			    current->link = true;
			    point = tmp;
			    state = 4; /* read the link target */
			} else {
			    /* It's a file! - field 2 is the mtime */
			    current->dir = false;
			    current->remotetime = atol( tmp );
			    point = tmp;
			    state = 2;
			}
		    } else {
			/* Corrupt line, we need at least two fields */
			/* FIXME: Report this to the user. */
			state = 5;
		    }			
		} else {
		    /* Within the second field */
		    *(point++) = *pos;
		}
		break;
	    case 2:
		if( *pos == '\n' ) {
		    /* End of the size field */
		    *point = '\0';
		    current->remotesize = atol( tmp );
		    state = 3;
		} else {
		    /* Within the file size field */
		    *(point++) = *pos;
		}
		break;
	    case 3: /* junk state */
		break;
	    case 4: /* read the link name */
		if( *pos == '\n' ) {
		    /* End of the field */
		    *point = '\0';
		    current->remotelink = strdup( tmp );
		    state = 3;
		} else {
		    *(point++) = *pos;
		}
	    case 5: /* error state */
		break;
	    }
	}
	if( this_file==NULL || state == 5 ) {
	    /* FIXME: Report this to the user */
	    DEBUG( DEBUG_FILES, "Corrupt line.\n" );
	    file_delete( the_site, current );
	    continue;
	}
	current->diff = file_deleted;
	current->filename = strdup( base_name( this_file ) );
	/* we are going to chop the leading / of this_file */
	if( strlen(current->filename) > strlen(this_file) ) {
	    /* FIXME: Handle these errors properly */
	    DEBUG( DEBUG_FILES, "Very corrupt line.\n" );
	    free( current->filename );
	    file_delete( the_site, current );
	    continue;
	}
	dir_length = strlen(this_file) - strlen(current->filename ) - 1;
	current->directory = malloc( dir_length + 1 );
	strncpy( current->directory, this_file + 1, dir_length );
	*(current->directory+dir_length) = '\0'; /* null-term */
	DEBUG( DEBUG_FILES, "Split %s is: Dir [%s], name [%s]\n",
	       this_file, current->directory, current->filename );
	the_site->numdeleted++;
	site_assignnames( current, the_site );
	DEBUG( DEBUG_FILES, "Remote: [%s] - file: size: %ld  time: %ld\n",
	       current->filename, current->remotesize, current->remotetime );
	DEBUG( DEBUG_FILES, 
	       "Directory: [%s]  Filename: %s\n"
	       "rel_local: %s  rel_remote: %s\n"
	       "full_local: %s  full_remote: %s\n",
	       current->directory, current->filename,
	       current->rel_local, current->rel_remote, 
	       current->full_local, current->full_remote );
	free( this_file );
    }
    fclose( fp );
    return 0;
}

/* Read the local site files... 
 * A stack is used for directories within the site - this is not recursive.
 * Each item on the stack is a FULL PATH to the directory, i.e., including
 * the local site root. */

int site_readlocalfiles( struct site_t *the_site ) {
    char *dirstack[MAXDIRS], *this;
    char *full = NULL, *fname, *temp;
    struct stat item;
    DIR *curdir;
    struct dirent *ent;
    int dirtop = 0;
    size_t dir_length;
    struct site_file_t *current;

    /* Push the root directory on to the stack */
    dirstack[dirtop++] = strdup( the_site->local_root );
    
    /* Now, for all items in the stack, process all the files, and
     * add the dirs to the stack. Everything we put on the stack is
     * temporary and gets freed eventually. */

    while( dirtop > 0 ) {
	/* Pop the stack */
	this = dirstack[--dirtop];
	
	DEBUG( DEBUG_FILES, "Dir: %s\n", this );
	curdir = opendir( this );
	if( curdir == NULL ) {
	    free( this );
	    continue;
	}

	/* Now read all the directory entries */

	while( (ent = readdir( curdir )) != NULL ) {
	    if( strcmp( ent->d_name, "." )==0 ||
		strcmp( ent->d_name, ".." )==0 ) {
		continue;
	    }

	    if( full != NULL ) 
		free( full );
	    full = malloc( strlen( this ) + strlen( ent->d_name ) + 1 );
	    strcpy( full, this );
	    strcat( full, ent->d_name );
	    DEBUG( DEBUG_FILES, "Item: %s - ", ent->d_name );
#ifndef __EMX__
 	    if( lstat( full, &item ) == -1 ) {
#else
	    /* There are no symlinks under OS/2, use stat() instead */
	    if( stat( full, &item ) == -1 ) {
#endif
		/* FIXME: FE warning here */
		continue;
	    }
#ifndef __EMX__
	    /* Is this a symlink? */
	    if( S_ISLNK( item.st_mode ) ) {
		DEBUG( DEBUG_FILES, "symlink - " );
		if( the_site->symlinks == sitesym_ignore ) {
		    /* Just skip it */
		    DEBUG( DEBUG_FILES, "ignoring.\n" );
		    continue;
		} else if( the_site->symlinks == sitesym_follow ) {
		    DEBUG( DEBUG_FILES, "followed - " );
		    /* Else, carry on as normal, stat the real file */
		    if( stat( full, &item ) == -1 ) {
			/* It's probably a broken link */
			DEBUG( DEBUG_FILES, "broken.\n" );
			continue;
		    }
		} else {
		    DEBUG( DEBUG_FILES, "maintained:\n" );
		}
	    }
#endif /* __EMX__ */
	    /* Now process it */

	    /* This is the rel_local of this file - i.e., everything
	     * apart from the local root */
	    fname = (char *)full+strlen(the_site->local_root)-1;

	    /* Check for excludes */
	    if( file_isexcluded( ent->d_name, fname, the_site ) )
		continue;

   
	    for( current=the_site->files; current!=NULL; current=current->next ) {
		if( strcmp( current->rel_local, fname ) == 0 )
		    /* Match found! */
		    break;
/*		last = current; */
	    }
	    
	    if( S_ISREG( item.st_mode ) ) {
		DEBUG( DEBUG_FILES, "file - " );

		if( current == NULL ) {
		    /* This is a new local file... */

		    DEBUG( DEBUG_FILES, "new - " );
		    the_site->numnew++;
		    /* Add a new file to the beginning of the list */
		    current = file_prepend( the_site );
		    current->filename = strdup( base_name( fname ) );
		    /* This is the length of the directory name...
		     * fname = "/full/file/name.ext", filename = "name.ext"
		     * we want directory = "full/file/"... do the maths */
		    dir_length = strlen(fname) - strlen(current->filename)-1;
		    DEBUG( DEBUG_FILES, "dir_length = %ld\n", (long) dir_length );
		    current->directory = malloc( dir_length + 1 );
		    strncpy( current->directory, fname+1, dir_length );
		    *(current->directory+dir_length) = '\0'; /* nullterm */
		    DEBUG( DEBUG_FILES, "directory = %s\n", current->directory );
		    site_assignnames( current, the_site );
		    current->localtime = item.st_mtime;
		    current->localsize = item.st_size;
		    current->diff = file_new;
		    current->mode = item.st_mode;
		    current->dir = false;
		    the_site->totalnew += item.st_size;
/*		    DEBUG( DEBUG_FILES, "new; size: %ld, time: %ld\n", item.st_size, item.st_mtime );  */
		} else {
		    /* Match found... compare it */
		    current->localtime = item.st_mtime;
		    current->localsize = item.st_size;
		    the_site->numdeleted--;
		    if( (current->localtime > current->remotetime) ||
			(current->localsize != current->remotesize) ) {
			/* Changed file */
			current->diff = file_changed;
			current->mode = item.st_mode;
			the_site->numchanged++;
			the_site->totalchanged += item.st_size;
			DEBUG( DEBUG_FILES, "changed (%ld -> %ld)\n", current->localtime, current->remotetime );
		    } else {
			current->diff = file_unchanged;
			current->updated = true;
			the_site->numunchanged++;
			DEBUG( DEBUG_FILES, "unchanged.\n" );
		    }
		    /* The names will already have been assigned */
		}
				    
		/* Assign the ASCII type value */
		current->isascii = file_isascii( fname, the_site );

	    } else if( S_ISDIR( item.st_mode ) ) {
		/* THIS IS A DIRECTORY! */
		DEBUG( DEBUG_FILES, "directory - " );
		if( dirtop < MAXDIRS ) {
		    temp = malloc( strlen( full ) + 2 );
		    strcpy( temp, full );
		    strcat( temp, "/" );
		    dirstack[dirtop++] = temp;
		} else {
		    /* No more room in stack
		     * FIXME: Report this to the user.*/
		}
		if( current == NULL ) {
		    /* New Directory */
		    the_site->numnew++;
		    current = file_append( the_site );
		    current->filename = strdup( base_name( fname ) );
		    /* Work out the directory name */
		    dir_length = strlen(fname) - strlen(current->filename )-1;
		    current->directory = malloc( dir_length + 1 );
		    strncpy( current->directory, fname+1, dir_length );
		    *(current->directory+dir_length) = '\0'; /* null-term */
		    site_assignnames( current, the_site );
		    current->dir = true;
		    current->diff = file_new;
		    current->mode = item.st_mode;
		    current->localsize = item.st_size;
		    DEBUG( DEBUG_FILES, "new.\n" ); 
		} else {
		    /* It's an existing directory */
		    the_site->numdeleted--;
		    the_site->numunchanged++;
		    current->diff = file_unchanged;
		    current->updated = true;
		    DEBUG( DEBUG_FILES, "existing.\n" ); 
		}
	    } 
#ifndef __EMX__
	    else if( S_ISLNK( item.st_mode ) ) {
		char tmp[BUFSIZ];
		memset( tmp, 0, BUFSIZ );
		DEBUG( DEBUG_FILES, "symlink being maintained.\n" );
		if( readlink( full, tmp, BUFSIZ ) == -1 ) {
		    DEBUG( DEBUG_FILES, "readlink failed: %s\n", 
			   strerror(errno) );
		    continue;
		}
		DEBUG( DEBUG_FILES, "Link target: %s\n", tmp );
		if( current == NULL ) {
		    /* New link */
		    the_site->numnew++;
		    current = file_append( the_site );
		    current->filename = strdup( base_name( fname ) );
		    dir_length = strlen(fname) - strlen(current->filename)-1;
		    DEBUG( DEBUG_FILES, "dir_length = %ld\n", (long) dir_length );
		    current->directory = malloc( dir_length + 1 );
		    strncpy( current->directory, fname+1, dir_length );
		    *(current->directory+dir_length) = '\0'; /* nullterm */
		    DEBUG( DEBUG_FILES, "directory = %s\n", current->directory );
		    current->locallink = strdup( tmp );
		    current->diff = file_new;
		    current->link = true;
		    site_assignnames( current, the_site );
		} else {
		    /* Existing link... compare */
		    the_site->numdeleted--;
		    current->locallink = strdup( tmp );
		    if( strcmp( current->remotelink, tmp ) == 0 ) {
			/* Links match */
			current->updated = true;
			current->diff = file_unchanged;
			the_site->numunchanged++;
		    } else {
			current->diff = file_changed;
			the_site->numchanged++;
		    }
		}
	    }
#endif /* __EMX__ */
	    else {
		DEBUG( DEBUG_FILES, "something else.\n" );
	    }
	}
	/* Close the open directory */
	closedir( curdir );
	/* And we're finished with this */
	free( this );
    }
    return 0;
}

static void site_checkmoved( struct site_t *the_site ) {
    /* Have any files been moved?
     * We look for new files, then see if we can find a deleted file
     * which matches filename, mod time, and size exactly.
     * 
     * This is simplistic, and it is possible to get it wrong in some
     * (unusual) circumstances:
     *    Old: /bar/file1.zip, /foo/file1.zip 
     *    New: /new/file1.zip.
     * If all the file1.zip's have the same size and mod time, it
     * IS NOT POSSIBLE to know which one has become /new/file1.zip without 
     * storing remembering the contents of /bar/file1.zip somehow and 
     * comparing, or using a filesystem which remembers stuff. In any case
     * it's pretty silly to try - so the whole shabang is optional.
     *
     * It would probably be better to search for DELETED files, then 
     * check them against NEW files, since files are more often added
     * to sites then they are deleted. But hey, this is C, who'll notice?
     */
    struct site_file_t *search, *current;
    /*  char *bname, *tmp; */
    /* First off, find the new files... */
    DEBUG( DEBUG_FILES, "Checking for moved files...\n" );
    for( search=the_site->files; search!=NULL; search=search->next ) {
	if( search->diff != file_new )
	    continue;
	/* OK, we have us a new file... now iterate through the deleted
	 * files and look for a match */
	DEBUG( DEBUG_FILES, "- New file: %s\nComparing:", search->rel_local );
	for( current=the_site->files; current!=NULL ; current=current->next ) {
	    if( (current->diff != file_deleted) || current->dir ) {
		/* Ignore files which aren't deleted, and dirs, completely.
		 * (search->diff==file_new, so we DON'T need to check
		 * that current != search as well) */
		continue;
	    }
	    DEBUG( DEBUG_FILES, " %s", current->rel_local );
	    if( (strcmp( search->filename, current->filename ) == 0) &&
		current->remotesize == search->localsize &&
		current->remotetime == search->localtime ) {
		DEBUG( DEBUG_FILES, " - matched!\n" );
		break;
	    }
	}
	if( current != NULL ) {
	    /* OK, current == the deleted file, search == the new file.
	     * First make sure search knows where it WAS before the move... */
	    search->old = current;
	    /* Remove the deleted file from the list so it doesn't
	     * actually get deleted by site_update */
	    file_delete( the_site, current );
	    /* Diddle the totals a bit */
	    search->diff = file_moved;
	    the_site->numdeleted--;
	    the_site->numnew--;
	    the_site->nummoved++;
	} else {
	    DEBUG( DEBUG_FILES, " - No match found.\n" );
	}
    }
    DEBUG( DEBUG_FILES, "Finished checking for moved files.\n" );
}

/* This writes the remote files list back to the file
 * and has some hairy logic in the middle, but is otherwise
 * pretty simple.
 * TODO: This is crap. What we should really do is only 
 * write out the remote file info, regardless... that is what the 
 * info file is for. This would GREATLY simplify this entire function.
 * Would mean simply that in site_update, copying over the
 * modtimes + sizes.
 */

#define WRITE_REMOTE_FILE( f ) \
fprintf( fp, "%s\t%ld\t%ld\n", f->rel_remote, f->remotetime, f->remotesize );
#define WRITE_LOCAL_FILE( f ) \
fprintf( fp, "%s\t%ld\t%ld\n", f->rel_local, f->localtime, f->localsize );
#define WRITE_REMOTE_LINK( f ) \
fprintf( fp, "%s\t%s\t%s\n", f->rel_remote, LINKWORD, f->remotelink );
#define WRITE_LOCAL_LINK( f ) \
fprintf( fp, "%s\t%s\t%s\n", f->rel_local, LINKWORD, f->locallink );

int site_writefiles( struct site_t *the_site ) {
    FILE *fp;
    struct site_file_t *current;
    fp = fopen( the_site->infofile, "w" );
    if( fp == NULL ) {
	return -1; 
    }
    for( current = the_site->files; current!=NULL; current=current->next ) {
	if( current->dir ) {
	    if( current->updated ) {
		/* DIR: Updated */
		switch( current->diff ) {
		case file_new:
		case file_unchanged:
		    /* It's new, or it's already there... */
		    fprintf( fp, "%s\t%s\n", current->rel_local, DIRWORD );
		    break;
		case file_moved: /* invalid */
		case file_changed: /* invalid */
		case file_deleted:/* it's not there now */
		default:
		    break;
		} 
	    } else {
		/* DIR: Not Updated */
		switch( current->diff ) {
		case file_unchanged: /* ... it was there anyway */
		case file_deleted: /* ... but it wasn't deleted */
		    /* So the directory exists remotely */
		    fprintf( fp, "%s\t%s\n", current->rel_remote, DIRWORD );
		    break;
		case file_new: /* ... but it wasn't created */
		default:
		    /* So the directory does not exist remotely */
		    break;
		}
	    }
	} else if( current->link ) {
	    if( current->updated ) {
		/* LINK: Updated */
		switch( current->diff ) {
		case file_new: /* it's there now */
		case file_unchanged: /* it's still there now */
		case file_changed: /* it's there now but differently */
		    WRITE_LOCAL_LINK( current );
		    break;
		case file_deleted: /* it's not there now */
		default: 
		    break;
		}
	    } else {
		/* LINK: Not Updated */
		switch( current->diff ) {
		case file_unchanged: /* it's still there anyway  */
		case file_changed:   /* it's still there anyway */
		case file_deleted: /* but actually still there */
		    WRITE_REMOTE_LINK( current );
		    break;
		case file_new: /* it's not there */
		default:
		    break;
		}
	    }
	} else {
	    if( current->updated ) {
		/* FILE: Updated */
		switch( current->diff ) {
		case file_unchanged:
		    /* Write state of remote file */
		    WRITE_REMOTE_FILE( current );
		    break;
		case file_moved:
		case file_changed:
		case file_new:
		    /* Write state of local file */
		    WRITE_LOCAL_FILE( current );
		    break;
		case file_deleted:
		    /* So don't write anything out */
		    break;
		}
	    } else {
		/* FILE: Not Updated */
		switch( current->diff ) {
		case file_changed: /* ... but it hasn't been updated */
		case file_unchanged: 
		case file_deleted: /* ... but it hasn't been deleted */
		    /* Write out the remote time */
		    WRITE_REMOTE_FILE( current );
		    break;
		case file_moved: 
		    /* But it hasn't actually been moved, so write out
		     * the 'deleted' file (where the moved file WAS), and 
		     * ignore the 'new' file (where it was moved TO) */
		    WRITE_REMOTE_FILE( current->old );
		    break;
		case file_new: /* So ignore it... */
		    break;
		}
	    }
	}
    }
    fclose( fp );
    return 0;
}

#undef WRITE_REMOTE_FILE
#undef WRITE_LOCAL_FILE

/* Simply marks all the files in the given site as 'updated'.
 * For 'dynamic updating', this will do, by diff==
 *   file_deleted: Remove the file from the list
 *   file_changed, file_new: Set diff=file_unchanged
 *   file_moved: Remove the ->old, set diff=file_unchanged
 *   file_unchanged: Do nothing.
 */
void site_catchup( struct site_t *the_site ) {
    struct site_file_t *current;
    for( current=the_site->files; current!=NULL; current=current->next ) {
	current->updated = true;
    }
    /* Mark it as mirrored */
    the_site->is_different = false;
}

/* Reinitializes the site - clears any remote files
 * from the list, and marks all other files as 'new locally'.
 */
void site_initialize( struct site_t *the_site ) {
    struct site_file_t *current, *next;
    
    current = the_site->files;

    while( current!= NULL ) {
	next = current->next;
	switch( current->diff ) {
	case file_deleted:
	    file_delete( the_site, current );
	    the_site->numdeleted--;
	    break;
	case file_moved:
	    /* Current is the NEW file, current->old is the DELETED
	     * file. So, simply nuke current->old. */
	    current->old = NULL;
	    current->updated = false;
	    the_site->nummoved--;
	    the_site->numnew++;
	    current->diff = file_new;
	    break;
	case file_changed:
	    current->updated = false;
	    the_site->numchanged--;
	    the_site->numnew++;
	    current->diff = file_new;
	    break;
	case file_unchanged:
	    current->updated = false;
	    the_site->numunchanged--;
	    the_site->numnew++;
	    current->diff = file_new;
	    break;
	case file_new:
	    current->updated = false;
	    break;
	}
	current = next;
    }
    
    if( the_site->files == NULL ) {
	/* There are no files remotely OR locally, so the
	 * sites are the same */
	the_site->is_different = false;
    } else {
	/* Otherwise, we KNOW that there are no files remotely
	 * since we just removed them from memory, so any 
	 * remaining files must be local ones. Hence, the sites
	 * ARE different */
	the_site->is_different = true;
    }

}

/* This walks the files list as returned by the protocol driver,
 * and converts it into a 'real' files list. */
void site_fetch_walk( struct site_t *the_site,
		      struct proto_file_t *files ) {
    struct proto_file_t *this_file, *next_file;
    struct site_file_t *file;
    char rel_local[BUFSIZ]; /* temporary */

    site_destroy( the_site );

    this_file = files;
    while( this_file != NULL ) {
	/* Check whether the filename is excluded.  Mock up a rel_local
	 * for it. Bit noddy */
	/* rel_local starts with a / */
	rel_local[0] = '/'; rel_local[1] = '\0';
	/* FIXME: Buffer ovverrun here. */
	strcat( rel_local, this_file->directory );
	strcat( rel_local, this_file->filename );
	
	if( !file_isexcluded( this_file->filename, rel_local, the_site ) ) {
	    /* Add the file to the list */
	    if( this_file->isdir ) {
		file = file_append( the_site );
	    } else {
		file = file_prepend( the_site );
	    }
	    
	    /* Pointer copy the filename and directory strings...  nothing
	     * to free out of this_file now. */
	    file->filename = this_file->filename;
	    file->directory = this_file->directory;
	    file->dir = this_file->isdir;
	    /* Sort out the names */
	    site_assignnames( file, the_site );
	    
	    file->diff = file_deleted;
	    file->updated = false;
	    /* ... so, it exists remotely only and hasn't been deleted.
	     * bit of a hack, but okay for the moment... */
	    
	    if( ! this_file->isdir ) {
		file->remotesize = this_file->size;
		file->remotetime = this_file->modtime;
	    }
	    
	    fe_fetch_found( file );
	    
	}

	next_file = this_file->next;
	free( this_file );
	this_file = next_file;
    }

}

/* Updates the remote file list... site_fetch_callback is called for
 * every remote file found.
 */
int site_fetch( struct site_t *the_site ) {
    struct proto_file_t *files = NULL;
    int ret;

    if( CALL(fetch_list) == NULL ) {
	return SITE_UNIMPL;
    }

    ret = proto_init( the_site );
    if( ret != SITE_OK )
	return ret;

    ret = CALL(fetch_list)( the_site->remote_root, &files );

    CALL(finish)( );
    
    if( ret == PROTO_OK ) {
	/* Copy over the list of files */
	site_fetch_walk( the_site, files );
	return SITE_OK;
    } else {
	return SITE_FAILED;
    }

}

/* Destroys a file */
void site_destroyfile( struct site_file_t *file ) {
    
    free( file->directory );
    free( file->filename );

    free( file->full_local );
    free( file->full_remote );

    free( file->rel_local );
    free( file->rel_remote );

    free( file );

}

/* Called to delete all the files associated with the site */
void site_destroy( struct site_t *the_site ) {
    struct site_file_t *current, *next;

    current = the_site->files;

    while( current != NULL ) {
	next = current->next;
	if( current->old != NULL ) {
	    site_destroyfile( current->old );
	}
	site_destroyfile( current );
	current = next;
    }

    the_site->files = NULL;
    the_site->files_tail = NULL;

}


/* Produces a section of the flat listing output, of all the items
 * with the given diff type in the given site, using the given section
 * name. */
void site_flatlist_items( FILE *f, struct site_t *the_site, 
			  const enum file_diff diff, const char *name ) {
    struct site_file_t *current;
    fprintf( f, "sectstart|%s", name );
    putc( '\n', f );
    for( current = the_site->files; current!=NULL; current=current->next) {
	if( current->diff == diff ) {
	    fprintf( f, "item|%s%s", current->rel_remote,
		    current->dir?"/":"" );
	    if( current->old !=NULL ) {
		fprintf( f, "|%s\n", current->old->rel_remote );
	    } else {
		putc( '\n', f );
	    }	    
	}
    }
    fprintf( f, "sectend|%s\n", name );
}

/* Produce the flat listing output for the given site */
void site_flatlist( FILE *f, struct site_t *the_site ) {
    fprintf( f, "sitestart|%s", the_site->name );
    if( the_site->url )	fprintf( f, "|%s", the_site->url );
    putc( '\n', f );
    if( the_site->numnew > 0 )
	site_flatlist_items( f, the_site, file_new, "added" );
    if( the_site->numchanged > 0 )
	site_flatlist_items( f, the_site, file_changed, "changed" );
    if( the_site->numdeleted > 0 )
	site_flatlist_items( f, the_site, file_deleted, "deleted" );
    if( the_site->nummoved > 0 )
	site_flatlist_items( f, the_site, file_moved, "moved" );
    fprintf( f, "siteend|%s\n", the_site->is_different?"changed":"unchanged" );
}

const char *site_pseudourl( struct site_t *the_site ) {
    static char urlbuf[256];
    snprintf( urlbuf, 256, 
	      "%s://%s%s",
	      the_site->driver->service_name, the_site->server, 
	      the_site->remote_root_user +
	      ( the_site->remote_isrel ? 1 : 0 ) );
    return urlbuf;
}
	     
