%{
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "main.h"
#include "utils.h"


#ifndef NOUSE_REGEX
#define REGCOMP_OPT (REG_NOSUB | REG_ICASE | REG_EXTENDED)
#include <regex.h>
#endif

#define DEFAULT_RANDOM 100
#define BUFFERLEN 200

#define BASE_TYPE 0

/* Function prototypes */

item *createitem(void);
item *add_exem(item *i, int type, str_list *keys);
item *add_sexem(item *i, int type, str_list *keys);
item *add_cont(item *i, int type, str_list *keys, item *sub);
item *add_scont(item *i, int type, str_list *keys, item *sub);
item *add_redir(item *i, redir_list *keys, int type);
item *add_rand(item *i, int rand);
item *add_log(item *i, char *filename);
item *addSub(item *i, item *sub);

str_list *newKey(char *s);
str_list *addKey(str_list *list, str_list *new);
str_list *getFileKeys(char *file);

redir_list *newRedir(char *s);
redir_list *addRedir(redir_list *list, redir_list *new);
redir_list *getFileRedir(char *file);

void yyerror(char *s, ...);

extern int yylex(void);


/* The lex filestream */
extern FILE *yyin;


/* Keep track of current file being read, and line number */

char *cfile = NULL;
extern int lineno;


/* Structure we are creating */
item *configuration;

%}

%union {
	int iValue;
	char *sPtr;
	str_list *kPtr;
	redir_list *rPtr;
	item *iPtr;
}

%token <iValue> INTEGER CONTAINS_TOK EXEMPT_TOK
%token <sPtr> STRING
%token LOGFILE_TOK REDIRECT_TOK RANDOM_TOK FILE_TOK SOURCE_TOK 
%token BROWSER_TOK NONE_TOK

%type <kPtr> key key_list;
%type <rPtr> redir redir_list;
%type <iPtr> stmt create_item config_file
%type <iValue> browser;
%%

config_file:
	  { $$ = configuration; } stmt_list			{ }
	;

stmt_list:
	  stmt stmt_list									{ }
	| /* NULL */										{ }
	;

stmt:
	  EXEMPT_TOK key_list ';'						{ $$ = add_exem($<iPtr>0,$1,$2); }
	| SOURCE_TOK EXEMPT_TOK key_list ';'		{ $$ = add_sexem($<iPtr>0,$2,$3);}
	| browser REDIRECT_TOK redir_list ';'		{ $$ = add_redir($<iPtr>0,$3,$1);}
	| LOGFILE_TOK STRING ';'						{ $$ = add_log($<iPtr>0, $2);    }
	| LOGFILE_TOK NONE_TOK ';'						{ $$ = add_log($<iPtr>0, NULL);  }
	| RANDOM_TOK INTEGER ';'						{ $$ = add_rand($<iPtr>0, $2);   }
	| SOURCE_TOK CONTAINS_TOK key_list '{' create_item stmt_list '}'
													{ $$ = add_scont($<iPtr>0, $2, $3, $5);}
	| CONTAINS_TOK key_list '{' create_item stmt_list '}'
													{ $$ = add_cont($<iPtr>0, $1, $2, $4); }
	| error ';'											{ $$ = $<iPtr>0; }
	| error '}'											{ $$ = $<iPtr>0; }
	;

browser:
	  BROWSER_TOK										{ $$ = 1; }
	| /* NULL */										{ $$ = 0; }
	;

key_list:
	  key													{ $$ = $1; }
	| key_list key										{ $$ = addKey($1, $2); }
	;

key:
	  STRING												{ $$ = newKey($1); }
	| FILE_TOK STRING                        	{ $$ = getFileKeys($2); }
	;

redir_list:
	  redir												{ $$ = $1; }
	| redir_list redir								{ $$ = addRedir($1, $2); }
	;

redir:
	  STRING												{ $$ = newRedir($1); }
	| FILE_TOK STRING									{ $$ = getFileRedir($2); }
	;

create_item:
															{ $$ = createitem(); }
	;

%%

item *createitem(void)
{
	item *nitem = (item *) safemalloc(sizeof(item));
	memset(nitem, 0, sizeof(item));
	nitem->random = DEFAULT_RANDOM;
	return nitem;
}

item *add_exem(item *i, int type, str_list *keys)
{
	item *sub = createitem();
	sub->keys = keys;
	sub->type = type;
	addSub(i, sub);
	return i;
}

item *add_sexem(item *i, int type, str_list *keys)
{
	if (type != EXEMPT) 
	{
		str_list *c;
		yyerror(
			"Keyword \"source\" can only be applied to \"exempt\" or \"contains\""
		);
		while(keys != NULL)
		{
			c = keys->next;
			free(keys);
			keys = c;
		}
		
		return i;
	}
	return add_exem(i, SOURCEEXEMPT, keys);
}

item *add_cont(item *i, int type, str_list *keys, item *sub)
{ 
	sub->keys = keys;
	sub->type = type;
	addSub(i, sub);
	return i;
}

item *add_scont(item *i, int type, str_list *keys, item *sub)
{
	if (type != CONTAINS)
	{
		str_list *c;
		yyerror(
			"Keyword \"source\" can only be applied to \"exempt\" or \"contains\""
		);
		while(keys != NULL)
		{
			c = keys->next;
			free(keys);
			keys = c;
		}
		
		return i;
	}
	return add_cont(i, SOURCECONTAINS, keys, sub);
}


item *add_redir(item *i, redir_list *redirs, int type)
{
	redir_list *r = redirs;
	while (r != NULL)
	{
		r->type = type;
		i->numredir++;
		r = r->next;
	}
	i->redirect = addRedir(i->redirect, redirs);
	return i;
}


item *add_log(item *i, char *filename)
{
	if (i->logfile != NULL) {
		yyerror("Logfile allready defined in this block");
		if (filename) free(filename);
		return i;
	}
	if (filename != NULL)
		i->logfile = filename;
	else
		i->logfile = strdup("");
	return i;
}

item *add_rand(item *i, int rand)
{
	i->random = rand;
	return i;
}

item *addSub(item *i, item *sub)
{
	item *cur = i->subs;

	if (cur == NULL) i->subs = sub;
	else
	{
		while(cur->next != NULL) cur = cur->next;
		cur->next = sub;
	}
	return i->subs;
}


str_list *newKey(char *s)
{
	str_list *nstr;

#ifndef NOUSE_REGEX
	int errcode;
	regex_t *regc = (regex_t *) safemalloc(sizeof(regex_t));
	if ((errcode = regcomp(regc, s, REGCOMP_OPT)) != 0)
	{
		char errbuf[BUFFERLEN];
		regerror(errcode, regc, errbuf, BUFFERLEN);
		yyerror("Invalid regular expression \"%s\" - %s", s, errbuf);
		free(regc);
		return NULL;
	}
#endif

	nstr = (str_list *) safemalloc(sizeof(str_list));
	nstr->data = s;
	nstr->next = NULL;
#ifndef NOUSE_REGEX
	nstr->regc = regc;
#elif !defined NOSAVESKIP
	/* for Boyer-Moore searching */
	nstr->skip = buildskip(s);
#endif

	return nstr;
}

str_list *getFileKeys(char *file)
{
	str_list *head = NULL;
	str_list *cur  = NULL;
	str_list *nstr = NULL;

	char buffer[BUFFERLEN];
	FILE *fp;

	int oldline = lineno;

	if ((fp = fopen(file, "r")) == NULL)
	{
		yyerror("Couldn't open file \"%s\"", file);
		return NULL;
	}

	cfile = file;

	/* Read line at a time, remove \n, put in a node and add to list */
	for(lineno=1; fgets(buffer, BUFFERLEN, fp); lineno++)
	{
		char *s;
		if (*(s = buffer + strlen(buffer)-1) == EOLN) *s = EOS;

		nstr = newKey(strdup(buffer));

		if (cur == NULL)
			head = cur = nstr;
		else if (nstr != NULL)
		{
			cur->next = nstr;
			cur = nstr;
		}
	}

	if (fclose(fp) == EOF)
		yyerror("Couldn't close file \"%s\"", file);
	
	lineno = oldline;
	cfile = NULL;

	return head;
}

str_list *addKey(str_list *list, str_list *new)
{
	str_list *cur = list;

	if (list == NULL) return new;

	while(cur->next != NULL) cur = cur->next;
	cur->next = new;
	
	return list;
}


redir_list *newRedir(char *s)
{
	redir_list *nstr = (redir_list *) safemalloc(sizeof(redir_list));
	nstr->data = s;
	nstr->next = NULL;
	nstr->type = 0;
	return nstr;
}

redir_list *getFileRedir(char *file)
{
	redir_list *head = NULL;
	redir_list *cur  = NULL;
	redir_list *nstr = NULL;

	char buffer[BUFFERLEN];
	FILE *fp;

	int oldline = lineno;

	if ((fp = fopen(file, "r")) == NULL)
	{
		yyerror("Couldn't open file \"%s\"", file);
		return NULL;
	}

	cfile = file;

	/* Read line at a time, remove \n, put in a node and add to list */
	for(lineno=1; fgets(buffer, BUFFERLEN, fp); lineno++)
	{
		char *s;
		if (*(s = buffer + strlen(buffer)-1) == EOLN) *s = EOS;

		nstr = newRedir(strdup(buffer));

		if (cur == NULL)
			head = cur = nstr;
		else if (nstr != NULL)
		{
			cur->next = nstr;
			cur = nstr;
		}
	}

	if (fclose(fp) == EOF)
		yyerror("Couldn't close file \"%s\"", file);
	
	lineno = oldline;
	cfile = NULL;

	return head;
}

redir_list *addRedir(redir_list *list, redir_list *new)
{
	redir_list *cur = list;
	if (list == NULL) return new;
	while (cur->next != NULL) cur = cur->next;
	cur->next = new;
	return list;
}


void yyerror(char *s, ...)
{
	va_list ap;
	
	va_start(ap, s);
	vfprintf(stderr, s, ap);
	if (cfile)
		fprintf(stderr, " at line %d (%s).\n", lineno, cfile);
	else
		fprintf(stderr, " at line %d.\n", lineno);
	va_end(ap);
}

item *load_config(const char *file)
{
	if ((yyin = fopen(file, "r")) == NULL) {
		fprintf(stderr, "ERROR: Couldn't open file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}

	configuration = createitem();
	configuration->type = BASE_TYPE;
	yyparse();

	if (fclose(yyin) == EOF) {
		fprintf(stderr, "ERROR: Couldn't close file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}

	return configuration;
}


/* Recursive function to destroy data structure. */

void delete_data(item *curr)
{
	str_list *curKey;
	redir_list *curRedir;

	if (curr == NULL) return;

	delete_data(curr->subs);
	delete_data(curr->next);

	if (curr->logfile != NULL) free(curr->logfile);

	curKey = curr->keys;
	while (curKey != NULL)
	{
		str_list *p;
		p = curKey;
		curKey = p->next;
		if (p->data != NULL) free(p->data);
#ifndef NOUSE_REGEX
		if (p->regc != NULL) free(p->regc);
#elif !defined NOSAVESKIP
		if (p->skip != NULL) free(p->skip);
#endif
		free(p);
	}

	curRedir = curr->redirect;
	while (curRedir != NULL)
	{
		redir_list *p;
		p = curRedir;
		curRedir = p->next;
		if (p->data != NULL) free(p->data);
		free(p);
	}
}

	
