/*
** 1999-03-13 -	A (new) module for dealing with user dialogs. The old one was written during
**		my first week of GTK+ programming (back in May 1998, if you really care), and
**		that had started to show a little too much. Hopefully, this is cleaner.
** 1999-06-19 -	Since the initial cleanliness quickly wore down due to (I guess) bad design,
**		I did a complete rewrite and redesign. This new shiny version features clear
**		separation of dialogs into synchronous and asynchronous versions. The former
**		are way (way) more useful.
*/

#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "dialog.h"

struct _Dialog {
	GtkWidget	*dlg;		/* Main GtkDialog object. */
	gint		last_button;
	gint		button;
	DlgAsyncFunc	func;		/* Used in asynchronous dialogs only. */
	gpointer	user;		/* This, too. */
};

#define	DLG_BUTTON_MAX		(32)	/* Max length of a single button label. */

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Build a set of action buttons (like "OK", "Cancel" and so on), and pack them into <bo>. Buttons are
**		defined by writing the desired labels separated by vertical bars (e.g. "OK|Cancel" is very common).
**		Will connect <func,user> to the "clicked" signal of all buttons. Each button will have its user data
**		pointer set to the index (counting from 0) of the button.
**		Returns the index of the last button built (e.g. 1 for the "OK|Cancel" case).
*/
static guint build_buttons(const gchar *buttons, GtkWidget *box, GtkSignalFunc func, gpointer user)
{
	gchar		*base, *ptr, label[DLG_BUTTON_MAX];
	gint		index = 0;
	GtkWidget	*btn;

	for(base = (gchar *) buttons, index = 0; *base; index++)
	{
		for(ptr = label; (ptr - label) < (sizeof label - 1) && *base && *base != '|';)
			*ptr++ = *base++;
		*ptr = '\0';
		if(*base == '|')
			base++;
		btn = gtk_button_new_with_label(label);
		GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(func), user);
		gtk_object_set_user_data(GTK_OBJECT(btn), GINT_TO_POINTER(index));
		gtk_box_pack_start(GTK_BOX(box), btn, TRUE, TRUE, 0);
		gtk_widget_show(btn);
		if(index == 0)
			gtk_widget_grab_default(btn);
	}
	return index - 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Close dialog down in response to some event, and report it as if button <button> was clicked. */
static void sync_close(Dialog *dlg, gint button)
{
	dlg->button = button;
	gtk_widget_hide(dlg->dlg);
	gtk_grab_remove(dlg->dlg);
	gtk_main_quit();
}

/* 1999-06-19 -	User clicked the dialog's window close button. Hide it, and return non-button code (-1) to caller. */
static gboolean evt_sync_delete(GtkWidget *wid, GdkEvent *evt, gpointer user)
{
	sync_close(user, -1);

	return TRUE;
}

static gboolean evt_sync_key_pressed(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	Dialog	*dlg = user;

	switch(evt->keyval)
	{
		case GDK_Return:
			sync_close(dlg, 0);
			return TRUE;
		case GDK_Escape:
			sync_close(dlg, dlg->last_button);
			return TRUE;
	}
	return FALSE;
}

/* 1999-06-19 -	This gets called when the user clicks a button in a synchronous dialog. */
static void evt_sync_button_clicked(GtkWidget *wid, gpointer user)
{
	sync_close(user, GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(wid))));
}

/* 1999-06-19 -	Create a new, synchronous dialog box. Creates and initializes the auxilary data as well as the
**		actual dialog window, with action buttons and stuff. Note that this window, when displayed by
**		dlg_dialog_sync_wait(), will be modal, and hang around (at least in memory) until you call
**		dlg_dialog_sync_destroy().
*/
Dialog * dlg_dialog_sync_new(GtkWidget *body, const gchar *title, const gchar *buttons)
{
	Dialog		*dlg;
	GtkWidget	*vbox;

	dlg = g_malloc(sizeof *dlg);
	dlg->dlg = gtk_dialog_new();
	gtk_window_set_position(&GTK_DIALOG(dlg->dlg)->window, GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "delete_event", GTK_SIGNAL_FUNC(evt_sync_delete), dlg);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "key_press_event", GTK_SIGNAL_FUNC(evt_sync_key_pressed), dlg);
	if(title != NULL)
		gtk_window_set_title(GTK_WINDOW(GTK_DIALOG(dlg->dlg)), title);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), body, TRUE, TRUE, 0);
	gtk_widget_show(body);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->dlg)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show(vbox);
	dlg->last_button = build_buttons(buttons, GTK_DIALOG(dlg->dlg)->action_area, evt_sync_button_clicked, dlg);

	return dlg;
}

/* 1999-06-19 -	Create a synchronous dialog whose body is a plain text label. Extremely sugary. */
Dialog * dlg_dialog_sync_new_simple(const gchar *body, const gchar *title, const gchar *buttons)
{
	return dlg_dialog_sync_new(gtk_label_new(body), title, buttons);
}

/* 1999-06-19 -	This (bulkily-named) function is very handy for simple confirm dialogs. */
gint dlg_dialog_sync_new_simple_wait(const gchar *body, const gchar *title, const gchar *buttons)
{
	Dialog	*dlg;
	gint	ret;

	dlg = dlg_dialog_sync_new_simple(body, title, buttons);
	ret = dlg_dialog_sync_wait(dlg);
	dlg_dialog_sync_destroy(dlg);

	return ret;
}

/* 1999-06-19 -	Display a previously created dialog, and wait for the user to click one of its buttons. Then
**		return the index (counting from zero which is the leftmost button) of the button that was
**		clicked.
*/
gint dlg_dialog_sync_wait(Dialog *dlg)
{
	if(dlg != NULL)
	{
		gtk_widget_show(dlg->dlg);
		gtk_grab_add(dlg->dlg);
		gtk_main();
		return dlg->button;
	}
	return -1;
}

/* 1999-06-19 -	Close an asynchronous dialog as if <button> was clicked. */
void dlg_dialog_sync_close(Dialog *dlg, gint button)
{
	if(dlg != NULL)
		sync_close(dlg, button);
}

/* 1999-06-19 -	Destroy a synchronous dialog. Don't expect to be able to access body widgets after calling this. */
void dlg_dialog_sync_destroy(Dialog *dlg)
{
	gtk_widget_destroy(dlg->dlg);

	g_free(dlg);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Execute the close-down policy of the asynchronous dialog. */
static void async_close(Dialog *dlg, gint button)
{
	dlg->button = button;
	if((dlg->button > -2 ) && (dlg->func != NULL))
		dlg->func(dlg->button, dlg->user);
	gtk_widget_destroy(dlg->dlg);
	g_free(dlg);
}

/* 1999-06-19 -	User closed an asynch. dialog window. */
static gboolean evt_async_delete(GtkWidget *wid, GdkEvent *evt, gpointer user)
{
	async_close(user, -1);

	return TRUE;
}

/* 1999-06-19 -	User pressed a key in an asynchronous dialog. */
static gboolean evt_async_key_pressed(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	Dialog	*dlg = user;

	switch(evt->keyval)
	{
		case GDK_Return:
			async_close(dlg, 0);
			return TRUE;
		case GDK_Escape:
			async_close(dlg, dlg->last_button);
			return TRUE;
	}
	return FALSE;
}

/* 1999-06-19 -	User clicked one of the action buttons in an asynchronous dialog. */
static void evt_async_button_clicked(GtkWidget *wid, gpointer user)
{
	async_close(user, GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(wid))));
}

/* 1999-06-19 -	Create, and immediately display, an asynchronous dialog. The supplied callback will be called
**		as the user clicks a button (or closes the window). At that point, the dialog (and any user-
**		supplied body widgets) are still around.
*/
Dialog * dlg_dialog_async_new(GtkWidget *body, const gchar *title, const gchar *buttons, DlgAsyncFunc func, gpointer user)
{
	Dialog		*dlg;
	GtkWidget	*vbox;

	dlg = g_malloc(sizeof *dlg);
	dlg->func = func;
	dlg->user = user;
	dlg->dlg = gtk_dialog_new();
	gtk_window_set_position(&GTK_DIALOG(dlg->dlg)->window, GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "delete_event", GTK_SIGNAL_FUNC(evt_async_delete), dlg);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "key_press_event", GTK_SIGNAL_FUNC(evt_async_key_pressed), dlg);
	if(title != NULL)
		gtk_window_set_title(GTK_WINDOW(GTK_DIALOG(dlg->dlg)), title);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), body, TRUE, TRUE, 0);
	gtk_widget_show(body);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->dlg)->vbox), vbox, TRUE, TRUE, 0);
	gtk_widget_show(vbox);
	dlg->last_button = build_buttons(buttons, GTK_DIALOG(dlg->dlg)->action_area, evt_async_button_clicked, dlg);
	gtk_widget_show(dlg->dlg);

	return dlg;
}

/* 1999-06-19 -	A simple sugary wrapper for text-only dialogs. */
Dialog * dlg_dialog_async_new_simple(const gchar *body, const gchar *title, const gchar *buttons, DlgAsyncFunc func, gpointer user)
{
	return dlg_dialog_async_new(gtk_label_new(body), title, buttons, func, user);
}

/* 1999-06-19 -	Provide a simple error reporting dialog. */
Dialog * dlg_dialog_async_new_error(const gchar *body)
{
	return dlg_dialog_async_new_simple(body, "Error", "OK", NULL, NULL);
}

/* 1999-06-19 -	Close an asynchronous dialog, as if button <button> was clicked. */
void dlg_dialog_async_close(Dialog *dlg, gint button)
{
	async_close(dlg, button);
}

/* 1999-06-19 -	Close an asynchronous dialog, but do not call the user's callback. */
void dlg_dialog_async_close_silent(Dialog *dlg)
{
	async_close(dlg, -2);
}
