/*
    main.c - Provide the main window and related routines
    Copyright (c) 1999  Martin Schulze <joey@infodrom.north.de>

    This file is part of the Gerstensaft package.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111, USA.
*/

/* #define DBL_CLICK_FILE / **/

#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <gtk/gtk.h>
#include "msgbox.h"
#include "main.h"
#include "io.h"
#include "beer.h"
#include "send.h"
#include "nls.h"
#include "config.h"
#include <stdio.h>

GdkPixmap *dir_pixmap = NULL,
          *linkdir_pixmap = NULL,
          *linkfile_pixmap = NULL,
          *updir_pixmap = NULL,
          *doc_pixmap = NULL;
GdkBitmap *dir_mask = NULL,
          *linkdir_mask = NULL,
          *linkfile_mask = NULL,
          *updir_mask = NULL,
          *doc_mask = NULL;

gint main_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  return (FALSE);
}

void main_destroy (GtkWidget *widget, gpointer data)
{
  char **cp;
  int i;

  if (((struct s_main_data *) data) != NULL) {
    if (((struct s_main_data *) data)->dir != NULL)
      free (((struct s_main_data *) data)->dir);
    if (((struct s_main_data *) data)->numfiles)
      free (((struct s_main_data *) data)->selected);
    if (((struct s_main_data *) data)->numstored) {
      if (((struct s_main_data *) data)->stored != NULL) {
	for (i=0,cp=((struct s_main_data *) data)->stored; i<((struct s_main_data *) data)->numstored; i++,((char **)cp)++)
	  free (*cp);
	free (((struct s_main_data *) data)->stored);
      }
      if (((struct s_main_data *) data)->size != NULL)
	free (((struct s_main_data *) data)->size);
    }
    free (((struct s_main_data *) data));
  }

  gtk_main_quit ();
}

void menu_clear_history (GtkWidget *widget, gpointer data)
{
  config_clear_history();
}

void load_pixmap (const char *name, GtkWidget *parent, GdkPixmap **pixmap, GdkBitmap **mask)
{
  char fname[1024];
  GtkStyle *style;
  
  snprintf (fname, sizeof(fname), "%s/%s", PIXMAP_DIR, name);
  style = gtk_widget_get_style (parent);

  *pixmap = gdk_pixmap_create_from_xpm (parent->window, mask, &style->bg[GTK_STATE_NORMAL], fname);
}

void read_directory (struct s_main_data *wdata, char *dir)
{
  struct s_file *files, *fp;
  char name[100], size[10], *dummy="";
  char *clp[3] = {dummy, name, size};
  GdkColor background;
  GdkColormap *colormap;

#ifdef DEBUG
  printf ("read_directory(%s)\n", dir);
#endif

  background.red = background.green = background.blue =  ~0;
  colormap = gdk_colormap_get_system();
  gdk_color_alloc (colormap, &background);

 /* Copy the new directory into local variable */
  if (wdata->dir)
    free (wdata->dir);
  wdata->dir = check_and_adjust_dir (dir);

 if ((files = get_dir (wdata->dir)) != NULL) {
   if (wdata->selected)
     free (wdata->selected);
   gtk_clist_freeze (GTK_CLIST(wdata->listbox));
   gtk_clist_clear (GTK_CLIST(wdata->listbox));
   wdata->numfiles=0;
   for (fp = files; fp; fp=fp->next) {
     snprintf (name, sizeof (name), "%s", fp->name);
     snprintf (size, sizeof (size), "%d", fp->size);
     gtk_clist_append (GTK_CLIST(wdata->listbox), clp);
     gtk_clist_set_background (GTK_CLIST(wdata->listbox), wdata->numfiles, &background);
     switch (fp->type)
       {
       case FILE_REGULAR:
	 gtk_clist_set_pixmap (GTK_CLIST(wdata->listbox), wdata->numfiles, 0, doc_pixmap, doc_mask);
	 break;
       case FILE_DIRECTORY:
	 gtk_clist_set_pixmap (GTK_CLIST(wdata->listbox), wdata->numfiles, 0, dir_pixmap, dir_mask);
	 break;
       case FILE_LINK:
	 gtk_clist_set_pixmap (GTK_CLIST(wdata->listbox), wdata->numfiles, 0, linkfile_pixmap, linkfile_mask);
	 break;
       case FILE_LINKDIR:
	 gtk_clist_set_pixmap (GTK_CLIST(wdata->listbox), wdata->numfiles, 0, linkdir_pixmap, linkdir_mask);
	 break;
       case FILE_UPDIR:
	 gtk_clist_set_pixmap (GTK_CLIST(wdata->listbox), wdata->numfiles, 0, updir_pixmap, updir_mask);
	 break;
       }
     wdata->numfiles++;
   }
   gtk_clist_thaw (GTK_CLIST(wdata->listbox));
   if ((wdata->selected = (int *)malloc (wdata->numfiles*sizeof(int))) != NULL)
     memset (wdata->selected, 0, wdata->numfiles*sizeof(int));
 }

 /* Now adjust the input field */
 gtk_entry_set_text (GTK_ENTRY(wdata->dirinput), wdata->dir);
}

void dirinput_change (GtkWidget *widget, gpointer data)
{
#ifdef DEBUG
  printf ("dirinput_change()\n");
  printf ("  Old value: %s\n", ((struct s_main_data *)data)->dir);
  printf ("  New value: %s\n", gtk_entry_get_text (GTK_ENTRY(widget)));
#endif

  read_directory ((struct s_main_data *)data, gtk_entry_get_text (GTK_ENTRY(widget)));
}

void main_row_select (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
#ifdef DEBUG
  printf ("select_row %d\n", row);
#endif
  if (row < ((struct s_main_data *) data)->numfiles)
    if ((row > 0) || (!strcmp(((struct s_main_data *) data)->dir,"/")))
      ((struct s_main_data *) data)->selected[row] = 1;

  /* Don't allow to select the .. file */
  if ((row == 0) && (strcmp(((struct s_main_data *) data)->dir,"/")))
    gtk_clist_unselect_row (GTK_CLIST(((struct s_main_data *) data)->listbox), row, column);
}

void main_row_unselect (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
#ifdef DEBUG
  printf ("unselect_row %d\n", row);
#endif
  if (row < ((struct s_main_data *) data)->numfiles)
    ((struct s_main_data *) data)->selected[row] = 0;
}

void main_row_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
  gint row, col;
  gchar *file;
  char *fname, *cp;
#ifdef DBL_CLICK_FILE
  char **tmp;
  unsigned long int *size;
  char name[1024], asize[10], *dummy="";
  char *clp[3] = {dummy, name, asize};
  GdkColor background;
  GdkColormap *colormap;
#endif

  gtk_clist_get_selection_info (GTK_CLIST (widget), event->x, event->y, &row, &col);

  if(event->type == GDK_2BUTTON_PRESS) {
    gtk_clist_get_text(GTK_CLIST(((struct s_main_data *) data)->listbox), row, 1, &file);
    if (!strcmp(file, "..")) {
      if ((fname = (char *)malloc (strlen (((struct s_main_data *) data)->dir) +1)) == NULL)
	return;
      strcpy (fname, ((struct s_main_data *) data)->dir);
      if ((cp = strrchr (fname, '/')) != NULL) {
	*cp = '\0';
	read_directory ((struct s_main_data *)data, fname);
	free (fname);
      }
    } else {
      if ((fname = (char *)malloc (strlen (((struct s_main_data *) data)->dir) + strlen (file) +2)) == NULL)
	return;
      sprintf (fname, "%s/%s", ((struct s_main_data *) data)->dir, file);

      if (is_dir (fname, NULL)) {
	read_directory ((struct s_main_data *)data, fname);
#ifdef DBL_CLICK_FILE
      } else {
	/*
	  Problem: The row doesn't get unselected.  The procedure
	  calls look fine but the result is not sufficient.  Any
	  hints?
	*/
        gtk_clist_freeze (GTK_CLIST(((struct s_main_data *) data)->listbox));
        gtk_clist_unselect_row (GTK_CLIST(((struct s_main_data *) data)->listbox), row, col);
        gtk_clist_thaw (GTK_CLIST(((struct s_main_data *) data)->listbox));
	if (!list_contains_file (((struct s_main_data *) data)->stored,
				 ((struct s_main_data *) data)->numstored,
				 ((struct s_main_data *) data)->dir,
				 file)) {
	  /* Set up a new array and copy old files if needed */
	  if ((tmp = (char **)malloc ((((struct s_main_data *) data)->numstored+1)*sizeof(char *))) == NULL)
	    return;
	  if ((size =(unsigned long int *)malloc ((((struct s_main_data *) data)->numstored+1)*sizeof(unsigned long int))) == NULL)
	    return;
	  memcpy (tmp, ((struct s_main_data *) data)->stored, ((struct s_main_data *) data)->numstored*sizeof(char *));
	  memcpy (size, ((struct s_main_data *) data)->size, ((struct s_main_data *) data)->numstored*sizeof(unsigned long int));

	  if ((tmp[((struct s_main_data *) data)->numstored] = (char *)malloc (strlen(fname)+1)) != NULL) {
	    background.red = background.green = background.blue =  ~0;
	    colormap = gdk_colormap_get_system();
	    gdk_color_alloc (colormap, &background);

	    strcpy (tmp[((struct s_main_data *) data)->numstored], fname);
	    size[((struct s_main_data *) data)->numstored] = get_filesize (fname);
	    strncpy (name, fname, sizeof (name));
	    snprintf (asize, sizeof (asize), "%lu", size[((struct s_main_data *) data)->numstored]);
	    row = gtk_clist_append (GTK_CLIST(((struct s_main_data *) data)->filebox), clp);
	    gtk_clist_set_background (GTK_CLIST(((struct s_main_data *) data)->filebox), row, &background);
	    gtk_clist_set_pixmap (GTK_CLIST(((struct s_main_data *) data)->filebox), row, 0, doc_pixmap, doc_mask);
	    ((struct s_main_data *) data)->numstored++;
	  }

	  /* Now free the old data and copy the pointer */
	  if (((struct s_main_data *) data)->stored != NULL)
	    free (((struct s_main_data *) data)->stored);
	  if (((struct s_main_data *) data)->size != NULL)
	    free (((struct s_main_data *) data)->size);
	  ((struct s_main_data *) data)->stored = tmp;
	  ((struct s_main_data *) data)->size = size;
	}
#endif
      }
      free (fname);
    }
  }
}

void main_button_select_all (GtkWidget *widget, gpointer data)
{
  int i;

  gtk_clist_freeze (GTK_CLIST(((struct s_main_data *) data)->listbox));
  for (i=0; i<((struct s_main_data *) data)->numfiles; i++)
    if (!((struct s_main_data *) data)->selected[i]) {
      ((struct s_main_data *) data)->selected[i] = 1;
      gtk_clist_select_row (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 0);
    }

  gtk_clist_thaw (GTK_CLIST(((struct s_main_data *) data)->listbox));
}

void main_button_unselect_all (GtkWidget *widget, gpointer data)
{
  int i;

  gtk_clist_freeze (GTK_CLIST(((struct s_main_data *) data)->listbox));
  for (i=0; i<((struct s_main_data *) data)->numfiles; i++)
    if (((struct s_main_data *) data)->selected[i]) {
      ((struct s_main_data *) data)->selected[i] = 0;
      gtk_clist_unselect_row (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 0);
    }
  gtk_clist_thaw (GTK_CLIST(((struct s_main_data *) data)->listbox));
}

void button_add (GtkWidget *widget, gpointer data)
{
  int i, n;
  char **tmp;
  unsigned long int *size;
  gchar *fname;
  char name[1024], asize[10], *dummy="";
  char *clp[3] = {dummy, name, asize};
  gint row;
  GdkColor background;
  GdkColormap *colormap;

  background.red = background.green = background.blue =  ~0;
  colormap = gdk_colormap_get_system();
  gdk_color_alloc (colormap, &background);

  /* Count additional elements and unselect items */
  for (i=0,n=0; i < ((struct s_main_data *) data)->numfiles; i++) {
    if (((struct s_main_data *) data)->selected[i]) {
      gtk_clist_get_text (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 1, &fname);

      if (list_contains_file (((struct s_main_data *) data)->stored,
			      ((struct s_main_data *) data)->numstored,
			      ((struct s_main_data *) data)->dir,
			      fname))
	gtk_clist_unselect_row (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 0);
      else
	n++;
    }
  }

  if (!n)
    return;

  /* Set up a new array and copy old files if needed */
  if ((tmp = (char **)malloc ((n+((struct s_main_data *) data)->numstored)*sizeof(char *))) == NULL)
    return;
  if ((size = (unsigned long int *)malloc ((n+((struct s_main_data *) data)->numstored)*sizeof(unsigned long int))) == NULL)
    return;
  memcpy (tmp, ((struct s_main_data *) data)->stored, ((struct s_main_data *) data)->numstored*sizeof(char *));
  memcpy (size, ((struct s_main_data *) data)->size, ((struct s_main_data *) data)->numstored*sizeof(unsigned long int));
  
  /* Add new files to array */
  gtk_clist_freeze (GTK_CLIST(((struct s_main_data *) data)->filebox));
  n = strlen (((struct s_main_data *) data)->dir)+2;  /* length + '/' + '\0' */

  for (i=0; i < ((struct s_main_data *) data)->numfiles; i++) {
    if (((struct s_main_data *) data)->selected[i]) {
      gtk_clist_get_text (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 1, &fname);
      gtk_clist_unselect_row (GTK_CLIST(((struct s_main_data *) data)->listbox), i, 0);

      if ((tmp[((struct s_main_data *) data)->numstored] = (char *)malloc (n+strlen(fname))) != NULL) {
	sprintf (tmp[((struct s_main_data *) data)->numstored], "%s/%s", ((struct s_main_data *) data)->dir, fname);
	size[((struct s_main_data *) data)->numstored] = get_filesize (tmp[((struct s_main_data *) data)->numstored]);
	snprintf (name, sizeof (name), "%s/%s", ((struct s_main_data *) data)->dir, fname);
	snprintf (asize, sizeof (asize), "%lu", size[((struct s_main_data *) data)->numstored]);
	row = gtk_clist_append (GTK_CLIST(((struct s_main_data *) data)->filebox), clp);
	gtk_clist_set_background (GTK_CLIST(((struct s_main_data *) data)->filebox), row, &background);
	if (is_dir (name, NULL))
	  gtk_clist_set_pixmap (GTK_CLIST(((struct s_main_data *) data)->filebox), row, 0, dir_pixmap, dir_mask);
	else
	  gtk_clist_set_pixmap (GTK_CLIST(((struct s_main_data *) data)->filebox), row, 0, doc_pixmap, doc_mask);

	((struct s_main_data *) data)->numstored++;
      }
    }
  }

  /* Now free the old data and copy the pointer */
  if (((struct s_main_data *) data)->stored != NULL)
    free (((struct s_main_data *) data)->stored);
  if (((struct s_main_data *) data)->size != NULL)
    free (((struct s_main_data *) data)->size);
  ((struct s_main_data *) data)->stored = tmp;
  ((struct s_main_data *) data)->size = size;

  gtk_clist_thaw (GTK_CLIST(((struct s_main_data *) data)->filebox));

  if (((struct s_main_data *) data)->numstored > 0) {
    gtk_widget_set_sensitive (((struct s_main_data *) data)->clearbutton, TRUE);
    gtk_widget_set_sensitive (((struct s_main_data *) data)->sendbutton, TRUE);
  }
}

void main_clear (gpointer data)
{
  if (data != NULL && ((struct s_main_data *) data)->stored != NULL)
    free (((struct s_main_data *) data)->stored);
  ((struct s_main_data *) data)->stored = NULL;
  ((struct s_main_data *) data)->numstored = 0;

  gtk_clist_clear (GTK_CLIST(((struct s_main_data *) data)->filebox));
  gtk_widget_set_sensitive (((struct s_main_data *) data)->clearbutton, FALSE);
  gtk_widget_set_sensitive (((struct s_main_data *) data)->sendbutton, FALSE);
}

void button_clear (GtkWidget *widget, gpointer data)
{
  gtk_clist_clear (GTK_CLIST(((struct s_main_data *) data)->filebox));

  main_clear (data);
}

void button_send (GtkWidget *widget, gpointer data)
{
  if (((struct s_main_data *) data)->numstored) {
    send_dialog (((struct s_main_data *) data)->numstored,
		 ((struct s_main_data *) data)->stored, 
		 ((struct s_main_data *) data)->size, 
		 data);
  }
}

/*
 * FIXME: Prevent the Size field from being size-increased if the dialog gets resized
 */
/**
 * main_window:
 *
 * Opens the main window with menu etc.
 * 
 * return values: nothing
+------+-------+-------+
| File | ..... | Help  |
+------+-------+-------+
| Open |       | Index |
|      |       |       |
| Quit |       | About |
+------+       +-------+
 */
void main_window()
{
  GtkWidget *window;
  GtkWidget *mainbox, *hbox, *vbox;
  GtkWidget *menubar, *menu, *menuitem;
  GtkWidget *frame;
  GtkWidget *button;
#if GTK_MAJOR_VERSION >= 1 && GTK_MINOR_VERSION >= 1
  GtkWidget *scrollwin;
#endif
  GtkTooltips *tooltips;
  char *titles[3] = {"", _("Filename"), _("Size")};
  char cwd[255];
  struct s_main_data *wdata;

  if ((wdata = (struct s_main_data *) malloc (sizeof(struct s_main_data))) == NULL)
    return;
  memset (wdata, 0, sizeof (struct s_main_data));

  /* Create a new dialog box */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Gerstensaft");
  gtk_widget_set_usize (GTK_WIDGET (window), 350, 350);

  /* Connect the appropriate signal handlers */
  gtk_signal_connect (GTK_OBJECT (window), "delete_event", GTK_SIGNAL_FUNC (main_delete), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy", GTK_SIGNAL_FUNC (main_destroy), (gpointer) wdata );

  mainbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), mainbox);
  menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX(mainbox), menubar, FALSE, FALSE, 2);

  menu = gtk_menu_new();
  menuitem = gtk_menu_item_new_with_label("File");
  gtk_menu_item_set_submenu( GTK_MENU_ITEM(menuitem), menu );
  gtk_menu_bar_append( GTK_MENU_BAR (menubar), menuitem );
  gtk_widget_show(menuitem);

    /* Should be moved into the Options menu */
    menuitem = gtk_menu_item_new_with_label("Clear History");
    gtk_signal_connect (GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(menu_clear_history), NULL);
    gtk_menu_append( GTK_MENU(menu), menuitem);
    gtk_widget_show(menuitem);

    menuitem = gtk_menu_item_new_with_label("Quit");
    gtk_signal_connect (GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(main_destroy), (gpointer) wdata);
    gtk_menu_append( GTK_MENU(menu), menuitem);
    gtk_widget_show(menuitem);

  menu = gtk_menu_new();
  menuitem = gtk_menu_item_new_with_label("Help");
  gtk_menu_item_set_submenu( GTK_MENU_ITEM(menuitem), menu );
  gtk_menu_item_right_justify( GTK_MENU_ITEM(menuitem) );
  gtk_menu_bar_append( GTK_MENU_BAR (menubar), menuitem );
  gtk_widget_show(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("Todo"));
    gtk_signal_connect ( GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(msgbox_todo), NULL);
    gtk_menu_append( GTK_MENU(menu), menuitem);
    gtk_widget_show(menuitem);

    menuitem = gtk_menu_item_new_with_label(_("About"));
    gtk_signal_connect ( GTK_OBJECT(menuitem), "activate", GTK_SIGNAL_FUNC(msgbox_about), NULL);
    gtk_menu_append( GTK_MENU(menu), menuitem);
    gtk_widget_show(menuitem);

  gtk_widget_show(menubar);

  /* Horizontal box containing Clist on the left and buttons on the right */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(mainbox), hbox);

  /* Listbox containing files to select */
  frame = gtk_frame_new (NULL);
  gtk_container_border_width (GTK_CONTAINER(frame), 5);
  gtk_container_add (GTK_CONTAINER(hbox), frame);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(vbox), 5);
  gtk_container_add (GTK_CONTAINER(frame), vbox);

  /* Input field for directory, will be updated automatically */
  wdata->dirinput = gtk_entry_new_with_max_length (60);
  gtk_signal_connect (GTK_OBJECT(wdata->dirinput), "activate", GTK_SIGNAL_FUNC(dirinput_change), (gpointer) wdata);
  gtk_box_pack_start (GTK_BOX(vbox), wdata->dirinput, FALSE, FALSE, FALSE);
  gtk_widget_show (wdata->dirinput);

  wdata->listbox = gtk_clist_new_with_titles (3, titles);
  gtk_widget_set_usize (GTK_WIDGET (wdata->listbox), 100, 100);
  gtk_clist_column_titles_passive (GTK_CLIST(wdata->listbox));
  gtk_clist_set_selection_mode (GTK_CLIST(wdata->listbox), GTK_SELECTION_MULTIPLE);
  gtk_clist_set_column_width (GTK_CLIST(wdata->listbox), 0, 16);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->listbox), 0, GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_width (GTK_CLIST(wdata->listbox), 1, 130);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->listbox), 1, GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_width (GTK_CLIST(wdata->listbox), 2, 30);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->listbox), 2, GTK_JUSTIFY_RIGHT);

  gtk_signal_connect(GTK_OBJECT(wdata->listbox), "select_row", GTK_SIGNAL_FUNC(main_row_select), (gpointer) wdata);
  gtk_signal_connect(GTK_OBJECT(wdata->listbox), "unselect_row", GTK_SIGNAL_FUNC(main_row_unselect), (gpointer) wdata);
  gtk_signal_connect(GTK_OBJECT(wdata->listbox), "button_press_event", GTK_SIGNAL_FUNC(main_row_dblclick), (gpointer) wdata);

#if GTK_MAJOR_VERSION >= 1 && GTK_MINOR_VERSION >= 1
  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER(scrollwin), wdata->listbox);
  gtk_widget_show (scrollwin);
#else
  gtk_box_pack_start (GTK_BOX(vbox) , wdata->listbox, TRUE, TRUE, 0);
#endif

  gtk_widget_show (window);

  load_pixmap ("updir.xpm", window, &updir_pixmap, &updir_mask);
  load_pixmap ("dir.xpm", window, &dir_pixmap, &dir_mask);
  load_pixmap ("doc.xpm", window, &doc_pixmap, &doc_mask);
  load_pixmap ("linkdir.xpm", window, &linkdir_pixmap, &linkdir_mask);
  load_pixmap ("linkfile.xpm", window, &linkfile_pixmap, &linkfile_mask);

  if (getcwd (cwd, sizeof(cwd)))
    read_directory (wdata, cwd);

  gtk_widget_show (wdata->listbox);

  gtk_widget_show (frame);
  gtk_widget_show (vbox); /* listbox */

  /* Vertical box containing buttons */
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(vbox), 5);
  gtk_box_pack_start (GTK_BOX(hbox) , vbox, FALSE, TRUE, 0);

  button = gtk_button_new_with_label (_(" Select all "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (main_button_select_all), (gpointer) wdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Select all files"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_(" Unselect all "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (main_button_unselect_all), (gpointer) wdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Unselected all files"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_(" Add "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (button_add), (gpointer) wdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Add selected files to sendlist"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  wdata->clearbutton = gtk_button_new_with_label (_(" Clear "));
  gtk_widget_set_sensitive (wdata->clearbutton, FALSE);
  gtk_signal_connect (GTK_OBJECT (wdata->clearbutton), "clicked", GTK_SIGNAL_FUNC (button_clear), (gpointer) wdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, wdata->clearbutton, _("Clear list of selected files"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), wdata->clearbutton, TRUE, FALSE, 0);
  gtk_widget_show (wdata->clearbutton);

  wdata->sendbutton = gtk_button_new_with_label (_(" Send "));
  gtk_widget_set_sensitive (wdata->sendbutton, FALSE);
  gtk_signal_connect (GTK_OBJECT (wdata->sendbutton), "clicked", GTK_SIGNAL_FUNC (button_send), (gpointer) wdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, wdata->sendbutton, _("Send the recorded files to receipant"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), wdata->sendbutton, TRUE, FALSE, 0);
  gtk_widget_show (wdata->sendbutton);

  gtk_widget_show (vbox);
  gtk_widget_show (hbox);

  /* Add another box so we get some space between object and border */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(mainbox), hbox);
  gtk_container_border_width (GTK_CONTAINER(hbox), 5);

  /* This should be an additional box containing all selected files */
  wdata->filebox = gtk_clist_new_with_titles (3, titles);
  gtk_clist_column_titles_passive (GTK_CLIST(wdata->filebox));
  gtk_clist_set_selection_mode (GTK_CLIST(wdata->filebox), GTK_SELECTION_SINGLE);
  gtk_clist_set_column_width (GTK_CLIST(wdata->filebox), 0, 16);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->filebox), 0, GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_width (GTK_CLIST(wdata->filebox), 1, 450);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->filebox), 1, GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_width (GTK_CLIST(wdata->filebox), 2, 40);
  gtk_clist_set_column_justification (GTK_CLIST(wdata->filebox), 2, GTK_JUSTIFY_RIGHT);

  /*
  gtk_signal_connect(GTK_OBJECT(wdata->filebox), "select_row", GTK_SIGNAL_FUNC(main_row_select), (gpointer) wdata);
  gtk_signal_connect(GTK_OBJECT(wdata->filebox), "unselect_row", GTK_SIGNAL_FUNC(main_row_unselect), (gpointer) wdata);
  gtk_signal_connect(GTK_OBJECT(wdata->filebox), "button_press_event", GTK_SIGNAL_FUNC(main_row_dblclick), (gpointer) wdata);
  */

#if GTK_MAJOR_VERSION >= 1 && GTK_MINOR_VERSION >= 1
  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX(hbox), scrollwin, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER(scrollwin), wdata->filebox);
  gtk_widget_show (scrollwin);
#else
  gtk_box_pack_start (GTK_BOX(hbox) , wdata->filebox, TRUE, TRUE, 0);
#endif
  gtk_widget_show (wdata->filebox);
  gtk_widget_show (hbox);

  gtk_widget_show (mainbox);
}
