/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"
#include <dirent.h>


static GtkWidget	*config_window;

#define	N_APPLYBUTTONS	10

static GtkWidget	*apply_buttons[N_APPLYBUTTONS];
static gint			n_applybuttons;

static gint			restart_flag;


  /* These names nust track the order of MON_ defines in gkrellm.h
  */
static gchar	*monitor_name[N_MONITORS] =
	{
	"CPU", "Proc", "Disk", "Net", "Inet",
	"MemInfo", "FS", "Mail", "Apm", "Uptime"	/* I don't use these yet */
	};


/* void gtk_adjustment_set_value( GtkAdjustment *adjustment, gfloat value); */


void
create_about_tab(GtkWidget *vbox)
	{
	GtkWidget	*label;
	gchar		buf[512];

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	sprintf(buf, "GKrellM %d.%d.%d\nGNU Krell Monitors\n\n"
				"Copyright (c) 1999 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License",
				VERSION_MAJOR, VERSION_MINOR, VERSION_REV);
	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	}



/* ------------------Misc Settings---------------------------------*/
static GtkWidget	*smp_button[3],
					*disk_spin_button,
					*load_spin_button;

extern Chart		disk;

static GtkWidget	*bar_graph_button,
					*clip_proc_button;


static gint		disk_map[] =
	{ 10, 20, 50, 100, 200, 500 };

static gint		disk_temp;

static void
cb_disk_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint	value;

	value = gtk_spin_button_get_value_as_int(spin);
	if (value != disk_temp)	/* Avoid recursion */
		{
		value = map_1_2_5(value, disk_map, sizeof(disk_map) / sizeof(int));
		disk_temp = value;
		gtk_spin_button_set_value(spin, (gfloat) value);
		}
	}


static void
apply_misc_config()
	{
	GtkSpinButton	*spin;
	gint			i;
	gfloat			f;

	if (smp_cpus > 0)
		for (i = 0; i < 3; ++i)
			if (GTK_TOGGLE_BUTTON(smp_button[i])->active)
				UC.smp_mode = i;

	spin = GTK_SPIN_BUTTON(disk_spin_button);
	i = gtk_spin_button_get_value_as_int(spin);
	UC.disk_resolution = i;

	spin = GTK_SPIN_BUTTON(load_spin_button);
	f = gtk_spin_button_get_value_as_float(spin);
	UC.load_resolution = (gint) (f * 100.0);

	UC.proc_load_bar_graph = GTK_TOGGLE_BUTTON(bar_graph_button)->active;
	UC.proc_clip_processes = GTK_TOGGLE_BUTTON(clip_proc_button)->active;
	}

static void
create_misc_tab(GtkWidget *vbox)
	{
	GtkWidget		*button;
	GtkWidget		*frame;
	GtkWidget		*hbox;
	GtkWidget		*vbox1;
	GtkWidget		*label;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	GSList			*group;
	gchar			buf[80];

	if (smp_cpus > 0)
		{
		frame = gtk_frame_new ("SMP CPU Charts");
		gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

		hbox = gtk_hbox_new (FALSE, 3);
		gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
		gtk_container_add(GTK_CONTAINER(frame), hbox);

		button = gtk_radio_button_new_with_label(NULL, "Real CPUs.");
		gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
		smp_button[0] = button;
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));

		button = gtk_radio_button_new_with_label(group, "Composite CPU.");
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
		smp_button[1] = button;

		button = gtk_radio_button_new_with_label(group, "Composite and real");
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		smp_button[2] = button;

		button = smp_button[UC.smp_mode];
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
		}

	/* Disk misc */
	frame = gtk_frame_new ("Disk");
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	vbox1 = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);
	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_add(GTK_CONTAINER(vbox1), hbox);
	adj = (GtkAdjustment *)gtk_adjustment_new(disk.scale_min,
							10.0, 500.0, 1.0, 1.0, 0);
	disk_spin_button = gtk_spin_button_new (adj, 0.95 /*climb rate*/, 0);
	gtk_widget_set_usize (disk_spin_button, 50, 0);
	spin = GTK_SPIN_BUTTON(disk_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			GTK_SIGNAL_FUNC (cb_disk_resolution), (gpointer) spin);
	gtk_box_pack_start(GTK_BOX(hbox), disk_spin_button, FALSE, TRUE, 5);
	sprintf(buf, "Chart resolution in disk_io's/sec per grid");
	label = gtk_label_new (buf);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 5);

	/* Proc misc */
	frame = gtk_frame_new ("Proc");
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

	vbox1 = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);

	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
	gtk_container_add(GTK_CONTAINER(vbox1), hbox);

	bar_graph_button = gtk_check_button_new_with_label("Load bar graph");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bar_graph_button),
			UC.proc_load_bar_graph);
	gtk_box_pack_start(GTK_BOX(hbox), bar_graph_button, TRUE, TRUE, 0);

	clip_proc_button = gtk_check_button_new_with_label("Clip process data");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(clip_proc_button),
			UC.proc_clip_processes);
	gtk_box_pack_start(GTK_BOX(hbox), clip_proc_button, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_add(GTK_CONTAINER(vbox1), hbox);
	adj = (GtkAdjustment *)gtk_adjustment_new(
				(gfloat) UC.load_resolution / 100.0, 0.5, 5.0, 0.5, 0.5, 0);
	load_spin_button = gtk_spin_button_new (adj, 0.95 /*climb rate*/, 1);
	gtk_widget_set_usize (load_spin_button, 50, 0);
	spin = GTK_SPIN_BUTTON(load_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), load_spin_button, FALSE, TRUE, 5);
	label = gtk_label_new(
		"Load resolution in average process load per minute per grid");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 5);

#if 0
	label = gtk_label_new (
	"Load resolution is 1 process/minute load per grid. Process creation\n"
	"has 20 forks/sec per grid. Krell full scale is 100 forks/sec rate.\n");
#endif

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	}


/* ------------------General Settings---------------------------------*/
static
GtkWidget	*enable_button[N_MONITORS],
			*enable_hst_button,
			*enable_clk_button,
			*clk_24hr_button,
			*clk_12hr_sec_button,
			*hostname_short_button,
			*enable_extra_info_button,
			*save_position_button,
			*on_top_button,
			*fixed_scale_spin_button,
			*update_HZ_spin_button;

void
apply_general_config()
	{
	gint	n;

	UC.enable_hostname = GTK_TOGGLE_BUTTON (enable_hst_button)->active;
	UC.hostname_short = GTK_TOGGLE_BUTTON (hostname_short_button)->active;
	UC.enable_clock = GTK_TOGGLE_BUTTON (enable_clk_button)->active;
	UC.clock_24hr_format = GTK_TOGGLE_BUTTON (clk_24hr_button)->active;
	UC.clock_12hr_seconds = GTK_TOGGLE_BUTTON (clk_12hr_sec_button)->active;
	UC.enable[MON_PROC] = GTK_TOGGLE_BUTTON (enable_button[MON_PROC])->active;
	UC.enable[MON_DISK] = GTK_TOGGLE_BUTTON (enable_button[MON_DISK])->active;

	UC.enable_extra_info = GTK_TOGGLE_BUTTON(enable_extra_info_button)->active;
	UC.save_position = GTK_TOGGLE_BUTTON(save_position_button)->active;
	UC.fixed_scale = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(fixed_scale_spin_button));
	n = UC.update_HZ;
	UC.update_HZ = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(update_HZ_spin_button));
	if (n != UC.update_HZ)
		start_timer();
	if (GK.on_top_hint_ok)
		{
		n = GTK_TOGGLE_BUTTON(on_top_button)->active;
		if (n != UC.on_top)
			set_on_top(n);
		UC.on_top = n;
		}
	}

static gchar	*general_info_text	=
"    Krells\n"
"Krells are the horizontally moving indicators below each chart and\n"
"on meter style monitors.  Depending on the monitor, they show fast\n"
"response data rates or a percentage of some capacity.\n\n"
"    Charts\n"
"Charts can be configured to auto scale or to have a fixed number of\n"
"1 to 5 grids. In either case, chart resolution is set by specifying\n"
"a value per grid.\n"
"   CPU charts - Data is plotted as a percentage.  In auto scale mode,\n"
"resolution is 20% per grid and the krell full scale value is 100%.\n"
"Fixed scale grid resolution is 100 divided by the number of grids.\n"
"User time is plotted in the \"in\" color, sys time in the \"out\" color.\n"
"   Proc chart - The krell shows process forks with a full scale value\n"
"of 10 forks.  The chart has a resolution of 10 forks/sec per grid in\n"
"auto scale mode and 50 forks/second max on fixed scale charts.\n"
"The process load resolution per grid is best left at 1.0 for auto\n"
"scaling, but can be set as high as 5 if you use fixed scale charts\n"
"with only 1 or 2 grids.  Forks are plotted in the \"in\" color, load\n"
"is plotted in the \"out\" color.\n"

"   Inet charts - The krell for this chart samples TCP port hits once\n"
"per second with a full scale value of 5 hits.  Grid resolutions for\n"
"the minute and hour chart are not linked to the krell full scale\n"
"value.  See Internet->Info\n"
"   Disk and Net charts - In autoscale mode a grid resolution should\n"
"be selected that is about 1/5 the expected maximum data rate\n"
"since the krell full scale value is set to 5 times the grid resolution.\n"
"For fixed scaling, the krell full scale value is set to the selected\n"
"number of grids times the grid resolution.  The \"in\" color is used\n"
"for disk read data and net receive bytes, while disk write data and\n"
"net transmit bytes are plotted in the \"out\" color.\n"
"Example: Your ppp0 net interface is a 56k modem and data transfer\n"
"rates are typically less than 10000 bytes/sec.  You could select\n"
"auto scaling charts and set grid resolution to 2000 bytes/sec per\n"
"grid, or you could select 2 fixed grids and set grid resolution to\n"
"5000 bytes/sec per grid. In both cases the krell full scale value is\n"
"10000 bytes/sec rate.\n\n"

"    Notes on Options\n"
"Draw extra info on charts at startup - The extra info can also be\n"
"toggled for each chart by left clicking on the chart.\n"
"Remember screen location - This is useful if you do not configure\n"
"your window manager to start GKrellM at a specific location.  This\n"
"option can be overridden with the -g geometry command line option.\n"
"Krell and LED updates per second - Set this to a lower value if you\n"
"run a slower CPU or run GKrellM over a network and want to save\n"
"some bandwith.\n\n"

"GkrellM can be put into a hard disk spindown friendly state by\n"
"turning off the extra info draw of \"users\" on the proc chart and\n"
"by muting the mail check with the \"Mute mode inhibits all mail\n"
"checking\" option in the Mail tab selected\n"
;

void
create_general_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox;
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Enables tab */
	vbox = create_tab(tabs, "Enables");
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	enable_hst_button = gtk_check_button_new_with_label("Hostname Label");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_hst_button),
						UC.enable_hostname);
	gtk_box_pack_start(GTK_BOX(hbox), enable_hst_button, FALSE, TRUE, 0);
	hostname_short_button = gtk_check_button_new_with_label(
						"Short hostname");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hostname_short_button),
						UC.hostname_short);
	gtk_box_pack_start(GTK_BOX(hbox), hostname_short_button, TRUE, TRUE, 10);


	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	enable_clk_button = gtk_check_button_new_with_label("Clock/Calendar");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_clk_button),
						UC.enable_clock);
	gtk_box_pack_start(GTK_BOX(hbox), enable_clk_button, FALSE, TRUE, 0);
	clk_12hr_sec_button = gtk_check_button_new_with_label(
						"Show seconds instead of am/pm");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(clk_12hr_sec_button),
						UC.clock_12hr_seconds);
	gtk_box_pack_start(GTK_BOX(hbox), clk_12hr_sec_button, TRUE, TRUE, 5);
	clk_24hr_button = gtk_check_button_new_with_label(
						"24 hour clock");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(clk_24hr_button),
						UC.clock_24hr_format);
	gtk_box_pack_start(GTK_BOX(hbox), clk_24hr_button, TRUE, TRUE, 5);


	enable_button[MON_PROC] = gtk_check_button_new_with_label("Proc Monitor");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_button[MON_PROC]),
						UC.enable[MON_PROC]);
	gtk_box_pack_start(GTK_BOX(vbox), enable_button[MON_PROC], TRUE, TRUE, 0);

	enable_button[MON_DISK] = gtk_check_button_new_with_label("Disk Monitor");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_button[MON_DISK]),
						UC.enable[MON_DISK]);
	gtk_box_pack_start(GTK_BOX(vbox), enable_button[MON_DISK], TRUE, TRUE, 0);

/* --Options tab */
	vbox = create_tab(tabs, "Options");
	enable_extra_info_button = gtk_check_button_new_with_label(
			"Draw extra info on charts at startup");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_extra_info_button),
						UC.enable_extra_info);
	gtk_box_pack_start(GTK_BOX(vbox), enable_extra_info_button, TRUE, TRUE, 0);

	save_position_button = gtk_check_button_new_with_label(
			"Remember screen location at exit and move to it at next startup");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(save_position_button),
						UC.save_position);
	gtk_box_pack_start(GTK_BOX(vbox), save_position_button, TRUE, TRUE, 0);

	if (GK.on_top_hint_ok)
		{
		on_top_button = gtk_check_button_new_with_label(
				"Set always on top window manager hint");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(on_top_button),
							UC.on_top);
		gtk_box_pack_start(GTK_BOX(vbox), on_top_button, TRUE, TRUE, 0);
		}

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new ((gfloat) UC.update_HZ,
			4.0, 10.0, 1.0, 1.0, 0.0);
	update_HZ_spin_button = gtk_spin_button_new(adj, 0.5, 0);
	spin = GTK_SPIN_BUTTON(update_HZ_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), update_HZ_spin_button, FALSE, TRUE, 0);
	label = gtk_label_new("Krell and LED updates per second.");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	adj = (GtkAdjustment *) gtk_adjustment_new ((gfloat) UC.fixed_scale,
			0.0, 5.0, 1.0, 1.0, 0.0);
	fixed_scale_spin_button = gtk_spin_button_new(adj, 0.5, 0);
	spin = GTK_SPIN_BUTTON(fixed_scale_spin_button);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), fixed_scale_spin_button, FALSE, TRUE, 0);
	label = gtk_label_new(
	"0: Autoscale charts    1-5: Number of grids for fixed scale charts");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);

/* --Info tab */
	vbox = create_tab(tabs, "Info");
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, general_info_text, -1);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
	}



/* ------------------Sizes Settings----------------------------------*/

static GtkWidget	*heights_equal_button,
					*spin_button_w,
					*spin_button_h[N_CHART_MONITORS];

void
cb_equal_button()
	{
	gint	value, i;

	value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_h[0]));
	if (value)
		for (i = 1; i < N_CHART_MONITORS; ++i)
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button_h[i]),
					(gfloat) value);
	}

void
cb_size_changed(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint	i, value;

	if (   spin != GTK_SPIN_BUTTON(spin_button_w)
		&& GTK_TOGGLE_BUTTON(heights_equal_button)->active
	   )
		{
		value = gtk_spin_button_get_value_as_int(spin);
		for (i = 0; i < N_CHART_MONITORS; ++i)
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button_h[i]),
							(gfloat) value);
		}
	if (restart_flag == FALSE)
		for (i = 0; i < n_applybuttons; ++i)
			gtk_widget_set_sensitive(apply_buttons[i], FALSE);
	restart_flag = TRUE;
	}

static void
apply_sizes_config()
	{
	gint	i;

	UC.chart_width =
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_w));
	for (i = 0; i < N_CHART_MONITORS; ++i)
		UC.chart_height[i] =
		   gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_button_h[i]));
	}

void
create_sizes_tab(GtkWidget *vbox)
	{
	GtkWidget		*frame;
	GtkWidget		*hbox, *hbox1;
	GtkWidget		*vbox1;
	GtkWidget		*label;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	gint			i, heights_equal;

	heights_equal = TRUE;
	for (i = 1; i < N_CHART_MONITORS; ++i)
		if (UC.chart_height[i] != UC.chart_height[0])
			heights_equal = FALSE;

	hbox1 = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox1), 5);
	gtk_container_add(GTK_CONTAINER(vbox), hbox1);

	vbox1 = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 0);
	
	frame = gtk_frame_new ("Chart Width");
	gtk_box_pack_start(GTK_BOX(vbox1), frame, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);

	adj = (GtkAdjustment *) gtk_adjustment_new ((gfloat)UC.chart_width,
		(gfloat) GK.chart_width_min, (gfloat) GK.chart_width_max,
		1.0 /*step*/, 5.0 /*page*/, 0.0);
	spin_button_w = gtk_spin_button_new (adj,
				0.5 /*gfloat climb_rate*/, 0 /*guint digits*/);
	spin = GTK_SPIN_BUTTON (spin_button_w);
	gtk_spin_button_set_numeric(spin, TRUE);
	gtk_spin_button_set_shadow_type(spin, GTK_SHADOW_ETCHED_IN);
	gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
			GTK_SIGNAL_FUNC (cb_size_changed), (gpointer) spin);

	gtk_box_pack_start (GTK_BOX (hbox), spin_button_w, FALSE, TRUE, 0);
	label = gtk_label_new ("All Charts");
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

	label = gtk_label_new(
			"Clicking OK after changing a size\n"
			"will force a restart.");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, TRUE, 0);

	frame = gtk_frame_new ("Chart Heights");
	gtk_box_pack_start(GTK_BOX(hbox1), frame, TRUE, TRUE, 0);
	vbox1 = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 2);
	gtk_container_add(GTK_CONTAINER(frame), vbox1);

	for (i = 0; i < N_CHART_MONITORS; ++i)
		{
		hbox = gtk_hbox_new (FALSE, 5);
		gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);
		gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 0);

		adj = (GtkAdjustment *) gtk_adjustment_new (
			(gfloat) UC.chart_height[i],
			(gfloat) GK.chart_height_min, (gfloat) GK.chart_height_max,
			1.0 /*step*/, 5.0 /*page*/, 0.0);
		spin_button_h[i] = gtk_spin_button_new (adj, 0.5, 0);
		spin = GTK_SPIN_BUTTON(spin_button_h[i]);
		gtk_spin_button_set_numeric(spin, TRUE);
		gtk_spin_button_set_shadow_type(spin, GTK_SHADOW_ETCHED_IN);
		gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
				GTK_SIGNAL_FUNC (cb_size_changed), (gpointer) spin);

		gtk_box_pack_start (GTK_BOX (hbox), spin_button_h[i], FALSE, TRUE, 0);
		label = gtk_label_new (monitor_name[i]);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
		}
	heights_equal_button = gtk_check_button_new_with_label(
				"Keep heights equal.");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(heights_equal_button),
				heights_equal);
	gtk_box_pack_start(GTK_BOX(vbox), heights_equal_button, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(heights_equal_button),"clicked",
				(GtkSignalFunc) cb_equal_button, NULL);
	}

/* ------------------Themes Tab----------------------------------*/

GtkWidget		*default_theme_entry;
static gint		theme_width, author_width;

static gchar *
get_theme_author(gchar *path)
	{
	static gchar	buf[80];
	FILE			*f;
	gchar			*s, *q, rcfile[128], line[128];

	buf[0] = '\0';
	sprintf(rcfile, "%s/%s", path, GKRELLMRC);
	if ((f = fopen(rcfile, "r")) == NULL)
		return buf;
	while (fgets(line, sizeof(line), f))
		{
		if (   (s = strtok(line, " :=\t\n")) == NULL
			|| strcmp(s, "author") != 0
		   )
			continue;
		s = strtok(NULL, "\n");		/* Rest of line is Author string */
		if (s)
			{
			while (   *s == ' ' || *s == '\t' || *s == '"' || *s == '='
					|| *s == ':')
				++s;
			q = strchr(s, (int) '"');
			if (q)
				*q = '\0';
			strcpy(buf, s);
			break;
			}
		}
	fclose(f);
	return buf;
	}

static void
scan_for_themes(GtkWidget *clist)
	{
	DIR				*dir;
	struct dirent	*dentry;
	gchar			path[256], gkrellmdir[128];
	gchar			*buf[3], *s;
	gint			row, l;

	sprintf(gkrellmdir, "%s/%s", homedir(), GKRELLM_DIR);

	buf[0] = "Default";
	buf[1] = "";
	buf[2] = NULL;
	theme_width = 10;
	author_width = 10;

	row = gtk_clist_append(GTK_CLIST(clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(clist), row, "");

	if ((dir = opendir(gkrellmdir)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] != '.' && dentry->d_ino > 0)
			{
			sprintf(path, "%s/%s", gkrellmdir, dentry->d_name);
			if (isdir(path) && strcmp(dentry->d_name, GKRELLM_DATA_DIR))
				{
				buf[0] = dentry->d_name;
				buf[1] = get_theme_author(path);
				row = gtk_clist_append(GTK_CLIST(clist), buf);
				s = g_strdup(path);
				gtk_clist_set_row_data_full(GTK_CLIST(clist), row,
							s, (GtkDestroyNotify) g_free);
				if ((l = strlen(dentry->d_name)) > theme_width)
					theme_width = l;
				if ((l = strlen(buf[1])) > author_width)
					author_width = l;
				}
			}
		}
	closedir(dir);
	gtk_clist_set_sort_column(GTK_CLIST(clist), 0);
	gtk_clist_sort(GTK_CLIST(clist));
	}


static void
cb_theme_selected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	gchar	*theme;
	gint	i;

	if ((theme = gtk_clist_get_row_data(GTK_CLIST(clist), row)) != NULL)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), theme);

	if (restart_flag == FALSE)
		for (i = 0; i < n_applybuttons; ++i)
			gtk_widget_set_sensitive(apply_buttons[i], FALSE);
	restart_flag = TRUE;
	}

void
apply_theme_config()
	{
	FILE	*f;
	gchar	*new_theme, path[128];

	new_theme = gtk_entry_get_text(GTK_ENTRY(default_theme_entry));
	if (strcmp(new_theme, GK.theme_path))
		{
		if (GK.debug)
			printf("Writting new gkrellm_theme.cfg <%s>\n", new_theme);
		sprintf(path, "%s/%s", homedir(), GKRELLM_THEME);
		restart_flag = TRUE;
		if ((f = fopen(path, "w")) != NULL)
			{
			fprintf(f, "%s\n", new_theme);
			fclose(f);
			}
		exec_theme_path = NULL;			/* Don't reuse command line theme */
		}
	}

gchar	*themes_titles[] = {"Theme", "Author" };

void
create_theme_tab(GtkWidget *vbox)
	{
	GtkWidget	*hbox;
	GtkWidget	*label;
	GtkWidget	*scrolled;
	GtkWidget	*theme_clist;
	gchar		buf[128];

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	label = gtk_label_new("Theme:");
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	default_theme_entry = gtk_entry_new_with_max_length(48);
	if (GK.theme_path)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), GK.theme_path);
	gtk_box_pack_start(GTK_BOX(hbox), default_theme_entry, TRUE, TRUE,0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

	theme_clist = gtk_clist_new_with_titles(2, themes_titles);
	gtk_signal_connect (GTK_OBJECT (theme_clist), "select_row",
			(GtkSignalFunc) cb_theme_selected, NULL);
	gtk_clist_set_shadow_type(GTK_CLIST(theme_clist), GTK_SHADOW_OUT);
	gtk_clist_column_titles_passive(GTK_CLIST(theme_clist));
	gtk_container_add (GTK_CONTAINER (scrolled), theme_clist);

	scan_for_themes(theme_clist);

	/* How can I get a string width for the default gtk font? */
	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 0, theme_width * 7 + 8);
	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 1, author_width * 7);

	sprintf(buf, "Untar your theme tar files in %s/.gkrellm", homedir());
	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label = gtk_label_new(
			"Download themes from the GKrellM theme site at www.muhri.net");
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	}


/* -------------------------------------------------------------------*/

void
close_config()
	{
	gtk_widget_destroy(config_window);
	config_window = NULL;
	}


static void
apply_config()
	{
	apply_general_config();
	apply_misc_config();

	/* These applies are in respective source files to handle their
	|  local data.
	*/
	apply_fs_config();
	apply_inet_config();
	apply_net_config();
	apply_sensors_config();
	apply_stat_config();
	apply_mail_config();
	apply_hostname_config();
	apply_clock_config();

	/* The FS and Inet monitors do their own internal applies when
	|  their save buttons are activated.  Might change this later??
	*/

	/* If these two applies have changes, then a restart will be forced.
	*/
	apply_sizes_config();
	apply_theme_config();

	save_user_config();
	}

static void
OK_config()
	{
	apply_config();
	close_config();
	if (restart_flag)
		{
		restart_gkrellm();
		/* NOT REACHED */
		}
	}


#define	OK_BUTTON		1
#define	APPLY_BUTTON	2
#define	CLOSE_BUTTON	4
#define	ALL_BUTTONS		(OK_BUTTON | APPLY_BUTTON | CLOSE_BUTTON)

void
default_control_buttons(GtkWidget *vbox, gint buttons)
	{
	GtkWidget	*hbox;
	GtkWidget	*button;

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 3);

	if (buttons & OK_BUTTON)
		{
		button = gtk_button_new_with_label("Ok");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) OK_config, NULL);
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);
		}
	if (buttons & APPLY_BUTTON)
		{
		button = gtk_button_new_with_label("Apply");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) apply_config, NULL);
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);
		if (n_applybuttons < N_APPLYBUTTONS - 1)
			apply_buttons[n_applybuttons++] = button;
		}
	if (buttons & CLOSE_BUTTON)
		{
		button = gtk_button_new_with_label("Close");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) close_config, NULL);
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);
		}
	}

GtkWidget *
create_tab(GtkWidget *tabs, char *name)
	{
	GtkWidget	*frame;
	GtkWidget	*label;
	GtkWidget	*vbox;

	frame = gtk_frame_new(NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 3);
	gtk_widget_show(frame);

	label = gtk_label_new(name);
	gtk_notebook_append_page(GTK_NOTEBOOK(tabs), frame, label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 3);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	return vbox;
	}

void
create_config_window()
	{
	GtkWidget	*vbox;
	GtkWidget	*tabs;
	
	if (config_window)
		{
		gdk_window_raise(config_window->window);
		return;
		}
	restart_flag = FALSE;
	n_applybuttons = 0;

	config_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(config_window), "delete_event",
			(GtkSignalFunc) close_config, NULL);
	gtk_window_set_policy(GTK_WINDOW (config_window), FALSE, FALSE, TRUE);
	gtk_window_set_title(GTK_WINDOW (config_window), "Configuration");
	gtk_container_border_width(GTK_CONTAINER(config_window), 3);
	
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(config_window), vbox);
	gtk_widget_show(vbox);


	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(vbox), tabs, TRUE, TRUE, 0);

	vbox = create_tab(tabs, "General");
	create_general_tab(vbox);
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "Misc");
	create_misc_tab(vbox);
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "FS");
	create_fs_tab(vbox);				/* In fs.c */
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "Internet");
	create_inet_tab(vbox);				/* In inet.c */
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "Net");
	create_net_tab(vbox);				/* In net.c */
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "Mail");
	create_mail_tab(vbox);
	default_control_buttons(vbox, ALL_BUTTONS);
	gtk_widget_show_all(vbox);

	if (GK.sensor_temp_files)
		{
		vbox = create_tab(tabs, "Sensors");
		create_sensors_tab(vbox);		/* In sensors.c */
		default_control_buttons(vbox, ALL_BUTTONS);
		gtk_widget_show_all(vbox);
		}

	vbox = create_tab(tabs, "Sizes");
	create_sizes_tab(vbox);
	default_control_buttons(vbox, OK_BUTTON | CLOSE_BUTTON);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "Themes");
	create_theme_tab(vbox);
	default_control_buttons(vbox, OK_BUTTON | CLOSE_BUTTON);
	gtk_widget_show_all(vbox);

	vbox = create_tab(tabs, "About");
	create_about_tab(vbox);
	default_control_buttons(vbox, OK_BUTTON | CLOSE_BUTTON);
	gtk_widget_show_all(vbox);


	gtk_widget_show(tabs);
	gtk_widget_show(config_window);
	}


GtkWidget *
create_popup_menu()
	{
	GtkWidget	*menu;
	GtkWidget	*button;

	menu = gtk_menu_new();

	button = gtk_menu_item_new_with_label("Configuration");
	gtk_signal_connect(GTK_OBJECT(button), "activate",
				(GtkSignalFunc) create_config_window, NULL);
	gtk_menu_append(GTK_MENU(menu), button);
	gtk_widget_show(button);

	button = gtk_menu_item_new_with_label("Quit");
	gtk_signal_connect(GTK_OBJECT (button), "activate",
				(GtkSignalFunc) gtk_main_quit, NULL);
	gtk_menu_append(GTK_MENU(menu),button);
	gtk_widget_show(button);

	return menu;
	}
