/* napster.c - all napster-related protocol stuff */

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include <gnome.h>

#include "napster.h"
#include "gnapster.h"
#include "network.h"
#include "ui.h"

extern CONN_INFO conn_info;
extern MOTD_TAB motd_tab;
extern SEARCH_TAB search_tab;
extern CONSOLE_TAB console_tab;
extern CHAT_TAB chat_tab;
extern USER_INFO user_info;

char *conn_table[] = {
   "Unknown", "14.4", "28.8", "33.6", "56k", "64k ISDN", "128k ISDN",
   "Cable", "DSL", "T1", "T3+", NULL
};

void napster_handle_data(char c, char *data) {
   char temp_str[1024];
   
   switch(c) {
    case NAPSTER_MESSAGE:
      sprintf(temp_str, "%s\n", data);
      gnapster_text_insert(motd_tab.text, motd_tab.vbox, temp_str);
      break;
    case NAPSTER_STATUS_MESSAGE:
	{
	   int songs, users, gigs;
      
	   sscanf(data, "%i %i %i", &users, &songs, &gigs);
	   g_snprintf(temp_str, sizeof(temp_str), 
		      "Connected (%s:%i): %i songs, %i users, %i gigs",
		      conn_info.ip, conn_info.port, songs, users, gigs);
	   gnapster_update_status(temp_str, 0);
	   if (conn_info.status) free(conn_info.status);
	   conn_info.status = strdup(temp_str);
	}
      break;
    case NAPSTER_USER_MESSAGE_IN:
	{
	   char *user, *message;
	   user = strdup(data);
	   message = strchr(user, ' ');
	   if (!message) break;
	   message[0] = '\0';
	   message++;
	   g_snprintf(temp_str, sizeof(temp_str), "<%s> %s\n", 
		      user, message);
	   gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	   free(user);
	}
      break;
    case NAPSTER_CHAN_JOIN:
      if (user_info.chan) free(user_info.chan);
      user_info.chan = strdup(data);
      gtk_clist_clear(GTK_CLIST(chat_tab.clist));
      gtk_clist_freeze(GTK_CLIST(chat_tab.clist));
      break;
    case NAPSTER_CHAN_USERLIST:
	{
	   char *chan, *user, *bs;
	   chan = strdup(data);
	   user = strchr(chan, ' ');
	   if (!user) break;
	   user[0] = '\0';
	   user++;
	   bs = strchr(user, ' ');
	   if (!bs) break;
	   bs[0] = '\0';
	   bs++;
	     {
		char *add_id[] = { user, NULL };
		gtk_clist_append(GTK_CLIST(chat_tab.clist), add_id);
	     }
	}
      break;
    case NAPSTER_CHAN_USERLIST_END:
      gtk_clist_thaw(GTK_CLIST(chat_tab.clist));
      break;
    case NAPSTER_CHAN_TOPIC:
      g_snprintf(temp_str, sizeof(temp_str), "Topic is: %s\n", data);
      gnapster_text_insert(chat_tab.text, chat_tab.vbox, temp_str);
      break;
    case NAPSTER_WHOIS_REPLY:
	{
	   int i = 0, flag = 0, x = 0, z = 0;
	   char values[10][80];
	   char *value_table[] = {
	      "User", "Account", "Idle Time", "Channels", "Status",
	      "Sharing", "Downloads", "Uploads", "Connection", "Client",
	      NULL
	   };

	   while(data[i]) {
	      if (data[i] == '\"' && !flag) flag = 1;
	      else if (data[i] == '\"' && flag) flag = 0;
	      else if (data[i] == ' ' && !flag) {
		 values[x][z] = '\0';
		 z = 0;
		 x++;
	      } else values[x][z++] = data[i];
	      i++;
	   }
	   values[x][z] = '\0';
	   
	   for(i=0; i<=9; i++) {
	      if (!values[i]) return;
	   }
	   
	   gtk_text_freeze(GTK_TEXT(console_tab.text));
	   for(i=0; i<=9; i++) {
	      if (!strcmp(value_table[i], "Idle Time")) {
		 int idle_time = atoi(values[i]);
		 g_snprintf(values[i], sizeof(values[i]), "%i m %i s", 
			    idle_time / 60, idle_time % 60);
	      } else if (!strcmp(value_table[i], "Connection")) {
		 int conn = atoi(values[i]);
		 g_snprintf(values[i], sizeof(values[i]), 
			    napster_conn_as_str(conn));
	      }
	      g_snprintf(temp_str, sizeof(temp_str), "| %s: %s\n", 
			 value_table[i], values[i]);
	      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	   }
	   gtk_text_thaw(GTK_TEXT(console_tab.text));
	}
      break;
    case NAPSTER_WHOIS_OFFLINE:
	{
	   char *user, *account;
	   
	   user = strdup(data);
	   account = strchr(user, ' ');
	   if (account) {
	      account[0] = '\0';
	      g_snprintf(temp_str, sizeof(temp_str), "%s is offline.\n", 
			 user);
	      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	   }
	   free(user);
	}
      break;
    case NAPSTER_USER_ERROR:
      g_snprintf(temp_str, sizeof(temp_str), "* Error: %s\n", data);
      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
      break;
    case NAPSTER_SEARCH_RESULT:
	{
	   NAP_SEARCH *search;
	   char temp_str2[5];
	   
	   search = gnapster_parse_search(data);
	   if (!search) break;
	   
	   g_snprintf(temp_str, sizeof(temp_str), "%.02fM", ((float)search->size / 1024) / 1024);
	   g_snprintf(temp_str2, sizeof(temp_str2), "%i", search->kbps);
	   gnapster_clist_append(search_tab.clist, search_tab.vbox,
				 search, search->trunc_file,
				 napster_conn_as_str(search->connection),
				 search->user, temp_str, temp_str2);
	}
      break;
    case NAPSTER_END_SEARCH:
      gnapster_update_status(conn_info.status, 0);
      gtk_entry_set_text(GTK_ENTRY(search_tab.entry), "");
      gtk_entry_set_editable(GTK_ENTRY(search_tab.entry), TRUE);
      gtk_clist_sort(GTK_CLIST(search_tab.clist));
      gtk_clist_thaw(GTK_CLIST(search_tab.clist));
      if (g_list_length(GTK_CLIST(search_tab.clist)->row_list) == 0) {
	 gnapster_text_insert(console_tab.text, console_tab.vbox,
			      "No search results found.\n");
      }
      break;
    case NAPSTER_BROWSE_RESULT:
	{
	   NAP_SEARCH *browse;
	   char temp_str2[5];
	   
	   browse = gnapster_parse_browse(data);
	   if (!browse) break;
	   
	   g_snprintf(temp_str, sizeof(temp_str), "%.02fM", ((float)browse->size / 1024) / 1024);
	   g_snprintf(temp_str2, sizeof(temp_str2), "%i", browse->kbps);
	   gnapster_clist_append(search_tab.clist, search_tab.vbox,
				 browse, browse->trunc_file,
				 napster_conn_as_str(browse->connection),
				 browse->user, temp_str, temp_str2);
	}
      break;
    case NAPSTER_END_BROWSE:
      gnapster_update_status(conn_info.status, 0);
      gtk_entry_set_text(GTK_ENTRY(search_tab.entry), "");
      gtk_entry_set_editable(GTK_ENTRY(search_tab.entry), TRUE);
      gtk_clist_thaw(GTK_CLIST(search_tab.clist));
      if (g_list_length(GTK_CLIST(search_tab.clist)->row_list) == 0) {
	 gnapster_text_insert(console_tab.text, console_tab.vbox,
			      "No mp3s found.\n");
      }
      break;
    case NAPSTER_BROWSE_ERROR:
      g_snprintf(temp_str, sizeof(temp_str), 
		 "* Error: User is not online\n");
      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
      break;
    case NAPSTER_DOWNLOAD_REPLY:
	{
	   NAP_DOWNLOAD *download;
	   
	   download = gnapster_parse_download(data);
	   if (!download) break;
	   
	   printf("Received download reply from %s (for %s)\n",
		  download->user, download->file);
	   
	   if (download->port == 0) {
	      /* They will connect to me... */
	      printf("download port is 0...using new method\n");
	      g_snprintf(temp_str, sizeof(temp_str), "%s \"%s\"", 
			 download->user, download->file);
	      napster_send(conn_info.sock, "\364\1", temp_str);
	      printf("they should be connecting soon...\n");
	   } else {
	      /* Connect to them */
	      napster_connect_user(download);
	   }
	}
      break;
    case NAPSTER_DOWNLOAD_ERROR:
	{
	   char *user, *bs;
	   
	   user = strdup(data);
	   bs = strchr(user, ' ');
	   if (bs) {
	      bs[0] = '\0';
	      g_snprintf(temp_str, sizeof(temp_str), "%s has disconnected.\n", user);
	      gnapster_text_insert(console_tab.text, console_tab.vbox, temp_str);
	   }
	}
      break;
    default:
      if (strstr(data, "username contains invalid characters")) {
	 gnome_error_dialog("Invalid Username!  Open the properties and try again");
      } else if (strstr(data, "invalid password")) {
	 gnome_error_dialog("Invalid Password!  Open the properties and try again");
      } else {
	 printf("unknown[%i] -> %s\n", c, data);
      }
      break;
   }
}

void napster_get_best_host() {
   int sock;
   struct sockaddr_in server;
   
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (sock < 0) {
      printf("socket error\n");
      /* Error handle */
      return;
   }
   
   server.sin_family = AF_INET;
   server.sin_port = htons(8875);
   /* FOR DEBUGGING PURPOSES ONLY */
   server.sin_addr.s_addr = inet_addr("208.184.216.223");
//   server.sin_addr.s_addr = inet_addr("208.178.163.58");
   
   fcntl(sock, F_SETFL, O_NONBLOCK);
   connect(sock, (struct sockaddr *)&server, sizeof(server));
   if (errno != EINPROGRESS) {
      gnapster_update_status("Error connecting to server.", 0);
      close(sock);
      return;
      printf("bad\n");
   }
   
   conn_info.sock_input = gdk_input_add(sock, GDK_INPUT_WRITE,
					network_conn_cb, NULL);
}

void napster_connect(char *ip, int port) {
   int sock;
   struct sockaddr_in server;
   
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (sock < 0) {
      /* error check */
      printf("bogus\n");
      return;
   }
   
   server.sin_family = AF_INET;
   server.sin_port = htons(port);
   server.sin_addr.s_addr = inet_addr(ip);
   
   fcntl(sock, F_SETFL, O_NONBLOCK);
   connect(sock, (struct sockaddr *)&server, sizeof(server));
   if (errno != EINPROGRESS) {
      gnapster_update_status("Error connecting to best server.", 0);
      printf("bad\n");
      close(sock);
      return;
   }

   conn_info.sock_input = gdk_input_add(sock, GDK_INPUT_WRITE,
					network_conn_real_cb, NULL);   
}

void napster_connect_user(NAP_DOWNLOAD *download) {   
   int sock;
   
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (sock < 0) {
      /* error check */
      printf("bogus\n");
      return;
   }
   
   download->saddr.sin_family = AF_INET;
   download->saddr.sin_port = htons(download->port);
   download->saddr.sin_addr.s_addr = download->ip;
   
   fcntl(sock, F_SETFL, O_NONBLOCK);
   connect(sock, (struct sockaddr *)&download->saddr, sizeof(download->saddr));
   if (errno != EINPROGRESS) {
//      gnapster_update_status("Error connecting to best server.", 0);
      printf("bad\n");
      close(sock);
      return;
   }
   
   download->sock = sock;
   download->header_val = 0;
   download->size = 0;
   download->sock_input = gdk_input_add(sock, GDK_INPUT_WRITE,
					network_conn_user_cb, download);
}

void napster_bind(int port) {
   int sock, len = 0;
   struct sockaddr_in server;
   
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (sock < 0) {
      printf("socket error!\n");
      return;
   }
   
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = htonl(INADDR_ANY);
   server.sin_port = htons(port);
   
   setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &len, 4);
   
   if (bind(sock, (struct sockaddr *)&server, sizeof(server)) < 0) {
      close(sock);
      perror("bind");
//      gnome_error_dialog("An error occured while attempting to bind port 6699.  It is possible that another napster client is running.");
      return;
   }
   listen(sock, 0);
   
   conn_info.bind_sock = sock;
   conn_info.bind_sock_input = gdk_input_add(conn_info.bind_sock,
					     GDK_INPUT_READ,
					     network_handle_connection,
					     NULL);
   
   return;
}

void napster_send_get(int sock, char *user, char *file, int conn) {
   char *temp_str;
   send(sock, "GET", 3, 0);
   temp_str = g_strdup_printf("%s \"%s\" %i", user, file, conn);
   send(sock, temp_str, strlen(temp_str), 0);
   free(temp_str);
}

void napster_send(int sock, char *cmd, char *data) {
   char size_str[2];
   
   sprintf(size_str, "%c", strlen(data));
   
   send(sock, size_str, 2, 0);
   send(sock, cmd, 2, 0);
   
   send(sock, data, strlen(data), 0);
}

int napster_conn_as_int(char *s) {
   int x = 0;
   
   while(conn_table[x]) {
      if (!strcmp(conn_table[x], s)) return x;
      x++;
   }
   
   return -1;
}

char *napster_conn_as_str(int i) {
   if (i < 0 || i > 10) return "Unknown";
   return conn_table[i];
}
