/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "prototypes.h"
#include <tcl.h>
#include <tk.h>
#include <string.h>

extern void debug(const char *fmt,...);		/* output to standard error stream command */
extern void time_copy();
extern int tx_fd;
queue sendq;
queue rtxq;
extern sliding_key skey;
extern user_data me;
extern char hostname[];
extern Tcl_Interp *interp;
extern u_int8 cur_col;
extern char sessionname[];


extern struct in_addr ipaddr;	/* variable storing our local ip address */

extern page *p;			/* variable that refers to the main page. */

extern char vanityname[];
extern char os_version[];
extern char nt_version[];
void init_jim_rtx();
extern int new_data;

/*used when we need to send a line message about a line we
   don't know about */
line unknown_line;

extern struct timeval rtx_time;

int send_queued_packet();
extern unsigned int subtract_time();
char *get_missing_id();

extern int checksum_active;

const int NORMAL_S_MESSAGE = 0;	/* normal session message type */
const int COLOUR_CHANGE = 1;	/* session message to signal user */
												/*   changing colour only. */

int s_message_type;

void init_header(ntheader * header, unsigned char type, struct timeval *t)
{
    header->checkbyte = (unsigned char) NT_PROTOCOL_CHECKBYTE;
    header->version_no = (unsigned char) CURRENT_PROTOCOL_VERSION;
    htontime(t, &(header->timestamp));
    header->msgtype = type;
}

void init_queue(q)
queue *q;
{
    q->data_rate = DEFAULT_DATA_RATE;
    q->queue_last = -1;
    q->queue_first = 0;
    q->queue_size = 0;
}

int send_line(fd, l, b)
int fd;
line *l;
block *b;
{
    ntheader header;
    netline nl;
    int len;

    init_header(&header, LINEMSG, &(l->last_mod));
    htonid(l->lineid, &(nl.lineid));
    if (b != NULL) {
	htonid(b->blockid, &(nl.blockid));
	nl.ypos = htons(b->ypos);
	nl.xpos = b->xpos;
	if (l->linenum == LINE_DELETED)
	    nl.linenum = htons(LINE_DELETED);
	else
	    nl.linenum = htons(find_line_num(b, l->lineid));
    } else {
	htonid("NULL", &(nl.blockid));
	nl.ypos = 0;
	nl.xpos = 0;
	nl.linenum = 0;
    }

    if (l->prev_line != NULL)
	htonid(l->prev_line->lineid, &(nl.previd));
    else
	htonid("NULL", &(nl.previd));

    if (l->next_line != NULL)
	htonid(l->next_line->lineid, &(nl.nextid));
    else
	htonid("NULL", &(nl.nextid));
    nl.no_of_chars = l->no_of_chars;
    if (l->no_of_chars > 0)
	strncpy(nl.line_data, l->line_data, MAX_LINE_LEN);
    else
	nl.line_data[0] = '\0';
    htonuser(&(l->modifier), &(nl.modifier));
    len = sizeof(netline) + (nl.no_of_chars + 1) - MAX_LINE_LEN;
    debug("Sending: >>%s<<\n", l->line_data);
    debug("line id: >>%s<<\n", l->lineid);
    xsocksend(fd, &header, (u_char *) & nl, len);
    return 0;
}

int send_block(fd, b)
int fd;
block *b;
{
    ntheader header;
    netblock nb;
    init_header(&header, BLOCKMSG, &(b->last_mod));
    htonid(b->blockid, &(nb.blockid));

    if (b->prev_block != NULL)
	htonid(b->prev_block->blockid, &(nb.previd));
    else
	htonid("NULL", &(nb.previd));

    if (b->next_block != NULL)
	htonid(b->next_block->blockid, &(nb.nextid));
    else
	htonid("NULL", &(nb.nextid));

    if ((b->first_line != NULL) & (b->status != DELETED))
	htonid(b->first_line->lineid, &(nb.first_line));
    else
	htonid("NULL", &(nb.first_line));

    htonuser(&(b->originator), &(nb.originator));
    htonuser(&(b->modifier), &(nb.modifier));
    typeface_copy(&(b->face), &(nb.face));
    nb.status = b->status;
    nb.xpos = b->xpos;
    nb.ypos = htons(b->ypos);
    nb.no_of_lines = htons(b->no_of_lines);
    debug("sending block id: >>%s<<\n", b->blockid);
    xsocksend(fd, &header, (u_char *) & nb, sizeof(netblock));
    return 0;
}



int send_pointer(fd, p)
int fd;
pointer *p;
{
    ntheader header;
    netpointer np;
    struct timeval t;
    get_cur_time(&t);
    debug("secs: %lu, usecs: %lu\n", t.tv_sec, t.tv_usec);
    init_header(&header, POINTERMSG, &t);
    htonid(p->id, &(np.id));
    strncpy(np.username, p->username, MAX_USER_NAME);
    np.x = htonl(p->x);
    np.y = htonl(p->y);
    np.col = p->col;
    strncpy(np.style, p->style, STYLE_LEN);
    xsocksend(fd, &header, (u_char *) & np, sizeof(netpointer));
    return 0;
}



int packets_queued(q)
queue *q;
{
    return (q->queue_size);
}

int in_queue(q, id)
char *id;
queue *q;
{
    int i;
    for (i = 0; i < q->queue_size; i++) {
	if (strncmp(q->qdata[i].id, id, ID_LEN) == 0)
	    return TRUE;
    }
    return FALSE;
}

int print_queue(q)
queue *q;
{
    int i;
    for (i = 0; i < q->queue_size; i++) {
	printf("id:%s, type:%d\n", q->qdata[i].id, q->qdata[i].type);
    }
    return 0;
}

void add_to_queue(q, id, buf, type)
queue *q;
char *id;
void *buf;
int type;
{
    if (buf == NULL) {
	debug("Warning: attempt to queue null packet\n");
    } else if (q->queue_size < MAX_PACKET_QUEUE_LEN) {
	q->queue_last++;
	if (q->queue_last == MAX_PACKET_QUEUE_LEN)
	    q->queue_last = 0;

	q->qdata[q->queue_last].ptr = buf;
	strncpy(q->qdata[q->queue_last].id, id, ID_LEN);
	q->qdata[q->queue_last].type = type;
	q->queue_size++;
    } else {
	debug("Warning: packet queue full - discarding change\n");
    }
}

void *get_from_queue(queue * q, int *type, char *id)
{
    if (q->queue_size > 0) {
	int tmp = q->queue_first;
	q->queue_size--;
	*type = q->qdata[q->queue_first].type;
	if (id != NULL)
	    strncpy(id, q->qdata[q->queue_first].id, ID_LEN);
	q->queue_first++;
	if (q->queue_first == MAX_PACKET_QUEUE_LEN)
	    q->queue_first = 0;
	return ((void *) (q->qdata[tmp].ptr));
    } else {
	debug("Warning: packet queue empty - returning null\n");
	return NULL;
    }
}

void delete_from_queue(q, id)
queue *q;
char *id;
{
    int i, j;
    fprintf(stderr, "******DELETE_FROM_QUEUE needs fixing\n");
    for (i = 0; i < q->queue_size; i++) {
	if (strncmp(q->qdata[i].id, id, ID_LEN) == 0) {
	    for (j = i; j < q->queue_size - 1; j++) {
		memcpy(&(q->qdata[j]), &(q->qdata[j + 1]), sizeof(packet));
	    }
	    q->queue_size--;
	    return;
	}
    }
}


void queue_packet_for_sending(fd, l, id, q, type)
int fd;
void *l;			/*Either a line or a block */
char *id;
queue *q;
int type;
{


    if (l == NULL)
	return;			/*this shouldn't ever happen */

    debug("requesting queuing\n");

    if (packets_queued(q) > 0) {
	/*there are packets queued */
	if (in_queue(q, id)) {
	    debug("already queued: %s\n", id);
	    /*this line is already queued for transmission, so do nothing */
	    return;
	} else {
	    debug("adding to queue\n");
	    /*this line must be queued */
	    add_to_queue(q, id, l, type);
	    return;
	}
    } else {
	struct timeval t;
	struct timezone tzp;
	unsigned long diff, futuretime;

	gettimeofday(&t, &tzp);
	diff = subtract_time(&t, &(q->last_time));
	/*diff is the difference between the last packet time and now in 
	   milliseconds, or MAXINT if it would overflow */
	if ((q->last_size * 1000) / (q->data_rate) > diff) {	/* don't fix the type cast error here *//*    as it screws up transmission   */
	    /*it's too early to send this packet */
	    debug("queue=0, but too early\n");
	    add_to_queue(q, id, l, type);
	    futuretime = ((q->last_size * 1000) / (q->data_rate)) - diff;
	    debug("future_time=%lu, diff=%lu\n", futuretime, diff);
	    q->event_token = Tcl_CreateTimerHandler(futuretime,
				     (Tk_TimerProc *) send_queued_packet,
						    (ClientData) 0);
	    return;
	}
	/*else drop through, and send the packet now */
	debug("sending packet immediately\n");
	htontime(&t, &(q->last_time));
	switch (type) {
	case LINEMSG:
	    q->last_size = sizeof(ntheader) + sizeof(netline)
		+ (((line *) l)->no_of_chars + 1) - MAX_LINE_LEN;
	    init_rtx_token(&skey);
	    send_line(fd, (line *) l, ((line *) l)->block);
	    break;
	case UNKNOWNLINEMSG:
	    strncpy(unknown_line.lineid, id, ID_LEN);
	    unknown_line.linenum = UNKNOWN_LINE;
	    send_line(tx_fd, &unknown_line, NULL);
	    break;
	case BLOCKMSG:
	    q->last_size = sizeof(ntheader) + sizeof(netblock);
	    init_rtx_token(&skey);
	    send_block(fd, (block *) l);
	    break;
	case POINTERMSG:
	    q->last_size = sizeof(ntheader) + sizeof(netpointer);
	    send_pointer(fd, (pointer *) l);
	}
    }
}


int send_queued_packet()
{
    void *l;
    queue *q;
    int type;
    struct timeval t;
    struct timezone tzp;

    debug("send_queued_packet\n");
    q = &sendq;
    l = get_from_queue(q, &type, unknown_line.lineid);
    if (l == NULL) {
	/*this shouldn't ever happen */
	fprintf(stderr, "NULL packet got into send queue\n");
	if (packets_queued(q) > 0)
	    send_queued_packet();
	return TCL_OK;
    }
    switch (type) {
    case LINEMSG:
	send_line(tx_fd, (line *) l, ((line *) l)->block);
	break;
    case UNKNOWNLINEMSG:
	unknown_line.linenum = UNKNOWN_LINE;
	send_line(tx_fd, &unknown_line, NULL);
	break;
    case BLOCKMSG:
	send_block(tx_fd, (block *) l);
	break;
    case POINTERMSG:
	send_pointer(tx_fd, (pointer *) l);
	break;
    }
    if (packets_queued(q) > 0) {
	/*schedule the transmission of the next one */
	q->event_token = Tk_CreateTimerHandler((q->last_size * 1000) / (q->data_rate),
				     (Tk_TimerProc *) send_queued_packet,
					       (ClientData) 0);
    } else {			/* packet queue is empty */
	q->event_token = 0;
	init_rtx_token(&skey);
    }
    gettimeofday(&t, &tzp);
    time_copy(&t, &(q->last_time));
    return TCL_OK;
}

void send_rtx_request(remote_addr, t)
struct in_addr *remote_addr;
struct timeval *t;
{
    ntheader header;
    netrtxrequest request;
    init_header(&header, RTX_REQUEST, t);
    memcpy(&(request.source), remote_addr, sizeof(struct in_addr));
    xsocksend(tx_fd, &header, (u_char *) & request, sizeof(netrtxrequest));
}

void send_specific_rtx_request(remote_addr)
struct in_addr *remote_addr;
{
    ntheader header;
    struct timeval t;
    struct timezone tzp;
    netrtxspecificrequest request;
    int i, no_missing;

    gettimeofday(&t, &tzp);
    init_header(&header, RTX_SPECIFIC_REQUEST, &t);
    memcpy(&(request.source), remote_addr, sizeof(struct in_addr));
    no_missing = number_missing();
    if (no_missing > MAX_NO_OF_REQUESTS)
	no_missing = MAX_NO_OF_REQUESTS;
    for (i = 0; i < no_missing; i++) {
	debug("requesting %d: %s\n", i, (get_missing_id(i)));
	htonid(get_missing_id(i), &(request.ids[i]));
    }
    request.no_of_items = htonl(no_missing);
    xsocksend(tx_fd, &header, (u_char *) & request, sizeof(struct in_addr) +
	      sizeof(int) + (ID_LEN * no_missing));

}

void send_page_request(remote_addr)
struct in_addr *remote_addr;
{
    ntheader header;
    struct timeval t;
    struct timezone tzp;
    netpagerequest request;

    gettimeofday(&t, &tzp);
    init_header(&header, PAGE_INFO_REQUEST, &t);
    memcpy(&(request.source), remote_addr, sizeof(struct in_addr));
    xsocksend(tx_fd, &header, (u_char *) & request, sizeof(netpagerequest));

}


void init_jim_rtx()
{				/* a little extra function to let me start a sliding key sequence *//* from outside of this */
    init_rtx_token(&skey);
}				/*end void */


void send_session_message()
{
    struct timeval cur_time;	/* used to store current time */
    static ntheader header;
    static netsession sm;
    static int first = 0;
    if (first == 0) {
	/*no point in setting all this up every time */
	struct timeval t;
	struct timezone tzp;
	char *str;
	gettimeofday(&t, &tzp);
	/*sure I'll think of a use for t eventually */
	init_header(&header, SESSIONMSG, &t);
	htonuser(&me, &sm.ud);
	if (strcmp(vanityname, "") == 0)
	    sprintf(sm.textname, "%s@%s", me.username, hostname);
	else
	    sprintf(sm.textname, "%s", vanityname);
	sm.textnamelen = strlen(sm.textname) + 1;
	str = sm.textname + sm.textnamelen;
	strcpy(str, nt_version);
	sm.textnamelen += strlen(nt_version) + 1;
	str = sm.textname + sm.textnamelen;
	strcpy(str, os_version);
	sm.textnamelen += strlen(os_version);
	first++;
	/* initialize retransmit time as now */
	get_cur_time(&rtx_time);

    }
    sm.col = cur_col;

    debug("sending a session message\n");

    xsocksend(tx_fd, &header, (u_char *) & sm, sizeof(user_data) +
	      sizeof(u_int8) + (sm.textnamelen + 1));

    timeout_participants();

    /* do not schedule another session message if sess-message being */
    /*   used to signify user's colour change */
    if (s_message_type == COLOUR_CHANGE) {
	s_message_type = NORMAL_S_MESSAGE;	/* reset session message type */
	/* to normal session message. */
    }
    /* end if */
    else {
	/*must make these session messages backoff eventually */
	Tk_CreateTimerHandler(5000,
			      (Tk_TimerProc *) send_session_message,
			      (ClientData) 0);
    }				/* end else */

    /* here is my piggy backed message to keep the redisplay system happy.. */
    /* it makes sure that all data has been updated.. */
    if (new_data == 1) {
	block *b;
	new_data = 0;
	/* now go through each block and find the block(s) that needs re-displaying. */
	b = p->first_block;

	while (b != NULL) {
	    debug("new_data = %d", b->new_data);
	    if (b->new_data == 1) {
		b->new_data = 0;
		redisplay_block(b);

	    }
	    b = b->next_block;
	}

    }
    /* newly added bit to check the time of the last rtx message received or sent */
    get_cur_time(&cur_time);

    if (((cur_time.tv_sec) - (rtx_time.tv_sec)) > 10) {		/* if more than 10 seconds *//* since last rtx req/send */
	int time;

	lbl_srandom(cur_time.tv_sec);	/* seed randomizer */
	time = (lbl_random() % 6000);	/* get random value 0->6000 */
	time = time - 3000;	/* r value from -3000 to 3000 */
	debug("timed out rtx - starting our own\n");
	/* send retransmission advert after 5 seconds random -3 -> +3 secs. */
	Tk_CreateTimerHandler(5000 + time,
			      (Tk_TimerProc *) init_jim_rtx,
			      (ClientData) 0);

	/* init_rtx_token(&skey); */



    }
    /* end if we have to start a new retransmission cycle */
    else {
	debug("rtx not timed out t = %d\n", ((cur_time.tv_sec) - (rtx_time.tv_sec)));
    }

}

void send_leave_message()
{
    static ntheader header;
    static netsession sm;
    struct timeval t;
    struct timezone tzp;

    gettimeofday(&t, &tzp);
    init_header(&header, LEAVE_CONF, &t);
    htonuser(&me, &sm.ud);
    sprintf(sm.textname, "%s@%s", me.username, hostname);

    sm.textnamelen = strlen(sm.textname) + 1;	/* by adding +1 - gets rid of dodgy */
    /* leaving message bits. */
    debug("sending leave message with name:%s\n", sm.textname);

    xsocksend(tx_fd, &header, (u_char *) & sm, sizeof(user_data) +
	      sizeof(u_int8) + (sm.textnamelen + 1));

}

void send_page_info()
{				/* new message to send page checksum */


    static ntheader header;
    static netpageinfo npi;
    struct timeval t;
    struct timezone tzp;

    if (checksum_active == 0) {
	return;
    }
    gettimeofday(&t, &tzp);	/* create the packet's header */
    init_header(&header, PAGE_INFO, &t);

    htontime(&(p->last_mod), &(npi.timestamp));

    calc_page_checksum(p);	/* re-calculate this page's checksum */

    strncpy(npi.checksum, p->checksum, 18);	/* set the packet's fields correctly */

    memcpy(&(npi.source), &ipaddr, sizeof(struct in_addr));

    npi.checksum_version = CHECKSUM_ALGORITHM_VERSION;

    npi.no_of_blocks = htons(p->no_of_blocks);

    debug("sending page info packet\n");

    /* send the packet */
    xsocksend(tx_fd, &header, (u_char *) & npi, sizeof(netpageinfo));

}				/* end send_page_info packet */

void send_blockinfo_request(block * b, struct in_addr *target)
{
    static ntheader header;
    static netblockinforequest nbir;
    struct timeval t;
    struct timezone tzp;


    if (checksum_active == 0) {
	return;
    }
    gettimeofday(&t, &tzp);
    init_header(&header, BLOCK_INFO_REQUEST, &t);

    memcpy(&(nbir.target), target, sizeof(struct in_addr));

    htonid(b->blockid, &(nbir.blockid));

    xsocksend(tx_fd, &header, (u_char *) & nbir, sizeof(netblockinforequest));
    debug("sending block info request packet\n");

}				/* end send_blockinfo_request */

void send_blockinfo_request2(struct in_addr *target)
{				/* version 2 - requests transmission *//* of info on all blocks in the page */
    static ntheader header;
    static netblockinforequest2 nbir;
    struct timeval t;
    struct timezone tzp;


    if (checksum_active == 0) {
	return;
    }
    gettimeofday(&t, &tzp);
    init_header(&header, BLOCK_INFO_REQUEST2, &t);

    memcpy(&(nbir.target), target, sizeof(struct in_addr));

    xsocksend(tx_fd, &header, (u_char *) & nbir, sizeof(netblockinforequest2));
    debug("sending block info request packet version 2\n");

}				/* end send_blockinfo_request */


void send_blockinfo(block * b)
{
    static ntheader header;	/* broadcast information about a block to everyone */
    static netblockinfo nbi;
    struct timeval t;
    struct timezone tzp;

    if (checksum_active == 0) {
	return;
    }
    gettimeofday(&t, &tzp);
    init_header(&header, BLOCK_INFO, &t);

    memcpy(&(nbi.source), &ipaddr, sizeof(struct in_addr));

    nbi.checksum_version = CHECKSUM_ALGORITHM_VERSION;

    htonid(b->blockid, &nbi.blockid);

    strncpy(nbi.checksum, b->checksum, 18);

    nbi.no_of_lines = htons(b->no_of_lines);

    xsocksend(tx_fd, &header, (u_char *) & nbi, sizeof(netblockinfo)	/*sizeof(struct in_addr) +
									   sizeof(u_int16) + (sizeof(u_char)*20) + sizeof(u_int16) + sizeof(netid) */ );

    debug("sending block info packet\n");
}				/* end send_blockinfo */

void send_block_rtx_request(char id[ID_LEN], struct in_addr *source)
{
    /* send a request to the site at address source to retransmit the block with idno==id, */
    /* as well as all of it's lines */


    static ntheader header;
    static netblockrtxrequest nbrr;
    struct timeval t;
    struct timezone tzp;

    if (checksum_active == 0) {
	return;
    }
    gettimeofday(&t, &tzp);
    init_header(&header, BLOCK_RTX_REQUEST, &t);

    memcpy(&(nbrr.target), source, sizeof(struct in_addr));

    htonid(id, &(nbrr.blockid));

    xsocksend(tx_fd, &header, (u_char *) & nbrr, sizeof(netblockrtxrequest));

    debug("sending block retransmission request\n");
}				/* end send_block_rtx_request */
