/***************************************************************************
                          qbrewdoc.cpp  -  client document class for QBrew
                             -------------------
    begin                : September 20th, 1999
    copyright            : (C) 1999 by David Johnson
    email                : arandir@meer.net

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include "qbrew.h"
#include "qbrewdoc.h"
#include "fileresource.h"

QBrewDoc::QBrewDoc(QWidget *parent=0)
{
	application = parent;
	initSerialization();
	// there is no document yet, so make one
	// shouldn't be calling newDoc() since
	// this calls documentChanged()
	newDoc();
}

QBrewDoc::~QBrewDoc() {}

void QBrewDoc::initSerialization()
{
	// set up group map
	gmap[groupRecipe] = gidRecipe;
	gmap[groupDocGrains] = gidDocGrains;
	gmap[groupDocHops] = gidDocHops;
	gmap[groupDocMisc] = gidDocMisc;
	
	// set up recipe attributes
	fmap[fieldTitle] = fidTitle;
	fmap[fieldBrewer] = fidBrewer;
	fmap[fieldStyle] = fidStyle;
	fmap[fieldSize] = fidSize;
	fmap[fieldMash] = fidMash;
}

///////////////////////////////////////////////////////////////////////////////
// file functions

void QBrewDoc::newDoc()
{
	// create a new document. assumes application knows
	// what it is doing by calling this
	
	// reset appropriate elements
	title = "";
	brewer = "";
	style = "";
	mash = false;
	// stupid hack to avoid circular includes of qbrew.h
	size = ((QBrew *)application)->getBatchSetting();
	if (!grainlist.isEmpty()) grainlist.clear();
	if (!hopslist.isEmpty()) hopslist.clear();
	if (!misclist.isEmpty()) misclist.clear();
	
	emit documentChanged();
	b_modified = false;
	b_new = true;
}

bool QBrewDoc::save(const QString &fname)
{
	// give file a new name and save it. assumes
	// that the application is passing a valid filename
	
	// open file for writing
	rfile = new Store(IO_WriteOnly, fname, QBREW_FORMAT, QBREW_VERSION);
	if (!rfile->good()) {
		warning("QBrewDoc: Error opening " + fname + "\n");
		delete rfile;
		return false;
	} else {
		// file opened without error
		// write recipe attributes
		rfile->putGroup(groupRecipe);
		rfile->putLine(fieldTitle, title);
		rfile->putLine(fieldBrewer, brewer);
		rfile->putLine(fieldStyle, style);
		rfile->putLine(fieldSize, size);
		rfile->putLine(fieldMash, ( (mash) ? "TRUE" : "FALSE" ));
		// write grains
		rfile->putGroup(groupDocGrains);
		GrainList::Iterator itg;
    	for (itg=grainlist.begin(); itg != grainlist.end(); ++itg) {
    		rfile->putLine((*itg).getName(), (*itg).serializeOut());
	    }
		// write hops
		rfile->putGroup(groupDocHops);
		HopsList::Iterator ith;
    	for (ith=hopslist.begin(); ith != hopslist.end(); ++ith) {
    		rfile->putLine((*ith).getName(), (*ith).serializeOut());
	    }
		// write misc ingredients
		rfile->putGroup(groupDocMisc);
		MiscIngredientList::Iterator itm;
    	for (itm=misclist.begin(); itm != misclist.end(); ++itm) {
    		rfile->putLine((*itm).getName(), (*itm).serializeOut());
	    }
	}
	delete rfile;	
	
	// document is saved, so set flags accordingly
	b_modified = false;
	b_new = false;
	return true;
}

bool QBrewDoc::load(const QString &fname)
{
	// open the named file. assumes that the application
	// has passed us a valid file name. we still have
	// to check that it's a valid recipe file, though.
	int ID;
	
	// open file for reading
	rfile = new Store(IO_ReadOnly, fname, QBREW_FORMAT, QBREW_VERSION);
	if (!rfile->good()) {
		warning("QBrewDoc: Error opening " + fname + "\n");
		delete rfile;
		return false;
	} else {
		// file opened without error
		if (rfile->getVersion() < QBREW_PREVIOUS) {
			warning("QBrewDoc: Unsupported recipe version " +  rfile->getVersion() + "\n");
			delete rfile;
			return false;
		}
		// clear lists
		grainlist.clear();
		// read in file line by line
		while (rfile->getLine()) {
			ID = gmap[rfile->getGroup()];
			if (rfile->getGroup() != rfile->getName()) {
				switch (ID) {
					case gidRecipe:
						processRecipe(rfile->getName(), rfile->getValue());
						break;
					case gidDocGrains:
						processGrain(rfile->getName(), rfile->getValue());
						break;
					case gidDocHops:
						processHops(rfile->getName(), rfile->getValue());
						break;
					case gidDocMisc:
						processMiscIngredient(rfile->getName(), rfile->getValue());
						break;
					default:
						warning("QBrewDoc: " + rfile->getGroup() + " is not a valid group\n");
						break;
				}
			}
		}		
	}
	delete rfile;
		
	// we have now opened a document
	emit documentChanged();
	emit recalc();
	b_modified = false;
	b_new = false;
	return true;
}

/////////////////////////////////////////////////////////////////////////////
// process input

void QBrewDoc::processRecipe(const QString &fname, const QString &fvalue)
{
	int ID = fmap[fname];
	switch (ID) {
		case fidTitle:
			title = fvalue.copy();
			break;
		case fidBrewer:
			brewer = fvalue.copy();
			break;
		case fidStyle:
			style = fvalue.copy();
			break;
		case fidSize:
			size = fvalue.toUInt();
			break;
		case fidMash:
			mash = (fvalue.lower() == "true");
			break;
		default:
			warning("QBrewDoc: " + fname + " is not a valid field\n");
			break;
	}
}

void QBrewDoc::processGrain(const QString &fname, const QString &fvalue)
{
	Grain newgrain;
	newgrain.serializeIn(fname, fvalue);
	grainlist.append(newgrain);	
}

void QBrewDoc::processHops(const QString &fname, const QString &fvalue)
{
	Hops newhop;
	newhop.serializeIn(fname, fvalue);
	hopslist.append(newhop);	
}

void QBrewDoc::processMiscIngredient(const QString &fname, const QString &fvalue)
{
	MiscIngredient newmisc;
	newmisc.serializeIn(fname, fvalue);
	misclist.append(newmisc);	
}

///////////////////////////////////////////////////////////////////////////////
// member access functions

bool QBrewDoc::isModified() { return b_modified; }

bool QBrewDoc::isNew() { return b_new; }

void QBrewDoc::setTitle(const QString &stitle)
{
	title = stitle;
	emit documentModified();
	b_modified = true;
}

const QString QBrewDoc::getTitle() { return title; }

void QBrewDoc::setBrewer(const QString &sbrewer)
{
	brewer = sbrewer;
	emit documentModified();
	b_modified = true;
}

const QString QBrewDoc::getBrewer() { return brewer; }

void QBrewDoc::setStyle(const QString &sstyle)
{
	style = sstyle;
	emit documentModified();
	emit styleChanged();
	b_modified = true;
}

const QString QBrewDoc::getStyle() { return style; }

void QBrewDoc::setSize(int ssize)
{
	size = ssize;
	emit documentModified();
	emit recalc();
	b_modified = true;
}

const int QBrewDoc::getSize() { return size; }

void QBrewDoc::setMash(bool smash)
{
	mash = smash;
	emit documentModified();
	emit recalc();
	b_modified = true;
}

const bool QBrewDoc::getMash() { return mash; }

GrainIterator QBrewDoc::addGrain(const QString &gname, const unsigned &gquant, const unsigned &gextr,
				const unsigned &gcolor, const QString &guse)
{
	GrainList::Iterator it;
	Grain newgrain = Grain(gname, gquant, gextr, gcolor, Grain::useStringToEnum(guse));
	it = grainlist.append(newgrain);
	emit documentModified();
	emit recalc();
	b_modified = true;
	return it;
}

void QBrewDoc::applyGrain(GrainIterator it, const QString &gname, const unsigned &gquant,
				const unsigned &gextr, const unsigned &gcolor, const QString &guse)
{
	Grain newgrain(gname, gquant, gextr, gcolor, Grain::useStringToEnum(guse));
	(*it) = newgrain;
	emit documentModified();
	emit recalc();
	b_modified = true;
}

void QBrewDoc::removeGrain(GrainIterator it)
{
	grainlist.remove(it);
	emit documentModified();
	emit recalc();
	b_modified = true;
}

GrainList *QBrewDoc::getGrainList()
{
	return &grainlist;
}	

HopIterator QBrewDoc::addHop(const QString &hname, const unsigned &hquant, const QString &hform,
				const unsigned &halpha, const unsigned &htime)
{
	HopsList::Iterator it;
	Hops newhop = Hops(hname, hquant, hform, halpha, htime);
	it = hopslist.append(newhop);
	emit documentModified();
	emit recalc();
	b_modified = true;
	return it;
}

void QBrewDoc::applyHop(HopIterator it, const QString &hname, const unsigned &hquant, const QString &hform,
				const unsigned &halpha, const unsigned &htime)
{
	Hops newhop = Hops(hname, hquant, hform, halpha, htime);
	(*it) = newhop;
	emit documentModified();
	emit recalc();
	b_modified = true;
}

void QBrewDoc::removeHop(HopIterator it)
{
	hopslist.remove(it);
	emit documentModified();
	emit recalc();
	b_modified = true;
}

HopsList *QBrewDoc::getHopsList()
{
	return &hopslist;
}	

MiscIngredientIterator QBrewDoc::addMiscIngredient(const QString &mname, const unsigned &mquant, const QString &mnotes)
{
	MiscIngredientList::Iterator it;
	MiscIngredient newmisc = MiscIngredient(mname, mquant, mnotes);
	it = misclist.append(newmisc);
	emit documentModified();
	emit recalc();
	b_modified = true;
	return it;
}

void QBrewDoc::applyMiscIngredient(MiscIngredientIterator it, const QString &mname,
		const unsigned &mquant, const QString &mnotes)
{
	MiscIngredient newmisc = MiscIngredient(mname, mquant, mnotes);
	(*it) = newmisc;
	emit documentModified();
	emit recalc();
	b_modified = true;
}

void QBrewDoc::removeMiscIngredient(MiscIngredientIterator it)
{
	misclist.remove(it);
	emit documentModified();
	emit recalc();
	b_modified = true;
}

MiscIngredientList *QBrewDoc::getMiscIngredientList()
{
	return &misclist;
}	

























