/***************************************************************************
                          qbrewview.cpp  -  client view class for QBrew
                             -------------------
    begin                : September 20th, 1999
    copyright            : (C) 1999 by David Johnson
    email                : arandir@meer.net

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include "resource.h"
#include "qbrewview.h"
#include "style.h"

QBrewView::QBrewView(QWidget *parent, const char* name, QBrewDoc* doc, QBrewCalc* calc) : QWidget(parent, name)
{
	document = doc;
	calcdata = calc;
	
	// initializations
	initWidgets();
	initLayouts();
	
	//post layout fixups
	mainLayout->activate();
	// the following allows enough room to display all widgets
	ingredientsTab->setMinimumHeight(148);
	
	// connect doc with the view
	connect(doc,SIGNAL(documentChanged()),this,SLOT(slotDocumentChanged()));
	connect(doc,SIGNAL(documentChanged()),grainPage,SLOT(slotDocumentChanged()));
	connect(doc,SIGNAL(documentChanged()),hopsPage,SLOT(slotDocumentChanged()));
	connect(doc,SIGNAL(documentChanged()),miscPage,SLOT(slotDocumentChanged()));
	// connect changing style to slotStyleChanged
	connect(doc,SIGNAL(styleChanged()),this,SLOT(slotStyleChanged()));	
	// set widgets to a new doc
	slotDocumentChanged();
}

QBrewView::~QBrewView()
{
	// child widgets and layouts automatically freed
}

///////////////////////////////////////////////////////////////////////////////
// initializations

void QBrewView::initWidgets()
{
	// think about linking labels to their "buddy" widgets

	///////////////////////////////////////////////////////////////////////////
	// recipe layout widgets
		
	// title
	titleLabel = new QLabel(this, "titleLabel");
	titleLabel->setText(tr("Title"));
	titleEdit = new QLineEdit(this, "titleEdit");
	connect(titleEdit,SIGNAL(textChanged(const QString &)),document,SLOT(setTitle(const QString &)));
	
	// style
	styleLabel = new QLabel(this, "styleLabel");
	styleLabel->setText(tr("Style"));
	styleCombo = new QComboBox(false, this, "styleCombo");
	styleCombo->insertStringList(calcdata->getStylesList());
	connect(styleCombo,SIGNAL(activated(const QString &)),document,SLOT(setStyle(const QString &)));
	
	// brewer
	brewerLabel = new QLabel(this, "brewerLabel");
	brewerLabel->setText(tr("Brewer"));
	brewerEdit = new QLineEdit(this, "brewerEdit");
	connect(brewerEdit,SIGNAL(textChanged(const QString &)),document,SLOT(setBrewer(const QString &)));
	
	// size
	sizeLabel = new QLabel(this, "sizeLabel");
	sizeLabel->setText(tr("Size"));
	sizeSpin = new FixedSpinBox(0, 10000, 25, 2, this, "sizeSpin");
	sizeSpin->setWrapping(true);
	sizeSpin->setSuffix(tr(" Gallons"));
	connect(sizeSpin,SIGNAL(valueChanged(int)),document,SLOT(setSize(int)));
	
	// mash
	mashCheck = new QCheckBox(tr("Mash"), this, "mashCheck");
	connect(mashCheck,SIGNAL(toggled(bool)),document,SLOT(setMash(bool)));
	
	///////////////////////////////////////////////////////////////////////////
	// style layout widgets

	styleBox = new QGroupBox(this, "styleBox");
	styleBox->setFrameStyle(QFrame::Box | QFrame::Sunken);
	ogMinimum = new QLabel(styleBox, "ogMinimum");
	ogMaximum = new QLabel(styleBox, "ogMaximum");
	ogYours = new QLabel(styleBox, "ogYours");
	ibuMinimum = new QLabel(styleBox, "ibuMinimum");
	ibuMaximum = new QLabel(styleBox, "ibuMaximum");
	ibuYours = new QLabel(styleBox, "ibuYours");
	srmMinimum = new QLabel(styleBox, "srmMinimum");
	srmMaximum = new QLabel(styleBox, "srmMaximum");
	srmYours = new QLabel(styleBox, "srmYours");
	
	vSeparator = new QFrame(styleBox, "vSeparator");
	vSeparator->setFrameStyle(QFrame::VLine | QFrame::Sunken);

	fgLabel = new QLabel(styleBox, "fgLabel");
	alcoholByVol =  new QLabel(styleBox, "alcoholByVol");
	alcoholByWgt =  new QLabel(styleBox, "alcoholByWgt");

	///////////////////////////////////////////////////////////////////////////
	// ingredient widgets
	ingredientsTab = new QTabWidget(this, "ingredientsTab");
	grainPage = new GrainPage(this, "grainPage", document, calcdata);
	hopsPage = new HopsPage(this, "hopsPage", document, calcdata);
	miscPage = new MiscPage(this, "miscPage", document, calcdata);
	// ingredients TabWidget setup
	ingredientsTab->addTab(grainPage, "&Grains");
	ingredientsTab->addTab(hopsPage, "Ho&ps");
	ingredientsTab->addTab(miscPage, "&Miscellaneous");
}

void QBrewView::initLayouts()
{
	mainLayout = new QVBoxLayout( this, 6, 6, "mainLayout" ); // main layout
	recipeLayout = new QGridLayout( 2, 10, 3, "recipeLayout" );
	styleLayout = new QGridLayout( styleBox, 4, 9, 9, 3, "styleLayout" );
	
	// mainLayout
	mainLayout->addLayout(recipeLayout, 0);
	mainLayout->addWidget(styleBox, 0);
	mainLayout->addWidget(ingredientsTab, 1);

	// recipeLayout
	recipeLayout->addColSpacing(1, 3);
	recipeLayout->addColSpacing(3, 6);
	recipeLayout->addColSpacing(5, 3);
	recipeLayout->addColSpacing(7, 3);
	recipeLayout->setColStretch(2, 3);
	recipeLayout->setColStretch(6, 1);
	recipeLayout->setColStretch(8, 1);
	recipeLayout->addWidget(titleLabel, 0, 0);
	recipeLayout->addWidget(titleEdit, 0, 2);
	recipeLayout->addWidget(styleLabel, 0, 4);
	recipeLayout->addMultiCellWidget(styleCombo, 0, 0, 6, 8); //span 4 cols
	recipeLayout->addWidget(brewerLabel, 1, 0);
	recipeLayout->addWidget(brewerEdit, 1, 2);
	recipeLayout->addWidget(sizeLabel, 1, 4);
	recipeLayout->addWidget(sizeSpin, 1, 6);
	recipeLayout->addWidget(mashCheck, 1, 8, AlignRight);
	
	// styleLayout
	styleLayout->addRowSpacing(0, 9);
	styleLayout->addColSpacing(1, 3);
	styleLayout->addColSpacing(3, 3);
	styleLayout->addColSpacing(5, 3);
	styleLayout->addColSpacing(7, 3);
	styleLayout->setColStretch(0, 1);
	styleLayout->setColStretch(2, 1);
	styleLayout->setColStretch(4, 1);
	styleLayout->setColStretch(8, 1);
	styleLayout->addWidget(ogMinimum, 1, 0);
	styleLayout->addWidget(ibuMinimum, 1, 2);
	styleLayout->addWidget(srmMinimum, 1, 4);
	styleLayout->addWidget(ogMaximum, 2, 0);
	styleLayout->addWidget(ibuMaximum, 2, 2);
	styleLayout->addWidget(srmMaximum, 2, 4);
	styleLayout->addWidget(ogYours, 3, 0);
	styleLayout->addWidget(ibuYours, 3, 2);
	styleLayout->addWidget(srmYours, 3, 4);
	styleLayout->addMultiCellWidget(vSeparator, 1, 3, 6, 6);
	styleLayout->addWidget(fgLabel, 1, 8);
	styleLayout->addWidget(alcoholByVol, 2, 8);
	styleLayout->addWidget(alcoholByWgt, 3, 8);
}

///////////////////////////////////////////////////////////////////////////////
// utility routines

void QBrewView::setComboItem(QComboBox *combo, const QString &txt)
{
	int item;
	// search combo box for style string
	for (item = 0; item < combo->count(); item++) {
		if (combo->text(item) == txt) {
			combo->setCurrentItem(item);
			return;
		}
	}
	// if item not found, will set it to the first one
	combo->setCurrentItem(0);
}

///////////////////////////////////////////////////////////////////////////////
// slot implementations

void QBrewView::slotDocumentChanged()
{
	// update recipe widgets
	titleEdit->setText(document->getTitle());
	brewerEdit->setText(document->getBrewer());
	styleCombo->clear();
	styleCombo->insertStringList(calcdata->getStylesList());
	setComboItem(styleCombo, document->getStyle());
	sizeSpin->setValue(document->getSize());
	mashCheck->setChecked(document->getMash());
	// update style widgets
	slotStyleChanged();
}

void QBrewView::slotStyleChanged()
{
	Style *style;
	style = calcdata->getStyle(document->getStyle());
		
	// beer style has changed, so update labels in styleLayout
	styleBox->setTitle(styleCombo->currentText() + tr(" Characteristics"));
	ogMinimum->setText(tr("Minimum Gravity: ") + QString::number(style->getOGLow() + 1000));
	ogMaximum->setText(tr("Maximum Gravity: ") + QString::number(style->getOGHi() + 1000));
	ogYours->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Gravity:") + CHAR_NBSP + QString::number(calcdata->getOG() + 1000, 'f', 0) + "</b>");
	ibuMinimum->setText(tr("Minimum Bitterness: ") + QString::number(style->getIBULow()));
	ibuMaximum->setText(tr("Maximum Bitterness: ") + QString::number(style->getIBUHi()));
	ibuYours->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Bitterness:") + CHAR_NBSP + QString::number(calcdata->getIBU(), 'f', 0) + "</b>");
	srmMinimum->setText(tr("Minimum Color: ") + QString::number(style->getSRMLow()) + CHAR_LATIN_DEGREE);
	srmMaximum->setText(tr("Maximum Color: ") + QString::number(style->getSRMHi()) + CHAR_LATIN_DEGREE);
	srmYours->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Color:") + CHAR_NBSP + QString::number(calcdata->getSRM(), 'f', 0) + CHAR_LATIN_DEGREE + "</b>");
	fgLabel->setText(tr("Estimated FG: ") + QString::number(calcdata->getFGEstimate() + 1000, 'f', 0));
	alcoholByVol->setText("<b>" + tr("Alcohol") + CHAR_NBSP + tr("by") + CHAR_NBSP + tr("Volume:") + CHAR_NBSP + QString::number(calcdata->getABV(), 'f', 1) + "%</b>");
	alcoholByWgt->setText("<b>" + tr("Alcohol") + CHAR_NBSP + tr("by") + CHAR_NBSP + tr("Weight:") + CHAR_NBSP + QString::number(calcdata->getABW(), 'f', 1) + "%</b>");		
}

































