// 
// $Id: qaregexpwidget.cpp,v 1.3 1999/08/16 02:01:39 amos Exp $
//
// Implementation of QRegExpWidget class
//
// Jan Borsodi <amos@abn.hibu.no>
// Created on: <21-Jul-1999 21:22:46 amos>
//
// Copyright (C) 1999 Jan Borsodi.  All rights reserved.
//

#include "qaregexpwidget.hpp"
#include <qaregexp.hpp>
#include <qaregexperror.hpp>

#include <qframe.h>
#include <qlayout.h>
#include <qwidget.h>
#include <qstring.h>
#include <qlineedit.h>
#include <qmultilinedit.h>
#include <qtextview.h>
#include <qsplitter.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qcolor.h>
#include <qstylesheet.h>
#include <qmainwindow.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qstatusbar.h>
#include <qvgroupbox.h>
#include <qlistbox.h>
#include <qpoint.h>
#include <qrect.h>

/*!
  \class QRegExpWidget qregexpwidget.hpp
  \brief A widget with the possibility to see regexps in action.

  Has a text field for regexp, a multiline edit for the text,
  and a output text view. When a regexp matches it will display
  the match in a different color, sub expressions can also be matched.
*/

/*!
  Default constructor
*/

QaRegExpWidget::QaRegExpWidget( QWidget *parent, const char *name )
    : QMainWindow( parent, name ), Reg( "" )
{
    setCaption( "RegExplorer - 0.1.3" );
    SubValue = 0;
    MatchCount = 0;
    SubMatch = 0;
    Multiple = false;
    Central = new QWidget( this );
    QVBoxLayout *clay = new QVBoxLayout( Central, 1 );
    QSplitter *MainSplit = new QSplitter( Horizontal, Central );
    clay->addWidget( MainSplit, 1 );
    MainSplit->setOpaqueResize( true );
    QWidget *LeftSplit = new QWidget( MainSplit, "left split widget" );
    Lay = new QVBoxLayout( LeftSplit, 0, 3 );
    {
        QHBoxLayout *hlay = new QHBoxLayout();
        Lay->addLayout( hlay, 0 );
        {
            QLabel *patlab = new QLabel( tr( "Pattern:" ), LeftSplit );
            hlay->addWidget( patlab, 0 );
            QLineEdit *line = new QLineEdit( LeftSplit );
            hlay->addWidget( line, 0 );
            connect( line, SIGNAL( textChanged( const QString & ) ),
                     this, SLOT( textChange( const QString & ) ) );
        }

        QSplitter *split = new QSplitter( Horizontal, LeftSplit );
        split->setOpaqueResize( true );
        Lay->addWidget( split, 1 );
        {
            Edit = new QMultiLineEdit( split );
            Edit->setText( "void test::setText( const QString &s )\n"
                           "{\n"
                           "}\n" );
            connect( Edit, SIGNAL( textChanged() ),
                     this, SLOT( multiTextChanged() ) );

            TextVu = new QTextView( split );
            TextVu->setPaper( QColor( 150, 152, 255 ) );
            TextVu->setText( "void test::setText( const QString &s )<br>"
                             "{<br>"
                             "}<br>" );
            TextVu->setFocusPolicy( NoFocus );
        }
        QValueList<int> ilst;
        ilst.append( 1 );
        ilst.append( 1 );
        split->setSizes( ilst );

        QSplitter *SubSplit = new QSplitter( Vertical, MainSplit );
        {
            Sub = new QListBox( SubSplit );
            connect( Sub, SIGNAL( highlighted( int ) ),
                     this, SLOT( valueChange( int ) ) );

            Matches = new QListBox( SubSplit );
            connect( Matches, SIGNAL( highlighted( int ) ),
                     this, SLOT( valueChangeMultiple( int ) ) );
            Matches->hide();
        }
        SubSplit->setOpaqueResize( true );

        ErrLabel = new QLabel( " ", statusBar() );
        statusBar()->addWidget( ErrLabel, 1 );
    }
    QValueList<int> ilst;
    ilst.append( 4 );
    ilst.append( 1 );
    MainSplit->setSizes( ilst );
    setCentralWidget( Central );

    FileMenu = new QPopupMenu( 0, "file menu" );
    {
        menuBar()->insertItem( tr( "&File" ), FileMenu );

        FileMenu->setItemEnabled( FileMenu->insertItem( tr( "&Load" ), this, SLOT( doLoad() ), CTRL + Key_L ), false );
        FileMenu->setItemEnabled( FileMenu->insertItem( tr( "&Save" ), this, SLOT( doSave() ), CTRL + Key_S ), false );
        FileMenu->insertSeparator();
        FileMenu->insertItem( tr( "&Exit" ), this, SLOT( close() ), CTRL + Key_Q );
    }

    EditMenu = new QPopupMenu( 0, "modes menu" );
    {
        menuBar()->insertItem( tr( "&Edit" ), EditMenu );

        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&New Group" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Delete Group" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Add RegExp" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Remove RegExp" ) ), false );
    }

    MatchMenu = new QPopupMenu( 0, "match menu" );
    {
        menuBar()->insertItem( tr( "&Match Type" ), MatchMenu );

        MatchNormalID = MatchMenu->insertItem( tr( "&Normal" ), this, SLOT( doNormal() ), CTRL + Key_N );
        MatchMultipleID = MatchMenu->insertItem( tr( "&Multiple" ), this, SLOT( doMultiple() ), CTRL + Key_M );
        MatchMenu->setItemEnabled( MatchMenu->insertItem( tr( "&Split" ) ), false );
        MatchMenu->insertSeparator();
        MatchCaseID = MatchMenu->insertItem( tr( "&Case Sensitive" ), this, SLOT( doCase() ), CTRL + Key_I );

        MatchMenu->setCheckable( true );
        MatchMenu->setItemChecked( MatchNormalID, true );
        MatchMenu->setItemChecked( MatchCaseID, true );
    }

    ModesMenu = new QPopupMenu( 0, "modes menu" );
    {
        menuBar()->insertItem( tr( "Em&ulation Mode" ), ModesMenu );

        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&None" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Lisp" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Perl" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Python" ) ), false );

        ModesMenu->setCheckable( true );
        ModesMenu->setItemChecked( ModesMenu->idAt( 0 ), true );
    }

    QString s;
    for ( int i = 0; i < Edit->numLines(); i++ )
    {
        s += Edit->textLine( i );
        if ( i < Edit->numLines() )
            s+= "\n";
    }
    EditString = s;
    textChange( "" );
}

void QaRegExpWidget::doNormal()
{
    if ( MatchMenu->isItemChecked( MatchNormalID ) )
        return;
    MatchMenu->setItemChecked( MatchNormalID, true );
    MatchMenu->setItemChecked( MatchMultipleID, false );
    Multiple = false;

    Matches->setEnabled( false );
    updateMatch();

    // Fix for listbox repaint bug
    Matches->hide();
//    Matches->show();
}

void QaRegExpWidget::doMultiple()
{
    if ( !MatchMenu->isItemChecked( MatchNormalID ) )
        return;
    MatchMenu->setItemChecked( MatchNormalID, false );
    MatchMenu->setItemChecked( MatchMultipleID, true );
    Multiple = true;

    Matches->setEnabled( true );
    updateMatch();

    // Fix for listbox repaint bug
//    Matches->hide();
    Matches->show();
}

void QaRegExpWidget::doLoad()
{
}

void QaRegExpWidget::doSave()
{
}

void QaRegExpWidget::doCase()
{
    MatchMenu->setItemChecked( MatchCaseID, !MatchMenu->isItemChecked( MatchCaseID ) );
    Reg.setCaseSensitive( MatchMenu->isItemChecked( MatchCaseID ) );
    updateMatch();
}

/*!
  Destroys the object
*/

QaRegExpWidget::~QaRegExpWidget()
{
    delete Lay;
}

/*!
  Text has changed
*/

void QaRegExpWidget::textChange( const QString &t )
{
    QString regstr = t;
    convertText( regstr );
    try
    {
        Reg.setExpression( regstr );
        Reg.setCaseSensitive( MatchMenu->isItemChecked( MatchCaseID ) );
        if ( Multiple )
        {
            if ( SubMatch >= MatchCount )
                SubMatch = MatchCount - 1;
        }
        if ( SubValue > Reg.subCount() )
            SubValue = Reg.subCount();
        if ( SubMatch < 0 )
            SubMatch = 0;
        updateMatch();
    }
    catch ( QaRegExpError &e )
    {
        switch ( e.errorCode() )
        {
            case QaRegExpError::UnmatchedBracket:
            case QaRegExpError::InvalidRange:
            case QaRegExpError::UnmatchedParenthesis:
                ErrLabel->setText( tr( "Incomplete input" ) );
                break;
            default:
                ErrLabel->setText( e.error() );
        }
    }
}

/*!
  Converts backslashified expressions to their correct ascii values
*/

void QaRegExpWidget::convertText( QString &s )
{
    int i = 0;
    while ( ( i = s.find( "\\", i ) ) != -1 )
    {
        QChar c = s.at( i+1 );
        if ( c == "t" )
            s.replace( i, 2, "\t" );
        else if ( c == "n" )
            s.replace( i, 2, "\n" );
        else if ( c == "r" )
            s.replace( i, 2, "\r" );
        else if ( c == "f" )
            s.replace( i, 2, "\f" );
        else if ( c == "d" )
        {
            s.replace( i, 2, "[0-9]" );
            i += 4;
        }
        else if ( c == "D" )
        {
            s.replace( i, 2, "[^0-9]" );
            i += 4;
        }
        else if ( c == "w" )
        {
            QString str( QString( "[" ) + wordCharacters() + QString( "]" ) );
            s.replace( i, 2, str );
            i += str.length() - 1;
        }
        else if ( c == "W" )
        {
            QString str( QString( "[^" ) + wordCharacters() + QString( "]" ) );
            s.replace( i, 2, str );
            i += str.length() - 1;
        }
        else if ( c == "s" )
        {
            QString str( "[ \t\n\r\f]" );
            s.replace( i, 2, str );
            i += str.length() - 1;
        }
        else if ( c == "S" )
        {
            QString str( "[^ \t\n\r\f]" );
            s.replace( i, 2, str );
            i += str.length() - 1;
        }
        else
            i++;
        i++;
    }
}

QString QaRegExpWidget::wordCharacters()
{
    return QString( tr( "a-zA-Z_0-9" ) );
}


/*!
  Adds <br> where \n is located.
*/

QString QaRegExpWidget::toHtml( const QString &s )
{
    return QStyleSheet::convertFromPlainText( s );
}

/*!
  Called whenever the sub expression value changes.
*/

void QaRegExpWidget::valueChange( int val )
{
    if ( SubValue == val )
        return;
    SubValue = val;
    if ( Multiple )
        updateMatch();
    else
        updateOutput();
}

/*!
  Called whenever the sub expression value changes.
*/

void QaRegExpWidget::valueChangeMultiple( int val )
{
    if ( SubMatch == val )
        return;
    SubMatch = val;
    updateMatch();
}

/*!
  Update the output
*/

void QaRegExpWidget::updateOutput()
{
    if ( Multiple )
    {
        if ( Reg.matchBeginning( SubMatch ) != -1 )
        {
            QString st = Reg.matchString( SubMatch );
            QString str = EditString;
            QString font( "<font color=#ffff00><b>" );
            QString fontback( "</b></font>" );
            QaRegExpRange range = Reg.matches()[SubMatch];
            int index1 = range.start();
            int index2 = range.start() + range.length();

            QString front,back,mid;
            front = str.mid( 0, index1 );
            mid = str.mid( index1, range.length() );
            back = str.mid( index2 );

            front = toHtml( front );
            front = front.mid( 3, front.length() - 7 );
            mid = toHtml( mid );
            mid = mid.mid( 3, mid.length() - 7 );
            back = toHtml( back );
            back = back.mid( 3, back.length() - 7 );
            str = front + font + mid + fontback + back;
            TextVu->setText( str );
            ErrLabel->setText( tr( "Match" ) );
        }
        else
        {
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "No Match" ) );
        }
    }
    else
    {
        if ( Reg.matchBeginning( SubValue ) != -1 )
        {
            QString st = Reg.matchString( SubValue );
            QString str = EditString;
            QString font( "<font color=#ffff00><b>" );
            QString fontback( "</b></font>" );
            QaRegExpRange range = Reg.matches()[SubValue];
            int index1 = range.start();
            int index2 = range.start() + range.length();

            QString front,back,mid;
            front = str.mid( 0, index1 );
            mid = str.mid( index1, range.length() );
            back = str.mid( index2 );

            front = toHtml( front );
            front = front.mid( 3, front.length() - 7 );
            mid = toHtml( mid );
            mid = mid.mid( 3, mid.length() - 7 );
            back = toHtml( back );
            back = back.mid( 3, back.length() - 7 );
            str = front + font + mid + fontback + back;
            TextVu->setText( str );
            ErrLabel->setText( tr( "Match" ) );
        }
        else
        {
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "No Match" ) );
        }
    }
}

/*!
  Text in the multilinedit widget has changed.
*/

void QaRegExpWidget::multiTextChanged()
{
    QString s;
    for ( int i = 0; i < Edit->numLines(); i++ )
    {
        s += Edit->textLine( i );
        if ( i < Edit->numLines() - 1 )
            s+= "\n";
    }
    EditString = s;
    updateMatch();
}

/*!
  Do a rematch of the regexp.
*/

void QaRegExpWidget::updateMatch()
{
    if ( Multiple )
    {
        if ( ( MatchCount = Reg.matchMultiple( EditString, SubValue ) ) > 0 )
        {
            if ( SubMatch >= MatchCount )
                SubMatch = MatchCount - 1;
            updateOutput();
        }
        else
        {
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "NoMatch" ) );
        }
    }
    else
    {
        if ( Reg.match( EditString, SubValue ) )
        {
            updateOutput();
        }
        else
        {
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "NoMatch" ) );
        }
    }
    Sub->clear();
    Matches->clear();
    if ( Multiple )
    {
        for ( int i = 0; i < MatchCount; i++ )
        {
            QString str;
            str = QString( "%1 - [%2,%3]" ).arg( i ).arg( Reg.matchBeginning(i) ).arg( Reg.matchEnd(i) );
            Matches->insertItem( str, i );
        }
        for ( int i = 0; i < Reg.subCount() + 1; i++ )
        {
            QString str;
            str = QString( "%1" ).arg( i );
            Sub->insertItem( str, i );
        }
        Matches->setSelected( SubMatch, true );
        Matches->setCurrentItem( SubMatch );
    }
    else
    {
        for ( int i = 0; i < Reg.subCount() + 1; i++ )
        {
            QString str;
            if ( Reg.matches()[i].start() != -1 )
                str = QString( "%1 - [%2,%3]" ).arg( i ).arg( Reg.matchBeginning(i) ).arg( Reg.matchEnd(i) );
            else
                str = QString( tr( "(%1) - No match" ) ).arg( i );
            Sub->insertItem( str, i );
        }
    }
    Sub->setSelected( SubValue, true );
    Sub->setCurrentItem( SubValue );
}
