/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "textlist.h"
#include <stdlib.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkvscrollbar.h>


/* the number rows memchunk expands at a time */
#define TLIST_OPTIMUM_SIZE 512

/* minimum allowed width */
#define MIN_WIDTH 5

/* this defigns the base grid spacing */
#define CELL_SPACING 1

/* added the horizontal space at the beginning and end of a row*/
#define COLUMN_INSET 3

/* scrollbar spacing class macro */
#define SCROLLBAR_SPACING(w) (TEXTLIST_CLASS (GTK_OBJECT (w)->klass)->scrollbar_spacing)

/* gives the top pixel of the given row in context of
 * the tlist's voffset */
#define ROW_TOP_YPIXEL(tlist, row) (((tlist)->row_height * (row)) - (tlist)->voffset)

/* returns the row index from a y pixel location in the 
 * context of the tlist's voffset */
#define ROW_FROM_YPIXEL(tlist, y)  (((y) + (tlist)->voffset) / (tlist)->row_height)

/* returns the top pixel of the given row in the context of
 * the list height */
#define ROW_TOP(tlist, row)        ((tlist)->row_height * (row))

/* returns the total height of the list */
#define LIST_HEIGHT(tlist)         ((tlist)->row_height * ((tlist)->rows))

/* returns the total width of the list */
#define LIST_WIDTH(tlist)          ((tlist)->tlist_window_width)


/* Signals */
enum
{
  ADD, /* container */
  SELECT_ROW,
  UNSELECT_ROW,
  ACTION,
  MENU_POPUP,
  LAST_SIGNAL
};

enum
{
  SYNC_REMOVE,
  SYNC_INSERT
};


typedef void (*TextListSignal1)(TextList *object, gint arg);
typedef void (*TextListSignal2)(TextList *object, guint button, guint32 time);

/* TextList Methods */
static void text_list_class_init (TextListClass * klass);
static void text_list_init (TextList * tlist);

/* GtkObject Methods */
static void text_list_destroy (GtkObject * object);
static void text_list_finalize (GtkObject * object);


/* GtkWidget Methods */
static void text_list_realize (GtkWidget * widget);
static void text_list_unrealize (GtkWidget * widget);
static void text_list_map (GtkWidget * widget);
static void text_list_unmap (GtkWidget * widget);
static void text_list_draw (GtkWidget * widget, GdkRectangle * area);
static gint text_list_expose (GtkWidget * widget, GdkEventExpose * event);
static gint text_list_button_press (GtkWidget * widget, GdkEventButton * event);
static gint text_list_button_release (GtkWidget * widget, GdkEventButton * event);
static gint text_list_motion (GtkWidget * widget,  GdkEventMotion * event);

static void text_list_size_request (GtkWidget * widget, GtkRequisition * requisition);
static void text_list_size_allocate (GtkWidget * widget, GtkAllocation * allocation);
static gint get_selection_info (TextList * tlist, gint x, gint y, gint * row);

/* GtkContainer Methods */
static void text_list_foreach (GtkContainer * container, GtkCallback callback, gpointer callback_data);

/* Drawing */
static void draw_row_area (TextList * tlist, GdkRectangle * area, gint row);
static void draw_rows_area (TextList * tlist, GdkRectangle * area);

/* Selection */
static void toggle_row (TextList * tlist, gint row);
static void select_row (TextList * tlist, gint row);
static void unselect_row (TextList * tlist, gint row);

static void real_select_row (TextList * tlist, gint row);
static void real_unselect_row (TextList * tlist, gint row);

static void select_this_row (gint row, TextList *tlist);
static void unselect_this_row (gint row, TextList *tlist);

/* Scrollbars */
static void create_scrollbars (TextList * tlist);
static void adjust_scrollbars (TextList * tlist);
static void check_exposures   (TextList * tlist);
static void vadjustment_changed (GtkAdjustment * adjustment, gpointer data);
static void vadjustment_value_changed (GtkAdjustment * adjustment, gpointer data);

/* Signals */
static void text_list_marshal_signal1(GtkObject *object, GtkSignalFunc func, gpointer func_data, GtkArg *args);
static void text_list_marshal_signal2(GtkObject *object, GtkSignalFunc func, gpointer func_data, GtkArg *args);

/* Fill in data after widget is realized and has style */
static void add_style_data (TextList * tlist);

/* Added by Pixel */
static gint text_list_key_press (GtkWidget *widget, GdkEventKey *event);
static void text_list_active_to_selection(TextList *tlist);
static void text_list_toggle_to_selection(TextList *tlist);
static void text_list_extend_up_list(TextList *tlist);
static void text_list_keep_selection_up_list(TextList *tlist);
static void text_list_up_list(TextList *tlist);
static void text_list_extend_down_list(TextList *tlist);
static void text_list_keep_selection_down_list(TextList *tlist);
static void text_list_down_list(TextList *tlist);
static void text_list_page_up(TextList *tlist);
static void text_list_page_down(TextList *tlist);
static void text_list_keep_selection_page_up(TextList *tlist);
static void text_list_keep_selection_page_down(TextList *tlist);
static void text_list_extend_page_up(TextList *tlist);
static void text_list_extend_page_down(TextList *tlist);
static void text_list_to_bottom(TextList *tlist);
static void text_list_to_top (TextList *tlist);
static void text_list_keep_selection_to_bottom(TextList *tlist);
static void text_list_keep_selection_to_top(TextList *tlist);
static void text_list_extend_to_bottom(TextList *tlist);
static void text_list_extend_to_top(TextList *tlist);


static gboolean ensure_active_row_is_visible(TextList *tlist, gfloat row_align);
static gboolean is_active_row_selected(TextList *tlist);
static gboolean select_active_row(TextList *tlist);
static void select_range_active_last_active(TextList *tlist);
static void select_range(TextList *tlist, gint row_min, gint row_max);
static void draw_active_row(TextList *list);
static void draw_last_active_row(TextList *list);
static gboolean select_active_row_and_show(TextList *tlist);
static gboolean is_selected(TextList *tlist, gint row);

static void text_list_unselect_all(TextList *tlist);

/* scroll the viewing area of the list to the given 
 *     row; row_align and col_align are between 0-1 representing the
 * location the row should appear on the screnn, 0.0 being top or left,
 * 1.0 being bottom or right; if row           is -1 then then there
 * is no change */
static void text_list_moveto (TextList * tlist, gint row, gfloat row_align);

/* returns whether the row is visible */
static GtkVisibility text_list_row_is_visible (TextList * tlist, gint row);

static void text_list_draw_row (TextList * tlist, gint row);
static void text_list_draw_rows (TextList * tlist);



static GtkContainerClass *parent_class = NULL;
static guint tlist_signals[LAST_SIGNAL] = {0};


guint text_list_get_type ()
{
  static guint tlist_type = 0;

  if (!tlist_type)
    {
      GtkTypeInfo tlist_info = {
	"TextList",
	sizeof (TextList),
	sizeof (TextListClass),
	(GtkClassInitFunc) text_list_class_init,
	(GtkObjectInitFunc) text_list_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      tlist_type = gtk_type_unique (gtk_container_get_type (), &tlist_info);
    }
  return tlist_type;
}

static void container_add (GtkContainer *container, GtkWidget *child)
{
  TEXTLIST(container)->child = child;
  
  gtk_widget_set_parent (child, GTK_WIDGET(container));  

  if (GTK_WIDGET_VISIBLE (GTK_WIDGET (container)))
    {
      if (GTK_WIDGET_REALIZED (GTK_WIDGET (container)) && !GTK_WIDGET_REALIZED (child))
	gtk_widget_realize (child);
      
      if (GTK_WIDGET_MAPPED (GTK_WIDGET (container)) && !GTK_WIDGET_MAPPED (child))
	gtk_widget_map (child);
      
      if (GTK_WIDGET_VISIBLE (child))
	gtk_widget_queue_resize (child);
    }
}

static void container_remove (GtkContainer *container, GtkWidget *child)
{
  gtk_widget_unparent (child);  
  TEXTLIST(container)->child = NULL;
}


static void text_list_class_init (TextListClass * klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;
  container_class = (GtkContainerClass *) klass;

  parent_class = gtk_type_class (gtk_container_get_type ());

  tlist_signals[SELECT_ROW] =
    gtk_signal_new ("select_row", GTK_RUN_FIRST, object_class->type,
		    GTK_SIGNAL_OFFSET (TextListClass, select_row),
		    text_list_marshal_signal1,
		    GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  tlist_signals[UNSELECT_ROW] =
    gtk_signal_new ("unselect_row", GTK_RUN_FIRST, object_class->type,
		    GTK_SIGNAL_OFFSET (TextListClass, unselect_row),
		    text_list_marshal_signal1,
		    GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  tlist_signals[ACTION] = 
    gtk_signal_new ("action", GTK_RUN_LAST, object_class->type, 
		    GTK_SIGNAL_OFFSET (TextListClass, action), 
		    gtk_signal_default_marshaller, 
		    GTK_TYPE_NONE, 0);
  tlist_signals[MENU_POPUP] = 
    gtk_signal_new ("menu_popup", GTK_RUN_LAST, object_class->type, 
		    GTK_SIGNAL_OFFSET (TextListClass, menu_popup), 
		    text_list_marshal_signal2,
		    GTK_TYPE_NONE, 2, GTK_TYPE_UINT, GTK_TYPE_ULONG);

  gtk_object_class_add_signals (object_class, tlist_signals, LAST_SIGNAL);

  object_class->destroy = text_list_destroy;
  object_class->finalize = text_list_finalize;

  widget_class->realize = text_list_realize;
  widget_class->unrealize = text_list_unrealize;
  widget_class->map = text_list_map;
  widget_class->unmap = text_list_unmap;
  widget_class->draw = text_list_draw;
  widget_class->key_press_event = text_list_key_press;
  widget_class->button_press_event = text_list_button_press;
  widget_class->button_release_event = text_list_button_release;
  widget_class->motion_notify_event = text_list_motion;
  widget_class->expose_event = text_list_expose;
  widget_class->size_request = text_list_size_request;
  widget_class->size_allocate = text_list_size_allocate;

  /*container_class->foreach = text_list_foreach;*/
  container_class->add = container_add;
  container_class->remove = container_remove;

  klass->select_row = real_select_row;
  klass->unselect_row = real_unselect_row;
  klass->action = NULL;
  klass->menu_popup = NULL;
  

  klass->scrollbar_spacing = 5;
}

static void text_list_marshal_signal1(GtkObject *object, GtkSignalFunc func, gpointer func_data, GtkArg *args)
{
  TextListSignal1 rfunc;

  rfunc = (TextListSignal1) func;
  (*rfunc) ((TextList *) object, GTK_VALUE_INT (args[0]));
}
static void text_list_marshal_signal2(GtkObject *object, GtkSignalFunc func, gpointer func_data, GtkArg *args)
{
  TextListSignal2 rfunc;

  rfunc = (TextListSignal2) func;
  (*rfunc) ((TextList *) object, GTK_VALUE_UINT(args[0]), GTK_VALUE_ULONG(args[1]));
}

static void text_list_init (TextList * tlist)
{
  tlist->child = NULL;

  tlist->flags = 0;

  GTK_WIDGET_UNSET_FLAGS (tlist, GTK_NO_WINDOW);
  GTK_WIDGET_SET_FLAGS(tlist, GTK_CAN_FOCUS);

  tlist->rows = 0;
  tlist->row_center_offset = 0;
  tlist->row_height = 0;
  tlist->row_array = NULL;

  tlist->tlist_window = NULL;
  tlist->tlist_window_width = 1;
  tlist->tlist_window_height = 1;

  tlist->voffset = 0;

  tlist->shadow_type = GTK_SHADOW_IN;
  tlist->vscrollbar_policy = GTK_POLICY_AUTOMATIC;

  tlist->fg_gc = NULL;
  tlist->bg_gc = NULL;

  tlist->selection = NULL;
}

/*
 * GTKTLIST PUBLIC INTERFACE
 *   text_list_new
 */
GtkWidget *text_list_new ()
{
  TextList *tlist;

  tlist = gtk_type_new (text_list_get_type ());

  /* create scrollbars */
  create_scrollbars (tlist);

  return GTK_WIDGET (tlist);
}

void text_list_moveto (TextList * tlist, gint row, gfloat row_align)
{
  gint x;

  g_return_if_fail (tlist != NULL);

  if (row < 0 || row >= tlist->rows) return;

  /* adjust vertical scrollbar */
  x = ROW_TOP (tlist, row) - (row_align * (tlist->tlist_window_height - tlist->row_height));
  
  if (x < 0) 
    GTK_RANGE (tlist->vscrollbar)->adjustment->value = 0.0;
  else if (x > LIST_HEIGHT (tlist) - tlist->tlist_window_height)
    GTK_RANGE (tlist->vscrollbar)->adjustment->value = LIST_HEIGHT (tlist) - 
      tlist->tlist_window_height;
  else
    GTK_RANGE (tlist->vscrollbar)->adjustment->value = x;
  
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (tlist->vscrollbar)->adjustment), "value_changed");
}

void text_list_set_text (TextList * tlist, gint row, gchar * text)
{
  g_return_if_fail (tlist != NULL);
  if (row < 0 || row >= tlist->rows) return;

  tlist->row_array[row].text = text;
}

void text_list_set_foreground (TextList * tlist, gint row, GdkColor * color)
{
  g_return_if_fail (tlist != NULL);
  if (row < 0 || row >= tlist->rows) return;

  tlist->row_array[row].foreground = color;
}

void text_list_clear (TextList * tlist)
{
  g_return_if_fail (tlist != NULL);

  free(tlist->row_array);
  g_list_free (tlist->selection);

  tlist->row_array = NULL;
  tlist->selection = NULL;
  //tlist->voffset = 0;
  tlist->rows = 0;
}

void text_list_unselect_row (TextList * tlist, gint row)
{
  g_return_if_fail (tlist != NULL);
  if (row < 0 || row >= tlist->rows) return;

  unselect_row (tlist, row);
}

GtkVisibility text_list_row_is_visible (TextList * tlist, gint row)
{
  gint top;

  g_return_val_if_fail (tlist != NULL, 0);

  if (row < 0 || row >= tlist->rows) return GTK_VISIBILITY_NONE;
  if (tlist->row_height == 0) return GTK_VISIBILITY_NONE;
  if (row < ROW_FROM_YPIXEL (tlist, 0)) return GTK_VISIBILITY_NONE;
  if (row > ROW_FROM_YPIXEL (tlist, tlist->tlist_window_height)) return GTK_VISIBILITY_NONE;

  top = ROW_TOP_YPIXEL (tlist, row);
  if ((top < 0) || ((top + tlist->row_height) >= tlist->tlist_window_height))
    return GTK_VISIBILITY_PARTIAL;

  return GTK_VISIBILITY_FULL;
}

GtkAdjustment * text_list_get_vadjustment (TextList * tlist)
{
  g_return_val_if_fail (tlist != NULL, NULL);
  g_return_val_if_fail (IS_TEXTLIST (tlist), NULL);

  return gtk_range_get_adjustment (GTK_RANGE (tlist->vscrollbar));
}

/*
 * GTKOBJECT
 *   text_list_destroy
 *   text_list_finalize
 */
static void text_list_destroy (GtkObject * object)
{
  TextList *tlist;

  g_return_if_fail (object != NULL);
  g_return_if_fail (IS_TEXTLIST (object));

  tlist = TEXTLIST (object);

  /* get rid of all the rows */
  text_list_clear (tlist);

  /* destroy the scrollbars */
  if (tlist->vscrollbar) {
    gtk_widget_unparent (tlist->vscrollbar);
    tlist->vscrollbar = NULL;
  }

  if (tlist->child) {
    gtk_widget_unparent (tlist->child);
    tlist->child = NULL;
  }

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void text_list_finalize (GtkObject * object)
{
  TextList *tlist;

  g_return_if_fail (object != NULL);
  g_return_if_fail (IS_TEXTLIST (object));

  tlist = TEXTLIST (object);

  if (GTK_OBJECT_CLASS (parent_class)->finalize)
    (*GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

/*
 * GTKWIDGET
 *   text_list_realize
 *   text_list_unrealize
 *   text_list_map
 *   text_list_unmap
 *   text_list_draw
 *   text_list_expose
 *   text_list_button_press
 *   text_list_button_release
 *   text_list_button_motion
 *   text_list_size_request
 *   text_list_size_allocate
 */
static void text_list_realize (GtkWidget * widget)
{
  TextList *tlist;
  GdkWindowAttr attributes;
  gint attributes_mask;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));

  tlist = TEXTLIST (widget);

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  add_style_data (tlist);
  border_width = GTK_CONTAINER (widget)->border_width;
  
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x + border_width;
  attributes.y = widget->allocation.y + border_width;
  attributes.width = widget->allocation.width - border_width * 2;
  attributes.height = widget->allocation.height - border_width * 2;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_KEY_PRESS_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  /* main window */
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, tlist);

  widget->style = gtk_style_attach (widget->style, widget->window);
  widget->style->bg_gc[GTK_STATE_NORMAL] = widget->style->white_gc;

  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

  /* tlist-window */
  attributes.x = tlist->internal_allocation.x + widget->style->klass->xthickness;
  attributes.y = tlist->internal_allocation.y + widget->style->klass->ythickness;
  attributes.width = tlist->tlist_window_width;
  attributes.height = tlist->tlist_window_height;
  
  tlist->tlist_window = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (tlist->tlist_window, tlist);

  gdk_window_set_background (tlist->tlist_window, &widget->style->white);
  gdk_window_show (tlist->tlist_window);
  gdk_window_get_size (tlist->tlist_window, &tlist->tlist_window_width, &tlist->tlist_window_height);

  /* GCs */
  tlist->fg_gc = gdk_gc_new (widget->window);
  tlist->bg_gc = gdk_gc_new (widget->window);
  
  /* We'll use this gc to do scrolling as well */
  gdk_gc_set_exposures (tlist->fg_gc, TRUE);
}

static void text_list_unrealize (GtkWidget * widget)
{
  TextList *tlist;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));

  tlist = TEXTLIST (widget);

  gdk_gc_destroy (tlist->fg_gc);
  gdk_gc_destroy (tlist->bg_gc);

  gdk_window_set_user_data (tlist->tlist_window, NULL);
  gdk_window_destroy (tlist->tlist_window);
  tlist->tlist_window = NULL;

  tlist->fg_gc = NULL;
  tlist->bg_gc = NULL;

  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void text_list_map (GtkWidget * widget)
{
  TextList *tlist;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));

  tlist = TEXTLIST (widget);

  if (!GTK_WIDGET_MAPPED (widget))
    {
      GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);

      gdk_window_show (widget->window);
      gdk_window_show (tlist->tlist_window);

      /* map vscrollbars */
      if (GTK_WIDGET_VISIBLE (tlist->vscrollbar) &&
	  !GTK_WIDGET_MAPPED (tlist->vscrollbar))
	gtk_widget_map (tlist->vscrollbar);

      if (tlist->child && GTK_WIDGET_VISIBLE (tlist->child) && !GTK_WIDGET_MAPPED (tlist->child))
	gtk_widget_map (tlist->child);
    }
}

static void text_list_unmap (GtkWidget * widget)
{
  TextList *tlist;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));

  tlist = TEXTLIST (widget);

  if (GTK_WIDGET_MAPPED (widget))
    {
      GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);

      gdk_window_hide (tlist->tlist_window);
      gdk_window_hide (widget->window);

      /* unmap scrollbars */
      if (GTK_WIDGET_MAPPED (tlist->vscrollbar))
	gtk_widget_unmap (tlist->vscrollbar);

      if (tlist->child && GTK_WIDGET_VISIBLE (tlist->child) && GTK_WIDGET_MAPPED (tlist->child))
	gtk_widget_unmap (tlist->child);
    }
}

static void text_list_draw (GtkWidget * widget, GdkRectangle * area)
{
  TextList *tlist;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      tlist = TEXTLIST (widget);
      border_width = GTK_CONTAINER (widget)->border_width;

      gdk_window_clear_area (widget->window,
			     area->x - border_width, 
			     area->y - border_width,
			     area->width, area->height);

      /* draw list shadow/border */
      gtk_draw_shadow (widget->style, widget->window,
		       GTK_STATE_NORMAL, tlist->shadow_type,
		       0, 0, 
		       tlist->tlist_window_width + (2 * widget->style->klass->xthickness),
		       tlist->tlist_window_height + (2 * widget->style->klass->ythickness));

      gdk_window_clear_area (tlist->tlist_window,
			     0, 0, -1, -1);

      text_list_draw_rows (tlist);

      if (tlist->child) gtk_widget_draw (tlist->child, area);      
    }
}

static gint text_list_expose (GtkWidget * widget, GdkEventExpose * event)
{
  TextList *tlist;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_TEXTLIST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      tlist = TEXTLIST (widget);

      /* draw border */
      if (event->window == widget->window)
	gtk_draw_shadow (widget->style, widget->window,
			 GTK_STATE_NORMAL, tlist->shadow_type,
			 0, 0,
			 tlist->tlist_window_width + (2 * widget->style->klass->xthickness),
			 tlist->tlist_window_height + (2 * widget->style->klass->ythickness));

      /* exposure events on the list */
      if (event->window == tlist->tlist_window)
	draw_rows_area (tlist, &event->area);
    }

  return FALSE;
}


static gint text_list_key_press (GtkWidget *widget, GdkEventKey *event)
{
  TextList *tlist;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_TEXTLIST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  tlist = TEXTLIST (widget);
  switch (event->keyval) 
    {
    case GDK_space:
      if (event->state & GDK_SHIFT_MASK)
	text_list_toggle_to_selection(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_toggle_to_selection(tlist);
      else
	text_list_active_to_selection(tlist);
      break;

    case GDK_Return:
      gtk_signal_emit(GTK_OBJECT(widget), tlist_signals[ACTION]);
      break;

    case GDK_Up: 
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_up_list(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_up_list(tlist);
      else
	text_list_up_list(tlist);
      break;
      
    case GDK_Down: 
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_down_list(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_down_list(tlist);
      else
	text_list_down_list(tlist);
      break;

    case GDK_Page_Up:
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_page_up(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_page_up(tlist);
      else
	text_list_page_up(tlist);
      break;
      
    case GDK_Page_Down:
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_page_down(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_page_down(tlist);
      else
	text_list_page_down(tlist);
      break;

    case GDK_Home:
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_to_top(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_to_top(tlist);
      else
	text_list_to_top(tlist);
      break;
      
    case GDK_End:
      if (event->state & GDK_SHIFT_MASK)
	text_list_extend_to_bottom(tlist);
      else if (event->state & GDK_CONTROL_MASK)
	text_list_keep_selection_to_bottom(tlist);
      else
	text_list_to_bottom(tlist);
      break;

      /*case GDK_F10:
      XWarpPointer(GDK_DISPLAY(), None, GDK_WINDOW_XWINDOW(tlist->tlist_window), 0, 0, 0, 0, 
		   tlist->tlist_window_width * 3 / 4, ROW_TOP_YPIXEL(tlist, tlist->active_row) + tlist->row_height / 2);
      gtk_signal_emit(GTK_OBJECT(widget), tlist_signals[MENU_POPUP], );
      */

    default: return FALSE;
    }
  return TRUE;
}


static gint text_list_button_press (GtkWidget * widget, GdkEventButton * event)
{
  TextList *tlist;
  gint x, y, row;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (IS_TEXTLIST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  tlist = TEXTLIST (widget);
  /* selections on the list */
  if (event->window == tlist->tlist_window)
    {
      x = event->x;
      y = event->y;

      if (get_selection_info (tlist, x, y, &row)) 
	{
	  text_list_set_active_row(tlist, row);

	  if (event->button == 1) 
	    {
	      if (event->state & GDK_SHIFT_MASK) 
		{
		  select_range_active_last_active(tlist);
		} 
	      else if (event->state & GDK_CONTROL_MASK) 
		{
		  toggle_row(tlist, row);
		  draw_last_active_row(tlist);
		  draw_active_row(tlist);
		} 
	      else 
		{
		  if (g_list_length(tlist->selection) == 1 && (int) tlist->selection->data == row) {
		    gtk_signal_emit(GTK_OBJECT(tlist), tlist_signals[ACTION]);
		  } else {
		    text_list_unselect_all(tlist);
		    select_active_row_and_show(tlist);
		  }
		}
	    }
	  else {
	    if (tlist->active_row != tlist->last_active_row) {
	      draw_last_active_row(tlist);
	      draw_active_row(tlist);
	    }
	    gtk_signal_emit(GTK_OBJECT(tlist), tlist_signals[MENU_POPUP], event->button, event->time);
	  }
	}

      return FALSE;
    }

  return FALSE;
}

static gint text_list_button_release (GtkWidget * widget, GdkEventButton * event)
{
  return FALSE;
}

static gint text_list_motion (GtkWidget * widget, GdkEventMotion * event)
{
  return TRUE;
}

static void text_list_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  TextList *tlist;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));
  g_return_if_fail (requisition != NULL);

  tlist = TEXTLIST (widget);

  add_style_data (tlist);

  requisition->height = tlist->row_height * 8;
  requisition->width = 0;

  /* add the vscrollbar space */
  if ((tlist->vscrollbar_policy == GTK_POLICY_AUTOMATIC) ||
      GTK_WIDGET_VISIBLE (tlist->vscrollbar))
    {
      gtk_widget_size_request (tlist->vscrollbar, &tlist->vscrollbar->requisition);

      requisition->width += tlist->vscrollbar->requisition.width + SCROLLBAR_SPACING (tlist);
      requisition->height = MAX (requisition->height,
				 tlist->vscrollbar->requisition.height);
    }
  if (tlist->child) gtk_widget_size_request (tlist->child, &tlist->child->requisition);  

  requisition->width += widget->style->klass->xthickness * 2 +
    GTK_CONTAINER (widget)->border_width * 2;
  requisition->height += widget->style->klass->ythickness * 2 +
    GTK_CONTAINER (widget)->border_width * 2;
}

static void text_list_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  TextList *tlist;
  GtkAllocation tlist_allocation;
  GtkAllocation child_allocation;
  gint vscrollbar_vis, hscrollbar_vis;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (IS_TEXTLIST (widget));
  g_return_if_fail (allocation != NULL);

  tlist = TEXTLIST (widget);
  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x + GTK_CONTAINER (widget)->border_width,
			      allocation->y + GTK_CONTAINER (widget)->border_width,
			      allocation->width - GTK_CONTAINER (widget)->border_width * 2,
			      allocation->height - GTK_CONTAINER (widget)->border_width * 2);
    }

  /* use internal allocation structure for all the math
   * because it's easier than always substracting the container
   * border width */
  tlist->internal_allocation.x = 0;
  tlist->internal_allocation.y = 0;
  tlist->internal_allocation.width = MAX (1, allocation->width -
    GTK_CONTAINER (widget)->border_width * 2);
  tlist->internal_allocation.height = MAX (1, allocation->height -
    GTK_CONTAINER (widget)->border_width * 2);
	
  /* allocate tlist window assuming no scrollbars */
  tlist_allocation.x = tlist->internal_allocation.x + widget->style->klass->xthickness;
  tlist_allocation.y = tlist->internal_allocation.y + widget->style->klass->ythickness;
  tlist_allocation.width = MAX (1, tlist->internal_allocation.width - 
    (2 * widget->style->klass->xthickness));
  tlist_allocation.height = MAX (1, tlist->internal_allocation.height -
    (2 * widget->style->klass->ythickness));
  
  /* here's where we decide to show/not show the scrollbars */
  vscrollbar_vis = 0;
  hscrollbar_vis = 0;
  
  if (LIST_HEIGHT (tlist) <= tlist_allocation.height &&
      tlist->vscrollbar_policy == GTK_POLICY_AUTOMATIC) vscrollbar_vis = 0;
  else
    {
      if (!vscrollbar_vis)
	{
	  vscrollbar_vis = 1;
	  tlist_allocation.width = MAX (1, tlist_allocation.width - 
					(tlist->vscrollbar->requisition.width +
					 SCROLLBAR_SPACING (tlist)));
	}  
    } 

  tlist->tlist_window_width = tlist_allocation.width;
  tlist->tlist_window_height = tlist_allocation.height;
  
  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (tlist->tlist_window,
			      tlist_allocation.x,
			      tlist_allocation.y,
			      tlist_allocation.width,
			      tlist_allocation.height);
    }
  
  adjust_scrollbars (tlist);
  
  /* allocate the vscrollbar */
  if (vscrollbar_vis)
    {
      if (!GTK_WIDGET_VISIBLE (tlist->vscrollbar))
	gtk_widget_show (tlist->vscrollbar);
      
      child_allocation.x = tlist->internal_allocation.x + 
	tlist->internal_allocation.width -
	tlist->vscrollbar->requisition.width;
      child_allocation.y = tlist->internal_allocation.y;
      child_allocation.width = tlist->vscrollbar->requisition.width;
      child_allocation.height = MAX (1, tlist->internal_allocation.height);
      
      gtk_widget_size_allocate (tlist->vscrollbar, &child_allocation);
    }
  else
    {
      if (GTK_WIDGET_VISIBLE (tlist->vscrollbar))
	gtk_widget_hide (tlist->vscrollbar);
    }

  if (tlist->child) {
    child_allocation.x = 0;
    child_allocation.y = 0;
    child_allocation.width = tlist->child->requisition.width;
    child_allocation.height = tlist->child->requisition.height;

    gtk_widget_size_allocate (tlist->child, &child_allocation);
  }
  
  /* set the vscrollbar adjustments */
  adjust_scrollbars (tlist);
}

/* 
 * GTKCONTAINER
 *   text_list_foreach
 */
static void text_list_foreach (GtkContainer * container, GtkCallback callback, gpointer callback_data)
{
  TextList *tlist;

  g_return_if_fail (container != NULL);
  g_return_if_fail (IS_TEXTLIST (container));
  g_return_if_fail (callback != NULL);

  tlist = TEXTLIST (container);

  /* callbacks for the scrollbars */
  if (tlist->vscrollbar) (*callback) (tlist->vscrollbar, callback_data);

  if (tlist->child) (*callback) (tlist->child, callback_data);
}

/*
 * DRAWING
 *   draw_row
 *   draw_rows
 */
static void draw_rectangle(TextList *tlist, GdkGC *bg_gc, GdkRectangle *rect)
{
  gdk_draw_rectangle (tlist->tlist_window, bg_gc, TRUE, rect->x, rect->y, rect->width, rect->height);
}
static void draw_black_rectangle(TextList *tlist, GdkRectangle *rect)
{
  draw_rectangle(tlist, GTK_WIDGET(tlist)->style->black_gc, rect);
}

void text_list_draw_row(TextList * tlist, gint row) 
{
  
  if (text_list_row_is_visible (tlist, row) != GTK_VISIBILITY_NONE)
    draw_row_area(tlist, NULL, row);
}

static void draw_row_area(TextList * tlist, GdkRectangle * area, gint row)
{
  TextListRow * tlist_row;
  GtkWidget *widget;
  GdkGC *fg_gc, *bg_gc;
  GdkRectangle row_rectangle, clip_rectangle, *rect, r;

  g_return_if_fail (tlist != NULL);

  /* bail now if we arn't drawable yet */
  if (!GTK_WIDGET_DRAWABLE (tlist)) return;
  if (row < 0 || row >= tlist->rows) return;

  widget = GTK_WIDGET (tlist);
  tlist_row = &tlist->row_array[row];

  /* select GC for background rectangle */
  if (tlist_row->state == GTK_STATE_SELECTED) {
    fg_gc = widget->style->fg_gc[GTK_STATE_SELECTED];
    bg_gc = widget->style->bg_gc[GTK_STATE_SELECTED];
  } else {
    if (tlist_row->foreground) {
      gdk_gc_set_foreground (tlist->fg_gc, tlist_row->foreground);
      fg_gc = tlist->fg_gc;
    } else fg_gc = widget->style->fg_gc[GTK_STATE_NORMAL];
    
    if (tlist_row->background) {
      gdk_gc_set_foreground (tlist->bg_gc, tlist_row->background);
      bg_gc = tlist->bg_gc;
    } else bg_gc = widget->style->bg_gc[(tlist->active_row == row) ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL];
  }
  
  /* rectangle of the entire row */
  row_rectangle.x = 0;
  row_rectangle.y = ROW_TOP_YPIXEL (tlist, row);
  row_rectangle.width = tlist->tlist_window_width;
  row_rectangle.height = tlist->row_height;

  /* draw the cell borders and background */
  if (area) {
    if (!gdk_rectangle_intersect (area, &row_rectangle, &r)) return;
    draw_rectangle(tlist, bg_gc, &r);  
  } else draw_rectangle(tlist, bg_gc, &row_rectangle);  

  if (row == tlist->active_row)
    {
      GdkRectangle underline;
      underline.x = COLUMN_INSET;
      underline.y = row_rectangle.y + tlist->row_height - CELL_SPACING;
      underline.width = row_rectangle.width;
      underline.height = CELL_SPACING;
      
      if (area) {
	if (gdk_rectangle_intersect(area, &underline, &r))
	  draw_black_rectangle(tlist, &r);
      } else draw_black_rectangle(tlist, &underline);
    }  

  /* rectangle used to clip drawing operations, it's y and height
   * positions only need to be set once, so we set them once here. 
   * the x and width are set withing the drawing loop below once per
   * column */
  clip_rectangle.x = COLUMN_INSET;
  clip_rectangle.y = row_rectangle.y;
  clip_rectangle.width = row_rectangle.width;
  clip_rectangle.height = row_rectangle.height;

  if (area) {
    gdk_rectangle_intersect (area, &clip_rectangle, &r);
    rect = &r;
  } else rect = &clip_rectangle;

  gdk_gc_set_clip_rectangle (fg_gc, rect);
  
  gdk_draw_string (tlist->tlist_window, 
		   widget->style->font,
		   fg_gc,
		   clip_rectangle.x,
		   row_rectangle.y + tlist->row_center_offset,
		   tlist_row->text);
  
  gdk_gc_set_clip_rectangle (fg_gc, NULL);
}

void text_list_draw_rows(TextList * tlist) { draw_rows_area(tlist, NULL); }
static void draw_rows_area (TextList * tlist, GdkRectangle * area)
{
  int i, first_row, last_row;

  g_return_if_fail (tlist != NULL);
  g_return_if_fail (IS_TEXTLIST (tlist));

  if (tlist->row_height == 0 || !GTK_WIDGET_DRAWABLE (tlist)) return;

  if (area)
    {
      first_row = ROW_FROM_YPIXEL (tlist, area->y);
      last_row = ROW_FROM_YPIXEL (tlist, area->y + area->height);
    }
  else
    {
      first_row = ROW_FROM_YPIXEL (tlist, 0);
      last_row = ROW_FROM_YPIXEL (tlist, tlist->tlist_window_height);
    }

  /* this is a small special case which exposes the bottom cell line
   * on the last row -- it might go away if I change the wall the cell spacings
   * are drawn */
  if (tlist->rows == first_row) first_row--;

  for (i = first_row; i < tlist->rows; i++) 
    {
      if (i > last_row) return;
      draw_row_area (tlist, area, i);
    }

  if (!area) gdk_window_clear_area (tlist->tlist_window, 0, ROW_TOP_YPIXEL (tlist, i), -1, -1);
}

/*
 * SELECTION
 *   select_row
 *   real_select_row
 *   real_unselect_row
 *   get_selection_info
 */
static void toggle_row (TextList * tlist, gint row)
{
  if (tlist->row_array[row].state == GTK_STATE_SELECTED)
    unselect_row(tlist, row);
  else
    select_row(tlist, row);
}

static void select_row (TextList * tlist, gint row)
{
  gtk_signal_emit (GTK_OBJECT (tlist), tlist_signals[SELECT_ROW], row);
}

static void unselect_row (TextList * tlist, gint row)
{
  gtk_signal_emit (GTK_OBJECT (tlist), tlist_signals[UNSELECT_ROW], row);
}

static void real_select_row (TextList * tlist, gint row)
{
  g_return_if_fail (tlist != NULL);
  if (row < 0 || row > (tlist->rows - 1)) return;

  if (tlist->row_array[row].state == GTK_STATE_NORMAL)
    {
      tlist->row_array[row].state = GTK_STATE_SELECTED;
      tlist->selection = g_list_append (tlist->selection, (gpointer) row);

      text_list_draw_row (tlist, row);
    }
}

static void real_unselect_row (TextList * tlist, gint row)
{
  g_return_if_fail (tlist != NULL);
  if (row < 0 || row > (tlist->rows - 1)) return;

  if (tlist->row_array[row].state == GTK_STATE_SELECTED)
    {
      tlist->row_array[row].state = GTK_STATE_NORMAL;
      tlist->selection = g_list_remove (tlist->selection, (gpointer) row);

      text_list_draw_row (tlist, row);
    }
}

static gint get_selection_info (TextList * tlist, gint x, gint y, gint * row)
{
  g_return_val_if_fail (tlist != NULL, 0);

  /* bounds checking, return false if the user clicked 
   * on a blank area */
  *row = ROW_FROM_YPIXEL (tlist, y);
  if (*row >= tlist->rows) return 0;

  /* TODO */
  /*if (x < tlist->tlist_window_width)
    return 0;*/

  return 1;
}

/* 
 * SCROLLBARS
 *
 * functions:
 *   create_scrollbars
 *   adjust_scrollbars
 *   vadjustment_changed
 *   hadjustment_changed
 *   vadjustment_value_changed
 *   hadjustment_value_changed 
 */
static void create_scrollbars (TextList * tlist)
{
  GtkAdjustment *adjustment;

  tlist->vscrollbar = gtk_vscrollbar_new (NULL);
  GTK_WIDGET_UNSET_FLAGS(tlist->vscrollbar, GTK_CAN_FOCUS);
  adjustment = gtk_range_get_adjustment (GTK_RANGE (tlist->vscrollbar));

  gtk_signal_connect (GTK_OBJECT (adjustment), "changed",
		      (GtkSignalFunc) vadjustment_changed,
		      (gpointer) tlist);

  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
		      (GtkSignalFunc) vadjustment_value_changed,
		      (gpointer) tlist);

  gtk_widget_set_parent (tlist->vscrollbar, GTK_WIDGET (tlist));
  gtk_widget_show (tlist->vscrollbar);
}

static void adjust_scrollbars (TextList * tlist)
{
  GTK_RANGE (tlist->vscrollbar)->adjustment->page_size = tlist->tlist_window_height;
  GTK_RANGE (tlist->vscrollbar)->adjustment->page_increment = tlist->tlist_window_height / 2;
  GTK_RANGE (tlist->vscrollbar)->adjustment->step_increment = 10;
  GTK_RANGE (tlist->vscrollbar)->adjustment->lower = 0;
  GTK_RANGE (tlist->vscrollbar)->adjustment->upper = LIST_HEIGHT (tlist);

  if (tlist->tlist_window_height + tlist->voffset > LIST_HEIGHT (tlist))
    {
      GTK_RANGE (tlist->vscrollbar)->adjustment->value = MAX (0, LIST_HEIGHT (tlist) - 
	tlist->tlist_window_height);
      gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (tlist->vscrollbar)->adjustment), 
			       "value_changed");
    }

  if (LIST_HEIGHT (tlist) <= tlist->tlist_window_height &&
      tlist->vscrollbar_policy == GTK_POLICY_AUTOMATIC)
    {
      if (GTK_WIDGET_VISIBLE (tlist->vscrollbar))
	{
	  gtk_widget_hide (tlist->vscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (tlist));
	}
    }
  else
    {
      if (!GTK_WIDGET_VISIBLE (tlist->vscrollbar))
	{
	  gtk_widget_show (tlist->vscrollbar);
	  gtk_widget_queue_resize (GTK_WIDGET (tlist));
	}
    }

  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (tlist->vscrollbar)->adjustment), "changed");
}

static void vadjustment_changed (GtkAdjustment * adjustment, gpointer data)
{
  TextList *tlist;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);

  tlist = TEXTLIST (data);
}

static void check_exposures (TextList *tlist)
{
  GdkEvent *event;

  if (!GTK_WIDGET_REALIZED (tlist))
    return;

  /* Make sure graphics expose events are processed before scrolling
   * again */
  while ((event = gdk_event_get_graphics_expose (tlist->tlist_window)) != NULL)
    {
      gtk_widget_event (GTK_WIDGET (tlist), event);
      if (event->expose.count == 0)
	{
	  gdk_event_free (event);
	  break;
	}
      gdk_event_free (event);
    }
}

static void vadjustment_value_changed (GtkAdjustment * adjustment, gpointer data)
{
  TextList *tlist;
  GdkRectangle area;
  gint diff, value;

  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (data != NULL);
  g_return_if_fail (IS_TEXTLIST (data));

  tlist = TEXTLIST (data);

  if (!GTK_WIDGET_DRAWABLE (tlist))
    return;

  value = adjustment->value;

  if (adjustment == gtk_range_get_adjustment (GTK_RANGE (tlist->vscrollbar)))
    {
      if (value > tlist->voffset)
	{
	  /* scroll down */
	  diff = value - tlist->voffset;

	  /* we have to re-draw the whole screen here... */
	  if (diff >= tlist->tlist_window_height)
	    {
	      tlist->voffset = value;
	      text_list_draw_rows (tlist);
	      return;
	    }

	  if ((diff != 0) && (diff != tlist->tlist_window_height))
	    gdk_window_copy_area (tlist->tlist_window,
				  tlist->fg_gc,
				  0, 0,
				  tlist->tlist_window,
				  0,
				  diff,
				  tlist->tlist_window_width,
				  tlist->tlist_window_height - diff);

	  area.x = 0;
	  area.y = tlist->tlist_window_height - diff;
	  area.width = tlist->tlist_window_width;
	  area.height = diff;
	}
      else
	{
	  /* scroll up */
	  diff = tlist->voffset - value;

	  /* we have to re-draw the whole screen here... */
	  if (diff >= tlist->tlist_window_height)
	    {
	      tlist->voffset = value;
	      text_list_draw_rows (tlist);
	      return;
	    }

	  if ((diff != 0) && (diff != tlist->tlist_window_height))
	    gdk_window_copy_area (tlist->tlist_window,
				  tlist->fg_gc,
				  0, diff,
				  tlist->tlist_window,
				  0,
				  0,
				  tlist->tlist_window_width,
				  tlist->tlist_window_height - diff);

	  area.x = 0;
	  area.y = 0;
	  area.width = tlist->tlist_window_width;
	  area.height = diff;

	}

      tlist->voffset = value;
      if ((diff != 0) && (diff != tlist->tlist_window_height))
	check_exposures (tlist);
    }

  draw_rows_area (tlist, &area);
}

/* 
 * Memory Allocation/Distruction Routines for TextList stuctures
 *
 */
void text_list_set_size(TextList * tlist, gint size)
{
  gint i;
  
  g_return_if_fail (tlist != NULL);
  g_return_if_fail (IS_TEXTLIST (tlist));

  if (size != tlist->rows) tlist->row_array = realloc(tlist->row_array, size * sizeof(TextListRow));
  tlist->rows = size;
  
  for (i = 0; i < size; i++) 
    {
      tlist->row_array[i].foreground = NULL;
      tlist->row_array[i].background = NULL;
      tlist->row_array[i].state = GTK_STATE_NORMAL;
      tlist->row_array[i].text = NULL;
    }
}


extern void text_list_first_display(TextList * tlist)
{
  if (!text_list_is_empty(tlist)) {
    text_list_set_active_row(tlist, tlist->active_row);
    if (tlist->selection) {
      g_list_foreach(tlist->selection, (GFunc) select_this_row, tlist);
    } else select_active_row(tlist);
    ensure_active_row_is_visible(tlist, 0.5);
  }
  adjust_scrollbars (tlist);
  text_list_draw_rows (tlist);
}

/* Fill in data after widget has correct style */
static void add_style_data (TextList * tlist)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(tlist);

  /* text properties */
  tlist->row_height = widget->style->font->ascent + widget->style->font->descent + 1 + CELL_SPACING;
  tlist->row_center_offset = widget->style->font->ascent + 1.5;
}


static void text_list_down_list(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->active_row + 1)) {
    ensure_active_row_is_visible(tlist, 1);
    text_list_unselect_all(tlist);
    select_active_row_and_show(tlist);
  }
}

static void text_list_up_list(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->active_row - 1)) {
    ensure_active_row_is_visible(tlist, 0);
    text_list_unselect_all(tlist);
    select_active_row_and_show(tlist);
  }
}

static void text_list_extend_down_list(TextList *tlist)
{
  gboolean b = select_active_row(tlist);

  if (text_list_set_active_row(tlist, tlist->active_row + 1)) {
    ensure_active_row_is_visible(tlist, 1);
    if (!select_active_row_and_show(tlist))
      if (!b && !is_selected(tlist, tlist->active_row - 2))
	text_list_unselect_row(tlist, tlist->active_row - 1);
  } else if (b) draw_active_row(tlist);
}

static void text_list_extend_up_list(TextList *tlist)
{
  gboolean b = select_active_row(tlist);

  if (text_list_set_active_row(tlist, tlist->active_row - 1)) {
    ensure_active_row_is_visible(tlist, 0);
    if (!select_active_row_and_show(tlist))
      if (!b && !is_selected(tlist, tlist->active_row + 2))
	text_list_unselect_row(tlist, tlist->active_row + 1);
  } else if (b) draw_active_row(tlist);
}

static void text_list_keep_selection_down_list(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->active_row + 1)) {
    ensure_active_row_is_visible(tlist, 1);
    draw_last_active_row(tlist);
    draw_active_row(tlist);    
  }
}

static void text_list_keep_selection_up_list(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->active_row - 1)) {
    ensure_active_row_is_visible(tlist, 0);
    draw_last_active_row(tlist);
    draw_active_row(tlist);    
  }
}

static void text_list_active_to_selection(TextList *tlist)
{
  text_list_unselect_all(tlist);
  select_active_row_and_show(tlist);
}
static void text_list_toggle_to_selection(TextList *tlist)
{
  if (!text_list_is_empty(tlist)) toggle_row(tlist, tlist->active_row);
}

static void text_list_page_updown(TextList *tlist, gint shift_amount)
{
  text_list_set_active_row(tlist, tlist->active_row + shift_amount);

  GTK_RANGE (tlist->vscrollbar)->adjustment->value = tlist->voffset + shift_amount * tlist->row_height;
  gtk_signal_emit_by_name (GTK_OBJECT (GTK_RANGE (tlist->vscrollbar)->adjustment), "value_changed");

  draw_last_active_row(tlist);
  draw_active_row(tlist);    
}
static void text_list_page_up(TextList *tlist)
{
  text_list_unselect_all(tlist);
  text_list_keep_selection_page_up(tlist);
  select_active_row_and_show(tlist);
}
static void text_list_page_down(TextList *tlist)
{
  text_list_unselect_all(tlist);
  text_list_keep_selection_page_down(tlist);
  select_active_row_and_show(tlist);
}
static void text_list_keep_selection_page_up(TextList *tlist)
{
  gint first_visible_row = ROW_FROM_YPIXEL(tlist, 0);
  gint window_height_in_row = tlist->tlist_window_height / tlist->row_height;

  if (first_visible_row - window_height_in_row >= 0) {
    text_list_page_updown(tlist, -window_height_in_row);
  } else {
    text_list_page_updown(tlist, -first_visible_row);
  }
}
static void text_list_keep_selection_page_down(TextList *tlist)
{
  gint last_visible = ROW_TOP(tlist, tlist->rows);
  gint window_height_in_row = tlist->tlist_window_height / tlist->row_height;
  gint window_height = window_height_in_row * tlist->row_height;

  if (tlist->tlist_window_height + tlist->voffset + window_height <= last_visible + 1) {
    text_list_page_updown(tlist, window_height_in_row);
  } else {
    text_list_page_updown(tlist, (last_visible - tlist->voffset - tlist->tlist_window_height) / tlist->row_height);
  }
}

static void text_list_extend_page_up(TextList *tlist)
{
  text_list_keep_selection_page_up(tlist);
  select_range_active_last_active(tlist);
}

static void text_list_extend_page_down(TextList *tlist)
{
  text_list_keep_selection_page_down(tlist);
  select_range_active_last_active(tlist);
}

static void text_list_to_bottom(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->rows - 1)) {
    ensure_active_row_is_visible(tlist, 1);
    text_list_unselect_all(tlist);
    select_active_row_and_show(tlist);
  }
}

static void text_list_to_top(TextList *tlist)
{
  if (text_list_set_active_row(tlist, 0)) {
    ensure_active_row_is_visible(tlist, 0);
    text_list_unselect_all(tlist);
    select_active_row_and_show(tlist);
  }
}

static void text_list_keep_selection_to_bottom(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->rows - 1)) {
    ensure_active_row_is_visible(tlist, 1);
    draw_last_active_row(tlist);
    draw_active_row(tlist);    
  }
}

static void text_list_keep_selection_to_top(TextList *tlist)
{
  if (text_list_set_active_row(tlist, 0)) {
    ensure_active_row_is_visible(tlist, 0);
    draw_last_active_row(tlist);
    draw_active_row(tlist);    
  }
}

static void text_list_extend_to_bottom(TextList *tlist)
{
  if (text_list_set_active_row(tlist, tlist->rows - 1)) {
    ensure_active_row_is_visible(tlist, 1);
    select_range_active_last_active(tlist);
  }  
}

static void text_list_extend_to_top(TextList *tlist)
{
  if (text_list_set_active_row(tlist, 0)) {
    ensure_active_row_is_visible(tlist, 0);
    select_range_active_last_active(tlist);
  }  
}


/*******************************************************************************/
/* functions for manipulating the active row and the selection */
/*******************************************************************************/

static void draw_active_row(TextList *list)
{
  text_list_draw_row(list, list->active_row);
}

static void draw_last_active_row(TextList *list)
{
  text_list_draw_row(list, list->last_active_row);
}

/* text_list_set_active_row
 *   return TRUE if the active_row changed
 */
extern gboolean text_list_set_active_row(TextList *tlist, gint row)
{
  gint last_active_row = tlist->active_row;

  if (text_list_is_empty(tlist)) return FALSE;
  
  if (row < 0)				tlist->active_row = 0;
  else if (row >= tlist->rows - 1)	tlist->active_row = tlist->rows - 1;
  else					tlist->active_row = row;
  
  if (last_active_row != tlist->active_row) tlist->last_active_row = last_active_row;
  return last_active_row != tlist->active_row;
}

/* ensure_active_row_is_visible
 *   return TRUE if the active_row wasn't visible (it means the view is moved)
 *   if row_align = 0, the active_row is moved first
 *      row_align = 1, the active_row is moved last 
 */
static gboolean ensure_active_row_is_visible(TextList *tlist, gfloat row_align)
{
  gboolean b;

  b = text_list_row_is_visible(tlist, tlist->active_row) != GTK_VISIBILITY_FULL;

  if (b) text_list_moveto(tlist, tlist->active_row, row_align);
  return b;
}

static gboolean select_active_row(TextList *tlist)
{
  if (text_list_is_empty(tlist)) return TRUE;

  if (!is_active_row_selected(tlist)) 
    {  
      tlist->row_array[tlist->active_row].state = GTK_STATE_SELECTED;
      tlist->selection = g_list_append (tlist->selection, (gpointer) tlist->active_row);
      return TRUE;
    }
  else return FALSE;
}

static gboolean is_active_row_selected(TextList *tlist)
{
  return tlist->row_array[tlist->active_row].state == GTK_STATE_SELECTED;
}


static void select_this_row (gint row, TextList *tlist) 
{
  tlist->row_array[row].state = GTK_STATE_SELECTED;
  text_list_draw_row(tlist, row);
}
static void unselect_this_row (gint row, TextList *tlist) 
{
  tlist->row_array[row].state = GTK_STATE_NORMAL;
  text_list_draw_row(tlist, row);
}
static void text_list_unselect_all(TextList *tlist)
{
  g_list_foreach(tlist->selection, (GFunc) unselect_this_row, tlist);
  g_list_free(tlist->selection);
  tlist->selection = NULL;
}

extern void text_list_select_all(TextList *tlist)
{
  select_range(tlist, 0, tlist->rows - 1);
}

static void select_range_active_last_active(TextList *tlist)
{
  select_range(tlist, MIN(tlist->last_active_row, tlist->active_row), MAX(tlist->last_active_row, tlist->active_row));
}

static void select_range(TextList *tlist, gint row_min, gint row_max)
{
  gint i;
  
  for (i = row_min; i <= row_max; i++) {
    if (tlist->row_array[i].state != GTK_STATE_SELECTED)
    tlist->selection = g_list_append(tlist->selection, (gpointer) i);
    tlist->row_array[i].state = GTK_STATE_SELECTED;
  }
  text_list_draw_rows (tlist);
}

extern void text_list_invert_selection(TextList *tlist)
{
  gint i;
  
  g_list_free(tlist->selection);
  tlist->selection = NULL;

  for (i = 0; i < tlist->rows; i++) {
    if (tlist->row_array[i].state == GTK_STATE_SELECTED) {
      tlist->row_array[i].state = GTK_STATE_NORMAL;
    } else {
      tlist->selection = g_list_append(tlist->selection, (gpointer) i);
      tlist->row_array[i].state = GTK_STATE_SELECTED;
    }
  }
  text_list_draw_rows (tlist);
}


extern void text_list_get_coords(TextList *tlist, gint row, gint column, gint *x, gint *y)
{
  *x = COLUMN_INSET;
  *y = ROW_TOP_YPIXEL(tlist, row);
}


extern void text_list_set_active_row_and_show(TextList *tlist, gint row, gfloat row_align)
{
  text_list_set_active_row(tlist, row);
  ensure_active_row_is_visible(tlist, row_align);
  draw_last_active_row(tlist);
  draw_active_row(tlist);    
}

static gboolean select_active_row_and_show(TextList *tlist)
{
  gboolean b;

  b = select_active_row(tlist);
  draw_last_active_row(tlist);
  draw_active_row(tlist);    

  return b;
}

extern gboolean text_list_is_empty(TextList *tlist)
{
  return tlist->rows == 0;
}

static gboolean is_selected(TextList *tlist, gint row)
{
  if (row < 0) return FALSE;
  if (row >= tlist->rows) return FALSE;
  return tlist->row_array[row].state & GTK_STATE_SELECTED;
}
