namespace eval scroll_listbox {
    variable scroll_listbox

    # Define the options of the listbox which are common
    # to all platforms.
    option add *Scrolllistbox.list.foreground #000000
    option add *Scrolllistbox.list.background #ffffff
    option add *Scrolllistbox.yscroll.width 12
    option add *Scrolllistbox.xscroll.width 12

    # Define the OS specific options of the listbox.
    switch $tkWorld(ostype) {
	windows {
	    option add *Scrolllistbox.list.relief flat
	    option add *Scrolllistbox.list.font {helvetica 10 bold}
	    set scroll_listbox(padx) 0
	    set scroll_listbox(pady) 0
	}
	default {
	    option add *Scrolllistbox.list.relief sunken
	    option add *Scrolllistbox.list.font {helvetica 12 bold}
	    set scroll_listbox(padx) 0
	    set scroll_listbox(pady) 0
	}
    }
}

# scroll_listbox::create --
#
#   Method which creates a scrollable listbox widget with no
#   user preferences
#
# Args
#
#   w - The parent widget.
#
# Returns
#
#   The parent widget indicating successful creation of the widget.

proc scroll_listbox::create { w {width 20} {height 20} {selmode single} } {
    variable scroll_listbox

    frame $w \
	    -relief flat \
	    -class Scrolllistbox

    listbox $w.list \
	    -relief sunken \
	    -width $width \
	    -height $height \
	    -selectmode $selmode \
	    -yscrollcommand [list $w.yscroll set] \
	    -xscrollcommand [list $w.xscroll set]

    scrollbar $w.yscroll \
	    -command "$w.list yview" \
	    -orient vertical
    scrollbar $w.xscroll \
	    -command "$w.list xview" \
	    -orient horizontal

    grid $w.list $w.yscroll \
	    -row 0 \
	    -sticky news \
	    -padx $scroll_listbox(padx) \
	    -pady 0
    grid $w.xscroll \
	    -row 1 \
	    -sticky ew \
	    -padx 0 \
	    -pady $scroll_listbox(pady)

    # Remember that a weight factor of 1 means allow expanding.  The
    # way this is set up, the text widget expands in both directions,
    # the y scrollbar in the y direction ONLY, and x scrollbar in the
    # x direction ONLY.
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    
    return $w
}

# scroll_listbox::interior --
#
# Args
#
#   w - The parent widget.
#
# Returns
#
#   The name of the child listbox widget.

proc scroll_listbox::interior { w } {
    return $w.list
}

#---------------------------------------------------------------------

namespace eval scroll_text { 
    variable scroll_text

    # Define the options of the text widget which are common
    # to all platforms.
    option add *Scrolltext.text.foreground #000000
    option add *Scrolltext.text.background #ffffff
    option add *Scrolltext.yscrollbar.width 12
    option add *Scrolltext.xscrollbar.width 12

    # Define the OS specific options of the text widget.
    switch $tkWorld(ostype) {
	windows {
	    option add *Scrolltext.text.relief flat
	    option add *Scrolltext.text.font {fixed 10}
	    set scroll_text(padx) 0
	    set scroll_text(pady) 0
	}
	default {
	    option add *Scrolltext.text.relief sunken
	    option add *Scrolltext.text.font {fixed 12}
	    set scroll_text(padx) 0
	    set scroll_text(pady) 0
	}
    }

}

# scroll_text::create --
#
#   Method which creates a scrollable text widget with no user
#   preferences for -height, -width, -wrap, -background, or -font.
#
# Args
#
#   w - The parent widget.
#
# Returns
#
#   The parent widget indicating successful creation of the widget.

proc scroll_text::create { w {width 70} {height 20} } {
    variable scroll_text

    frame $w \
	    -relief flat \
	    -class Scrolltext
    
    scrollbar $w.yscrollbar \
	    -command "$w.text yview" \
	    -orient vertical
    scrollbar $w.xscrollbar \
	    -command "$w.text xview" \
	    -orient horizontal
    text $w.text \
	    -width $width \
	    -height $height \
	    -wrap none \
	    -yscrollcommand [list $w.yscrollbar set] \
	    -xscrollcommand [list $w.xscrollbar set]
    
    grid $w.text $w.yscrollbar \
	    -row 0 \
	    -sticky news \
	    -padx $scroll_text(padx) \
	    -pady 0
    grid $w.xscrollbar \
	    -row 1 \
	    -sticky ew \
	    -padx 0 \
	    -pady $scroll_text(pady)

    # See the comments in scroll_listbox.
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    
    return $w
}

# scroll_text::interior --
#
# Args
#
#   w - The parent widget of scroll_text
#
# Returns
#
#   The name of the child text widget

proc scroll_text::interior { w } {
    return $w.text
}

#---------------------------------------------------------------------

namespace eval seperator {
    variable sep
}

# seperator::create --
#
#   Method used to create a generic frame which can be used with
#   pack, grid, and place to create a seperator between your
#   application frames
#
# Args
#
#   w      - The name of the parent frame widget
#   relief - Choose a valid option
#
# Returns
#
#   The name of the parent frame widget indicating
#   successful creation

proc seperator::create { w {relief sunken} } {
    variable sep

    set f [frame $w \
	    -relief $relief \
	    -width 2 \
	    -height 2 \
	    -bd 4]

    if [info exists sep(list)] {
	lappend sep(list) $f
    } else {
	set sep(list) $f
    }

    return $f
}

# seperator::list --
#
#   Method which lists all object names of the seperators known
#   to the seperator object
#
# Args
#
#   None
#
# Returns
#
#   List of the object names of all the seperators known to the
#   seperator object

proc seperator::list { } {
    variable sep

    return $sep(list)
}

#---------------------------------------------------------------------

namespace eval ddlistbox {
    variable ddlistbox

    #Define the ddlistbox data structure.
    set ddlistbox(downarrow) [image create photo \
	    -format gif \
	    -file [file join $tkWorld(image_dir) downarrow.gif]]
    set ddlistbox(framenum) 0
    set ddlistbox(max_menu_items) 35
}

# ddlistbox::create  --
#
#   Method to create a drop-down listbox.
#
# Args
#
#   w     - Object name of the parent widget.
#   var   - Name of the variable associated with the entry
#           in the ddlistbox.
#   width - Width of the entry widget.
#
# Returns
#
#   The object name of the successfully created ddlistbox.

proc ddlistbox::create { w var {vlist ""} {width 10} } {
    variable ddlistbox

    # Create a unique indentifier for the frame to be created.
    incr ddlistbox(framenum)
    
    # Steal the Tk parent directory if that is the ddlistbox parent.
    if {$w == "."} {
	set w ""
    }
    set f [frame $w.ddlistbox$ddlistbox(framenum) \
	    -borderwidth 1 \
	    -relief sunken \
	    -class Dropdownlistbox]
    
    # Build the interface components.
    set e [entry $f.entry \
	    -width $width \
	    -borderwidth 0 \
	    -state disabled \
	    -textvariable $var]
    set m [menubutton $f.menubutton \
	    -image $ddlistbox(downarrow) \
	    -menu $f.menubutton.menu \
	    -borderwidth 1 \
	    -height 14 \
	    -width 10 \
	    -relief raised]
    
    # Build the ddlistbox.
    grid $e $m \
	    -sticky ew \
	    -padx 0 \
	    -pady 0

    # Define the menu in the ddlistbox.
    set ddlistbox(menulist.$m.menu) [menu $m.menu]

    # Build the initial list of values.
    ddlistbox::add $f $var $vlist

    # Return the object name of the ddlistbox.
    return $f
}

# ddlistbox::interior --
#
#   Method which communicates the menu object name from the
#   parent widget.
#
# Args
#
#   w - The object name of the parent.
#
# Returns
#
#   The object name of the child menu to add entries to.

proc ddlistbox::interior { w } {
    return $w.menubutton.menu
}

# ddlistbox::delete --
#
#   Method to delete elements from the selection list.
#
# Args
#
#   w - The object name of the parent ddlistbox.
#
# Returns
#
#   None.

proc ddlistbox::delete { w } {
    variable ddlistbox

    set m [ddlistbox::interior $w]

    foreach x [array names ddlistbox "menulist*"] {
	if [regexp "menulist.$m" $x] {
	    $ddlistbox($x) delete 0 end
	}
    }
}

# ddlistbox::add --
#
#   Method to add elements to the selection.
#
# Args
#
#   w     - The object name of the parent ddlistbox.
#   var   - Name of the textvariable for the entry widget.
#   vlist - List of values.
#
# Returns
#
#   None.

proc ddlistbox::add { f var vlist } {
    variable ddlistbox

    set m [ddlistbox::interior $f]

    # Count how many menus will be needed
    set menu_cnt [expr int([llength $vlist]/$ddlistbox(max_menu_items)) + 1]

    # Build the additional menus that will be needed.
    set x $m
    if {$menu_cnt > 1} {
	for {set i 1} {$i < $menu_cnt} {incr i 1} {
	    # Build the sub-menu if it does not already exist.
	    if ![info exists ddlistbox(menulist.$x.$i)] {
		set ddlistbox(menulist.$x.$i) [menu $x.$i \
			-activebackground #0000aa \
			-activeforeground #ffffff \
			-activeborderwidth 0 \
			-tearoff 1]
	    }
		
	    # Insert the cascade menu in the parent.
	    $x add cascade \
		    -label "More..." \
		    -menu $x.$i
	    
	    # Define the new parent for the next child.
	    set x "$x.$i"
	}
    }

    # Insert each item in the list.
    set i 0; set menu_level 0
    foreach v $vlist {
	$m add radio \
		-label $v \
		-variable $var \
		-hidemargin 1

	# Increment the counter
	incr i

	# Check if we need a submenu.
	if {[expr fmod($i,$ddlistbox(max_menu_items))] == "0.0"} {
	    incr menu_level
	    set m $m.$menu_level
	}
    }
}

#---------------------------------------------------------------------

namespace eval picklist {
    variable picklist

    array set picklist {
	status 0
    }
}

# picklist::create --
#
#   Method to create a modal dialog box where the user choose from
#   a list of items.
#
# Args
#
#   vlist - List of values to choose from.
#
# Returns
#
#   None.

proc picklist::create { vlist } {
    variable picklist

    # See if the picklist dialog already exists.
    if [winfo exists .picklist] {
	focus .picklist
    } else {
	set w [toplevel .picklist \
	    -borderwidth 1 \
	    -relief flat]
	wm title .picklist "Pick List"
    }

    # Find the width of the largest string in the list.
    set len 0
    set maxw -5
    foreach v $vlist {
	set len [string length $v]
	if {$len > $maxw} {
	    set maxw $len
	}
    }

    # Create the listbox to display the selections in.
    grid [scroll_listbox::create $w.sl $maxw 10 multiple] \
	    -columnspan 2 \
	    -sticky we \
	    -padx 0 \
	    -pady 2
    grid [seperator::create $w.sep] \
	    -columnspan 2 \
	    -sticky we \
	    -padx 0 \
	    -pady 2

    # Insert the selection options in the listbox.
    set lb [scroll_listbox::interior $w.sl]
    foreach v $vlist {
	$lb insert end $v
    }

    # Grid up the dialog box.
    button $w.ok \
	    -text Ok \
	    -width 5 \
	    -command {set picklist(status) 1}
    button $w.close \
	    -text Close \
	    -width 5 \
	    -command {set picklist(status) 0}
    grid $w.ok $w.close \
	    -padx 5 \
	    -pady 5
}

# picklist::wait --
#
#   Method to keep the picklist dialog open and grab focus until
#   the status has been changed.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc picklist::wait { } {
    variable picklist

    bind .picklist <Destroy> \
	    [list set picklist::picklist(status) 0]

    set old [focus -displayof .picklist]
    focus .picklist
    catch {tkwait visibility .picklist}
    catch {grab .picklist}

    tkwait variable picklist(status)
    catch {grab release .picklist}
    focus $old
}

# picklist::cancel --
#
#   Method to destroy the dialog box and return the selected
#   items list.
#
# Args
#
#   None.
#
# Returns
#
#   Selected items list.

proc picklist::cancel { } {
    # Define the picklist dialog listbox.
    set lb [scroll_listbox::interior .picklist.sl]

    #Get the list of current selection indices.
    set clist [$lb curselection]

    # Build the list of selected items.
    set slist ""
    foreach x $clist {
	append slist "[$lb get $x] "
    }

    # Destroy the picklist dialog before we go...
    destroy .picklist
    
    # Give up the list of items that the user selected.
    return $slist
}

#---------------------------------------------------------------------

namespace eval box {
    # No private variables
}

# box::create  --
#
#   Create a titled box for widgets to be put in.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc box::create { w {t ""} } {
    # Build the parent frame which gives the box.
    frame $w

    # Insert the title if given.
    if [string length $t] {
	grid [label $w.title \
		-text $t] \
		-sticky ew
    }

    grid [frame $w.border \
	    -borderwidth 2 \
	    -relief groove] \
	    -padx 2 \
	    -pady 2 \
	    -ipadx 2 \
	    -ipady 2 \
	    -sticky news

    return $w
}

# box::interior --
#
#   Return the object name of the child box
#
# Args
#
#   w - The object name of the parent widget.
#
# Returns
#
#   The object name of the child widget.

proc box::interior { w } {
    return $w.border
}

#---------------------------------------------------------------------

namespace eval radiobox {
    variable radiobox

    # Define the data structure.
    array set radiobox {
	rb_count 0
    }
}

# radiobox::create --
#
#   Method to create a radiobox.
#
# Args
#
#   w - The object name of the radiobox widget.
#   t - The title of the box.
#
# Returns
#
#   The object name of the radiobox widget indicating
#   successful creation.

proc radiobox::create { w {t ""} } {
    return [box::create $w $t]
}

# radiobox::add --
#
#   Method to add a button to the radiobox.
#
# Args
#
#   w      - The object name of the parent radiobox widget.
#   choice - Label of the option.
#   val    - Value when the option is selected.
#   var    - Variable associated with the selection
#
# Returns
#
#   None.

proc radiobox::add { w choice val var } {
    variable radiobox

    grid [radiobutton [box::interior $w].rb[incr radiobox(rb_count)] \
	    -text $choice \
	    -value $val \
	    -variable [list $var]] \
	    -sticky w
}

# radiobox::destroy --
#
#   Method to destroy the radiobox widget.
#
# Args
#
#   w - The object name of the radiobox widget.
#
# Returns
#
#   None.

proc radiobox::destroy { w } {
    variable radiobox

    # Kill the widget.
    destroy $w

    # Free up the memory associated with it.
    foreach i [array names radiobox] {
	unset radiobox($i)
    }
}

#---------------------------------------------------------------------

namespace eval checkbox {
    variable checkbox

    # Define the data structure.
    array set checkbox {
	rb_count 0
    }
}

# checkbox::create --
#
#   Method to create a checkbox.
#
# Args
#
#   w - The object name of the checkbox widget.
#   t - The title of the box.
#
# Returns
#
#   The object name of the checkbox widget indicating
#   successful creation.

proc checkbox::create { w {t ""} } {
    return [box::create $w $t]
}

# checkbox::add --
#
#   Method to add a button to the checkbox.
#
# Args
#
#   w      - The object name of the parent checkbox widget.
#   choice - Label of the option.
#   onval  - Value when the option is selected.
#   offval - Value when the option is NOT selected.
#   var    - Variable associated with the selection.
#
# Returns
#
#   None.

proc checkbox::add { w choice onval offval var } {
    variable checkbox

    grid [checkbutton [box::interior $w].rb[incr checkbox(rb_count)] \
	    -text $choice \
	    -onvalue $onval \
	    -offvalue $offval \
	    -variable [list $var]] \
	    -sticky w
}

# checkbox::destroy --
#
#   Method to destroy the checkbox widget.
#
# Args
#
#   w - The object name of the checkbox widget.
#
# Returns
#
#   None.

proc checkbox::destroy { w } {
    variable checkbox

    # Kill the widget.
    destroy $w

    # Free up the memory associated with it.
    foreach i [array names checkbox] {
	unset checkbox($i)
    }
}