/*
   File     : http.c
   Author   : Lionel ULMER
   Creation : 19/02/97
   Modification: 20/12/97 - Philippe Dax

   Functions "utils" network : call nameserver, connection to
   a TCP/IP server, URL parsing.
*/


#include "global.h"
#include "initui.h"	/* resources */
#include "http.h"	/* Pour avoir les codes d'erreur */

char systemname[9];
char releasename[9];
char machinename[9];

#ifdef WITHTHREAD
/* Variables globales "protges" par nbsimconlock */
WaitFIFO *fifofirst,*fifolast;
#endif /* WITHTHREAD */

/* Table de haschage */
ResCache *ht[HTABLE];


/* inithash : initialise la table de haschage */
void inithash()
{
  int i;

  for (i=0; i < HTABLE; ht[i++] = NULL)
    ;
}

void initsysinfo()
{
#ifdef HAVE_SYS_UTSNAME_H
  struct utsname sysinfo;

  uname(&sysinfo);
  strcpy(systemname, sysinfo.sysname);
  strcpy(releasename, sysinfo.release);
  strcpy(machinename, sysinfo.machine);
#endif
}

void initThreads()
{
  maxsimcon = resources.maxsimcon;
#ifdef WITHTHREAD
  pthread_mutex_init(&nbsimconlock, NULL);
#endif /* WITHTHREAD */
  inithash();
  nbsimcon = 0;
}

static int hfunc(char *name)
{
  int ret = 0;

  while (*name!='\0')
    ret = ((ret<<8)+((unsigned char) *name)) % HTABLE;
  return ret;
}

/* http dowloading by threads */
void *launchThread(void *info)
{
  ThreadLaunch *tl;
  NetCon *nc;
  struct sockaddr_in sa;
  char host[80], envproxy[90], domnoproxy[16], hostproxy[80], portproxy[5], scheme[8], path[MAXBUF],req[MAXBUF],httpmess[MAXBUF];
  char ln[MAXBUF],*p;
  int type,ret,ret2,eoheader,i,fl,httperr,major,minor, proxy=0, noproxy=0;
  struct hostent *ph;

  tl = (ThreadLaunch *) info;
  nc = (NetCon *) malloc(sizeof(NetCon));

  if (tl->block == NO_NONBLOCK) {
#ifdef WITHTHREAD
    trace(DBG_THRD, " -> launchThread %s",tl->url);
    /* Attends l'authorisation de commencer */
    if (tl->wait != NULL) {
#ifdef DEBUG
      trace(DBG_THRD, " -> wait unblock %s",tl->url);
#endif /* WITHTHREAD */
      pthread_mutex_lock(&nbsimconlock);
      pthread_cond_wait(&(tl->wait->cond),&nbsimconlock);
#ifdef DEBUG
      trace(DBG_THRD, " -> block %s",tl->url);
#endif

      /* Update nbsimcon */
      nbsimcon++;
      
      /* Enlve l'lment de la FIFO */
      fifofirst = tl->wait->next;
      free(tl->wait);
      /* Libre la gestion de la FIFO */
      pthread_mutex_unlock(&nbsimconlock);
    }
#endif
  }
  
  /* test if proxy */
  p = getenv("http_proxy");
  if (p && *p) {
    if (p[strlen(p) - 1] == '/')
      p[strlen(p) - 1] = '\0';
    strcpy(envproxy, p);
    p = strrchr(envproxy, ':');
    *p = '\0';
    strcpy(portproxy, ++p);
    p = strrchr(envproxy, '/');
    strcpy(hostproxy, ++p);
    proxy = 1;
    trace(DBG_HTTP, "hostproxy=%s portproxy=%s", hostproxy, portproxy);
  }
  p = getenv("no_proxy");
  if (p && *p) {
    strcpy(domnoproxy, p);
    noproxy = 1;
  }

  trace(DBG_HTTP, " -> Opening url %s",tl->url);
  /* Parse l'URL */
  type = parseURL(tl->url, host, scheme, path);

  switch (type) {
  case URLFILE:
    if ((ret = open(path, O_RDONLY)) < 0) {
      tl->callback(tl->handle, NULL);
    } else {
      nc->fd = ret;
      nc->bptr = -1;
      tl->callback(tl->handle, (void *) nc);
    }
    goto fin;

  case URLHTTP:
    trace(DBG_HTTP, " -> HTTP: %s %s %s", host, scheme, path);

    if (proxy && strstr(host, domnoproxy) == 0) {
      if ((ph = gethostbyname(hostproxy)) == NULL) {
        tl->callback(tl->handle, NULL);
        trace(DBG_HTTP, "gethostbyname host=%s", hostproxy);
        ret = -1;
      }
      memset(&sa, 0, sizeof(sa));
      sa.sin_family = AF_INET;
      sa.sin_port = htons(atoi(portproxy));
      memcpy(&sa.sin_addr, ph->h_addr_list[0], ph->h_length);
      ret = 0;
    }
    else {
      ret = resolve(host, scheme, &sa);
    }
    if (ret < 0) {	
      tl->callback(tl->handle, NULL);
      goto fin;
    }
#ifdef DEBUG
    trace(DBG_HTTP, " -> resolve done %s", tl->url);
#endif

    if ((ret = httpConnect(&sa)) < 0) {
      trace(DBG_HTTP, "httpConnect: %d", ret);
      tl->callback(tl->handle, NULL);
      goto fin;
    }
#ifdef DEBUG
    trace(DBG_HTTP, " -> connect done %s", tl->url);
#endif

    /* send GET request with adding useful infos */
    if (proxy && strstr(host, domnoproxy) == 0)
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        tl->url, VERSION, machinename, systemname, releasename);
    else
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        path, VERSION, machinename, systemname, releasename);
    if ((ret2 = httpSend(ret, req, strlen(req))) < 0) {
      tl->callback(tl->handle, NULL);
      close(ret);
      goto fin;
    }

    /* parse header HTTP/1.0 */
#ifdef DEBUG
    trace(DBG_HTTP, " -> parsing header %s", tl->url);
#endif
    eoheader=0;
    fl = 1; /* position first line */
    p = nc->buf;
    i = 0;
    do {
/*******************************
      signal(SIGALRM, sigalrm);
      alarm(TIMEOUT_HTTP);
      if (setjmp(jmphttp)) {
        trace(DBG_HTTP, "Timeout on server");
        tl->callback(tl->handle, NULL);
        goto fin;
      }
*******************************/
      nc->bl = read(ret, p, MAXNCBUF);
      nc->bptr = 0;

#ifdef DEBUG
      trace(DBG_HTTP, "-> %d", nc->bl);
#endif
      while (1) {	
	/* line in ln */
	while ((nc->bptr < nc->bl) && (p[nc->bptr] != '\n'))
	  ln[i++] = p[nc->bptr++];
	if (nc->bptr < nc->bl) {
	  /* test end of header */
	  if (ln[0] == '\r') {
	    nc->bptr++;
	    eoheader = 1;
	    break;
	  }
	  /* null before last (\r) */
	  ln[i-1] = '\0';
	  nc->bptr++;
	  trace(DBG_HTTP, " ->%s", ln);
	  if (fl) {
	    /* first line => get error code */
	    sscanf(ln, "HTTP/%d.%d %d", &major, &minor, &httperr);
	    strcpy(httpmess, ln+12);
	    trace(DBG_HTTP, " -> Code err (%d.%d) : %d - %s %s",
		   major, minor, httperr, httpmess, tl->url);
	    if (httperr != 200) {	      
	      tl->callback(tl->handle, NULL);
	      close(ret);
	      goto fin;
	    }
	    fl = 0;
	  }
          i = 0;
	} else {
	  /* get datas by httpRead() by readCfgDatas() in wmgt */
	  break;
	}
      }
    } while (!eoheader);    
    nc->fd = ret;
    
    /* Call the callback */
    tl->callback(tl->handle, (void *) nc);
    break;

  case URLFTP:
    warning("FTP protocol not supported.");
    tl->callback(tl->handle, NULL);
    goto fin; 

  default:
    warning("unknown scheme = %d", type);
    tl->callback(tl->handle, NULL);
    goto fin;
  }
  close(ret);

  /* End */
 fin:
  if (tl->block == NO_NONBLOCK) {
#ifdef WITHTHREAD
    /* access to global variable nbsimcon */
    pthread_mutex_lock(&nbsimconlock);
    nbsimcon--;
    /* Si y'a kkchose dans la FIFO, le rveille */
    if (fifofirst != NULL) {
      trace(DBG_THRD, " -> awake an (%d) %s",nbsimcon,tl->url);
      pthread_cond_signal(&fifofirst->cond);
    }
    pthread_mutex_unlock(&nbsimconlock);
#endif /* WITHTHREAD */
  }
  free(tl);
  free(nc);
  return NULL;
}

int httpOpen(char *url, int (*callback)(void *,void *), void *handle, int block)
{
  ThreadLaunch *tl;
#ifdef WITHTHREAD
  pthread_t dummy;
  WaitFIFO *wf;
#endif /* WITHTHREAD */
  
  trace(DBG_HTTP, " -> httpOpen: %s", url);
  /* fill the structure */
  tl = (ThreadLaunch *) malloc(sizeof(ThreadLaunch));
  tl->handle = handle;
  tl->callback = callback;
  tl->block = block;
#ifdef WITHTHREAD
  tl->wait = NULL;
#endif
  strcpy(tl->url, url);

  if (block == NO_NONBLOCK) {
#ifdef DEBUG
    trace(DBG_THRD, " -> No blocking: %s", url);
#endif
#ifdef WITHTHREAD
    /* access to global variable nbsimcon */
    pthread_mutex_lock(&nbsimconlock);
    /* test number of active connections */
    if (nbsimcon == maxsimcon) {
      trace(DBG_THRD, " -> Nb threads: %d %s", nbsimcon, url);
      /* Place le "thread" dans la file d'attente */
      wf = (WaitFIFO *) malloc(sizeof(WaitFIFO));
      pthread_cond_init(&(wf->cond), NULL);
      wf->next = NULL;
      if (fifofirst == NULL)
	fifofirst = wf;
      if (fifolast != NULL)
	fifolast->next = wf;
      fifolast = wf;
      
      /* Unblock the global variable */
      pthread_mutex_unlock(&nbsimconlock);
      /* Block the thread */
      tl->wait = wf;
      trace(DBG_THRD, " -> Too many threads, waiting... %s", url);
    } else {
      /* Add a connection */
      nbsimcon++;
      trace(DBG_THRD, " -> going now (%d) %s", nbsimcon, url);
      pthread_mutex_unlock(&nbsimconlock);
      /* Thread "non bloqu" */
      tl->wait = NULL;
    }

    /* Start new Thread */
    return pthread_create(&dummy, NULL, launchThread, (void *) tl);

#else
    launchThread((void *) tl);
    return 0;
#endif /* WITHTHREAD */
  }
  else {
#ifdef DEBUG
    trace(DBG_THRD, " -> Thread bloquant %s", url);
#endif
    launchThread((void *) tl);
    return 0;
  }
}

void httpClose(void *handle)
{
  NetCon *nc;
  
  nc = (NetCon *) handle;
  close(nc->fd);
  /* Warning at bugs... */
  free(nc);
}

int httpRead(void *handle, char *buf, int maxl)
{
  NetCon *nc;
  int l, len;

  nc = (NetCon *) handle;

  /* Modifie par Fabrice, l= longueur lue, maxl= longueur restante a recevoir */
  l = 0;
  if (nc->bptr < 0)
    len = 0;
  else
    len = nc->bl - nc->bptr;
  if (len > 0) {
    if (len>maxl) len=maxl;
    memcpy(buf, nc->buf+nc->bptr, len);
    l += len;
    maxl -= len;
    nc->bptr += len;
  }
  while (maxl > 0) {
    if ((len = read(nc->fd, buf + l, maxl)) <= 0)
      break;
    l += len;
    maxl -= len;
  }
  return l;
}

/* remplit le buffer answer avec (au plus) max octets */
int httpRecv(int s, char *answer, int max)
{
  return (read(s, answer, max));
}

/* send request to server */
int httpSend(int s, char *req, int nbytes)
{
  int sent, ret;

  for (sent = 0; sent < nbytes;) {
    if ((ret = write(s, req + sent, nbytes - sent)) == -1)
      return NETSERV;
    sent += ret;
  }
  return 0;
}

/* connect to server defined by serveraddr */
int httpConnect(struct sockaddr_in *serveraddr)
{
  int s;
  
  if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    perror("httpConnect: socket");
    return NETSOCKET;
  }
  if (connect(s, (struct sockaddr*)serveraddr, sizeof(struct sockaddr_in)) <0) {
    perror("httpConnect: connect");
    close(s);
    return NETCONNECT;
  }
  return s;
}

/* converts host/port into sockaddr */
int resolve(char *host, char *service, struct sockaddr_in *sadd)
{
#ifdef WITHTHREAD
  struct hostent h;
  struct servent s;
  int res;
  char buf[512], bufserv[512];
#endif

  struct hostent *ph;
  struct servent *ps;
  struct hostent host_hack;  
  unsigned short port;
  char addr[4];
  char *addr_hack;
  
  if (isdigit(host[0])) { /* IPv4 address */
    int a0,a1,a2,a3;
    sscanf(host, "%d.%d.%d.%d", &a0,&a1,&a2,&a3);
    addr[0]=a0;
    addr[1]=a1;
    addr[2]=a2;
    addr[3]=a3;

#ifdef WITHTHREAD    
    if ((ph = gethostbyaddr_r(addr, 4, AF_INET, &h, buf, 512, &res)) == NULL) {
      /*** Bletcherous hack in case you do not have a nameserver. */
      h = host_hack;
      h.h_addrtype = AF_INET;
      addr_hack = addr;
      h.h_addr_list = &addr_hack;
      h.h_length = 4;
    }
#else
    if ((ph = gethostbyaddr(addr, 4, AF_INET)) == NULL) {
      /*** Bletcherous hack in case you do not have a nameserver. */
      ph = &host_hack;
      ph->h_addrtype = AF_INET;
      addr_hack = addr;
      ph->h_addr_list = &addr_hack;
      ph->h_length = 4;
    }
#endif

  }
  else { /* hostname */

#ifdef WITHTHREAD
    if ((ph = gethostbyname_r(host, &h, buf, sizeof(buf), &res)) == NULL)
#else
    if ((ph = gethostbyname(host)) == NULL)
#endif
      return NETBADNAME;

  }
  if (isdigit(service[0])) {
    port = htons(atoi(service));
  }
  else {

#ifdef WITHTHREAD
    if ((ps = getservbyname_r(service, NULL, &s, bufserv, 512)) == NULL) {
/*** BUG
      if (!strcmp(service, "www"))
        port = htons(HTTPPORT);
      else
        return NETBADSERV;
***/
     port = htons(HTTPPORT);
    }
    else
      port = s.s_port;
#else
    if ((ps = getservbyname(service, NULL)) == NULL) {
      if (!strcmp(service, "www"))
        port = htons(HTTPPORT);
      else
        return NETBADSERV;
    }
    else
      port = ps->s_port;
#endif
  }

#ifdef WITHTHREAD
  sadd->sin_family = h.h_addrtype;
  sadd->sin_port = port;
  memcpy(&sadd->sin_addr, h.h_addr_list[0], h.h_length);
#else
  sadd->sin_family = ph->h_addrtype;
  sadd->sin_port = port;
  memcpy(&sadd->sin_addr, ph->h_addr_list[0], ph->h_length);
#endif

  return 0;
}

/* URL parsing */
int parseURL(char *url, char *server, char *scheme, char *path)
{
  int type;
  
  /* Recherche d'abord le type d'URL */
  if (!strncmp(url, "http://", 7)) {    
    type = URLHTTP;
    url += 7;
  } else if (!strncmp(url, "file:", 5)) {
    type = URLFILE;
    url += 5;
  } else if (!strncmp(url, "ftp://", 6)) {
    type = URLFTP;
    url += 6;
  } else {
    trace(DBG_HTTP, "bad URL: %s", url);
    return URLBAD;
  }

  if (type != URLFILE) {
    /* then parse server's name */
    while ((*url != ':') && (*url != '/') && (*url != '\0'))
      *server++ = *url++;
    *server = '\0';
    
    /* scheme */
    if (*url == ':') {
      url++;
      while (*url != '/')
	*scheme++ = *url++;
    } else {
      switch(type) {
      case URLHTTP:
	strcpy(scheme, "www");
	break;
      case URLFTP:
	strcpy(scheme, "ftp");
      default:
	break;
      }
    }
  }
  if (*url == '\0')
    strcpy(path, "/");
  else /* filename */
    strcpy(path, url);
  return type;
}
