/*
 *  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 *
 *  Seriously modified by Fredrik Hbinette <hubbe@hubbe.net>
 */

/*
 * x.c - functions to deal with X display.
 */

#include <x2vnc.h>
#include <X11/X.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>


#define SCROLLBAR_SIZE 10
#define SCROLLBAR_BG_SIZE (SCROLLBAR_SIZE + 2)

#define INVALID_PIXEL 0xffffffff
#define COLORMAP_SIZE 256


Display *dpy;
static Window topLevel;
static int topLevelWidth, topLevelHeight;

static Atom wmProtocols, wmDeleteWindow, wmState;
static Bool modifierPressed[256];

static Bool HandleTopLevelEvent(XEvent *ev);
static Bool HandleRootEvent(XEvent *ev);
static int displayWidth, displayHeight;
static int grabbed;
Cursor  grabCursor;

enum edge_enum edge = EDGE_EAST;


/*
 * CreateXWindow.
 */

Bool CreateXWindow(void)
{
    XSetWindowAttributes attr;
    XEvent ev;
    char defaultGeometry[256];
    XSizeHints wmHints;
    XGCValues gcv;
    int i;

    Pixmap    nullPixmap;
    XColor    dummyColor;


    if (!(dpy = XOpenDisplay(displayname))) {
	fprintf(stderr,"%s: unable to open display %s\n",
		programName, XDisplayName(displayname));
	return False;
    }

    for (i = 0; i < 256; i++)
	modifierPressed[i] = False;

    /* Try to work out the geometry of the top-level window */

    displayWidth = WidthOfScreen(DefaultScreenOfDisplay(dpy));
    displayHeight = HeightOfScreen(DefaultScreenOfDisplay(dpy));

    topLevelWidth = si.framebufferWidth;
    topLevelHeight = si.framebufferHeight;

    topLevelWidth=1;
    topLevelHeight=1;
    wmHints.x=0;
    wmHints.y=0;

    switch(edge)
    {
      case EDGE_EAST: wmHints.x=displayWidth-1;
      case EDGE_WEST: topLevelHeight=displayHeight;
	break;

      case EDGE_SOUTH: wmHints.y=displayHeight-1;
      case EDGE_NORTH: topLevelWidth=displayWidth;
	break;
    }


    wmHints.flags = PMaxSize | PMinSize |PPosition |PBaseSize;
    wmHints.max_width = topLevelWidth;
    wmHints.max_height = topLevelHeight;

    sprintf(defaultGeometry, "%dx%d+%d+%d", topLevelWidth, topLevelHeight,
	    wmHints.x, wmHints.y);

    XWMGeometry(dpy, DefaultScreen(dpy), geometry, defaultGeometry, 0,
		&wmHints, &wmHints.x, &wmHints.y,
		&topLevelWidth, &topLevelHeight, &wmHints.win_gravity);

    /* Create the top-level window */

    attr.border_pixel = 0; /* needed to allow 8-bit cmap on 24-bit display -
			      otherwise we get a Match error! */
    attr.background_pixel = BlackPixelOfScreen(DefaultScreenOfDisplay(dpy));
    attr.event_mask = ( LeaveWindowMask|
			StructureNotifyMask|
			ButtonPressMask|
			ButtonReleaseMask|
			PointerMotionMask|
			KeyPressMask|
			KeyReleaseMask|
			EnterWindowMask|
			(resurface?VisibilityChangeMask:0) );
      
    attr.override_redirect=True;

    topLevel = XCreateWindow(dpy, DefaultRootWindow(dpy), wmHints.x, wmHints.y,
			     topLevelWidth, topLevelHeight, 0, CopyFromParent,
			     InputOutput, CopyFromParent,
			     (CWBorderPixel|
			      CWEventMask|
			      CWOverrideRedirect|
			      CWBackPixel),
			     &attr);

    wmHints.flags |= USPosition; /* try to force WM to place window */
    XSetWMNormalHints(dpy, topLevel, &wmHints);

    wmProtocols = XInternAtom(dpy, "WM_PROTOCOLS", False);
    wmDeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, topLevel, &wmDeleteWindow, 1);

    XStoreName(dpy, topLevel, desktopName);


    XMapRaised(dpy, topLevel);

    XSelectInput(dpy, DefaultRootWindow(dpy), PropertyChangeMask);

    myFormat.bitsPerPixel = 8;
    myFormat.depth = 8;
    myFormat.trueColour = 1;
    myFormat.bigEndian = 0;
    myFormat.redMax = 7;
    myFormat.greenMax = 7;
    myFormat.blueMax = 3;
    myFormat.redShift = 0;
    myFormat.greenShift = 3;
    myFormat.blueShift = 6;

    nullPixmap = XCreatePixmap(dpy, DefaultRootWindow(dpy), 1, 1, 1);
    grabCursor = 
      XCreatePixmapCursor(dpy, nullPixmap, nullPixmap,
			  &dummyColor, &dummyColor, 0, 0);
    
    return True;
}



/*
 * ShutdownX.
 */

void
ShutdownX()
{
    XCloseDisplay(dpy);
}


/*
 * HandleXEvents.
 */

Bool
HandleXEvents()
{
    XEvent ev;

    while (XCheckIfEvent(dpy, &ev, AllXEventsPredicate, NULL)) {

      if (ev.xany.window == topLevel) {

	    if (!HandleTopLevelEvent(&ev))
		return False;

	} else if (ev.xany.window == DefaultRootWindow(dpy)) {

	    if (!HandleRootEvent(&ev))
		return False;

	} else if (ev.type == MappingNotify) {

	    XRefreshKeyboardMapping(&ev.xmapping);
	}
    }

    return True;
}

#define EW (edge == EDGE_EAST || edge==EDGE_WEST)
#define NS (edge == EDGE_NORTH || edge==EDGE_SOUTH)
#define CINVERSE(X,Y,Z) ((X)?((Z)==(Y)?1:!(Z)?(Y)-2:(Y)-(Z)):(Z))

#define SCALEX(X) ((X)-(edge==EDGE_WEST))*si.framebufferWidth/(displayWidth - EW)
#define SCALEY(Y) ((Y)-(edge==EDGE_NORTH))*si.framebufferHeight/(displayHeight - EW)

/*
 * HandleTopLevelEvent.
 */

static Bool
HandleTopLevelEvent(XEvent *ev)
{
  int i;
  
  int buttonMask;
  KeySym ks;
  char keyname[256];
  
  switch (ev->type) {
    case VisibilityNotify:
      if (ev->xvisibility.state != VisibilityUnobscured)
	XRaiseWindow(dpy, topLevel);
      return 1;
      
    case EnterNotify:
      if(!grabbed && ev->xcrossing.mode==NotifyNormal)
      {
	XGrabPointer(dpy, topLevel, True,
		     PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
		     GrabModeAsync, GrabModeAsync,
		     None, grabCursor, CurrentTime);
	XGrabKeyboard(dpy, topLevel, True, 
		      GrabModeAsync, GrabModeAsync,
		      CurrentTime);
	XFlush(dpy);
	XWarpPointer(dpy,None, DefaultRootWindow(dpy),0,0,0,0,
		     CINVERSE(EW,displayWidth,ev->xcrossing.x_root),
		     CINVERSE(NS,displayHeight,ev->xcrossing.y_root));
	
	SendPointerEvent(SCALEX(ev->xcrossing.x_root),
			 SCALEY(ev->xcrossing.y_root),
			 (ev->xmotion.state & 0x1f00) >> 8);
	grabbed=1;
      }
      return 1;
      
      
    case MotionNotify:
      
      if(grabbed)
      {
	int i, d;
	while (XCheckTypedWindowEvent(dpy, topLevel, MotionNotify, ev))
	  ;	/* discard all queued motion notify events */
	
	i=SendPointerEvent(SCALEX(ev->xmotion.x_root),
			   SCALEY(ev->xmotion.y_root),
			   (ev->xmotion.state & 0x1f00) >> 8);
	
	switch(edge)
	{
	  case EDGE_NORTH: d=ev->xmotion.y_root == displayHeight-1; break;
	  case EDGE_SOUTH: d=ev->xmotion.y_root == 0; break;
	  case EDGE_EAST: d=ev->xmotion.x_root == 0; break;
	  case EDGE_WEST: d=ev->xmotion.x_root == displayWidth-1; break;
	}
	if(d)
	{
	  XUngrabKeyboard(dpy, CurrentTime);
	  XUngrabPointer(dpy, CurrentTime);
	  XWarpPointer(dpy,None, DefaultRootWindow(dpy),0,0,0,0,
		       CINVERSE(EW,displayWidth,ev->xmotion.x_root),
		       CINVERSE(NS,displayHeight,ev->xmotion.y_root));
	  XFlush(dpy);
	  
	  for (i = 0; i < 256; i++) {
	    if (modifierPressed[i]) {
	      if (!SendKeyEvent(XKeycodeToKeysym(dpy, i, 0), False))
		return False;
	      modifierPressed[i]=False;
	    }
	  }
	  
	  grabbed=0;
	}
	return i;
      }
      return 1;
      
    case ButtonPress:
    case ButtonRelease:
      if (ev->type == ButtonPress) {
	buttonMask = (((ev->xbutton.state & 0x1f00) >> 8) |
		      (1 << (ev->xbutton.button - 1)));
      } else {
	buttonMask = (((ev->xbutton.state & 0x1f00) >> 8) &
		      ~(1 << (ev->xbutton.button - 1)));
      }
      
      return SendPointerEvent(SCALEX(ev->xbutton.x_root),
			      SCALEY(ev->xbutton.y_root),
			      buttonMask);
      
    case KeyPress:
    case KeyRelease:
      XLookupString(&ev->xkey, keyname, 256, &ks, NULL);
      
      if (IsModifierKey(ks)) {
	ks = XKeycodeToKeysym(dpy, ev->xkey.keycode, 0);
	modifierPressed[ev->xkey.keycode] = (ev->type == KeyPress);
      }
      
      return SendKeyEvent(ks, (ev->type == KeyPress));
      
      
    case ClientMessage:
      if ((ev->xclient.message_type == wmProtocols) &&
	  (ev->xclient.data.l[0] == wmDeleteWindow))
	{
	    ShutdownX();
	    exit(0);
	}
	break;
    }

    return True;
}



/*
 * HandleRootEvent.
 */

static Bool
HandleRootEvent(XEvent *ev)
{
    char *str;
    int len;

    switch (ev->type) {

    case PropertyNotify:
	if (ev->xproperty.atom == XA_CUT_BUFFER0) {

	    str = XFetchBytes(dpy, &len);
	    if (str) {
		if (!SendClientCutText(str, len))
		    return False;
		XFree(str);
	    }
	}
	break;
    }

    return True;
}


/*
 * AllXEventsPredicate is needed to make XCheckIfEvent return all events.
 */

Bool
AllXEventsPredicate(Display *dpy, XEvent *ev, char *arg)
{
    return True;
}



