/*
 *  Apple II emulator by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: defs.H,v 1.9 1997/06/29 22:14:08 chernabog Exp $
 *
 * MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *
 *  global #defines, typedefs, enums, etc for assembly and C source
 *  files.
 **/


#ifdef __ASSEMBLY__

#define FLAG(a,b) a ## b

#else

#define FLAG(a,b) b

#define	K *1024				/* kilobytes */
typedef void (*Function)();
typedef enum { False, True } Tr;	/* --- Domain of Truth values
					       (using the notation by
					       John Allen - "The
					       Anatomy of LISP") --- */

typedef enum {JOY_OFF, JOY_KYBD, JOY_DIGITAL, JOY_PCJOY} joystick_mode;

#ifdef DEBUGGER
#define BUF_X		39
#define BUF_Y		22
/* debugger commands */
enum token_type { MEM, DIS, REGS, SETMEM, STEP, FINISH, UNTIL, GO,
		  BREAK, WATCH, CLEAR, IGNORE, STATUS, LC, DRIVE,
		  SEARCH, GRAPHICS, SOUND, HELP, LOG, SAVE, UNKNOWN };
#define MAX_BRKPTS	16

#endif

#endif


/* -------------------------------------------------------------------------
    Macros and equates
   ------------------------------------------------------------------------- */

#define RebootSig	FLAG($,0x01)
#define ResetSig	FLAG($,0x02)
#define DebugStepSig	FLAG($,0x04)
#define EnterDebugSig	FLAG($,0x08)

#define C_Flag		FLAG($,0x1)		/* 6502 Carry	 	   */
#define X_Flag		FLAG($,0x2)		/* 6502 Xtra		   */
#define I_Flag		FLAG($,0x4)		/* 6502 Interrupt disable  */
#define V_Flag		FLAG($,0x8)		/* 6502 Overflow	   */
#define B_Flag		FLAG($,0x10)		/* 6502 Break		   */
#define D_Flag		FLAG($,0x20)		/* 6502 Decimal mode	   */
#define Z_Flag		FLAG($,0x40)		/* 6502 Zero		   */
#define N_Flag		FLAG($,0x80)		/* 6502 Neg		   */

#define NZ_Flag		FLAG($,0xC0)
#define NZC_Flag	FLAG($,0xC1)
#define NVZC_Flag	FLAG($,0xC9)

#define C_Flag_Not	FLAG($,0xFE)
#define Z_Flag_Not      FLAG($,0xBF)
#define NZ_Flag_Not	FLAG($,0x3F)
#define NZC_Flag_Not	FLAG($,0x3E)
#define NVZC_Flag_Not	FLAG($,0x36)
#define NVZ_Flag_Not	FLAG($,0x37)

#define C_Flag_Bit	FLAG($,8)		/* 6502 Carry		   */
#define X_Flag_Bit	FLAG($,9)		/* 6502 Xtra		   */
#define I_Flag_Bit	FLAG($,10)		/* 6502 Interrupt disable  */
#define V_Flag_Bit	FLAG($,11)		/* 6502 Overflow	   */
#define B_Flag_Bit	FLAG($,12)		/* 6502 Break		   */
#define D_Flag_Bit	FLAG($,13)		/* 6502 Decimal mode	   */
#define Z_Flag_Bit	FLAG($,14)		/* 6502 Zero		   */
#define N_Flag_Bit	FLAG($,15)		/* 6502 Neg		   */

#define X_Reg		%bl			/* 6502 X register in %bl  */
#define Y_Reg		%bh			/* 6502 Y register in %bh  */
#define A_Reg		%cl			/* 6502 A register in %cl  */
#define F_Reg		%ch			/* 6502 flags in %ch	   */
#define FF_Reg		%ecx			/* 6502 flags for bt	   */
#define SP_Reg		%edx			/* 6502 Stack pointer	   */
#define SP_Reg_L	%dl			/* 6502 Stack pointer low  */
#define SP_Reg_H	%dh			/* 6502 Stack pointer high */
#define PC_Reg		%si			/* 6502 Program Counter    */
#define PC_Reg_E	%esi			/* 6502 Program Counter    */
#define EffectiveAddr	%di			/* Effective address	   */
#define EffectiveAddr_E	%edi			/* Effective address	   */


/* -------------------------------------------------------------------------
    Video
   ------------------------------------------------------------------------- */

#define Video_Reg	%ebx			/* Video mode register	   */

#define Video_Text_Bit	FLAG($,24)		/* Text mode bit	   */
#define Video_Mixed_Bit FLAG($,25)		/* Mixed mode bit	   */
#define Video_Page2_Bit FLAG($,26)		/* Page2 mode bit	   */
#define Video_Hires_Bit FLAG($,27)		/* Hires mode bit	   */
#define SVGA_Page_Bit	FLAG($,31)		/* SVGA page bit	   */

/* -------------------------------------------------------------------------
    Disk drive
   ------------------------------------------------------------------------- */

#define Disk_Old		FLAG($,0x1)
#define	Disk_Write_Protect	FLAG($,0x2)
#define Disk_Present		FLAG($,0x4)
#define Disk_Motor_On		FLAG($,0x8)
#define Disk_Output_Mode	FLAG($,0x10)
#define Disk_Track_Modified	FLAG($,0x20)
#define Disk_Buffer_Valid	FLAG($,0x40)

#define Disk_Not_Output_Mode	FLAG($,0xEF)
#define Disk_Not_Buffer_Valid	FLAG($,0xBF)
#define Disk_Not_Motor_On	FLAG($,0xF7)

/* -------------------------------------------------------------------------
    CPU (6502) Routines
   ------------------------------------------------------------------------- */

#define GetFromPC_B	xorl	%eax, %eax;				  \
			movw	%si, EffectiveAddr;			  \
			call	*SYMBOL_NAME(table_read_memory)(,%esi,4); \
			incw	%si;

#define GetFromPC_W	cmpw	$0xcfff, %si;				  \
			jl	100f;					  \
			xorl	%eax, %eax;				  \
			movw	%si, EffectiveAddr;			  \
			call	*SYMBOL_NAME(table_read_memory)(,%esi,4); \
			xchgb	%al, %ah;				  \
			incw	%si;					  \
			incw	EffectiveAddr;				  \
			call	*SYMBOL_NAME(table_read_memory)(,%esi,4); \
			xchgb	%al, %ah;				  \
			jmp	101f;					  \
		100:	movzbl	SYMBOL_NAME(apple_ii_64k)(,%esi,1), %eax; \
			incw	%si;					  \
			movb	SYMBOL_NAME(apple_ii_64k)(,%esi,1), %ah;  \
		101:	incw	%si;

#define GetFromMem_B(x)						\
			movl	x, EffectiveAddr_E;		\
			call	*SYMBOL_NAME(table_read_memory)	\
					(,EffectiveAddr_E,4);

#define GetFromMem_W(x)							\
			movl	x, EffectiveAddr_E;			\
			cmpw	$0xcfff, EffectiveAddr;			\
			jl	200f;					\
/*			xorl	%eax, %eax;*/				\
			call	*SYMBOL_NAME(table_read_memory)		\
					(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;				\
			incw	EffectiveAddr;				\
			call	*SYMBOL_NAME(table_read_memory)		\
					(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;				\
			jmp	201f;					\
		200:	movzbl	SYMBOL_NAME(apple_ii_64k)		\
					(,EffectiveAddr_E,1), %eax;	\
			incw	EffectiveAddr;				\
			movb	SYMBOL_NAME(apple_ii_64k)		\
					(,EffectiveAddr_E,1), %ah;	\
		201:

#define Wait		movw	SYMBOL_NAME(apple_speed), %ax;		      \
		   0:							      \
			decw	%ax;					      \
			jnz	0b;

#ifdef DEBUGGER

#define SaveState	movl	PC_Reg, SYMBOL_NAME(debug_PC);		 \
			movl	EffectiveAddr, SYMBOL_NAME(debug_EA);	 \
			movl	SP_Reg, SYMBOL_NAME(debug_SP);		 \
			movb	X_Reg, SYMBOL_NAME(debug_X);		 \
			movb	Y_Reg, SYMBOL_NAME(debug_Y);		 \
			movb	A_Reg, SYMBOL_NAME(debug_A);		 \
			movb	F_Reg, SYMBOL_NAME(debug_F);

#define ReplaceState	movl	SYMBOL_NAME(debug_PC), PC_Reg;		\
			movl	SYMBOL_NAME(debug_EA), EffectiveAddr;	\
			movl	SYMBOL_NAME(debug_SP), SP_Reg;		\
			movb	SYMBOL_NAME(debug_X), X_Reg;		\
			movb	SYMBOL_NAME(debug_Y), Y_Reg;		\
			movb	SYMBOL_NAME(debug_A), A_Reg;		\
			movb	SYMBOL_NAME(debug_F), F_Reg;
#endif

/* DEBUGGER version of continue.  (we have other exception_flag
   values) */
#ifdef DEBUGGER
#define Continue	Wait						   \
			testb   $0xFF, SYMBOL_NAME(exception_flag);	   \
			jnz     1f;					   \
			GetFromPC_B					   \
			jmp	table_opcodes(,%eax,4);			   \
									   \
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	   \
			jz	4f;/*reboot*/				   \
			cmpb	ResetSig, SYMBOL_NAME(exception_flag);	   \
			jz	5f;/*reset*/				   \
			cmpb	DebugStepSig, SYMBOL_NAME(exception_flag); \
			jz	2f;/*debug step*/			   \
			jmp	3f;/*enter debugger*/			   \
									   \
/* step */	2:	SaveState;					   \
			ret;						   \
									   \
/* enter */	3:	SaveState;					   \
			pushal;						   \
/* do debugging */	call	SYMBOL_NAME(c_update_keyboard);		   \
			popal;						   \
			ReplaceState					   \
			GetFromPC_B					   \
			jmp	table_opcodes(,%eax,4);			   \
									   \
/* reboot */	4:	movb	$0, SYMBOL_NAME(exception_flag);	   \
			ret;						   \
/* reset */	5:	movb	$0, SYMBOL_NAME(exception_flag);	   \
			movw	$0xfffc, EffectiveAddr;			   \
			call	read_memory_word;			   \
			movw    %ax, PC_Reg;				   \
			GetFromPC_B					   \
			jmp	table_opcodes(,%eax,4);
#else
#define Continue	Wait						\
			testb   $0xFF, SYMBOL_NAME(exception_flag);	\
			jnz     1f;					\
			GetFromPC_B					\
			jmp	table_opcodes(,%eax,4);			\
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	\
			jz	2f;/*reboot*/				\
			movb	$0, SYMBOL_NAME(exception_flag);	\
			movw	$0xfffc, EffectiveAddr;			\
			call	read_memory_word;			\
			movw    %ax, PC_Reg;				\
			GetFromPC_B					\
			jmp	table_opcodes(,%eax,4);			\
/* reboot */	2:	movb	$0, SYMBOL_NAME(exception_flag);	\
			ret;
#endif


#define FlagC		lahf;						      \
  			andb	C_Flag, %ah;			      	      \
  			andb	C_Flag_Not, F_Reg;	      	      	      \
			orb	%ah, F_Reg;

#define FlagZ		lahf;						      \
  			andb	Z_Flag, %ah;			      	      \
  			andb	Z_Flag_Not, F_Reg;	      	      	      \
			orb	%ah, F_Reg;

#define FlagN		lahf;						      \
  			andb	N_Flag, %ah;			      	      \
  			andb	N_Flag_Not, F_Reg;	      	      	      \
			orb	%ah, F_Reg;

#define FlagNZ		lahf;						      \
  			andb	NZ_Flag, %ah;			      	      \
  			andb	NZ_Flag_Not, F_Reg;	      	      	      \
			orb	%ah, F_Reg;

#define FlagNZC		lahf;						      \
			andb	NZC_Flag, %ah;		      		      \
			andb	NZC_Flag_Not, F_Reg;      		      \
			orb	%ah, F_Reg;

#define FlagNVZC	lahf;					      	      \
			seto	%al;					      \
			shlb    $3, %al;				      \
			andb	NZC_Flag, %ah;		      		      \
			andb	NVZC_Flag_Not, F_Reg;      		      \
			orb	%al, %ah;				      \
			orb	%ah, F_Reg;

#define Push(x)		movb	x, SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1);      \
			decb	SP_Reg_L;

#define Pop(x)		incb	SP_Reg_L;				      \
			movb	SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1), x;

/* Immediate Addressing - the operand is contained in the second byte of the
   instruction. */
#define GetImm		movw	PC_Reg, EffectiveAddr;			      \
			incw	PC_Reg;

/* Absolute Addressing - the second byte of the instruction is the low
   order address, and the third byte is the high order byte. */
#define GetAbs		GetFromPC_W;					      \
			movw	%ax, EffectiveAddr;

/* Zero Page Addressing - the second byte of the instruction is an
   address on the zero page */
#define GetZPage	GetFromPC_B;					      \
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Addressing - The effective address is calculated by
   adding the second byte to the contents of the index register.  Due
   to the zero page addressing nature of this mode, no carry is added
   to the high address byte, and the crossing of page boundaries does
   not occur. */
#define GetZPage_X	GetFromPC_B;					      \
			addb	X_Reg, %al;				      \
			movw	%ax, EffectiveAddr;

#define GetZPage_Y	GetFromPC_B;					      \
			addb	Y_Reg, %al;				      \
			movw	%ax, EffectiveAddr;

/* Absolute Indexed Addressing - The effective address is formed by
   adding the contents of X or Y to the address contained in the
   second and third bytes of the instruction. */
#define GetAbs_X	GetFromPC_W;					      \
			addb	X_Reg, %al;				      \
			adcb	$0, %ah;				      \
			movw	%ax, EffectiveAddr;

#define GetAbs_Y	GetFromPC_W;					      \
			addb	Y_Reg, %al;				      \
			adcb	$0, %ah;				      \
			movw	%ax, EffectiveAddr;

/* Absolute Indirect Addressing - The second and third bytes of the
   instruction are the low and high bytes of an address, respectively.
   The contents of the fully specified memory location is the
   low-order byte of the effective address.  The next memory location
   contains the high order byte of the effective address. */
/*HACK - used?*/
#define GetInd		GetFromPC_W;					      \
			GetFromMem_W(%eax)

/* Zero Page Indirect Addressing (65C02) - The second byte of the
   instruction points to a memory location on page zero containing the
   low order byte of the effective address.  The next location on page
   zero contains the high order byte of the address. */
#define GetIndZPage	GetFromPC_B;					      \
			movw	%ax, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			incw	%di;					      \
			andw	$0xFF, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Indirect Addressing - The second byte is added to
   the contents of the X index register; the carry is discarded.  The
   result of this addition points to a memory location on page zero
   whose contents is the low order byte of the effective address.  The
   next memory location in page zero contains the high-order byte of
   the effective address.  Both memory locations specifying the high
   and low-order bytes must be in page zero. */
#define GetIndZPage_X	GetFromPC_B;					      \
			addb	X_Reg, %al;				      \
			movw	%ax, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			incw	%di;					      \
			andw	$0xFF, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			movw	%ax, EffectiveAddr;

/* Indirect Indexed Addressing - The second byte of the instruction
   points to a memory location in page zero.  The contents of this
   memory location are added to the contents of the Y index register,
   the result being the low order byte of the effective address.  The
   carry from this addition is added to the contents of the next page
   zero memory location, the result being the high order byte of the
   effective address. */
#define GetIndZPage_Y   GetFromPC_B;					      \
			movw	%ax, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			incw	%di;					      \
			andw	$0xFF, %di;				      \
			GetFromMem_B(%edi);				      \
			xchgb	%al, %ah;				      \
			addb	Y_Reg, %al;				      \
			adcb	$0, %ah;				      \
			movw	%ax, EffectiveAddr;

#define DoADC_b		call	read_memory;				      \
			bt	C_Flag_Bit, FF_Reg;			      \
			adcb	%al, A_Reg;				      \
			FlagNVZC

#define	DoADC_d		call	read_memory;				      \
			bt	C_Flag_Bit, FF_Reg;			      \
			adcb	A_Reg, %al;				      \
			daa;						      \
			movb	%al, A_Reg;				      \
			FlagNVZC

#define DoAND		call	read_memory;				      \
			andb	%al, A_Reg;				      \
			FlagNZ

#define DoASL		call	read_memory;				      \
			shlb	$1, %al;				      \
			FlagNZC						      \
			call	write_memory;

#define DoBIT		andb	NVZ_Flag_Not, F_Reg;	      		      \
			call	read_memory;				      \
			testb	%al, A_Reg;				      \
			FlagZ						      \
			movb    %al, %ah;				      \
			andb	$0x40, %al;				      \
			shrb	$3, %al;				      \
			orb	%al, F_Reg;				      \
			andb    $0x80, %ah;				      \
			orb	%ah, F_Reg;

#define DoCMP		call	read_memory;				      \
			cmpb	%al, A_Reg;				      \
			cmc;						      \
			FlagNZC

#define DoCPX		call	read_memory;				      \
			cmpb	%al, X_Reg;				      \
			cmc;						      \
			FlagNZC

#define DoCPY		call	read_memory;				      \
			cmpb	%al, Y_Reg;				      \
			cmc;						      \
			FlagNZC

#define DoDEC		call	read_memory;				      \
			decb	%al;					      \
			FlagNZ						      \
			call	write_memory;

#define DoEOR		call	read_memory;				      \
			xorb	%al, A_Reg;				      \
			FlagNZ

#define DoINC		call	read_memory;				      \
			incb	%al;					      \
			FlagNZ						      \
			call	write_memory;

#define DoJMP		movw	EffectiveAddr, PC_Reg;

#define DoJSR		movw	PC_Reg, %ax;				     \
			decw	%ax;					     \
			Push(%ah)					     \
			Push(%al)					     \
			movw	EffectiveAddr, PC_Reg;

#define DoLDA		call	read_memory;				     \
			movb	%al, A_Reg;				     \
			orb	%al, %al;				     \
			FlagNZ

#define DoLDX		call	read_memory;			  	     \
			movb	%al, X_Reg;				     \
			orb	%al, %al;				     \
			FlagNZ

#define DoLDY		call	read_memory;				     \
			movb	%al, Y_Reg;				     \
			orb	%al, %al;				     \
			FlagNZ

#define DoLSR		call	read_memory;				     \
			shrb	$1, %al;				     \
			FlagNZC						     \
			call	write_memory;

#define DoORA		call	read_memory;				     \
			orb	%al, A_Reg;				     \
			FlagNZ

#define DoROL(L,CONT)	call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			rclb	$1, %al;				     \
			jc	L;				   	     \
			orb	%al, %al;				     \
			clc;						     \
			FlagNZC						     \
			call	write_memory;				     \
			CONT						     \
	  L##:		orb	%al, %al;				     \
			stc;						     \
			FlagNZC						     \
			call	write_memory;				     \
			CONT

#define DoROR(L,CONT)	call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			rcrb	$1, %al;				     \
			jc	L;					     \
			orb	%al, %al;				     \
			clc;						     \
			FlagNZC						     \
			call	write_memory;				     \
			CONT						     \
	  L##:		orb	%al, %al;				     \
			stc;						     \
			FlagNZC						     \
			call	write_memory;				     \
			CONT

#define DoSBC_b		call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			cmc;						     \
			sbbb	%al, A_Reg;				     \
			cmc;						     \
			FlagNVZC

#define DoSBC_d		call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			cmc;						     \
			xchgb	A_Reg, %al;				     \
			sbbb	A_Reg, %al;				     \
			das;						     \
			movb	%al, A_Reg;				     \
			cmc;						     \
			FlagNVZC

#define DoSTA		movb	A_Reg, %al;				     \
			call	write_memory;

#define DoSTX		movb	X_Reg, %al;				     \
			call	write_memory;

#define DoSTY		movb	Y_Reg, %al;				     \
			call	write_memory;

/* -------------------------------------------------------------------------
    Undocumented 6502 (Illegal instructions)
   ------------------------------------------------------------------------- */

        /* AAX = A AND X -> M */
#define DoAAX		movb	A_Reg, %al;				     \
			andb	X_Reg, %al;				     \
			FlagNZ						     \
			call	write_memory;

	/* AMA = ORA 238, AND M, TAX */
#define DoAMA		orb	$238, A_Reg;				     \
			call	read_memory;				     \
			andb	%al, A_Reg;				     \
			movb	A_Reg, X_Reg;				     \
			FlagNZ

	/* ANA = AND M, Carry = BIT 7 */
#define DoANA(CONT)	call	read_memory;				     \
			andb	%al, A_Reg;				     \
			js	1f;					     \
			clc;						     \
			FlagNZC						     \
			CONT						     \
		1:	stc;						     \
			FlagNZC						     \
			CONT

	/* ANB = same as ANA */
#define DoANB(CONT)	call	read_memory;				     \
			andb	%al, A_Reg;				     \
			js	1f;					     \
			clc;						     \
			FlagNZC						     \
			CONT						     \
		1:	stc;						     \
			FlagNZC						     \
			CONT

	/* AXM = (A AND X) - M -> X */
#define DoAXM		call	read_memory;				     \
			andb	A_Reg, X_Reg;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			cmc;						     \
			sbbb	%al, X_Reg;				     \
			cmc;						     \
			FlagNVZC

	/* AXS = (A AND X) -> S, A AND X AND 17 -> M */
			/* HACK!!!!!!!!!!!!!!! */
#define DoAXS		movb	A_Reg, SP_Reg_L;	\
			andb	X_Reg, SP_Reg_L;	\
			movb	SP_Reg_L, %al;		\
			andb	$17, %al;		\
			FlagNZ	/* \ wasn't here */	\
			call	write_memory; 

	/* DCP = DEC M, CMP M */
#define	DoDCP		call	read_memory;				     \
			decb	%al;					     \
			cmpb	%al, A_Reg;				     \
			cmc;						     \
			FlagNZC						     \
			call	write_memory;

	/* ISB = INC M, SBC M */
#define DoISB_b		call	read_memory;				     \
			incb	%al;					     \
			call	write_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			cmc;						     \
			sbbb	%al, A_Reg;				     \
			cmc;						     \
			FlagNVZC

#define DoISB_d		call	read_memory;				     \
			incb	%al;					     \
			call	write_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			cmc;						     \
			xchgb	A_Reg, %al;				     \
			sbbb	A_Reg, %al;				     \
			das;						     \
			movb	%al, A_Reg;				     \
			cmc;						     \
			FlagNVZC

	/* LAN = ROL M, AND M */
#define DoLAN(CONT)	call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			rclb	$1, %al;				     \
			jc	1f;					     \
			call	write_memory;				     \
			andb	%al, A_Reg;				     \
			clc;						     \
			FlagNZC						     \
			CONT						     \
		   1:	call	write_memory;				     \
			andb	%al, A_Reg;				     \
			stc;						     \
			FlagNZC						     \
			CONT

	/* LAS = LDA M, TAX, TXS */
#define DoLAS		call	read_memory;				     \
			movb	%al, A_Reg;				     \
			movb	%al, X_Reg;				     \
			movb	%al, SP_Reg_L;				     \
			orb	%al, %al;				     \
			FlagNZ

	/* LAX = LDA M, TAX */
#define DoLAX		call	read_memory;				     \
			movb	%al, A_Reg;				     \
			movb	%al, X_Reg;				     \
			orb	%al, %al;				     \
			FlagNZ

	/* LOR = ASL M, ORA M */
#define DoLOR		call	read_memory;				     \
			shlb	$1, %al;				     \
			FlagC						     \
			call	write_memory;				     \
			orb	%al, A_Reg;				     \
			FlagNZ

	/* RAD = ROR M, ADC M */
#define DoRAD_b		call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			rcrb	$1, %al;				     \
			adcb	%al, A_Reg;				     \
			pushl	%eax;					     \
			FlagNVZC					     \
			popl	%eax;					     \
			call	write_memory;

#define DoRAD_d		call	read_memory;				     \
			bt	C_Flag_Bit, FF_Reg;			     \
			rcrb	$1, %al;				     \
			pushfl;						     \
			call	write_memory;				     \
			popfl;						     \
			adcb	A_Reg, %al;				     \
			daa;						     \
			movb	%al, A_Reg;				     \
			FlagNVZC

	/* RAM = AND M, LSR A */
#define DoRAM		call	read_memory;				     \
			andb	%al, A_Reg;				     \
			shrb	$1, A_Reg;				     \
			FlagNZC

	/* RBM = same as RAM */
#define DoRBM		DoRAM

	/* REO = LSR M, EOR M */
#define DoREO		call	read_memory;				     \
			shrb	$1, %al;				     \
			xorb	%al, A_Reg;				     \
			FlagNZC						     \
			call	write_memory;

	/* DoZBC = same as SBC */
#define DoZBC_b		DoSBC_b
#define DoZBC_d		DoSBC_d

	/* TEA = (A AND X AND (OP+2)+1) -> M */
#define	DoTEA		pushl	EffectiveAddr_E;			     \
			movw	PC_Reg, EffectiveAddr;			     \
			decw	EffectiveAddr;			     	     \
			call	read_memory;				     \
			popl	EffectiveAddr_E;			     \
			incb	%al;					     \
			andb	A_Reg, %al;				     \
			andb	X_Reg, %al;				     \
			FlagNZ						     \
			call	write_memory;

	/* TEX = (X AND (OP+2)+1) -> M */
#define DoTEX		pushl	EffectiveAddr_E;			     \
			movw	PC_Reg, EffectiveAddr;			     \
			decw	EffectiveAddr;			     	     \
			call	read_memory;				     \
			popl	EffectiveAddr_E;			     \
			incb	%al;					     \
			andb	X_Reg, %al;				     \
			FlagNZ						     \
			call	write_memory;

	/* TEY = (Y AND 1) -> M */
#define DoTEY		movb	Y_Reg, %al;				     \
			andb	$1, %al;				     \
			FlagNZ						     \
			call	write_memory;

	/* XMA = (X AND M) AND (A OR 238) -> A */
			/* HACK!!!!!!!!!!!!!!! */
#define DoXMA	/* the \ wasn't here before */	\
			call	read_memory;	\
			andb	X_Reg, %al;	\
			orb	$238, A_Reg;	\
			andb	%al, A_Reg;	\
			FlagNZ
