// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <stdio.h>
}
#include <qmsgbox.h>
#include <qstrlist.h>
#include "SgmlParser.h"
#include "StyleSheet.h"
#include "Url.h"

StyleSheet::StyleSheet()
{
    _styles.setAutoDelete( TRUE );
}

StyleSheet::StyleSheet( const StyleSheet& src )
        : _soi( src._soi.copy() )
{
    QListIterator<Style> i( src._styles );
    for ( i.toFirst(); i.current(); ++i ) {
        _styles.append( new Style( *i.current() ) );
    }
}

void StyleSheet::addStyle( Style*  style )
{
    _styles.append( style );
}

void StyleSheet::deleteStyle( Style* style )
{
//    delete style;
    _styles.remove( style );
}

Style* StyleSheet::getStyle( const Style* parent, const STag& tag )
{
    // Find the best matching style for the given start tag.
    int    factor = 0;
    int    tmp;
    Style* style = 0;
    for ( _styles.first(); _styles.current(); _styles.next() ) {
        if ( ( tmp = _styles.current()->matchFactor( &tag ) ) > factor ) {
            style = _styles.current();
            factor = tmp;
        }
    }

    if ( !style ) {
        return new Style( tag.name );
    }

    style = inherit( parent, style );

    return style;
}

Style* StyleSheet::inherit( const Style* base,
                            const Style* style )
{
    Style *s = new Style( style->_name );
    s->setDisplay( style->_display );

    StyleAttr* attr;
    StyleAttr* a;
    QListIterator<StyleAttr> i( style->_attrs );
    for ( attr = i.toFirst(); attr; attr = ++i ) {
        switch ( attr->type() ) {
            case StyleAttrProto::String:
                a = new StringStyleAttr( *((StringStyleAttr*)attr) );
                break;

            case StyleAttrProto::Number:
                a = new NumberStyleAttr( *((NumberStyleAttr*)attr) );
                break;

            case StyleAttrProto::Enumerated:
                a = new EnumeratedStyleAttr( *((EnumeratedStyleAttr*)attr) );
                break;

            case StyleAttrProto::List:
                a = new ListStyleAttr( *((ListStyleAttr*)attr) );
                break;

            default:
                a = 0;
                break;
        }
        if ( a ) {
            s->_attrs.append( a );
        }
    }

    if ( base ) {
        i = base->_attrs;
        for ( attr = i.toFirst(); attr; attr = ++i ) {
            if ( ( attr->proto()->inherit() ) && ( !s->find( attr->proto()->token() ) ) ) {
                switch ( attr->type() ) {
                    case StyleAttrProto::String:
                        a = new StringStyleAttr( *((StringStyleAttr*)attr) );
                        break;
                        
                    case StyleAttrProto::Number:
                        a = new NumberStyleAttr( *((NumberStyleAttr*)attr) );
                        break;
                        
                    case StyleAttrProto::Enumerated:
                        a = new EnumeratedStyleAttr( *((EnumeratedStyleAttr*)attr) );
                        break;
                        
                    case StyleAttrProto::List:
                        a = new ListStyleAttr( *((ListStyleAttr*)attr) );
                        break;
                        
                    default:
                        a = 0;
                        break;
                }
                if ( a ) {
                    s->_attrs.append( a );
                }
            }
        }
    }

    return s;
}

void StyleSheet::save()
{
    // Open the style sheet file for writing.
    Url url( soi() );
    if ( url.method() != "file" ) {
        QString error;
        error.sprintf( "Don't known how to write style sheet with method '%s'.", url.method().data() );
        QMessageBox::message( "QWeb: Error", error );
        return;
    }

    // Open the file for writing.
    FILE* fptr = fopen( url.path().data(), "w" );
    if ( !fptr ) {
        QString error;
        error.sprintf( "Failed to open file '%s' for writing.", url.path().data() );
        QMessageBox::message( "QWeb: Error", error );
        return;
    }

    Style* s;
    StyleAttr* attr;
    for ( s = _styles.first(); s; s = _styles.next() ) {
        fprintf( fptr, "%s\n", s->selectorString().data() );
        fprintf( fptr, "{\n" );
        fprintf( fptr, "    display: %s\n", s->_display->token.data() );
        for ( attr = s->_attrs.first(); attr; attr = s->_attrs.next() ) {
            switch ( attr->type() ) {
                case StyleAttrProto::String:
                    fprintf( fptr, "    %s: %s\n", attr->proto()->token().data(), ((StringStyleAttr*)attr)->value().data() );
                    break;
                    
                case StyleAttrProto::Number:
                    fprintf( fptr, "    %s: %d\n", attr->proto()->token().data(), ((NumberStyleAttr*)attr)->value() );
                    break;

                case StyleAttrProto::Enumerated:
                    fprintf( fptr, "    %s: %s\n", attr->proto()->token().data(), attr->proto()->valueToToken( ((EnumeratedStyleAttr*)attr)->value() )->data() );
                    break;

                case StyleAttrProto::List:
                    fprintf( fptr, "    %s: %s\n", attr->proto()->token().data(), ((ListStyleAttr*)attr)->value().data() );
                    break;

                case StyleAttrProto::Flag:
                    fprintf( fptr, "    %s: true\n", attr->proto()->token().data() );
                    break;
            }
        }
        fprintf( fptr, "}\n\n" );
    }

    fclose( fptr );
}
