/*
 * aboot.c
 *
 * This file is part of aboot, the SRM bootloader for Linux/Alpha
 * Copyright (C) 1996 Linus Torvalds, David Mosberger, and Michael Schwingen.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/a.out.h>
#include <linux/elf.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>

#include <asm/console.h>
#include <asm/hwrpb.h>
#include <asm/system.h>

#include <alloca.h>

#include <aboot.h>
#include <config.h>
#include <cons.h>
#include <setjmp.h>
#include <utils.h>

#define PASTE(x)		#x
#define VERSION(maj,min)	PASTE(maj) "." PASTE(min)

#ifndef elf_check_arch
# define elf_check_arch(e)	1
#endif

struct bootfs *	bfs = 0;		/* filesystem to boot from */
char *		dest_addr = 0;
long		bytes_to_copy = 0;
jmp_buf		jump_buffer;

char		kernel_args[256] = "";
unsigned long	start_addr = START_ADDR;
char *		bss_start = 0;
long		bss_size = 0;		/* size of bss */

static unsigned long	entry_addr = START_ADDR;


#ifdef DEBUG

void wait_key (void)
{
	while (cons_getchar() != 13) /* skip */;
}

void
test_key (void)
{
	int c;
	do {
		c = cons_getchar();
		if (c != 0) printf("%d ",c);
	} while (c != 13);
}

#endif /* DEBUG */


/*
 * The decompression code calls this function after decompressing the
 * first block of the object file.  The first block must contain all
 * the relevant header information.
 */
long
first_block (const char *buf, long blocksize)
{
	struct exec *aout;
	struct elfhdr *elf;
	struct elf_phdr *phdr;
	long text_offset, nbytes;

	aout = (struct exec *) buf;
	elf  = (struct elfhdr *) buf;
	
	if (elf->e_ident[0] == 0x7f
	    && strncmp(elf->e_ident + 1, "ELF", 3) == 0)
	{
		/* looks like an ELF binary: */
		if (elf->e_type != ET_EXEC) {
			printf("aboot: not an executable ELF file\n");
			return -1;
		}
		if (!elf_check_arch(elf->e_machine)) {
			printf("aboot: ELF executable not for this machine\n");
			return -1;
		}
		if (elf->e_phnum != 1) {
			printf("aboot: expected 1, not %d program headers\n",
			       elf->e_phnum);
			return -1;
		}
		if (elf->e_phoff + sizeof(*phdr) > (unsigned) blocksize) {
			printf("aboot: "
			       "ELF program header not in first block (%ld)\n",
			       (long) elf->e_phoff);
			return -1;
		}
		phdr = (struct elf_phdr *) (buf + elf->e_phoff);

		entry_addr	= elf->e_entry;
		start_addr	= phdr->p_vaddr;
		text_offset	= phdr->p_offset;
		bytes_to_copy	= phdr->p_filesz;
		bss_size	= phdr->p_memsz - phdr->p_filesz;
	} else if (N_MAGIC(*aout) == OMAGIC) {
		/* looks like an ECOFF binary: */
		entry_addr	= aout->a_entry;
		start_addr	= N_TXTADDR(*aout);
		text_offset	= N_TXTOFF(*aout);
		bytes_to_copy	= aout->a_text + aout->a_data;
		bss_size	= aout->a_bss;
	} else {
		printf("aboot: unexpected object file format %lo\n",
		       N_MAGIC(*aout));
		return -1;
	}

	if (start_addr < 0xfffffc0000000000UL
	    || entry_addr < start_addr
	    || entry_addr >= start_addr + bytes_to_copy)
	{
		printf("aboot: bad program header "
		       "(start=%lx, entry=%lx, size=%lx)\n",
		       start_addr, entry_addr, bytes_to_copy);
		return -1;
	}
	dest_addr = (char *) start_addr;
	bss_start = dest_addr + bytes_to_copy;

	nbytes = blocksize - text_offset;
	if (nbytes > 0) {
		memcpy(dest_addr, buf + text_offset, nbytes);
		dest_addr += nbytes;
		bytes_to_copy -= nbytes;
	} else if (nbytes < 0) {
		printf("aboot: block-size smaller than anticipated!\n");
		return -1;
	}
	return 0;
}


/*
 * Start the kernel.
 */
static void
run_kernel(void)
{
	__asm__ __volatile__(
		"bis %1,%1,$30\n\t"
		"bis %0,%0,$26\n\t"
		"ret ($26)"
		: /* no outputs: it doesn't even return */
		: "r" (entry_addr),
		  "r" (PAGE_SIZE + INIT_STACK));
}


/*
 * Head transfers control to this function.  Don't call it main() to avoid
 * gcc doing magic initialization things that we don't want.
 */
void
main_ (void)
{
	extern long load_kernel (void);
	long i, result;

	cons_init();

	printf("aboot: Linux/Alpha SRM bootloader version "
	       VERSION(ABOOT_MAJOR, ABOOT_MINOR) "\n");

	if (INIT_HWRPB->pagesize != 8192) {
		printf("aboot: expected 8kB pages, got %ldkB\n",
		       INIT_HWRPB->pagesize >> 10);
		return;
	}

	pal_init();

	result = load_kernel();
	if (result < 0) {
		printf("aboot: load failed (%lx)\n", result);
		return;
	}

	strcpy((char*)ZERO_PGE, kernel_args);
	printf("aboot: ok, now starting the kernel...\n");
	run_kernel();
	printf("aboot: kernel returned unexpectedly.  Halting slowly...\n");
	for (i = 0 ; i < 0x100000000 ; i++)
		/* nothing */;
	halt();
}
