//  UCS2Conv.h version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
// This file defines several UCS2Converter classes
//
#ifndef __ucs2_conv_h__
#define __ucs2_conv_h__

#include <UMap.h>

typedef unsigned char		QUOTE8;
typedef unsigned short		QUOTE16;
typedef unsigned int		QUOTE32;

#define DEFAULT_CONVERTER 	"UTF8"
#define ESC			27

//
// convert from UCS2 to char with encoding standard.
// the return array of this function should be deleted by
// the caller. The programmers are forced to reimplement this when adding new
// converters
// decoding here always measn making unicode.
class UCS2Converter
{
public:
				UCS2Converter ();
	virtual			~UCS2Converter();
	char			*name;

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);

protected:
				UCS2Converter (const char *name);
	void 			append (const UCS2);
	void 			append (const unsigned char in);
	void 			append8 (const QUOTE8 quote);
	void 			append16 (const QUOTE16 quote);
	void 			append32 (const QUOTE32 quote);
	void			append (const unsigned int in);

	// to build output automatically
	void 			init (const UCS2* input);
	void 			init (const unsigned char* input);
	int			inputSize;
	int			bufferSize;
	int			size;
	UCS2*			ucs2;
	unsigned char*		uchar;

};

//
// 8Bit conversion with a conversion map
//
class UUTF7Conv :  public UCS2Converter
{
public:
				UUTF7Conv ();
	virtual			~UUTF7Conv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
protected:
	void			mencode (int start);
	void			mdecode (int start);
};

//
// A converter that decodes mixed UTF8 and Java \uxxxx escapes, and
// encodes \uxxxx escapes.
//
class UJavaConv :  public UCS2Converter
{
public:
				UJavaConv ();
	virtual			~UJavaConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};

//
// JIS conversion 
//
class UJISConv :  public UCS2Converter
{
public:
				UJISConv();
	virtual			~UJISConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
				
	UMap			*mJIS0201; // JIS Roman
	UMap			*mJIS0208;
	UMap			*mJIS0212;
	int			initialized;
	void 			initialize();
protected:
				UJISConv(const char* name);
};

//
// SHIFT-JIS conversion 
//
class USJISConv :  public UCS2Converter
{
public:
				USJISConv();
	virtual			~USJISConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
				
	UMap			*mJIS0201; // JIS Roman
	UMap			*mJIS0208;
	int			initialized;
	void 			initialize();
};


//
// EUC conversion 
//
class UEUC_JPConv :  public UJISConv
{
public:
				UEUC_JPConv ();
	virtual			~UEUC_JPConv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};

//
// COMPUND_TEXT Japanese  conversion 
//
class CTEXT_JAConv :  public UJISConv
{
public:
				CTEXT_JAConv ();
	virtual			~CTEXT_JAConv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};



//
// GB2312 7 BIT cconversion 
//
class UGB2312_7Conv :  public UCS2Converter
{
public:
				UGB2312_7Conv();
	virtual			~UGB2312_7Conv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
				
	UMap			*mGB2312; // GB2312
	int			initialized;
	void 			initialize();
protected:
				UGB2312_7Conv(const char* name);
};

//
// GB2312 8 BIT cconversion 
//
class UGB2312_8Conv :  public UGB2312_7Conv
{
public:
				UGB2312_8Conv ();
	virtual			~UGB2312_8Conv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};

//
// HZ 7 BIT conversion 
//
class UHZConv :  public UGB2312_7Conv
{
public:
				UHZConv ();
	virtual			~UHZConv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};


//
// KSC5601 7 Bit conversion 
// ISO-2022-KR  conversion 
//
class UKSC5601Conv :  public UCS2Converter
{
public:
				UKSC5601Conv();
	virtual			~UKSC5601Conv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
				
	UMap			*mKSC5601; // KSC5601
	int			initialized;
	void 			initialize();
protected:
				UKSC5601Conv(const char* name);
};

//
// EUC Korean conversion 
//
class UEUC_KRConv :  public UKSC5601Conv
{
public:
				UEUC_KRConv ();
	virtual			~UEUC_KRConv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};

class UUHCConv :  public UKSC5601Conv
{
public:
				UUHCConv ();
	virtual			~UUHCConv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
};

//
// JOHAB(Korean) conversion 
//
class UJOHABConv :  public UKSC5601Conv
{
public:
                                UJOHABConv();
        virtual                 ~UJOHABConv ();
        virtual unsigned char*  encode (const UCS2 *in);
        virtual UCS2*           decode (const unsigned char *in);
};

//
// BIG5 Chinese conversion
//
class UBIG5Conv :  public UCS2Converter
{
public:
				UBIG5Conv ();
	virtual			~UBIG5Conv ();
	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);

	UMap			*mBIG5; // BIG5
	int			initialized;
	void 			initialize();
};

//
// 8Bit conversion with a conversion map
//
class U8BitConv :  public UCS2Converter
{
public:
				U8BitConv (const char* mapName);
	virtual			~U8BitConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
				
	UMap			*map; // an 8 bit map 
	int			initialized;
	void 			initialize();
};

//
// 16 bit plain conversion base class
// The input and output is double null terminated!
//
class U16BitConv :  public UCS2Converter
{
public:
	enum Endian	{UCS2BE, UCS2LE};
			U16BitConv (const char* mapName, Endian en);
	virtual		~U16BitConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
protected:
	Endian		endian;
};

//
// 16 bit plain conversion big endian
//
class U16BEConv :  public U16BitConv
{
public:
			U16BEConv (const char* mapName);
	virtual		~U16BEConv ();
};

//
// 16 bit plain conversion little endian
//
class U16LEConv :  public U16BitConv
{
public:
			U16LEConv (const char* mapName);
	virtual		~U16LEConv ();
};

//
// 16 bit plain conversion base class
//
//
class UUniConv :  public UCS2Converter
{
public:
	enum Endian	{UCS2BE, UCS2LE};
			UUniConv (const char* mapName);
	virtual		~UUniConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
protected:
	Endian		endian;
};

//
// 16 bit plain conversion base class
// It converts '\r' , U_PARA_SEP and U_LINE_SEP 
//
class UMSTXTConv :  public UCS2Converter
{
public:
	enum Endian	{UCS2BE, UCS2LE};
			UMSTXTConv (const char* mapName);
	virtual		~UMSTXTConv ();

	virtual unsigned char*	encode (const UCS2 *in);
	virtual UCS2*		decode (const unsigned char *in);
protected:
	Endian		endian;
};

#endif /* __ucs2_conv_h__ */
