/*
    SABRE Fighter Plane Simulator 
    Copyright (c) 1997 Dan Hammer
    Portions Donated By Antti Barck

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*************************************************
 *           SABRE Fighter Plane Simulator              *
 * Version: 0.1                                  *
 * File   : game.C                               *
 * Date   : March, 1997                          *
 * Author : Dan Hammer                           *
 *************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <float.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <signal.h>
#include <ctype.h>
#include <iostream.h>
#include <fstream.h>
#include <setjmp.h>
#include <stdarg.h>
#include "sim.h"
#include "siminc.h"
#include "game.h"

extern void buffer2ppm();
float time_expired;

GameSession::GameSession(char *world_file,
	      char *flight_file,
	      char *ground_unit_file,
	      Mouse *mouse, 
	      Joystick *yoke_jstk, 
	      Joystick *rudder_jstk, 
	      Joystick *throttle_jstk, 
	      int mouse_throttle, 
	      int mouse_rudder,
	      char *instr_path,
	      char *hud_path,
	      int demo_mode,
	      int (*messageHook)(void))
  : 
    clist(&info_manager),
    hud(g_font,hud_path),
    cpk(NULL),
    fi(mouse,yoke_jstk,rudder_jstk,throttle_jstk,
       mouse_throttle,mouse_rudder)
{
  this->demo_mode = demo_mode;
  pe = NULL;
  this->world_file = world_file;
  this->flight_file = flight_file;
  this->ground_unit_file = ground_unit_file;
  this->instr_path = instr_path;
  this->messageHook = messageHook;
  the_earth = &earth;
  time_expired = 0.0;
}

GameSession::~GameSession()
{
  if (pe != NULL)
    delete pe;
  if (cpk != NULL)
    delete cpk;
}

void GameSession::doGame()
{
  setup();
  if (!demo_mode)
    play();
  else
    demo();
}

void GameSession::readWorldFile()
{
  ifstream is;
  char c;
  Port_3D port;
  float pixl_ratio;
  char buff[100];
  if (open_is(is,world_file))
    {
      // Get path to palette file
      get_line(is,buff,sizeof(buff));
      printf("reading palette file %s\n",
	     buff);
      read_palette_file(buff);
      // Get path to texture file
      get_line(is,buff,sizeof(buff));
      printf("reading texture file %s\n",
	     buff);
      read_texture_file(buff);
      READ_TOKI('{',is,c)
	is >> world_scale >> time_factor >> hit_scaler >> 
	max_time >> shadow_level;
      is >> Port_3D::fovx >> Port_3D::fovy;
      READ_TOK('}',is,c)
	shadow_level *= world_scale;
	is >> world_light_source;
	world_light_source *= world_scale;
	is >> port;
	pixl_ratio = SCREEN_WIDTH / 320.0;
	Port_3D::fovx *= pixl_ratio;
	Port_3D::fovy *= pixl_ratio;
	is >> earth;
	is >> info_manager;
	is >> clist;
    }
  else
    {
      perror("reading world file");
      exit(1);
    }
  cpk = new Cockpit("f9fcpk",instr_path);
}

void GameSession::setup()
{
  pflag = 0;
  zz = 0;
  Weapons_Manager::the_weapons_manager = &fm.wm;

  printf("reading world file %s\n",world_file);
  readWorldFile();
  printf("reading flight info %s\n",flight_file);
  fm.read_file(flight_file);
  if (ground_unit_file != NULL)
    gm.read_file(ground_unit_file);
  clist.get_targets(target_list);
  for (int i=0;i<fm.n_flights;i++)
    target_list.add_target(fm.get_viewer(i));
  gm.addTargetList(target_list);
  pe = new Palette_Effect();

  Unguided_Manager::initUnguided();
  Pilot::initPilot();

  printf("%d textures\n",
	 map_man->n_maps);
  printf("%d terrain shapes\n",
	 earth.ntshapes);
  printf("%d static shape definitions\n",
	 info_manager.ninfos);
  printf("%d static shape objects\n",
	 clist.n_objects);
  printf("%d dynamic shape definitions\n",
	 fm.n_managers);
  printf("%d flights\n",
	 fm.n_flights);
  printf("%d ground units\n",
	gm.nunits);
}

void GameSession::play()
{
  Port_3D port;
  clear_scr(0);
  get_time();
  fm.start();
  do
    {
      if (messageHook && !messageHook())
	break;
      get_time();
      if (dump_screen)
	{
	  buffer2ppm();
	  dump_screen = 0;
	}
      if (gpause == 0)
	{
	  pflag = 0;
	  update_flight();
	  draw_flight();
	}
      else
	{
	  if (!pflag)
	    {
	      pflag = 1;
	      fm.set_flight_view(port);
	    }
	  if (gpause == 1)
	    draw_flight(&port);
	  if (!port_key(port))
	    break;
	}
    } while (flight_key(fm.get_view_node(),fm,fi,NULL));
  clear_scr(0);
}

inline int GETSTATE(Pilot *pil, char *state, int n)
{
  return(!memcmp(pil->dbg,state,n));
}

inline int INTERESTING(Flight_Node &nde)
{
  int result = 0;
  if (nde.viewer->hitme)
    result = 4;
  else if (GETSTATE(nde.pilot,"PRST0A",6))
    result = 3;
  else if (GETSTATE(nde.pilot,"PRST0B",6))
    result = 4;
  else if (GETSTATE(nde.pilot,"PRST1",5))
    result = 2;
  else if (GETSTATE(nde.pilot,"TRGDWLL",6))
    result = 4;
  else if (GETSTATE(nde.pilot,"LOCKON",6))
    result = 5;
  if (nde.flight->state.crashed)
    result = 0;
  return (result);
}

inline int DECIDEVIEW(int , int )
{
  int dec = RANDOM(3);
  switch (dec)
    {
    case 0:
      return(outside_2);
      break;
      
    case 1:
      return(outside_4);
      break;
      
    case 2:
      return(front);
      break;

    default:
      return(outside_2);
      break;
    }
}

void GameSession::demo()
{
  Port_3D port;
  Flight *fc;
  float view_time = 0;
  float maxv_time;
  int flg;
  int maxX,ix,ii,x;

  clear_scr(0);

  for (int i=0;i<fm.n_flights;i++)
    {
      fc = fm.get_flight(i);
      fc->controls.autopilot = 1;
    }

  view_time = 0;
  maxv_time = ((float) RANDOM(6)) + 3.0;
  get_time();
  fm.start();
  do
    {
      if (messageHook && !messageHook())
	break;
      if (gpause)
	{
	  if (!port_key(port))
	    break;
	  continue;
	}
      flg = 0;
      get_time();
      view_time += raw_time;
      Flight_Node &fn = fm.get_view_node();
      if (fn.flight->controls.autopilot)
	{
	  maxX = INTERESTING(fn);
	  ix = maxX;
	  ii = -1;
	  if (view_time >= 2.0)
	    {
	      for (int i=0;i<fm.n_flights;i++)
		{
		  if ((x = INTERESTING(*fm.get_node(i))))
		    {
		      if (x > maxX)
			{
			  ii = i;
			  maxX = x;
			}
		    }
		}
	    }

	  if (ii >= 0)
	    {
	      flg = 1;
	      view_time = 0.0;
	      fm.set_view_node(ii);
	      fc = fm.get_flight(ii);
	      fc->controls.view = (flight_view) DECIDEVIEW(maxX,ii);
	    }
      
	  if (view_time >= maxv_time && !flg && ix == 0)
	    {
	      view_time = 0.0;
	      maxv_time = ((float) RANDOM(6)) + 3.0;
	      int found_flight = 0;
	      for (int i=0;i<fm.n_flights;i++)
		{
		  fm.set_view_node(fm.view_node + 1);
		  fc = fm.get_flight(fm.view_node);
		  if (!fc->state.crashed)
		    {
		      found_flight = 1;
		      break;
		    }
		}
	      if (!found_flight)
		break;
	      else
		{
		  fc = fm.get_flight(fm.view_node);
		  fc->controls.view = (flight_view)
		    DECIDEVIEW(INTERESTING(*fm.get_node(fm.view_node)),fm.view_node);
		}
	    }
	}
      update_flight();
      draw_flight();

      if (dump_screen)
	{
	  buffer2ppm();
	  dump_screen = 0;
	}
    } 
  while (flight_key(fm.get_view_node(),fm,fi,NULL));
  clear_scr(0);
}

void GameSession::get_time()
{
  /* 
     This is silly, anyway. Just 
     set to a minimum value
     while (!timer.check_time());
     */
  time_frame = timer.get_time();
  if (time_frame < 0.01)
    time_frame = 0.01;
  raw_time = time_frame;
  if (time_frame > max_time)
    time_frame = max_time;
  time_expired += time_frame;
}

void GameSession::update_flight()
{
  Unguided_Manager::setUnguidedManager(&um);
  Unguided_Manager::setTargetList(target_list.targets,
				  target_list.idx);
  fm.update(&um);
  gm.update(&um,target_list);
  doPaletteEffect();
  mouse.Update();
  if (joystick0)
    joystick0->Update();
  if (joystick1)
    joystick1->Update();
  if (select_next)
    {
      fm.select_next_target(fm.view_node);
      select_next = 0;
    }
}

void GameSession::draw_flight(Port_3D *pport)
{
  Port_3D vport;
  DrawList dlist;
  DrawList dlist2;
  Pilot    *pilot;
  
  Flight &fc = fm.get_view_flight();
  Flight_ZViewer &vwr = fm.get_view_viewer();
  
  int pflag;

  zz++;
  if (zz == 30)
    {
      clear_zbuff();
      zz = 0;
    }
  else
    ztrans += (3L << Z_PREC);

  if (pport == NULL)
    {
      pflag = 0;
      /* If this plane has crashed, force outside view */
      if (fc.state.crashed)
	{
	  fc.controls.view = outside_1;
	  fc.controls.vdist = 400;
	  fc.controls.vphi = 0.57;
	}
      else if (!demo_mode)
	{
	  if (vwr.hitme && vwr.isHistory())
	    fc.controls.view = outside_4;
	}
      fm.set_flight_view(vport);
    }
  else
    {
      pflag = 1;
      vport = *pport;
    }

  earth.drawBackDrop(vport);
  /* 
     Set on/off texture mapping for objects of type "SCLOUD",
     to match that of the terrain
     */ 
  clist.setVisibleFlag(display_flags & CLOUDS_ON,"SCLOUD",6);
  clist.add_draw_list(dlist,vport);
  dlist.drawz(vport);
  fm.add_draw_list(dlist2,vport);
  um.add_draw_list(dlist2,vport);
  gm.add_draw_list(dlist2,vport);
  dlist2.drawz(vport);

  // Try drawing the airplane shape so that it
  // surrounds the pilot
  if (!pflag && fc.controls.view <= back && fc.controls.vextern)
    {
      /*
       * The problem is that with the z-buffer, we have to clip at z < 1.0.
       * We need to keep the world_scale around 0.03, to avoid wobbly texture mappings
       * on the larger shapes. So attempting to show the surrounding exterior of the
       * plane would fail, as it would be clipped out.
       * The solution is to scale up the plane and then move the view point. This
       * is accomplished by the code below. 
       */
      REAL_TYPE scl;
      R_3DPoint spc;
      R_3DPoint spcw;
      Vector    v;
      R2D_TYPE  zt;

      Port_3D pp = vport;
      // Turn off drawing the cockpit shape & shadow
      vwr.cpk_flg = 1;
      vwr.draw_shadow = 0;

      /*
      zbff_flag = 0;
      vwr.draw_prep(pp);
      vwr.draw(pp);
      zbff_flag = 1;
      */

      // Scale up the plane to world_scale = 1
      // by inversing the current world_scale
      scl = 1.0 / world_scale;
      set_co_scaler(scl);
      // Get the original view point in port coords
      spc = vwr.flt->specs->view_point;
      // Times z_manager's scaler gives us world_scale = 1 in port coords
      spc *= vwr.z_manager->scaler;
      // Convert it to world coords
      vwr.flt->state.flight_port.port2world(spc,&spcw);
      // Find the delta vector between it and the origin of the airplane shape
      v = Vector(spcw - vwr.flt->state.flight_port.look_from);
      // Move our view point that distance
      pp.delta_look_from(v);
      // Get the current zbias
      zt = zbias;
      // bias the zbuffer
      zbias = R2D_TYPE(2L << Z_PREC);
      vwr.draw_prep(pp);
      // Undo any distance-abstraction
      vwr.da_level = Z_Viewer::da_full;
      vwr.draw(pp);
      // Restore zbuffer bias
      zbias = zt;
      // Restore scale
      set_co_scaler(1.0);
      vwr.cpk_flg = 0;
      vwr.draw_shadow = 1;
    }

  earth.terrain_on = display_flags & TERRAIN_ON;
  earth.render_ground(vport);
  if (pflag)
    show_port_vars(vport,vport,g_font);
  else
    {
      pilot = (fm.get_view_node()).pilot;

      if (fc.controls.autopilot && !pilot->message.flag)
	show_message(0,"%s:%s",pilot->handle,fc.specs->model);
      if (pilot->message.flag)
	{
	  if (pilot->message.frm != NULL)
	    show_message(0,"%s:%s",pilot->message.frm->handle,
			 pilot->message.mssg);
	  else
	    show_message(0,pilot->message.mssg);
	}
      if (fc.controls.view == front)
	{
	  if (fm.view_node == 0 && fc.controls.cockpit && cpk != NULL)
	    cpk->draw((Flight &)fc);
	  if (fc.controls.hud_on)
	    hud.do_hud(fm.get_view_node());
	  if (fc.controls.armed_w)
	    hud.show_gunsight(fm.get_view_node(),vport);
	  else if (fc.controls.vect_on)
	    hud.show_vector(fc,vport);
	  if (pilot->waypoint && pilot->waypoint->mode == mnothing)
	    hud.show_waypoint(pilot,vport);
	}
      /* combat track view */
      if (fc.controls.view == outside_2 && pilot->target != NULL)
	{
	  int x,y;
	  R_3DPoint w0 = *pilot->target->get_position();
	  vport.transform(w0,&x,&y);
	  if (!vport.over_flow)
	    g_font->put_char(133,x-4,y-4,hud.hud_color,8);
	}
    }
  blit_buff();
  vwr.hitme = 0;
}

void GameSession::doPaletteEffect()
{
  if (pe == NULL)
    return;

  if (pe->effect_on)
    pe->is_done(time_frame,1);
  else
    {
      Flight_ZViewer &zv = fm.get_view_viewer();
      if (zv.hitme)
	{
	  pe->do_effect(red_effect,0.1);
	}
      else
	{
	  Flight &f = fm.get_view_flight();
	  if (f.state.crashed)
	    {
	      pe->do_effect(red_effect,2.0);
	    }
	}
    }
}

inline void PRINTLN(ostream &os, char *s, ...)
{
  char buf[512];
  va_list ap;
  va_start(ap,s);
  vsprintf(buf,s,ap);
  va_end(ap);
  os << buf;
}


void GameSession::printResults(ostream &os)
{
  int i;
  PRINTLN(os, "\nMission Debrief\n");
  PRINTLN(os, "Flight File: %s\n",flight_file);
  for (i=0;i<fm.n_flights;i++)
    {
      PRINTLN(os,"----------------------------------\n");
      PRINTLN(os,"Flight: %d\n",i);
      printResult(*fm.get_node(i),os);
    }
}


void GameSession::printResult(Flight_Node &fn, ostream &os)
{

  Flight *flight = fn.flight;
  Flight_ZViewer *viewer = fn.viewer;

  PRINTLN(os,"Aircraft type: %s, pilot: %s\n",
	  flight->specs->model,
	  fn.handle);
  if (viewer->bagged)
    {
      PRINTLN(os,"Aircraft destroyed by enemy fire\n");
    }
  else if (flight->state.on_ground && fn.lr.landing_attempted)
    {
      /* Rate the landing, and if bad, give reason why */
      if (fn.lr.score >= 0)
	PRINTLN(os,"Safe Landing\n");
      else if (fn.lr.score < 0)
	{
	  if (flight->state.crashed)
	    PRINTLN(os,"Crash Landing\n");
	  else
	    PRINTLN(os,"Bad Landing\n");
	  if (fn.user_node)
	    {
	      if (fn.lr.gear_up)
		PRINTLN(os," * Landing Gear Not Engaged\n");
	      if (fn.lr.l_z)
		{
		  PRINTLN(os," * Vertical Speed: %3.2f ft/s\n",
			  fn.lr.l_z_value);
		  PRINTLN(os,"          Maximum: %3.2f ft/s\n",
			  -flight->specs->l_z);
		}
	      if (fn.lr.v)
		{
		  PRINTLN(os," * Air Speed: %7.2f kts\n",
			  fn.lr.v_value * 3600.0 / 6000.0);
		  PRINTLN(os,"     Maximum: %7.2f kts\n", 
			  flight->specs->lspeed * 3600.0 / 6000.0);
		}
	      if (fn.lr.phi)
		{
		  PRINTLN(os," * Pitch : %3.2f degrees\n",
			  (_PI2 - fn.lr.phi_value) / _PI * 180);
		}
	      if (fn.lr.roll)
		{
		  PRINTLN(os," * Roll : %3.2f degrees\n",
			  (fn.lr.roll_value) / _PI * 180);
		}
	    }
	}
    }
  else
    PRINTLN(os,"Flight Interrupted\n");

  if (viewer->hurt > 0)
    {
      PRINTLN(os,"Damage to Aircraft\n");
      PRINTLN(os,"Total Damage: %d of Maximum %d\n",
	      viewer->hurt,viewer->max_damage);
    }

  if (viewer->ground_kills > 0)
    {
      PRINTLN(os,"%d ground targets destroyed\n",viewer->ground_kills);
    }

  if (viewer->air_kills > 0)
    {
      PRINTLN(os,"%d kill(s) recorded\n",viewer->air_kills);
    }

  if (viewer->shots > 0)
    PRINTLN(os,"rounds fired: %d  hits: %d  ratio: %3.0f%%\n",
	    viewer->shots,viewer->o_hits,
	    100.0 * (float) ((float)viewer->o_hits) / ((float)viewer->shots));
}

void GameSession::show_message(int bot, char *str, ...)
{
  char buf[512];
  va_list ap;

  va_start(ap,str);
  vsprintf(buf,str,ap);
  va_end(ap);

  int poly[10];
  int l = strlen(buf) * 6;
  int y = bot ?  SCREEN_HEIGHT - 8 : 0;
  int yd = bot ? SCREEN_HEIGHT - 1 : 7;
  if (l > 0)
    {
      int x = (SCREEN_WIDTH / 2) - l / 2;
      poly[0] = x - 1;
      poly[1] = y;
      poly[2] = x + l + 1;
      poly[3] = y;
      poly[4] = x + l + 1;
      poly[5] = yd;
      poly[6] = x - 1;
      poly[7] = yd;
      rendply(poly,4,1,&cliprect);
      g_font->font_sprintf(x,y,11,NORMAL,buf);
    }
}








