/*
    SABRE Fighter Plane Simulator 
    Copyright (c) 1997 Dan Hammer
    Portions Donated By Antti Barck

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*************************************************
 *           SABRE Fighter Plane Simulator       *
 * File   : pilot.C                              *
 * Date   : March, 1997                          *
 * Author : Dan Hammer                           *
 * Artificial dumbs                              *
 *************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <values.h>
#include <stdarg.h>
#include "defs.h"
#include "sim.h"
#include "simfile.h"
#include "vmath.h"
#include "port_3d.h"
#include "cpoly.h"
#include "copoly.h"
#include "flight.h"
#include "fltlite.h"
#include "weapons.h"
#include "unguided.h"
#include "pilot.h"
#include "rtkey.h"
#include "sim.h"

#define FLTSTATE flight->state
#define FLTSPCS  flight->specs
#define FLTPORT  FLTSTATE.flight_port
#define FLTPNT   FLTPORT.look_from
#define FLTSPNT  FLTPORT.slook_from
#define TRGPORT  target_flight->state.flight_port
#define TRGPNT   target_flight->state.flight_port.look_from
#define ROLL     FLTPORT.roll
#define RESET_AL_STATE { al_state.stage = 0; \
			       al_state.done = 0; }
#define FLTCTRLS flight->controls
#define AILERONS FLTCTRLS.ailerons
#define AIL_MIN FLTCTRLS.aileron_min
#define AIL_MAX FLTCTRLS.aileron_max 
#define ELEVATORS FLTCTRLS.elevators
#define EL_MIN FLTCTRLS.elevator_min
#define EL_MAX FLTCTRLS.elevator_max
#define RUDDER FLTCTRLS.rudder
#define RU_MIN FLTCTRLS.rudder_min
#define RU_MAX FLTCTRLS.rudder_max
#define FLAPS  FLTCTRLS.flaps
#define GEAR   FLTCTRLS.landing_gear
#define WBRAKES FLTCTRLS.wheel_brakes
#define SBRAKES FLTCTRLS.speed_brakes
#define THROTTLE FLTCTRLS.throttle
#define TRIM FLTCTRLS.trim
#define BANG_BANG { FLTCTRLS.bang_bang = 1; }

#define RIGHT_AILERONS { if (AILERONS < 0) \
  AILERONS = 0;  \
  else if (AILERONS < AIL_MAX) {\
  AILERONS += (control_step * t); if (AILERONS > AIL_MAX) AILERONS=AIL_MAX; } \
  if (FLTSPCS->adv_yaw != 0.0)\
  coord(); \
}

#define LEFT_AILERONS { if (AILERONS > 0) \
  AILERONS = 0;  \
  else if (AILERONS > AIL_MIN) {\
  AILERONS -= (control_step * t); if (AILERONS < AIL_MIN) AILERONS=AIL_MIN; }\
  if (FLTSPCS->adv_yaw != 0.0) \
     coord();\
}

#define UP_ELEVATORS  { if (ELEVATORS < 0) \
 ELEVATORS = 0;  \
 else if (ELEVATORS < EL_MAX){ \
 ELEVATORS += (control_step * t);\
 if (ELEVATORS > EL_MAX) \
 ELEVATORS = EL_MAX;} \
  }

#define DOWN_ELEVATORS     { if (ELEVATORS > 0) \
  ELEVATORS = 0;  \
  else if (ELEVATORS > EL_MIN){ \
 ELEVATORS -= (control_step * t);\
 if (ELEVATORS < EL_MIN) \
 ELEVATORS = EL_MIN;} \
}

#define R_AILERONS  { if (AILERONS < 30){ \
  AILERONS += (control_step * t); if (AILERONS > AIL_MAX) AILERONS=AIL_MAX;}\
  if (FLTSPCS->adv_yaw != 0.0)\
  coord();\
}

#define L_AILERONS  { if (AILERONS > -30) {\
  AILERONS -= (control_step * t); if (AILERONS < AIL_MIN) AILERONS=AIL_MIN;}\
  if (FLTSPCS->adv_yaw != 0.0)\
  coord();\
 }

#define U_ELEVATORS { if (ELEVATORS < 30) \
  ELEVATORS += (control_step * t);\
  if (ELEVATORS > EL_MAX) \
 ELEVATORS = EL_MAX; }

#define D_ELEVATORS { if (ELEVATORS > -30){ \
  ELEVATORS -= (control_step * t);\
  if (ELEVATORS < EL_MIN) \
  ELEVATORS = EL_MIN;}}

#define L_RUDDER  { if (RUDDER < RU_MAX) {\
  RUDDER  += (control_step * t); \
  if (RUDDER > RU_MAX) RUDDER = RU_MAX;} \
}

#define R_RUDDER  { if (RUDDER > RU_MIN) \
  RUDDER -= (control_step * t); \
  if (RUDDER < RU_MIN) RUDDER = RU_MIN; \
}

#define LEFT_RUDDER { if (RUDDER < 0) \
 RUDDER = 0; \
 else if (RUDDER < RU_MAX) {\
 RUDDER  += (control_step * t); if (RUDDER > RU_MAX) RUDDER = RU_MAX;}\
}

#define RIGHT_RUDDER { if (RUDDER > 0) \
 RUDDER = 0; \
 else if (RUDDER > RU_MIN){ \
 RUDDER -= (control_step * t); if (RUDDER < RU_MIN) RUDDER = RU_MIN;} \
}

#define SET_RUDDER(i) { if ((i) < RU_MIN) (i) = RU_MIN;\
  else if ((i) > RU_MAX) (i) = RU_MAX; \
  RUDDER = (i); \
 if (FLTSPCS->adv_roll != 0.0) coord(); \
 }


/*****************************************
 Note limit of 64 pilots                 *
 *****************************************/   
#define MAXPILOTS 64
Pilot *Pilot::pilots[MAXPILOTS];
int Pilot::npilots = 0;
int Pilot::maxpilots = MAXPILOTS;
float Pilot::xpixel_adjust = 1.0;
float Pilot::ypixel_adjust = 1.0;
float Pilot::bank_30 = 0.0;
float Pilot::bank_45 = 0.0;
float Pilot::rcontrol_max = 32.0;
float Pilot::gcontrol_max = 16.0;
float Pilot::g_range = 15.0;
float PITCH_MIN = 0.0;

#define ANGELS(x) (((x) / world_scale) / 1000.0)

inline REAL_TYPE angels(Port_3D &port)
{
  return (ANGELS(port.look_from.z));
} 

inline REAL_TYPE angels(Flight *flt)
{
  return (angels(flt->state.flight_port));
}

inline float get_diff(float port_angle, float goal_angle)
{
  float result;
  result = fabs(port_angle - goal_angle);
  if (result > _PI)
    result = fmod(_2PI,result);
  return result;
}

//   get which direction to turn to reach a given angle
//   return 0 if angle achieved within min
inline int get_dir(float port_angle, float goal_angle, float min)
{
  int result;
  float d1 = port_angle - goal_angle;
  if (fabs(d1) <= min)
    {
      result = 0;
    }
  else
    {
      float d2;
      if (d1 > 0)
	d2 = goal_angle + (_2PI - port_angle);
      else
	d2 = port_angle + (_2PI - goal_angle);
      if (fabs(d1) > fabs(d2))
	{
	  if (d1 < 0)
	    result = -1;
	  else
	    result = 1;
	}
      else
	{
	  if (d1 > 0)
	    result = -1;
	  else
	    result = 1;
	}
    }
  return result;
}

inline void set_cntrl(Control_State &cs, float gl, float rte, float mn)
{
  cs.goal = gl;
  cs.rate = rte;
  cs.min = mn;
}

istream &operator >>(istream &is, Pilot_Params &ps)
{
  char c= ' ';
  READ_TOK('(',is,c);
  is >> ps.skill_level >> ps.affiliation >> ps.update_time;
  is >> ps.mparam1 >> ps.mparam2 >> ps.mparam3;
  is >> ps.mparam4 >> ps.mparam5 >> ps.mparam6;
  is >> ps.mdampy >> ps.mdampx;
  is >> ps.mparam7 >> ps.mparam8 >> ps.mparam9 >> ps.mparam10;
  is >> ps.c_step1 >> ps.c_step2 >> ps.c_step3 >> ps.c_step4;
  is >> ps.v_acquire_range;
  is >> c;
  READ_TOK(')',is,c);
  ps.v_acquire_range = m2f(ps.v_acquire_range);
  return is;
}


/**********************************************************
 * initialize static members for Pilot class              *
 **********************************************************/
int Pilot::initPilot()
{
  /* all the pilots have 320x200 vga screen in their heads */
  xpixel_adjust = ((float) SCREEN_WIDTH) / 320.0;
  ypixel_adjust = ((float) SCREEN_HEIGHT) / 200.0;
  bank_30 = _DEGREE * 30.0;
  bank_45 = _DEGREE * 45.0;
  rcontrol_max = 82.0;
  gcontrol_max = 12.0;
  g_range      = 15.0;
  PITCH_MIN = _DEGREE * 10.0;
  return (0);
}

Pilot *Pilot::getPilot(char *hndl)
{
  Pilot *result = NULL;
  for (int i=0;i<npilots;i++)
    {
      if (!strcmp(hndl,pilots[i]->handle))
	{
	  result = pilots[i];
	  break;
	}
    }
  return (result);
}

Pilot::Pilot(Flight *flt, Pilot_Params *prms,
	     Weapon_Instance *wi, int nw,
	     char *hndl, Target *target_obj)
  : params(prms),
    flight(flt),
    weapons(wi),
    n_weaps(nw),
    handle(hndl),
    functioning(1),
    max_gs(9.0),
    target(NULL),
    target_flight(NULL)
{
  this->target_obj = target_obj;
  set_cntrl(pitch_control,_PI2,1.0,0.1);
  set_cntrl(speed_control,flt->state.z_vel,1.0,0.1);
  set_cntrl(bank_control,0.0,1.0,0.1);

  timer_max = timer_elapsed = 0.0;
  target = NULL;
  target_flight = NULL;
  target_pilot = NULL;
  flag1 = -1;
  t = 1.0;
  control_step = 30;
  gp_scx = gp_scy = 0.0;
  sel_wpn = 0;
  sel_weapon = &weapons[0];
  time_to_target = 0.0;
  gun_time = 1.0;
  in_range = 0;
  engaged = 0;
  dbg = "INIT";
  new_damage = 0;
  message.time_limit = 3.0;
  checkout();
  waypoint = NULL;
  attkr_cnt = 0;
  tdwell_time = 0;
  tdwell_max = 4.0;
  tdwell_flg = 0;
  ground_flag = 0;
  gh_height = 0.0;
}

void Pilot::start()
{
  // center the flight's controls
  AILERONS = 0;
  ELEVATORS = 0;
  RUDDER = 0;
  TRIM = 0;
  SBRAKES = 0;
  RESET_AL_STATE
}

void Pilot::set_waypoint(WayPoint *wp)
{
  waypoint = wp;
  flag1 = 0;
  timer_max = 0.0;
  m_state.done = 0;
  m_state.maneuver = 0;
  engaged = 0;
  if (waypoint != NULL)
    {
      if (waypoint->mode == mmaneuver)
	set_maneuver(waypoint->mode_x,waypoint->mode_y);
      speed_control.goal = waypoint->average_speed;
    }
  RESET_AL_STATE
}


void Pilot::checkout()
{
  has_fuel_tanks = 0;
  for (int i=0;i<n_weaps;i++)
    {
      if (weapons[i].getType() == fueltank_t)
	{
	  has_fuel_tanks = 2;
	  ft_idx = i;
	  break;
	}
    }
}

#define HARD_BANK 1.49
static float e_elapsed = 0.0;

void Pilot::update(int flg)
{
  R_KEY_BEGIN(400) // Pilot::update
    // Get elapsed time in microseconds
    t = time_frame;
 
  // flg == 0 means we're not on
  // "autopilot" so just update
  // weapons, navigation & communications info
  if (!flg)
    {
      track_target();
      get_waypoint_position();
      message.update();
      R_KEY_END
	return;
    }

  timer_elapsed += t;
  e_elapsed += t;
  track_target();
  message.update();
  if (!functioning)
    {
      R_KEY_END
	return;
    }
  damage_check();
  if (!functioning)
    {
      R_KEY_END
	return;
    }
  control_step = 1.0 / t;
  if (timer_elapsed < params->update_time)
    return;
  else
    timer_elapsed = 0.0;
  if (waypoint == NULL || waypoint->mode > mlanding)
    {
      if (stall_check())
	{
	  dbg = "STALL";
	  R_KEY_END
	  return;
	}

      if (ground_check())
	{
	  dbg = "GROUND";
	  reset_maneuver();
	  R_KEY_END
	  return;
	}
    }

  int mode;
  if (waypoint == NULL)
    mode = masis;
  else
    mode = waypoint->mode;

  // Check six!
  if (mode != mdrone && mode != mtakeoff)
    {
      Pilot *tpilot = threat_check();
      if (tpilot && tpilot != target_pilot)
	{
	  set_target(tpilot);
	  brdcst(5,params->affiliation,
		 "Engaging %s at angels %2.0f",
		 target_flight->specs->model,
		 angels(target_flight));
	  engaged = 1;
	}
      if (tpilot)
	{
	  jink();
	  return;
	}
    }


  switch (mode)
    {
    case mmtakeoff:
      do_takeoff();
      break;

    case mjink:
      jink();
      break;

    case mpursuit:
      do_pursuit();
      break;

    case mnothing:
    case mdrone:
      do_waypoint();
      break;

    case mmaneuver:
      do_maneuver();
      if (m_state.done)
	set_maneuver(waypoint->mode_x,waypoint->mode_y);
      break;
      
    case mcap:
      do_cap();
      break;

    case masis:
    default:
      dbg = "DEFLT";
      set_cntrl(bank_control,0.0,3.0,0.1);
      set_cntrl(pitch_control,_PI2,3.0,0.1);
      set_cntrl(speed_control,flight->specs->corner_speed,10,1.0);
      bank_update();
      pitch_update();
      speed_update();
      break;
    }
  R_KEY_END
    }

void Pilot::track_target()
{
  float gtime;
  gun_time = 1.0;
  if (target != NULL)
    {
      get_target_position();
      if (
	  g_trk.p_position.z > 0 &&
	  g_trk.distance >= WPSPECS(sel_weapon)->min_range &&
	  g_trk.distance <= WPSPECS(sel_weapon)->max_range
	  )
	{
	  in_range = 1;
	  gtime = time_to_target;
	}
      else
	{
	  in_range = 0;
	  gtime = gun_time;
	}
    }
  else
    {
      in_range = 0;
      gtime = gun_time;
    }
  gun_point = sel_weapon->predict_path(*flight,gtime);
  world_2_port(gun_point,&gun_point_p,&gp_scx,&gp_scy,FLTPORT);
  sel_weapon->target_position = g_trk.w_position;
}

/**************************************************************
 * Take a good hard look at how much punishment we've taken,  *
 * and get out if it's just too much to take!                 *
 **************************************************************/
int Pilot::damage_check()
{
  if (flight->mods.battle_damage > 0)
    {
      if (flight->mods.battle_damage >=
	  flight->specs->max_damage)
	{
	  if (functioning)
	    {
	      broadcast("Ejecting!",6,params->affiliation,NULL);
	      dbg = "EJECT";
	      functioning = 0;
	      set_target(NULL);
	    }
	  return (1);
	}
      else
	{
	  if (flight->mods.battle_damage > new_damage)
	    {
	      new_damage = flight->mods.battle_damage;
	      float d1 = ((float)new_damage) / ((float)flight->specs->max_damage);
	      if (target_obj->who_got_me != NULL &&
		  target_obj->who_got_me->getType() == GROUND_UNIT_T)
		brdcst(5,params->affiliation,"Taking flak: %3.0f%%%%",d1 * 100.0);
	      else
		{
		  if (d1 >= 0.75)
		    brdcst(5,params->affiliation,"Heavy damage: %3.0f%%%%!",d1 * 100.0);
		  else
		    brdcst(5,params->affiliation,"Taking fire %3.0f%%%%",d1 * 100.0);
		}
	    }
	}
    }
  return (0);
}

/**********************************************************
 * See if we're about to stall, and try and prevent it if *
 * if we are.                                             *
 **********************************************************/
int Pilot::stall_check()
{
  int result = 0;
  if (flight->state.stalled ||
      flight->state.near_stall <= 0.003)
    {
      result = 1;
      set_cntrl(aoa_control,0.0,1.0,0.1);
      aoa_update();
      set_cntrl(speed_control,flight->specs->max_speed,100,1.0);
      speed_update();
    }
  return result;
}

void Pilot::set_target(Pilot *pl)
{
  if (target_pilot != NULL && target_pilot != pl
      && target_pilot->params->affiliation != params->affiliation)
    {
      if (target_pilot->attkr_cnt > 0)
	target_pilot->attkr_cnt--;
    }
  target_pilot = pl;
  if (pl != NULL)
    {
      target_flight = pl->flight;
      target = pl->target_obj;
      if (pl->params->affiliation != params->affiliation)
	{
	  engaged = 1;
	  pl->attkr_cnt++;
	}
    }
}

#define MAXT 32;
/****************************************************
 * search the friendly skies for non-friendlies     *
 * flg == 0 : pick target with least # of attackers *
 * flg == 1 : only if targeting us                  *
 * flg == 2 : only if targeting us or 0 attackers   * 
 ****************************************************/
int Pilot::acquire_target(int flg)
{
  int result = 0;
  Pilot *targets[64];
  int n = 0;
  int mn;
  Pilot *pl;

  
  for (int i=0;i<npilots;i++)
    {
      Pilot *pl = pilots[i];
      /* potential target? */
      if ((pl != this ) && (pl->params->affiliation != params->affiliation)
	  && (pl != target_pilot))
	{
	  if (!(pl->target_obj->isHistory()))
	    {
	      Target_Geometry tg;
	      get_target_geometry(tg,*pl->flight);
	      if (tg.range <= params->v_acquire_range)
		{
		  /* he's in range -- now, if he happens to be targeting us.. */
		  if (flg && pl->target_pilot == this &&
		      target_pilot != NULL && 
		      target_pilot->target_pilot != this)
		    {
		      /* drop our current target, if we have one, since */
                      /* this guy is more of a threat                  */
		      set_target(pl);
		      return (1);
		    }
		  else if (flg == 2)
		    {
		      /* check if < 2 attackers on this guy .. if so, pick him */
		      if (pl->attkr_cnt < 2)
			{
			  set_target(pl);
			  return(1);
			}
		    }
		  else
		    /* otherwise, add him to potential target list */
		    targets[n++] = pl;
		}
	    }
	}
    }

  if (flg != 0)
    /* No immediate threats found and that's all we want to look for */
    return(0);
  
  /* Now, examine the potential target list, and, to be good team player,
     pick the guy who has the least # of attackers.
     */
  if (n > 0)
    {
      pl = NULL;
      mn = 9999;
      for (int i=0;i<n;i++)
	{
	  if (targets[i]->attkr_cnt < mn)
	    {
	      pl = targets[i];
	      mn = targets[i]->attkr_cnt;
	    }
	}
      if (pl != NULL)
	{
	  set_target(pl);
	  result = 1;
	}
    }
  new_target_flag = result;
  return result;
}

/* see if we're in immediate danger of having our pants shot off */
Pilot *Pilot::threat_check()
{
  Target_Geometry trg;
  Pilot *result = NULL;
  for (int i=0;i<npilots;i++)
    {
      Pilot *pl = pilots[i];
      if ((pl != this ) && (pl->params->affiliation != params->affiliation) )
	{
	  if (pl->flight == target_flight)
	    {
	      trg = tg;
	      trg.p = trk.p_position;
	    }
	  else
	    get_target_geometry(trg,*pl->flight);
	  // If we're targeted, take a peek and see if we're in his sights
	  if (pl->target_flight == flight &&
	      pl->trk.p_position.z > 0.0)
	    {
	      if (
		  pl->in_range &&
		  (fabs(pl->g_trk.sc_x - pl->gp_scx) <= 20) &&
		  (fabs(pl->g_trk.sc_y - pl->gp_scy) <= 20)
		  )
		{
		  result = pl;
		  break;
		}
	    }
	  if ((trg.aspect >= _PI - 0.174) && (trg.angle_off <= 0.174) &&
	      (trg.range <= 3000.0))
	    {
	      result = pl;
	      break;
	    }
	}
    }
  return result;
}

// Check if we're heading into the ground --
// If we are, pull up!
int Pilot::ground_check()
{
  if (ground_flag ||
      (FLTPNT.z + (FLTSTATE.d_agl * params->mparam5)) <= FLTSTATE.ground_height)
    {
      // 20 seconds to ground collision!
      set_cntrl(bank_control,0.0,_2PI,0.1);
      control_step = 10;
      bank_update();
      set_cntrl(speed_control,flight->specs->max_speed,100,1);
      speed_update();

      if (!flight->state.inverted)
	{
	  if (
	       flight->state.delta_climb_vel >= -1.0
	       &&
	       flight->state.delta_z_vel > -10.0
	       &&
	       flight->state.flight_port.slook_from.phi < 2.0
	       )
	    U_ELEVATORS
	      else
		{
		  extend();
		}
	 }
      
      if (ground_flag)
	{
	  if (gh_height >= 5.0)
	    ground_flag = 0;
	  else
	    gh_height += t;
	}
      else
	{
	  ground_flag = 1;
	  gh_height = 0.0;
	}
    }
  else
    ground_flag = 0;
  return (ground_flag);
}


void Pilot::jink()
{
  RUDDER = 0;
  FLAPS = 0;
  WBRAKES = 0;
  GEAR = 0;
  if (e_elapsed >= timer_max)
    {
      flag1 = !flag1;
      e_elapsed = 0.0;
      //     timer_max = (RANDOM(10) * 1.6) + 2.3;
      timer_max = (RANDOM(1000) * 1.6) + 200.3;
    }
  if (flag1)
    {
      hard_turn(0);
      UP_ELEVATORS
	}
  else
    {
      hard_turn(1);
    }
  dbg = "JINK";
}

void Pilot::extend()
{
  dbg = "EXTEND";
  RUDDER=0;
  SETCONTROL(pitch_control,_PI2 - _DEGREE * 2.0,_2PI,0);
  SETCONTROL(bank_control,0,_2PI,PITCH_MIN,1);
  SETCONTROL(speed_control,FLTSPCS->max_speed,100,0);
  bank_update();
  pitch_update();
  speed_update();
}

void Pilot::trackToX(Tracking_Target &ttrk)
{

  float s_x = ttrk.sc_x;
  float abs_x = fabs(s_x);
  float r_limit = abs_x * params->mparam7;
  control_step = ( params->c_step1 * abs_x / params->c_step2);
  if (control_step > params->c_step1)
    control_step = params->c_step1;
  else if (control_step <= 0)
    control_step = 1.0;
  if (s_x < 0)
    {
      if (ttrk.x_rate < r_limit)
	{
	  R_RUDDER
	    }
    }
  else
    {
      if (ttrk.x_rate < r_limit)
	{
	  L_RUDDER
	    }
    }
}

void Pilot::trackToY(Tracking_Target &ttrk)
{
  float r_limit,maxy,miny;
  float s_y = ttrk.sc_y;
  float abs_y = fabs(s_y);
 
  r_limit = abs_y * params->mparam7 ;

  maxy = miny = 0.0;
  control_step =  ( params->c_step3 * fabs(ttrk.y_rate) / params->c_step4);
  if (control_step > (int) params->c_step3)
    control_step = (int) params->c_step3;
  else if (control_step <= 0)
    control_step = 1.0 / t;
  
  if (s_y > maxy)
	{
	  if (ttrk.y_rate < r_limit)
	    {
	      if (abs_y > 15 && ttrk.y_rate < 0)
		UP_ELEVATORS
		  else
		    U_ELEVATORS
		  }
	  else // if (ttrk.y_rate > fabs(ttrk.vel_scy) )
	    {
	      D_ELEVATORS
		}
	}
  else if (s_y  < miny)
    {
      if (ttrk.y_rate < r_limit)
	{
	  if (abs_y  > 15 && ttrk.y_rate < 0 )
	    DOWN_ELEVATORS
	      else
		D_ELEVATORS
	      }
      else // if (ttrk.y_rate > fabs(ttrk.vel_scy) )
	{
	  U_ELEVATORS
	    }
    }
}

void Pilot::setTrackingInfo(Tracking_Target &ttrk, R_3DPoint &w)
{
  float tt = t;
  if (tt == 0.0)
    tt = eps;
  Vector v;

  // Get actual position of target. Record change rates for the
  // x,y projections, & distance
  ttrk.prev_w_position = ttrk.w_position;
  ttrk.w_position = w;
  ttrk.prev_sc_x = ttrk.sc_x;
  ttrk.prev_sc_y = ttrk.sc_y;
  ttrk.prev_p_position = ttrk.p_position;
  world_2_port(ttrk.w_position, &ttrk.p_position, &ttrk.sc_x, &ttrk.sc_y,FLTPORT);
  v = ttrk.p_position;
  ttrk.prev_distance = ttrk.distance;
  ttrk.distance = v.Magnitude() / world_scale;
  // We want a positive rate to indicate movement towards us,
  // negative away
  ttrk.x_rate = fabs(ttrk.prev_sc_x) - fabs(ttrk.sc_x);
  ttrk.y_rate = fabs(ttrk.prev_sc_y) - fabs(ttrk.sc_y);
  ttrk.d_rate = fabs(ttrk.prev_distance) - fabs(ttrk.distance);
  ttrk.x_rate /= tt;
  ttrk.y_rate /= tt;
  ttrk.d_rate /= tt;

  extern float calcRollForPoint2(Port_3D &, R_3DPoint &,
				 R_3DPoint *, float *, float *);
  ttrk.al_w = ttrk.w_position;
  ttrk.al_ang = calcRollForPoint2(FLTPORT,
				    ttrk.w_position,
				    &ttrk.al_p,
				    &ttrk.al_scx,
				    &ttrk.al_scy);
}

void Pilot::get_waypoint_position()
{
    if (!waypoint)
      return;
    R_3DPoint wl = waypoint->location;
    //    wl.z = FLTPORT.look_from.z;
    setTrackingInfo(wp_trk,wl);
    wp_trk.tol_x = wp_trk.tol_y = 4.0;
}

void Pilot::do_waypoint()
{
  flight->controls.armed_w = 0;
  flight->controls.vect_on = 1;
  setTrackingInfo(wp_trk,waypoint->location);

  if (distance_squared(FLTPNT,waypoint->location) <= 6000.0 * world_scale)
    {
      if (waypoint->next)
	{
	  set_waypoint(waypoint->next);
	  get_waypoint_position();
	  al_state.stage = 0;
	  al_state.done = 0;
	}
    }
  wp_trk.tol_x = 4.0;
  wp_trk.tol_y = 4.0;
  align_flight(wp_trk);
  set_cntrl(speed_control,waypoint->average_speed,10,1.0);
  speed_update();
 }

void Pilot::break_turn(int dir)
{
  R_KEY_BEGIN(500) //Pilot::break_turn

  float d2;
  switch(m_state.stage)
    {
    case 0:
      RUDDER = 0;
      FLAPS = 0;
      WBRAKES = 0;
      GEAR = 0;
      dbg = "BRK0";
      m_state.data0 = _DEGREE * 80.0;
      if (dir == right_turn)
	m_state.data0 = _2PI - m_state.data0;
      SETCONTROL(pitch_control,_PI2,_2PI*10,PITCH_MIN,0);
      SETCONTROL(speed_control,FLTSPCS->max_speed,100,0,0);
      speed_update();
      pitch_update();
      if (pitch_control.flag)
	m_state.stage = 1;
      break;

    case 1:
      dbg="BRK1";
      SETCONTROL(bank_control,m_state.data0,_2PI*10,PITCH_MIN,1);
      bank_update();
      speed_update();
      if (bank_control.flag)
	m_state.stage = 2;
      break;

    case 2:
      dbg="BRK2";
      SETCONTROL(g_control,FLTSPCS->load_limit + 0.5,1000,1.5,0);
      bank_update();
      g_update();
      speed_update();
      if (g_control.flag)
	m_state.stage = 3;
      break;

    case 3:
      dbg = "BRK3";
      m_state.data1 = flight->specs->weight / (flight->forces.lift.magnitude + eps);
      if (m_state.data1 < 0.75)
	{
	  d2 = arc_cos(m_state.data1);
	  if (d2 < 0)
	    d2 = _2PI + d2;
	  if (d2 > 1.57)
	    d2 = 1.57;
	  if (d2 < 0 )
	    d2 = 0;
	  if (dir)
	    bank_control.goal = _2PI - d2;
	  else
	    bank_control.goal = d2;
	  bank_update();
	  set_cntrl(g_control,max_gs,10,0);
	  g_update();
	  speed_update();
	}
      else
	{
	  dbg = "BRK4";
	  reset_maneuver();
	}
      break;
    }
  R_KEY_END
    }


void Pilot::standard_turn(int dir, float bank_angle)
{
  R_KEY_BEGIN(499)
    switch (m_state.stage)
      {
      case 0:
	RUDDER = 0;
	FLAPS = 0;
	GEAR = 0;
	if (bank_angle < 0.0)
	  bank_angle = 0.0;
	if (bank_angle > _PI)
	  bank_angle = _PI;
	if (dir == left_turn)
	  m_state.data0 = bank_angle;
	else
	  m_state.data0 = _2PI - bank_angle;
	/* Lift for level flight at given bank angle */
	m_state.data1 = 1.0 / (cos(bank_angle) + eps);
	SETCONTROL(pitch_control,_PI2,_2PI,_DEGREE,0);
	pitch_update();
	if (pitch_control.flag)
	  m_state.stage = 1;
	dbg="TURN0";
	break;

      case 1:
	SETCONTROL(bank_control,m_state.data0,_2PI,_DEGREE,1);
	bank_update();
	pitch_update();
	if (bank_control.flag)
	  m_state.stage = 2;
	dbg="TURN1";
	break;

      case 2:
	bank_update();
	SETCONTROL(g_control,m_state.data1,100,0);
	g_update();
	dbg="TURN3";
	break;
      }

    R_KEY_END
}

const float rll_limit_l = 1.5707963 / 2.0;
const float rll_limit_r = 6.2831853 - rll_limit_l;

int Pilot::align_flight(Tracking_Target &ttrk)
{
  if (al_state.stage == 0)
    {
      if (ttrk.p_position.z < 0)
	al_state.stage = 1;
      else if (fabs(ttrk.sc_x) > 8.0
	       || fabs(ttrk.sc_y) > 8.0 )
	al_state.stage = 2;
      else
	al_state.stage = 3;
    }

  switch (al_state.stage)
    {
      
    case 1:
      {
	al_state.done = 0;
	if (!m_state.maneuver)
	  {
	    set_maneuver(mturn,
			 ttrk.p_position.x > 0 ? right_turn : left_turn);
	    standard_turn(m_state.dir,_DEGREE * 60.0);
	  }
	else
	  {
	    standard_turn(m_state.dir,_DEGREE * 60.0);
	    if ( ttrk.p_position.z > 0 &&
		 fabs(ttrk.al_scx) <= 200 )
	      {
		m_state.done = 1;
		m_state.maneuver = 0;
		al_state.stage = 2;
	      }
	  }
	break;
      }

    case 2:
      {
	RUDDER = 0.0;
 	al_state.done = 0;
	if (ttrk.sc_x <= 0.0)
	  {
	    if (ttrk.al_ang > rll_limit_l)
	      ttrk.al_ang = rll_limit_l;
	  }
	else if (ttrk.al_ang < rll_limit_r)
	  ttrk.al_ang = rll_limit_r;
	dbg = "AL2";
	SETCONTROL(bank_control,ttrk.al_ang,_2PI,_DEGREE,1);
	bank_update();
	SETCONTROL(pitch_control,_PI2,_2PI,_DEGREE*5.0,0);
	pitch_update();
	if (pitch_control.flag)
	  {
	    SETCONTROL(g_control,1 / (cos(ttrk.al_ang) + eps),10,0.0,0);
	    g_update();
	  }
	if (fabs(ttrk.al_scx) <= ttrk.tol_x)
	  al_state.stage = 3;
	break;
      }

    case 3:
      {
	dbg = "AL3";
	SETCONTROL(bank_control,0.0,_2PI,_DEGREE,1);
	bank_update();
	if (bank_control.flag)
	  al_state.stage = 4;
      }
    break;


    case 4:
      {
	bank_update();
	dbg = "AL4";
	trackToY(ttrk);
	trackToX(ttrk);
	if (fabs(ttrk.al_scx) > ttrk.tol_x)
	  al_state.stage = 2;
      }
    break;

    }
  
  return (1);
}


void Pilot::do_takeoff()
{
  if (m_state.maneuver)
    do_maneuver();
  else if (m_state.done)
    set_waypoint(waypoint->next);
  else
    set_maneuver(mtakeoff);
}

void Pilot::do_cap()
{
  /* 
     If we are not engaged, or if we
     are engaged but there are > 2 other
     attackers on our target, see if there
     are better targets ... ie somebody
     targeting us, or somebody with < 2
     attackers
     */
  if (!engaged || target_pilot &&
      target_pilot->attkr_cnt > 2)
    {
      int flag;
      if (!engaged)
	flag = 0;
      else if (target_pilot && target_pilot->attkr_cnt > 2)
	flag = 2;
      if (acquire_target(flag))
	{
	  brdcst(5,params->affiliation,
		 "Engaging %s at angels %2.0f",
		 target_flight->specs->model,
		 angels(target_flight));
	  engaged = 1;
	}
    }
  else if (target->hurt >= target->max_damage ||
	   target_flight->state.crashed)
    {
      /* view the target a bit */
      if (tdwell_flg == 0)
	{
	  tdwell_time = tdwell_max;
	  tdwell_flg = 1;
	  dbg = "TRGDWLL";
	}
      else
	{
	  tdwell_time -= t;
	  if (tdwell_time < 0.0)
	    {
	      engaged = 0;
	      tdwell_flg = 0;
	    }
	}
      if (target->who_got_me == target_obj)
	{
	  brdcst(16,params->affiliation,
		 "Kill on %s at Angels %2.0f",
		 target_flight->specs->model,
		 angels(target_flight));
	}
      else if (target_flight->state.crashed && !target->who_got_me)
	{
	  brdcst(2,params->affiliation,
		 "%s Augered In!",
		 target_flight->specs->model);
	}
      else
	{
	  brdcst(2,params->affiliation,
		 "Good Kill On %s at Angels %2.0f",
		 target_flight->specs->model,
		 angels(target_flight));
	}
      engage();
      if (tdwell_flg)
	dbg = "TRGDWLL";
      return;
    }
  if (engaged)
    do_pursuit();
  else
    {
      dbg = "TRGXX";
      flight->controls.armed_w = 0;
      if (waypoint)
	do_waypoint();
    }
}

void Pilot::do_pursuit()
{
  if (target_pilot != NULL &&
      target_pilot->target_pilot != this)
    {
      if (acquire_target(1))
	{
	  brdcst(5,params->affiliation,
		 "Switching to %s at angels %2.0f.",
		 target_flight->specs->model,
		 angels(target_flight));
	  engaged = 1;
	}
    }
  if (target)
    engage();
}

int Pilot::dropTanks()
{
  if (has_fuel_tanks)
    {
      selWeapon(ft_idx);
      BANG_BANG
	has_fuel_tanks--;
      return (1);
    }
  return (0);
}

void Pilot::engage()
{
  R_KEY_BEGIN(409) // Pilot::engage
    flight->controls.radar = flight->controls.armed_w = 1;
  if (dropTanks())
    {
      brdcst(5,params->affiliation,"Dropping Tanks");
      return;
    }
  selWeapon(0);

  if ((trk.p_position.z <= (flight->specs->view_point.z + 10 * world_scale))
      || m_state.maneuver != 0)
    {
      if (!m_state.maneuver)
	{
	  int n = RANDOM(3) + 1;
	  switch (n)
	    {
	    case 1:
	      set_maneuver(mbreak_turn,
			   trk.p_position.x > 0 ? right_turn : left_turn);
	      break;

	    case 2:
	      if (flight->state.z_vel >= flight->specs->corner_speed - 100)
		{
		  set_maneuver(mimmelman);
		  break;
		}
	    case 3:
	      if (flight->state.flight_port.look_from.z - 
		  flight->state.ground_height > (5000 * world_scale))
		set_maneuver(msplit_s);
	      else if (flight->state.z_vel >= flight->specs->corner_speed - 100)
		set_maneuver(mimmelman);
	      else set_maneuver(mbreak_turn,
				trk.p_position.x > 0 ? right_turn : left_turn);
	      break;

	    default:
	      break;
	    }
	  do_maneuver();
	  return;
	}
      else
	{
	  do_maneuver();
	  if (!m_state.done)
	    {
	      if ( (trk.p_position.z >
		    (flight->specs->view_point.z + 10 * world_scale))
		   &&
		   fabs(trk.sc_x) <= 300 && fabs(trk.sc_y) <= 200)
		{
		  m_state.done = 1;
		  m_state.maneuver = 0;
		  pursue();
		}
	    }
	}
      R_KEY_END
	}
  else
    pursue();
}

/********************************************************************
 * Try to get lined up for a shot, without violating the laws of    *
 * physics too much. Needs much work, probably a whole new approach.*
 * Cheating (i.e., tweaking the plane's motion to help the pilot)   *
 * is dutifully noted.                                              *
 ********************************************************************/
void Pilot::pursue()
{
  R_KEY_BEGIN(4091) // Pilot::pursue
   
  float r_limit;
  float s_x,s_y;
  float abs_x,abs_y;
  float maxy,miny;

  FLAPS = 0;
  WBRAKES = 0;
  GEAR = 0;

  if (in_range)
    {
      s_x = g_trk.sc_x - gp_scx;
      s_y = g_trk.sc_y - gp_scy;
      c_trk = &g_trk;
    }
  else
    {
      s_x = l_trk.sc_x;
      s_y = l_trk.sc_y;
      c_trk = &l_trk;
    }

  abs_x = fabs(s_x);
  abs_y = fabs(s_y);

  if (in_range)
    {
      if (abs_x <= 20 &&
	  abs_y <= 20 )
	dbg = "PRST0B";
      else if (abs_x <= SCREEN_WIDTH / 2 &&
	       abs_y <= SCREEN_HEIGHT / 2)
	dbg = "PRST0A";
      else
	dbg = "PRST1";
    }
  else
    dbg = "PRST2";


  // Attempt to align the lift vector with the
  // target
  /*
  if (tg.angle_off < _PI2 - params->mparam4 ||
      tg.angle_off > _PI2 + params->mparam4 )
    {
      dv = DVector(0,tg.range * world_scale / params->mparam8, 0);
      dv2 = target_flight->to_world(dv);
      v6 = dv2.to_R_3DPoint();
      p6 = trk.w_position + v6;
      h_angle = 0;
    }
  else
    {
      p6 = l_trk.w_position;
      h_angle = 1;
    }
    */

  if (g_trk.distance > WPSPECS(sel_weapon)->max_range * 4.0)
    {
      extern float calcRollForPoint2(Port_3D &, R_3DPoint &,
				 R_3DPoint *, float *, float *);
      R_3DPoint tmp;
      float tmp0,tmp1;
      tg.b_angle = calcRollForPoint2(FLTPORT,
				    g_trk.w_position,
				    &tmp,
				    &tmp0,
				    &tmp1);
      tg.bp = g_trk.w_position;
    }
  else
    {
      tg.bp = g_trk.w_position;
      tg.b_angle = calcRollForPoint(FLTPORT,g_trk.w_position);
    }
  r_limit = abs_x * params->mparam7;
  SETCONTROL(bank_control,tg.b_angle,_2PI*10.0,PITCH_MIN,1);
  bank_update();

  /*********************************************************
   * Rudder                                                *
   *********************************************************/
  control_step = ( params->c_step1 * fabs(trk.x_rate) / params->c_step2);
  if (control_step > params->c_step1)
    control_step = params->c_step1;
  else if (control_step <= 0)
    control_step = 1.0;

  if (abs_x <= params->mparam1)
     {
       if (s_x < 0)
	 {
	   if (trk.x_rate < r_limit)
	     {
	       R_RUDDER
		 }
	   else if (trk.x_rate > fabs(trk.vel_scx))
	     {
	       L_RUDDER
		 }
	}
      else
	{
	  if (trk.x_rate < r_limit)
	    {
	      L_RUDDER
		  }
	  else if (trk.x_rate > fabs(trk.vel_scx))
	    {
	      R_RUDDER
		}
	}
     }

  if (fabs(trk.y_rate) > params->mparam2 ||
      abs_y > params->mparam3)
    {
      r_limit = abs_y * params->mparam7 ;
      maxy = miny = 0.0;
      control_step =  ( params->c_step3 * fabs(trk.y_rate) / params->c_step4);
      if (control_step > (int) params->c_step3)
	control_step = (int) params->c_step3;
      else if (control_step <= 0)
	control_step = 1.0 / t;

      if (s_y > maxy)
	{
	  if (trk.y_rate < r_limit)
	    {
	      if (in_range && abs_y > 15 && trk.y_rate < 0)
		UP_ELEVATORS
		  else
		    U_ELEVATORS
		  }
	  else // if (trk.y_rate > fabs(trk.vel_scy) )
	    {
	      D_ELEVATORS
		}
	}
      else if (s_y  < miny)
	{
	  if (trk.y_rate < r_limit)
	    {
	      if (in_range && abs_y  > 15 && trk.y_rate < 0 )
		DOWN_ELEVATORS
		  else
		    D_ELEVATORS
		  }
	  else // if (trk.y_rate > fabs(trk.vel_scy) )
	    {
	      U_ELEVATORS
		}
	}
    }
  if (( abs_y <= params->mdampy ) ) 
    {
      flight->state.pitch_rate = DAMP(flight->state.pitch_rate,abs_y * 0.01 * t);
    }

  /* Oooh ... you cheater ... you're moving the gun! */
  if (target->hurt < target->max_damage && 
      abs_x <= params->mparam6 &&
      abs_y <= params->mparam6 && 
      in_range)
    {
      sel_weapon->flags = WP_SETVIEW;
      sel_weapon->target_position = g_trk.w_position;
      flight->controls.bang_bang = 1;
      broadcast("Guns! Guns!",1,params->affiliation,NULL);
    }

  float wpr =  WPSPECS(sel_weapon)->min_range +
	  WPSPECS(sel_weapon)->max_range / 3.0;

  if ((tg.angle_off <= (_PI2 / 3.0)) && (tg.aspect < (_PI2 / 3.0)) 
      && (tg.range < wpr))
    //      && (tg.d_range < wpr))
    speed_control.goal = target_flight->state.z_vel;
  else
    speed_control.goal = flight->specs->max_speed;
  speed_update();

  if (!flight->controls.bang_bang)
    shoot();
  control_step = 1.0 / t;
  R_KEY_END
    }

void Pilot::hard_turn(int dir)
{
  dbg = "HTURN";
  if (dir == right_turn)
    SETCONTROL(bank_control,_2PI-HARD_BANK,_2PI*10.0,PITCH_MIN,1);
  else
    SETCONTROL(bank_control,HARD_BANK,_2PI*10.0,PITCH_MIN,1);
  bank_update();
  bank_control.snap_to = 0;

  if (bank_control.flag)
    {
      set_cntrl(g_control,max_gs,100.0,0);
      g_update();
    }
  else
    {
      set_cntrl(pitch_control,_PI2+_DEGREE*15.0,_2PI,0);
      pitch_update();
    }
  set_cntrl(speed_control,flight->specs->max_speed,
	    10,1.0);
  speed_update();
}

void Pilot::pause()
{

}

void Pilot::bank_update()
{
  bank_control.flag = 0;
  if (bank_control.rate > _2PI)
    bank_control.rate = _2PI;
  int dir = get_dir(ROLL,
		    bank_control.goal,
		    bank_control.min);

  float diff = get_diff(ROLL, bank_control.goal);

  float rate = (diff / _PI ) * bank_control.rate ;
  control_step = rcontrol_max * (fabs(FLTSTATE.roll_rate - rate) / _2PI);

  switch (dir)
    {
    case 0:
      AILERONS = 0;
      bank_control.flag = 1;
      if (bank_control.snap_to)
	{
	  flight->state.flight_port.set_roll(bank_control.goal);
	  flight->state.roll_rate = 0.0;
	}
      break;

    case 1:
      if (flight->state.roll_rate <= rate)
	RIGHT_AILERONS
	  else
	    L_AILERONS
	      break;

    case -1:
      if (flight->state.roll_rate >= -rate)
	LEFT_AILERONS
	  else
	    R_AILERONS
	      break;
    }
  bank_control.snap_to = 0;
}

void Pilot::pitch_update()
{
  float r;
  float rate;
  float dd;

  if (pitch_control.goal < 0.0)
    pitch_control.goal = _PI - pitch_control.goal;
  if (pitch_control.goal > _PI)
    pitch_control.goal = _2PI - pitch_control.goal;
  r = fabs(pitch_control.goal - FLTSPNT.phi) / _PI;
  if (r > 1.0)
    r = 1.0;
  if (r <= 0.0)
    r = 0.1;
  
  rate = r * pitch_control.rate;
  if (rate <= 0.0)
    rate = -rate;

  dd = fabs(rate - FLTSTATE.d_pitch_rate);
  if (dd > _2PI)
    r = _2PI;
  r = dd / _2PI;
  control_step = rcontrol_max * r;
  if (control_step > rcontrol_max)
    control_step = rcontrol_max;
  else if (control_step <= 0.0)
    control_step = 0.1;

  int dir = get_dir(FLTSPNT.phi,
		    pitch_control.goal,
		    pitch_control.min);
  pitch_control.flag = 0;
  if (flight->state.inverted)
    dir = -dir;
  if (dir == 0)
    {
      pitch_control.flag = 1;
      if (pitch_control.snap_to)
	{
	  flight->state.flight_port.set_phi(pitch_control.goal);
	  flight->forces.pitch_acc = 0.0;
	  flight->state.pitch_rate = 0.0;
	}
    }
  else if (dir == 1)
    {
      if (pitch_control.snap_to)
	{
	  FLTSTATE.pitch_rate = -rate;
	  flight->forces.pitch_acc = 0.0;
	  pitch_control.snap_to = 0;
	}
      else
	{
	  /*
	  if (FLTSTATE.pitch_rate > 0.0)
	    {
	      UP_ELEVATORS
		}
	  else
	  */
	    if (flight->state.pitch_rate >= -rate)
	      {
		U_ELEVATORS
		  }
	    else 
	      {
		DOWN_ELEVATORS
		  }
	}
    }
  else
    {
      if (pitch_control.snap_to)
	{
	  FLTSTATE.pitch_rate = rate;
	  flight->forces.pitch_acc = 0.0;
	  pitch_control.snap_to = 0;
	}
      else
	{
	  /*
	  if (flight->state.pitch_rate < 0.0)
	    {
	      DOWN_ELEVATORS
		}
		
	  else */ if (flight->state.pitch_rate <= rate)
	    {
	      D_ELEVATORS
		}
	  else 
	    {
	      UP_ELEVATORS
		}
	}
    }
  pitch_control.snap_to = 0;
}

void Pilot::aoa_update()
{
  float d = flight->state.angle_of_attack - aoa_control.goal;
  if ( (fabs(d) - eps) <= aoa_control.min )
    {
      aoa_control.flag = 1;
      //		flight->state.pitch_rate = 0.0;
      return;
    }
  aoa_control.flag = 0;
  int dir = d < 0.0;
  if (flight->state.inverted)
    dir = -dir;
  if (dir > 0)
    {
      if (flight->state.pitch_rate >= -aoa_control.rate)
	UP_ELEVATORS
	  else
	    D_ELEVATORS
	  }
  else
    {
      if (flight->state.pitch_rate <= aoa_control.rate)
	DOWN_ELEVATORS
	  else
	    U_ELEVATORS
	  }
}

void Pilot::speed_update()
{
  speed_control.flag = 0;
  if (flight->state.z_vel > speed_control.goal + speed_control.min)
    {
      if (speed_control.rate > 0.0 &&
	  flight->state.delta_z_vel <= -speed_control.rate)
	return;
      {
	if (!SBRAKES)
	  SBRAKES = 1;
	else if (THROTTLE > 20)
	  THROTTLE--;
      }
    }
  else if (flight->state.z_vel < speed_control.goal - speed_control.min)
    {
      if (speed_control.rate > 0.0 &&
	  flight->state.delta_z_vel >= speed_control.rate)
	return;
      {
	if (SBRAKES)
	  SBRAKES = 0;
	else if (THROTTLE < 100)
	  THROTTLE++;
      }
    }
  else
    speed_control.flag = 1;
}

void Pilot::g_update()
{
  g_control.flag = 0;
  if (g_control.snap_to)
    {
      /* force lift vector */
      if (g_control.goal < 0.0)
	{
	  g_control.goal = -g_control.goal;
	  flight->forces.lift.direction = DVector(0,-1,0);
	}
      else
	flight->forces.lift.direction = DVector(0,1,0);
      flight->forces.lift.magnitude =
	flight->state.weight * g_control.goal;
      g_control.snap_to = 0;
      return;
    }
  
  float rate;
  float r;
  float dd = fabs(FLTSTATE.load - g_control.goal);
  if (dd <= g_control.min)
    {
      g_control.flag = 1;
      return;
    }

  r = dd / g_range;
  rate = g_control.rate * r;
  dd = fabs(rate - FLTSTATE.d_load);
  if (dd > g_range)
    dd = g_range;
  r = dd / g_range;
  control_step = r * gcontrol_max;
  if (control_step < 0.0)
    control_step = 0.0;
  if (control_step > gcontrol_max)
    control_step = gcontrol_max;

  if (flight->state.load < g_control.goal)
    {
      if (flight->state.d_load < rate)
	{
	  U_ELEVATORS
	    }
	  else
	    D_ELEVATORS
    }
  else if (flight->state.load > g_control.goal)
    {
      if (flight->state.d_load > -rate)
	D_ELEVATORS
	  else
	    U_ELEVATORS
    }
}


void Pilot::shoot()
{
  if (!in_range || target->isHistory())
    return;
  if (trk.p_position.z > 0.0)
    {
      float shoot_boxx;
      float shoot_boxy;
      /*
      if (trk.distance > 4000.0)
	shoot_box = 2.0;
      if (trk.distance > 3000.0)
	shoot_box = 4.0;
	*/
      if (trk.distance > 1500.0)
	shoot_boxx = shoot_boxy = 4.0;
      else if (trk.distance > 1000.0)
	shoot_boxx = shoot_boxy = 6.0;
      else if (trk.distance > 400.0)
	shoot_boxx = shoot_boxy = 8.0;
      else
	shoot_boxx = shoot_boxy = 12.0;
      /*
      shoot_boxx *= ((float)SCREEN_WIDTH) / 320.0;
      shoot_boxy *= ((float)SCREEN_HEIGHT) / 200.0;
      */
      if ((fabs(g_trk.sc_x - gp_scx) <= shoot_boxx) &&
	  (fabs(g_trk.sc_y - gp_scy) <= shoot_boxy))
	{
	  if (time_to_target <= sel_weapon->weapon->w_specs->flt_specs.time_limit)
	    {
	      flight->controls.bang_bang = 1;
	      broadcast("Guns! Guns! Guns!",1,params->affiliation,NULL);
	    }
	}
    }
}

void Pilot::world_2_port(R_3DPoint &w, R_3DPoint *lp,
			 float *sc_x, float *sc_y, Port_3D &port)
{
  if (world2portN(w,lp,sc_x,sc_y,port))
    {
      *sc_x /= xpixel_adjust;
      *sc_y /= ypixel_adjust;
    }
}

void Pilot::get_target_position()
{
  R_KEY_BEGIN(901) // Pilot::get_target_position
    Vector v,v1;
  DVector hv;
  // Calc how long it will take for a round to reach
  // the target's position (horizontal)
  time_to_target = sel_weapon->calc_hvtime(*flight,*target->get_position());

  // Get actual position of target. Record change rates for the
  // x,y projections, & distance
  trk.prev_w_position = trk.w_position;
  trk.w_position = *(target->get_position());
  trk.prev_sc_x = trk.sc_x;
  trk.prev_sc_y = trk.sc_y;
  trk.prev_p_position = trk.p_position;
  world_2_port(trk.w_position, &trk.p_position, &trk.sc_x, &trk.sc_y,FLTPORT);
  v = trk.p_position;
  trk.prev_distance = trk.distance;
  trk.distance = v.Magnitude() / world_scale;
  // We want a positive rate to indicate movement towards us,
  // negative away
  trk.x_rate = fabs(trk.prev_sc_x) - fabs(trk.sc_x);
  trk.y_rate = fabs(trk.prev_sc_y) - fabs(trk.sc_y);
  trk.x_rate /= t + eps;
  trk.y_rate /= t + eps;
  trk.d_rate = fabs(trk.prev_distance) - fabs(trk.distance);
  trk.d_rate /= t + eps;
  // Get the target's velocity vector in port coords, if a
  // flight
  if (target_flight)
    {
      target_vel = to_vector(target_flight->state.velocity);
      target_vel = flight->to_port(target_vel);
      // Now, get 1-second lead/lag position
      l_trk.w_position = trk.w_position;
      hv = to_vector(target_flight->state.velocity);
      v1 = hv.to_vector();
      v1 *= world_scale;
      l_trk.w_position.x += v1.X;
      l_trk.w_position.y += v1.Y;
      l_trk.w_position.z += v1.Z;
      world_2_port(l_trk.w_position, &l_trk.p_position, &l_trk.sc_x,
		   &l_trk.sc_y,FLTPORT);
      // Velocity in screen coords,
      // based on lead of one second
      // + means moving towards us,
      // - away
      trk.vel_scx = fabs(trk.sc_x) - fabs(l_trk.sc_x);
      trk.vel_scy = fabs(trk.sc_y) - fabs(l_trk.sc_y);
      v = l_trk.p_position;
      l_trk.prev_distance = l_trk.distance;
      l_trk.distance = v.Magnitude() / world_scale;

      // Get position after predicted flight time of gun round
      g_trk.w_position = trk.w_position;
      v1 = hv.to_vector();
      v1 *= time_to_target;
      v1 *= world_scale;
      g_trk.w_position.x += v1.X;
      g_trk.w_position.y += v1.Y;
      g_trk.w_position.z += v1.Z;
      world_2_port(g_trk.w_position, &g_trk.p_position, &g_trk.sc_x,
		   &g_trk.sc_y,FLTPORT);
      v = g_trk.p_position;
      g_trk.prev_distance = g_trk.distance;
      g_trk.distance = v.Magnitude() / world_scale;

      // Get target geometry
      get_target_geometry(tg,(Flight &) *target_flight);
    }
  else
    {
      // target not a flight
      g_trk = trk;
      l_trk = trk;
      target_vel = DVector(0,0,0);
    }
  R_KEY_END
    }

void Pilot::get_target_geometry(Target_Geometry &trg, Flight &t_flt)
{
  float s_x,s_y;
  world_2_port(t_flt.state.flight_port.look_from,&trg.p,&s_x,&s_y,FLTPORT);
  float tmp = flight->state.flight_port.view_normal.Dot(
							t_flt.state.flight_port.view_normal);
  trg.angle_off = arc_cos(tmp);
  Vector v = Vector(t_flt.state.flight_port.look_from -
		    flight->state.flight_port.look_from );
  float r = v.Magnitude() / world_scale ;
  trg.d_range = (trg.range - r) / ( t + eps);
  trg.range = r;
  v.Normalize();
  tmp = v.Dot(t_flt.state.flight_port.view_normal);
  trg.aspect = arc_cos(tmp);
  if (trg.p.x >= 0)
    trg.l_r = 'L';
  else
    trg.l_r = 'R';
  if (trg.angle_off > _PI2)
    {
      if (trg.l_r == 'L')
	trg.l_r = 'R';
      else
	trg.l_r = 'L';
    }
}

void Pilot::do_maneuver()
{
  if (!m_state.done)
    {
      switch (m_state.maneuver)
	{
	case mtakeoff:
	  takeoff();
	  break;

	case mbreak_turn:
	  break_turn(m_state.dir);
	  break;

	case mimmelman:
	  immelman();
	  break;

	case msplit_s:
	  split_s();
	  break;

	case mbarrel_roll:
	  barrel_roll();
	  break;

	case mroll_out:
	  roll_out();
	  break;

	}
    }
}

void Pilot::roll_out()
{
  dbg = "RLLOUT0";
  bank_control.goal = 0.0;
  bank_control.min = 0.05;
  bank_control.rate = _2PI;
  bank_update();
  if (bank_control.flag)
    {
      dbg = "";
      reset_maneuver();
    }
}

void Pilot::immelman()
{
  switch(m_state.stage)
    {
    case 0:
      dbg = "IMML0";
      RUDDER = 0;
      FLAPS = 0;
      WBRAKES = 0;
      GEAR = 0;

      if (flight->state.z_vel < flight->specs->corner_speed)
	extend();
      else
	{
	  RUDDER = 0;
	  SETCONTROL(bank_control,0.0,_2PI,_DEGREE,1);
	  SETCONTROL(pitch_control,_PI2,_2PI,0,1);
	  set_cntrl(speed_control,flight->specs->max_speed,900,1.0);
	  bank_update();
	  pitch_update();
	  speed_update();
	  if (bank_control.flag)
	    m_state.stage = 1;
	}
      break;

    case 1:
      dbg = "IMML1";
      UP_ELEVATORS
	speed_update();
      if (flight->state.inverted)
	m_state.stage = 2;
      break;

    case 2:
      dbg = "IMML2";
      SETCONTROL(pitch_control,_PI2,_2PI*10.0,_DEGREE,0);
      pitch_update();
      if (pitch_control.flag)
	m_state.stage = 3;
      speed_update();
      break;

    case 3:
      dbg = "IMML3";
      SETCONTROL(bank_control,0.0,_2PI,_DEGREE*2,1);
      SETCONTROL(pitch_control,_PI2,_2PI,_DEGREE*2,0);
      bank_update();
      speed_update();
      pitch_update();
      if (bank_control.flag && pitch_control.flag)
	{
	  dbg = "";
	  reset_maneuver();
	}
      break;
    }
}

void Pilot::split_s()
{
  switch(m_state.stage)
    {
    case 0:
      dbg = "SPLS0";
      RUDDER = 0;
      FLAPS = 0;
      WBRAKES = 0;
      GEAR = 0;
      SETCONTROL(pitch_control,_PI2,_2PI,PITCH_MIN,0);
      pitch_update();
      if (pitch_control.flag)
	m_state.stage = 1;
      break;

    case 1:
      dbg = "SPLS1";
      SETCONTROL(bank_control,_PI,_2PI,PITCH_MIN,1);
      SETCONTROL(speed_control,FLTSPCS->max_speed,1000,0);
      bank_update();
      pitch_update();
      speed_update();
      if (bank_control.flag)
	m_state.stage = 2;
      break;

    case 2:
      dbg = "SPLS2";
      UP_ELEVATORS
	speed_update();
      if (!flight->state.inverted)
	m_state.stage = 3;
      break;

    case 3:
      dbg = "SPLS3";
      SETCONTROL(bank_control,0,_2PI,PITCH_MIN,1);
      SETCONTROL(pitch_control,_PI2,_2PI*100.0,PITCH_MIN,0);
      bank_update();
      pitch_update();
      if (pitch_control.flag)
	m_state.stage = 4;
      speed_update();
      break;

    case 4:
      dbg = "SPLS4";
      bank_update();
      speed_update();
      SETCONTROL(pitch_control,_PI2,_2PI,0,0);
      pitch_update();
      if (bank_control.flag)
	{
	  dbg = "";
	  reset_maneuver();
	}
      break;
    }
}

void Pilot::barrel_roll()
{
  FLAPS = 0;
  WBRAKES = 0;
  GEAR = 0;
}


void Pilot::takeoff()
{
  R_KEY_BEGIN(509) //Pilot::takeoff

    switch(m_state.stage)
      {
      case 0:
	dbg = "TKOFF1";
	RUDDER = 0;
	FLAPS = 0;
	WBRAKES = 1;
	GEAR = 1;
	SBRAKES = 0;
	flight->controls.engine_on = 1;
	flight->controls.armed_w = 0;
	if (THROTTLE < 100)
	  THROTTLE += 4;
	else
	  {
	    WBRAKES = 0;
	    m_state.stage = 1;
	  }
	break;

      case 1:
	dbg = "TKOFF1";
	if (flight->state.z_vel >= flight->specs->lspeed)
	  {
	    control_step = 30;
	    UP_ELEVATORS
	      m_state.stage = 4;
	  }
	break;

      case 2:
	if (flight->state.delta_climb_vel <= 0.0)
	  {
	    dbg = "TKOFF2";
	    if (flight->state.near_stall > 0.02)
	      {
		UP_ELEVATORS
		  }
	    else
	      {
		D_ELEVATORS
		  }
	  }
	else
	  {
	    pitch_control.goal = flight->state.flight_port.slook_from.phi;
	    pitch_control.min = 0.1;
	    pitch_control.rate = _2PI;
	    bank_control.goal = 0.0;
	    bank_control.min = 0.05;
	    m_state.stage = 3;
	  }
	break;

      case 3:
	dbg = "TKOFF3";
	if (flight->state.near_stall > 0.002)
	  {
	    pitch_update();
	  }
	else
	  D_ELEVATORS
	    bank_update();
	if (flight->state.flight_port.look_from.z >= 300 * world_scale)
	  {
	    GEAR = 0;
	    m_state.stage = 4;
	  }
	break;

      case 4:
	{
	  dbg = "TKOFF4";
	  float climb_rate;
	  if (flight->state.flight_port.look_from.z >= 300 * world_scale)
	    {
	      GEAR = 0;
	      climb_rate = 20000;
	      control_step = 20.0;
	    }
	  else
	    {
	      climb_rate = 12000;
	      control_step = 2.0;
	    }
	  set_cntrl(bank_control,0.0,_2PI,0.1);
	  bank_update();
	  if (stall_check())
	    {
	      dbg = "TSTALL";
	      return;
	    }
	  climb(climb_rate);
	  if (flight->state.flight_port.look_from.z >= waypoint->location.z)
	    {
	      dbg = "";
	      reset_maneuver();
	    }
	}
	break;
      }
  R_KEY_END
    }

int Pilot::broadcast(char *mss, int priority, int freq, char *who)
{
  int result = 0;
  for (int i=0;i<npilots;i++)
    {
      Pilot *pl = pilots[i];
      if ((freq != -1) && (pl->params->affiliation != freq))
	continue;
      if (who != NULL && strcmp(pl->handle,who))
	continue;
      if (pl->message.set_message(mss,priority,this))
	result++;
    }
  return result;
}

void Pilot::climb(float rate)
{
  set_cntrl(bank_control,0.0,_2PI,0.1);
  bank_update();
  if (stall_check())
    {
      dbg = "CSTALL";
      return;
    }
  dbg = "CLIMB";
  if (
      (
       flight->state.climb_vel > rate
       &&
       flight->state.delta_climb_vel >= 0.0
       )
      ||
      flight->state.delta_z_vel < 0.0
      )
    pitch_control.goal -= _DEGREE * t;
  else if (
	   flight->state.climb_vel <= rate
	   &&
	   flight->state.delta_climb_vel <= (rate / 2.0)
	   )
    pitch_control.goal += _DEGREE * t;
  pitch_control.rate = _DEGREE;
  pitch_control.min = _DEGREE;
  pitch_update();
}


void Pilot::descend(float rate)
{
  set_cntrl(bank_control,0.0,_2PI,0.1);
  bank_update();
  if (stall_check())
    {
      dbg = "DSTALL";
      return;
    }

  dbg = "DSCND";
  if (flight->state.climb_vel >= rate &&
      flight->state.delta_climb_vel >= 0.0 &&
      pitch_control.goal >= 0.2)
    {
      pitch_control.goal -= _DEGREE * t;
    }
  else if (flight->state.climb_vel <= rate &&
	   flight->state.delta_climb_vel <= 0.0)
    pitch_control.goal += _DEGREE * t;

  if (pitch_control.goal > _PI2)
    pitch_control.goal = _PI2;
  pitch_control.rate = _DEGREE;
  pitch_control.min = _DEGREE;
  pitch_update();
}

int Pilot::brdcst(int priority, int freq, char *mss, ...)
{
  va_list ap;
  va_start(ap,mss);
  vsprintf(mssbuff,mss,ap);
  va_end(ap);
  MYCHECK(strlen(mssbuff) <= 127);
  return (broadcast(mssbuff,priority,freq));
}

// Coordinate rudders & ailerons
void Pilot::coord()
{
  if (AILERONS == 0)
    RUDDER = 0;
  else
    {
      float torque,cntrl;
      torque = flight->specs->adv_yaw * AILERONS;
      cntrl = -(torque / flight->specs->control_yaw);
      {
	SET_RUDDER(cntrl)
	  }
    }
}

void Pilot::selectNextWeapon(int dir)
{
  sel_wpn += dir;
  if (sel_wpn >= n_weaps)
    sel_wpn = 0;
  else if (sel_wpn < 0)
    sel_wpn = n_weaps - 1 ;
  sel_weapon = &weapons[sel_wpn];
}

void Pilot::selWeapon(int n)
{
  if (n >= 0 && n < n_weaps)
    {
      sel_wpn = n;
      sel_weapon = &weapons[sel_wpn];
    }
}

void PilotMessage::update()
{
  if (!flag)
    return;
  elapsed_time += time_frame;
  if (elapsed_time >= time_limit)
    {
      flag = 0;
      mssg = NULL;
    }
}

int PilotMessage::set_message(char *mss, int priority, Pilot *fr)
{
  if (!flag || flag <= priority)
    {
      elapsed_time = 0.0;
      flag = priority;
      mssg = mss;
      frm = fr;
      return (1);
    }
  else
    return (0);
}
