#include <gtk/gtk.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "external.h"

static ChildRec *child_rec_new (pid_t pid, int fd, GdkInputFunction
				callback, gpointer callback_data,
				GdkInputCondition cond, GtkWidget
				*toplevel);

static GHashTable *children = NULL;
/* a pipe to myself, for notification when children exit */
static int childpipe[2];

extern gint debug_level;

void reaper (gint signum)
{
  int data[2];

  data[0] = wait (&data[1]);

  /* fail silently, don't ask me why... */
  if (data[0] == -1)
    return;
  
  /* tell myself the bad news */
  write (childpipe[1], &data, sizeof(int) * 2);
}

gchar *find_in_path (gchar *prog)
{
  gchar *path, *colon, *progpath;

  /* FIXME: should probably check to see if it actually exists */
  if (prog[0] == '/')
    return prog;
  
  path = getenv ("PATH");
  
  for (;;) {
    struct stat buf;
    size_t pathlen;

    colon = strchr(path, ':');
    if (colon == NULL)
      pathlen = strlen (path);
    else
      pathlen = colon - path;
	
    progpath = g_new0 (gchar, pathlen + strlen (prog) + 2);
    strncat (progpath, path, pathlen);
    strcat (progpath, "/");
    strcat (progpath, prog);

    /* FIXME: should maybe do some taint checking here */
    if (stat (progpath, &buf) == 0)
      break;
    else if (colon == NULL){
      fprintf (stderr, "Can't find %s, bailing out\n", &progpath[1]);
      exit (1);
    }
    path = ++colon;
  }
  
  return progpath;
}

static void cb_infanticide (GtkWidget *w, gpointer data)
{
  static int severity[3] = {SIGINT, SIGTERM, SIGKILL};
  int i, status;
  pid_t child_pid = GPOINTER_TO_INT(data);
  
  for (i = 0; i < 3; i++){
    gint rv;
    /* get progressively harsher */
    if (debug_level)
      g_print ("Sending signal %d to pid %d\n", severity[i], child_pid);
    kill (child_pid, severity[i]);

    /* doesn't respond in 2 seconds, try again */
    /* FIXME: probably not the best policy, though */
    sleep (2);
    
    if ((rv = waitpid (child_pid, &status, WNOHANG))){
      if (debug_level)
	g_print ("waitpid on %d returned %d, status %d\n", rv, child_pid,
		 status);
      break;
    } 
  }
}

static ChildRec *child_rec_new (pid_t pid, int fd, GdkInputFunction
				callback, gpointer callback_data,
				GdkInputCondition cond, GtkWidget
				*toplevel)
{
  ChildRec *child = g_new (ChildRec, 1);
  
  child->pipefd = fd;
  if (callback)
    child->tag = gdk_input_add (fd, cond, callback,
				callback_data);
  child->toplevel = toplevel;
  child->exit_jobs = NULL;
  g_hash_table_insert (children, GINT_TO_POINTER(pid), child);

  return child;
}

void cb_childpipe (gpointer data, gint childfd, GdkInputCondition
		   condition)
{
  pid_t child_pid;
  int exit_status;
  ChildRec *child;

  read (childfd, &child_pid, sizeof (pid_t));
  read (childfd, &exit_status, sizeof (int));
  
  child = g_hash_table_lookup (children, GINT_TO_POINTER(child_pid));

  if (child == NULL){
    fprintf (stderr, "Child's pid %d is invalid\n", child_pid);
    return;
  }

  if (child->pipefd >= 0)
    close (child->pipefd);
  gdk_input_remove (child->tag);
  if (child->toplevel)
    gtk_widget_destroy (child->toplevel);
  g_hash_table_remove (children, GINT_TO_POINTER(child_pid));

  child->exit_status = exit_status;

  while (child->exit_jobs) {
    ExitJob *exit_job = (ExitJob *) child->exit_jobs->data;
    
    exit_job->proc (child, exit_job->data1, exit_job->data2);
    child->exit_jobs = g_slist_remove (child->exit_jobs, exit_job);
    g_free (exit_job);
  }
  /* last job done, free the ChildRec */
  g_free (child);
}

void on_child_exit (ChildRec *child, ExitJobProc proc, gpointer data1, 
		    gpointer data2)
{
  ExitJob *job = g_new (ExitJob, 1);

  job->proc = proc;
  job->data1 = data1;
  job->data2 = data2;
  
  child->exit_jobs = g_slist_append (child->exit_jobs, job);
}

gint init_external (void)
{
  children = g_hash_table_new (g_direct_hash, NULL);
  signal (SIGCHLD, reaper);
  
  if (pipe (childpipe)){
    perror ("Can't pipe");
    abort();
  }

  /* always active */
  gdk_input_add (childpipe[0], GDK_INPUT_READ,
		 (GdkInputFunction) cb_childpipe,
		 NULL);
  
  return childpipe[0];
}

void cb_type1inst (GtkWidget *progress, gint logfd, GdkInputCondition 
		   condition)
{
  gchar line[81], *index;
  gint nbytes;
  gint nfonts = GPOINTER_TO_INT (gtk_object_get_data
				 (GTK_OBJECT(progress), "nfonts"));
  gint ndone = GPOINTER_TO_INT (gtk_object_get_data
				(GTK_OBJECT(progress), "ndone"));

  nbytes = read (logfd, line, 80);
  line[nbytes] = '\0';

  if (line[0] == '['){
    ndone = atoi (&line[1]);
    if (debug_level)
      g_print ("%d%% done\n", ndone * 100 / nfonts);
    gtk_progress_bar_update (GTK_PROGRESS_BAR(progress), (gfloat)
			     ndone / nfonts);
    gtk_object_set_data (GTK_OBJECT(progress), "ndone",
			 GINT_TO_POINTER(ndone));
  }
  else if (line[0] == '-'){
    if (debug_level)
      g_print ("All done!\n");
    gtk_progress_bar_update (GTK_PROGRESS_BAR(progress), 1.0);
  }
  else if (strncmp ("There are", line, 9) == 0){
    index = line + strcspn (line, "01234567890");

    if (index != line){
      nfonts = atoi (index);
      ndone = 0;

      gtk_object_set_data (GTK_OBJECT(progress), "nfonts",
			   GINT_TO_POINTER(nfonts));
      gtk_object_set_data (GTK_OBJECT(progress), "ndone",
			   GINT_TO_POINTER(ndone));
      if (debug_level)
	g_print ("There are %d fonts\n", nfonts);
    }
  }
}

/* FIXME: should write a "spawn" function */

ChildRec *type1inst (gchar *directory)
{
  GtkWidget *window, *progress, *button;
  
  /* this shouldn't happen */
  g_return_val_if_fail (chdir(directory) != -1, NULL);
  
  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_container_border_width (GTK_CONTAINER(window), 5);
  
  {
    GtkWidget *vbox, *label;
    gchar message[strlen(directory) + 22];
    
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_border_width (GTK_CONTAINER(vbox), 5);
    
    sprintf (message, "Running type1inst in %s", directory);
    label = gtk_label_new (message);

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);

    progress = gtk_progress_bar_new();
    gtk_widget_show (progress);
    gtk_box_pack_start (GTK_BOX(vbox), progress, FALSE, FALSE, 0);

    button = gtk_button_new_with_label ("Cancel");
    gtk_widget_show (button);
    gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 0);    

    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER(window), vbox);
  }
  
  gtk_widget_show (window);
  
  {
    /* back to using pipes, thanks to nifty perl command-line hack */
    int fd[2], pid;
    ChildRec *child;      

    if (pipe(fd)){
      perror ("Can't open pipe");
      return NULL;
    }

    pid = fork();
    
    if (pid){
      /* parent */
      if (pid < 0){
	perror ("Can't fork");
	return NULL;
      }
    }
    else {
      /* child */
      gchar *script, *perlpath, *t1instpath;
      
      close (fd[0]);
      dup2 (fd[1], STDOUT_FILENO);

      /* find perl and type1inst */
      perlpath = find_in_path ("perl");
      t1instpath = find_in_path ("type1inst");
      
      script = g_new (gchar, strlen (t1instpath) + strlen
			   ("$|=1;do \"\"") + 1);

      strcpy (script, "$|=1;do \"");
      strcat (script, t1instpath);
      strcat (script, "\"");
      
      execl (perlpath, perlpath, "-e", script, "--", "-nogs",
	      "-nolog", NULL);
      
      /* shouldn't get here */
      perror ("Can't exec type1inst");
      exit (1);
    }
    /* parent */
    
    close (fd[1]);
    
    child = child_rec_new (pid, fd[0], (GdkInputFunction) cb_type1inst,
			   progress, GDK_INPUT_READ, window);

    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_infanticide),
			GINT_TO_POINTER(pid));
    return child;
  }
  
  return NULL;
}

/* this is rather lousy */
#define PS_HEADER "\
%%!Adobe-PS
%%%%DocumentSuppliedResources font /%s
%%%%BeginResource: font /%s"

/* FIXME: well, you know, fix me... */
#define PS_SAMPLE "\
%%EndResource
/Courier-Medium findfont 18 scalefont setfont
50 660 moveto
(%s) show
50 640 moveto
(%s) show
/%s findfont 36 scalefont setfont
25 500 moveto
(The quick brown fox) show
25 450 moveto
(jumps over the lazy) show
25 400 moveto
(dog...) show
/%s findfont 60 scalefont setfont
25 330 moveto
(60 point) show
/%s findfont 48 scalefont setfont
25 270 moveto
(48 point) show
/%s findfont 24 scalefont setfont
25 240 moveto
(24 point) show
/%s findfont 18 scalefont setfont
25 210 moveto
(18 point) show
showpage\n"

/* sort of primitive - should really look inside the file */
static gint is_pfb (gchar *name)
{
  gint len;

  len = strlen (name);
  
  return ((name[len-1] == 'b') &&
	  (name[len-2] == 'f') &&
	  (name[len-3] == 'p') &&
	  (name[len-4] == '.'));
}

static void cb_sample_page (ChildRec *child, gchar *tfile)
{
  unlink (tfile);
  free (tfile);
}

void sample_sheet (gchar *cmd, gchar *arg, gchar *font, gchar *fontname)
{
  gchar *cmdpath, *pfbtopspath;
  int pfbtopsfd[2];
  pid_t pfbtopspid, cmdpid;
  FILE *infont, *outsample;
  gchar buf[256];
  gchar *tfile;
  ChildRec *child;

  cmdpath = find_in_path (cmd);
  pfbtopspath = find_in_path ("pfbtops");

  if (!(cmdpath && pfbtopspath)){
    fprintf (stderr, "Can't find %s or pfbtops in path\n", cmd);
    return;
  }

  /* totally random name */
  tfile = tempnam (NULL, NULL);
  outsample = fopen (tfile, "a");

  if (outsample == NULL){
    perror ("Can't open temporary file");
    return;
  }
  
  fprintf (outsample, PS_HEADER, fontname, fontname);

  /* if it's a .pfb font, convert it first */
  if (is_pfb (font)){
    
    if (pipe (pfbtopsfd)){
      perror ("Can't open pipe");
      return;
    }
    
    pfbtopspid = fork();
    if (pfbtopspid){
      /* parent */
      if (pfbtopspid < 0){
	perror ("Can't fork");
	return;
      }
    }
    else {
      /* child */
      close (pfbtopsfd[0]);
      dup2 (pfbtopsfd[1], STDOUT_FILENO);
      execl (pfbtopspath, pfbtopspath, font, NULL);
      exit (1);
    }
    /* parent */
    close (pfbtopsfd[1]);
    infont = fdopen (pfbtopsfd[0], "r");
    child_rec_new (pfbtopspid, pfbtopsfd[0], NULL,
		   NULL, GDK_INPUT_READ, NULL);

    if (infont == NULL){
      perror ("Can't reopen pipe for reading");
      return;
    }
  }
  else {
    /* just assume it's a PFA file, then */
    infont = fopen (font, "r");
    if (infont == NULL){
      perror ("Can't reopen font file for reading");
      return;
    }
  }

  /* slurp the font into the document */
  while (fgets (buf, 255, infont)){
    fputs (buf, outsample);
  }
  
  fclose (infont);

  /* lousy sample page - yes, this is a ridiculous invocation of
     fprintf, too... */
  fprintf (outsample, PS_SAMPLE, fontname, font,
	   fontname, fontname, fontname, fontname, fontname);
  fclose (outsample);

  /* now, call the command */
  cmdpid = fork();
  
  if (cmdpid){
    /* parent */
    if (cmdpid < 0){
      perror ("Can't fork");
      return;
    }
  }
  else {
    /* child */
    if (arg)
      execl (cmdpath, cmdpath, arg, tfile, NULL);
    else
      execl (cmdpath, cmdpath, tfile, NULL);
    /* shouldn't be here */
    exit (1);
  }

  /* cb_sample_page will unlink the tempfile */
  child = child_rec_new (cmdpid, -1, NULL,
			 NULL, GDK_INPUT_READ, NULL);
  on_child_exit (child, (ExitJobProc) cb_sample_page, tfile, NULL);
}
