/* GGlyph - a graphical utility for managing one's collection of Type
   1 fonts (and eventually, other sorts of fonts) under X11.

   Copyright (c) 1998 David Huggins-Daines
   <bn711@freenet.carleton.ca>.

   You are permitted to copy, distribute, and modify this file under the
   terms of the GNU General Public License, version 2, or any later
   version.  See the file COPYING for more details. */

/* fonts.c - methods for manipulating the database of fonts and font
   paths, as well as the X11 font path */

#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <dirent.h>
#include <ctype.h>
#include <gdk/gdkx.h>
#include "fonts.h"
#include "external.h"
#include "gdk_t1lib.h"

extern gint debug_level;
extern gchar *viewer_cmd, *lp_arg;

gint is_psfont (gchar *pathname)
{
  gint len;
  
  /* just assume that it is if it has a .pf[ab] extension */
  len = strlen (pathname);
  if (pathname[len-4] == '.' &&
      pathname[len-3] == 'p' &&
      pathname[len-2] == 'f' &&
      (pathname[len-1] == 'a' || 
       pathname[len-1] == 'b'))
    return TRUE;
  /* also assume that it isn't if it has a .bdf, .pcf, .spd, or .gsf
     extension.  These last ones will totally fuck up t1lib even
     though they look like normal type 1 fonts at first glance - it
     should check for this (I'll patch it to do so) */
  else if (strstr (pathname, ".pcf") ||
	   strstr (pathname, ".bdf") ||
	   strstr (pathname, ".gsf") ||
	   strstr (pathname, ".spd"))
    return FALSE;
  /* other cases - hopefully this isn't too slow.  note that this
     duplicates some functionality of type1inst */
  else {
    FILE *fontfile;
    
    fontfile = fopen (pathname, "r");
    if (fontfile == NULL){
      return FALSE;
    }
    
    /* first line should have the magic message */
    {
      gchar line[81];
      gint pos;
      
      if (fgets (line, 80, fontfile) == NULL)
	return FALSE;
      /* fix zeros (to make binary fonts recognizable) */
      /* FIXME: this is slightly wrong (inefficient) */
      for (pos = 0; line[pos] != '\n' && pos < 80; pos++)
	if (line[pos] == '\0')
	  line[pos] = '-';

      /* some broken freeware fonts have a different magic stringy */
      if (strstr (line, "%!PS-AdobeFont-1.0") ||
	  strstr (line, "%!FontType1-1.0")){
	fclose (fontfile);
	return TRUE;
      }
    }
    fclose (fontfile);
    return FALSE;  
  }
}

/* note that this doesn't actually copy the file itself, just the
   struct (and fixes the name) */
void copy_ffile_rec (FontDirRec *new_fdir, FontFileRec *ffile)
{
  /* it could already be there */
  if (g_hash_table_lookup (new_fdir->fonts, ffile->basename))
    return;
  
  {
    FontFileRec *new_ffile = g_new (FontFileRec, 1);
    gint pathlen = strlen (new_fdir->path);
    
    new_ffile->pathname = g_new (gchar, pathlen + strlen 
				 (ffile->basename) + 2);
    
    strcpy (new_ffile->pathname, new_fdir->path);
    strcat (new_ffile->pathname, "/");
    strcat (new_ffile->pathname, ffile->basename);
    
    new_ffile->basename = new_ffile->pathname + pathlen +
      sizeof(gchar);

    if (debug_level)
      g_print ("copy_ffile_rec: Copied %s to %s - %s\n", ffile->basename,
	       new_ffile->pathname, new_ffile->basename);

    if (ffile->xlfd)
      new_ffile->xlfd = strdup (ffile->xlfd);
    else
      new_ffile->xlfd = NULL;
    
    new_fdir->dirty = TRUE;

    /* the important part */
    new_ffile->t1_fontid = ffile->t1_fontid;

    g_hash_table_insert (new_fdir->fonts, new_ffile->basename,
			 new_ffile);
  }
}

FontFileRec *add_fontfile (FontDirRec *fontdir, gchar *pathname, gchar
			   *xlfd)
{
  FontFileRec *fontrec = NULL;
  gchar *basename = strrchr (pathname, '/') + sizeof (gchar);

  fontrec = g_hash_table_lookup (fontdir->fonts,
				 basename);
  if (fontrec) {
    if (debug_level)
      g_print ("add_fontfile: Font %s already in dir, updating info\n",
	       fontrec->basename);
      
    if (xlfd){
      if (fontrec->xlfd)
	g_free (fontrec->xlfd);
      fontrec->xlfd = strdup (xlfd);
    }
    return fontrec;
  }

  fontrec = g_new (FontFileRec, 1);
  fontrec->pathname = strdup (pathname);
  fontrec->basename = fontrec->pathname + (basename - pathname);
  if (xlfd)
    fontrec->xlfd = strdup (xlfd);
  else
    fontrec->xlfd = xlfd;
  
  fontrec->t1_fontid = NOT_LOADED;

  /* FIXME: should probably hash on full pathnames (since length is
     relatively unimportant).  Actually, this is a big mess in many
     other ways */
  g_hash_table_insert (fontdir->fonts, fontrec->basename, fontrec);

  if (debug_level){
    g_print ("add_fontfile: %p{%s} = %p\n", fontdir->fonts,
	     fontrec->basename, fontrec);
  }

  return fontrec;
}

void rescan_fontdir (FontDirRec *fdir)
{
  FILE *fontsdir;
  gchar fdpath[strlen(fdir->path) + 13];

  fdir->problems = NO_PROBLEMO;
  strcpy (fdpath, fdir->path);
  strcat (fdpath, "/fonts.scale");
    
  fontsdir = fopen (fdpath, "r");
  if (fontsdir == NULL){
    /* no fonts.scale, then try fonts.dir...  Debian 'freefont' and
       'sharefont' packages don't seem to have fonts.scale, yet X11
       works fine, wtf...  at least this prefers fonts.scale... */
    fdir->problems |= NO_FONTSSCALE;
    strcpy (fdpath, fdir->path);
    strcat (fdpath, "/fonts.dir");
    fontsdir = fopen (fdpath, "r");
  }
  
  /* Add all X-visible fonts first */
  if (fontsdir == NULL){
    fdir->problems |= NO_FONTSDIR;
  }
  else {
    /* auto variables == Scheme envy */
    gchar line[256];
    gint fc;

    fgets (line, 255, fontsdir);
    fc = atoi (line);

    while (fc && fgets (line, 255, fontsdir)){
      gint start, fnlen;

      start = strspn (line, WHITESPACE);
      fnlen = strcspn (line + start, WHITESPACE);

      {
	gint pathlen = strlen (fdir->path);
	gchar fontpath[pathlen + fnlen + 1];
	struct stat buf;

	strcpy (fontpath, fdir->path);
	strcat (fontpath, "/");
	strncat (fontpath, line + start, fnlen);
	
	/* check that it exists first  */
	if (stat (fontpath, &buf) == -1){
	  if (errno == ENOENT){
	    /* mark directory as "dirty" */
	    fdir->problems |= FONTS_NOT_INSTALLED;
	  }
	  else {
	    perror (fontpath);
	    return;
	  }
	}

	if (is_psfont (fontpath)){
	  gchar *xlfd;

	  xlfd = line + start + fnlen;
	  xlfd += strspn (xlfd, WHITESPACE);

	  if (debug_level)
	    g_print ("rescan_fontdir: Adding %s to %s\n", fontpath,
		     fdir->path);
	  
	  add_fontfile (fdir, fontpath, xlfd);
	}
	fc--;
      }
    }

    fclose (fontsdir);
  }

  /* now catch any remaining fonts (this might pick up some junk, but
     oh well - type1inst does a good job of sifting it out.) */
  {
    /* shell envy */
    DIR *dir;
    struct dirent *dent;

    dir = opendir (fdir->path);

    if (dir == NULL){
      perror (fdir->path);
      return;
    }

    if (debug_level)
      g_print ("chdir to %s\n", fdir->path);
    
    if (chdir (fdir->path) == -1){
      perror (fdir->path);
      return;
    }

    while ((dent = readdir (dir))){
      /* this is getting to be a recurring theme */
      struct stat buf;

      stat (dent->d_name, &buf);
      if (S_ISLNK (buf.st_mode) | S_ISREG (buf.st_mode)){
	/* hash table exploitation == Perl envy */
	FontFileRec *ffile = g_hash_table_lookup
	  (fdir->fonts, dent->d_name);

	if (ffile == NULL &&
	    is_psfont (dent->d_name)) {
	  gchar fontpath[strlen (fdir->path) + strlen (dent->d_name) + 2];

	  strcpy (fontpath, fdir->path);
	  strcat (fontpath, "/");
	  strcat (fontpath, dent->d_name);

	  if (debug_level)
	    g_print ("rescan_fontdir: adding invisible basename %s to %s\n",
		     dent->d_name, fdir->path);
	  
	  /* NULL indicates that it isn't visible to X (yet) */
	  add_fontfile (fdir, fontpath, NULL);
	  fdir->problems |= FONTS_NOT_INSTALLED;
	}
      }
    }
    closedir (dir);
  }
}

FontDirRec *add_fontdir (GSList **fdir_list, gchar *path)
{
  FontDirRec *fontdir = NULL;

  fontdir = g_new (FontDirRec, 1);
  fontdir->fonts = g_hash_table_new (g_str_hash, g_str_equal);

  *fdir_list = g_slist_append (*fdir_list, fontdir);

  fontdir->path = strdup(path);
  /* FIXME: this is illogical - I should merge the update_ and fill_
     functions, and then this can start out as TRUE */
  fontdir->dirty = FALSE;
  rescan_fontdir (fontdir);

  return fontdir;
}

/* returns a normalized value for a field of an XLFD */
/* the fact that these are all malloc'ed strings is somewhat wasteful, 
   since many of them will be totally identical */
gchar *get_xlfd_field (gchar *xlfd, gint field)
{
  gchar *index, *fieldstr, *dash;

  /* skip leading dash */
  index = xlfd + 1;
  
  while (field--){
    dash = strchr (index, '-');
    
    if (dash)
      index = ++dash;
    else
      return NULL;
  }
  
  dash = strchr (index, '-');
  
  {
    gint len = dash ? (dash - index) : strlen (index);
    
    fieldstr = g_new (gchar, len + 1);
    fieldstr[0] = '\0';
    strncat (fieldstr, index, len);

    /* first letter uppercase */
    if (fieldstr[0]){
      fieldstr[0] = toupper (fieldstr[0]);
    }
  }
  
  return fieldstr;
}

/* turns an XLFD slant character into something
   human-understandable...*/
gchar *get_xlfd_slant (gchar *xlfd)
{
  gint i;
  gchar *index, *dash;

  index = xlfd;
  /* slant is field 4 */
  for (i = 0; i < 4; i++){
    dash = strchr (index, '-');
    
    if (dash)
      index = ++dash;
    else
      return NULL;
  }

  switch (*index){
  case 'r':
    return "Roman";
    break;
  case 'i':
    return "Italic";
    break;
  case 'o':
    return "Oblique";
    break;
  default:
    return "Unknown";
  }
}


gint is_xlfd_monospaced (gchar *xlfd)
{
  gint i;
  gchar *index, *dash;

  index = xlfd;
  /* spacing is field 10 */
  for (i = 0; i < 11; i++){
    dash = strchr (index, '-');
    
    if (dash)
      index = ++dash;
    else
      return FALSE;
  }

  return !(*index == 'p');
}

void flush_XFontPath (void)
{
  char **paths;
  gint npaths;

  /* get the X font path */
  paths = XGetFontPath (GDK_DISPLAY(), &npaths);
  /* set the X font path (flushes the cache, etc...) */
  XSetFontPath (GDK_DISPLAY(), paths, npaths);
  XFreeFontPath (paths);
}

void add_to_XFontPath (gchar *dir)
{
  char **paths;
  gint i, len, npaths;

  /* get the X font path */
  paths = XGetFontPath (GDK_DISPLAY(), &npaths);

  /* make darn sure that dir/fonts.dir exists and has non-zero size
     (that's all we can reasonably check for) */
  {
    gchar fdir[strlen(dir) + 11];
    struct stat buf;

    strcpy (fdir, dir);
    strcat (fdir, "/fonts.dir");

    if (stat (fdir, &buf) == -1){
      perror ("Can't stat fonts.dir");
      return;
    }
    else if (buf.st_size == 0){
      fprintf (stderr, "%s is empty, you moron!\n", fdir);
      return;
    }
  }

  /* now make sure the directory isn't already in the path
     (disregarding final slashes! unlike xset!!) */
  
  len = strlen (dir);
  if (dir[len-1] == '/')
    len--;
  
  for (i = 0; i < npaths; i++){
    if (strncmp (paths[i], dir, len) == 0){
      if (debug_level)
	g_print ("add_to_XFontPath: %s is already in path\n", dir);
      XFreeFontPath (paths);
      return;
    }
  }

  paths = (char **) realloc (paths, (npaths + 1) * sizeof (char **));
  paths[npaths] = g_malloc (len);
  strcpy (paths[npaths], dir);
  npaths++;
  XSetFontPath (GDK_DISPLAY(), paths, npaths);
  XFreeFontPath (paths);
}

void remove_from_XFontPath (gchar *dir)
{
  char **paths;
  gint i, npaths, len;
  
  /* get the X font path */
  paths = XGetFontPath (GDK_DISPLAY(), &npaths);

  if (npaths == 1){
    /* FIXME: should alert the user that this is not a good idea */
    XFreeFontPath (paths);
    return;
  }

  /* disregard final slash, unlike xset... */
  len = strlen (dir);
  if (dir[len-1] == '/')
    len--;

  for (i = 0; i < npaths; i++)
    if (strncmp (paths[i], dir, len) == 0)
      break;

  if (i == npaths){
    fprintf (stderr, "%s not in X font path!\n", dir);
    return;
  }

  /* hmm.  Apparently, one should not free the strings themselves...
     I won't know for sure until I get my Debian 2.0 source CD and can 
     examine what it is that XGetFontPath actually returns */
  
/*   free (paths[i]); */

  if (i < npaths - 1)
    memmove (&paths[i], &paths[i+1], (npaths - i - 1) * sizeof (char **));
  
  paths = (char **) realloc (paths, (npaths - 1) * sizeof (char **));
  npaths--;
  
  XSetFontPath (GDK_DISPLAY(), paths, npaths);
  XFreeFontPath (paths);
}

void do_sample_sheet (FontFileRec *ffile, SampleAction act)
{
  /* something is seriously wrong, so die now */
  g_assert (T1_LoadFont (ffile->t1_fontid) != -1);
  
  switch (act){
  case VIEW_SAMPLE:
    sample_sheet (viewer_cmd, NULL, ffile->pathname, T1_GetFontName
		  (ffile->t1_fontid)); 
    break;
  case PRINT_SAMPLE:
    sample_sheet ("lpr", lp_arg, ffile->pathname, T1_GetFontName
		  (ffile->t1_fontid)); 
    break;
  }
}

