/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     cutarea.c                                                      *
 *            Funktions to cut rectangular areas from an image               *
 * Autor:     Andreas Tille                                                  *
 * Datum:     27.02.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

PICTURE *CreatePictureFromSample(PICTURE *sample)
/* Erzeugt Bildstruktur mit Kopie der Vorlage mit Ausnahme der Bilddaten
 * --- Parameter: ---
 * PICTURE *sample                    : Vorlage
 * --- R"uckgabe: ---
 * PICTURE *CreatePictureFromSample() : neues Bild
 */
{
   PICTURE *zw;
   CHUNK   *sp, *zp;
   int      nchunks;
   
   if ( sample->id != PAUL_ID ) {
      g_warning("Invalid PICTURE structure in file \"%s\" line %i", __FILE__, __LINE__);
      return NULL;
   }
   
   assert ( (zw = malloc(sizeof(PICTURE))) != NULL );

   zw->id       = sample->id;
   zw->im       = sample->im;
   zw->W        = sample->W;
   zw->H        = sample->H;
   /*   zw->DATA  soll nicht kopiert werden !!!!  */
   if (sample->file) assert ( (zw->file = strdup(sample->file)) );
   if (sample->ext)  assert ( (zw->ext  = strdup(sample->ext)) );
   else zw->ext = NULL;
   if (sample->dir) assert ( (zw->dir      = strdup(sample->dir)) );
   else zw->dir = NULL;
   zw->size     = sample->size;   
   zw->spp      = sample->spp;
   zw->storepix = sample->storepix;
   zw->trans    = sample->trans;
   zw->res      = sample->res;
   zw->x_offset = sample->x_offset;
   zw->y_offset = sample->y_offset;
   if ( (zw->n_gamma = sample->n_gamma) && sample->gamma ) { 
      assert ( (zw->gamma = malloc(zw->n_gamma*sizeof(png_color))) );
      memcpy(zw->gamma, sample->gamma, zw->n_gamma*sizeof(png_color));
   } else 
      zw->gamma = NULL;
   zw->his      = NULL;   /* wenn Daten nicht bernommen werden ist Histogramm ungltig */
   zw->zeit     = sample->zeit;
   for ( sp = sample->spec, nchunks = 1; sp->key != NULL; sp++ ) nchunks++;
   assert ( (zw->spec = malloc(nchunks*sizeof(CHUNK))) );
   for ( sp = sample->spec, zp = zw->spec; (zp->key = sp->key) != NULL; sp++, zp++ ) {
      if ( !sp->value ) zp->value = NULL;
      else              assert ( (zp->value = strdup(sp->value)) );
   }
   zw->roff_x   = sample->roff_x;
   zw->roff_y   = sample->roff_y;
   zw->phys     = sample->phys;
   if (sample->physname) { assert ( (zw->physname = strdup(sample->physname)) ); }
   else zw->physname = NULL;
   zw->flag     = sample->flag;
   zw->label    = sample->label;
   /* at least when more than one file is created (extrema) this will fail!!!!!!!!!!!!!!!! */
   return zw;
}


PICTURE *CutArea(PICTURE *bild, BOX *cut, long flag, char *was, char *ext, int del_old)
/* Ausschneiden eines Gebietes aus einem Bild
 * Dabei wird das Originalbild berschrieben.
 * --- Parameter: ---
 * PICTURE *bild     : Bildstrukturen
 * BOX     *cut      : auszuschneidende BOX
 * int      flag     : flag fr gewhlten Ausschnitt oder Extrema
 * char    *was      : Art des Auschneidens (cut, min, max)
 * char    *ext      : Erweiterung (cut, min, max)
 * int      del_old  : flag, whether the old image is to be freed or not
 * --- R"uckgabe: ---
 * PICTURE *bild     : Bildstrukturen mit Bildausschnitten
 * int CutArea()     : -1 falls Fehler, sonst 0
 */
{
#define NAMSIZ  200
   PICTURE                *zw;
   char                    name[NAMSIZ];
   register unsigned char *ap, *bp, *fap;
   register int            storepix;

   if ( CheckInBounds(bild, cut) < 1 ) return NULL;
      /* Bilddimensionen berschritten ... */

   if ( SaveExtrema(flag) ) { 
             /* Create a shell script which cuts images according to the   *
              * extrema of this image.                                     */
      FILE *fp;

      strcpy(name, bild->file);
      strcat(name, ext + 1);
      strcat(name, ".sh");
      Backup(name);
      assert ( (fp = fopen(name, "wt")) != NULL );
      fprintf(fp, "#!/bin/sh\n# Ausschneiden aller Bild-Dateien nach dem %s\n", was);
      fprintf(fp, "%s -c%ix%i+%i+%i $@\n", 
                   exename, cut->x2-cut->x1, cut->y2-cut->y1, cut->x1, cut->y1);
      fclose(fp);
      chmod(name,  S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
   }  

   zw       = CreatePictureFromSample(bild);
   zw->im   = NULL;
   zw->size = (zw->W = cut->x2 - cut->x1) * (zw->H = cut->y2 - cut->y1);
   assert ( (zw->DATA = malloc((zw->size)*3)) != NULL );
   storepix = bild->storepix;

   for ( fap = (ap = bild->DATA + storepix*(cut->y1*bild->W + cut->x1)) + storepix*(zw->H)*(bild->W),
         bp  = zw->DATA; ap < fap; ap += storepix*bild->W, bp += storepix*zw->W )
      memcpy(bp, ap, zw->storepix*zw->W);

   memset(name, 0, NAMSIZ);
   strcpy(name, was);
   sprintf(name+strlen(name), ": {%i,%i}{%i,%i}", cut->x1, cut->y1, cut->x2, cut->y2);
   CopySpec(zw->spec, ChunkNameDescription, name);
   CopySpec(zw->spec, ChunkNameTyp, TypAusschnitt);
   zw->flag |= SAFEFLAG; /*there is no cut flag. other solution: check for p->opt->cut when saving*/
   NewFileName(zw, ext);
   if ( del_old ) {
      GList *zl = g_list_append(NULL, bild);
      FreeBild(zl);
      g_list_free(zl);
   }
   
   return zw;
#undef NAMSIZ
}

int CutAreas(GList *piclist, OPTIONS *p)
/* cut box from set of images, replace the old images by the cuts
 * --- Parameter: ---
 * GList   *piclist   : list of images
 * OPTIONS *p         : cut: coordinates of box to cut from image
 * --- R"uckgabe: ---
 * int      CutAreas(): 0 if OK, -1 otherwise
 */
{
   char     buf[128];
   PICTURE *bild, *zw;
   GList   *pl;

   if ( !NBILDER(piclist) ) return 0;

   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      strcpy(buf, "Section from");
      if ( bild->file ) strcat(buf, bild->file);
      if ( (zw = CutArea(bild, p->cut, p->f, buf, APPCUT, 1)) == NULL ) {
         g_warning("%s wasn't cut due to wrong box coordinates", bild->file);
         continue;
      }
      pl->data = zw;
   }
   return 0;
}





