#/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     delbordr.c                                                     *
 *            Funktion zum Ausschneiden des eigentlichen Bildes aus einem    *
 *            Bild mit "dunklem" Rand                                        *
 * Autor:     Andreas Tille                                                  *
 * Datum:     27.02.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

typedef void (*CountThreshFunc)(PICTURE *, int, int, int *, int *, int, int);

static int TestLeftRight(int *lb, int *rb, int width, int pa, int pleft, int stp, int *found)
/* ermittelt, ob left und ap "beiderseits" width liegen
 * --- Parameter: ---
 * int           *lb    : Feld der linken Grenzen
 * int           *rb    : Feld der rechten Grenzen
 * int            width : Breite des Bildes
 * int            pa    : aktuelle Position
 * int            pleft : Begin des kritischen Randbereiches
 * int            stp   : Als 3 oder 1 Byte per Pixel gespeichert ?
 * int            found : Anzahl der bereits gefundenen Zeilen
 * --- R"uckgabe: ---
 * int           *lb    : neue linke Grenze eingetragen
 * int           *rb    : neue rechte Grenze eingetragen
 * int TestLeftRight()  : falls Randbereich erwischt, Offset bis zum 
 *                        voraussichtlich nchsten (halbe Bildbreite), sonst 0
 */
{
   register int i, j, storepix = stp;

   if ( storepix != 1 ) {
      pa    /= storepix;
      pleft /= storepix;
   }
   if ( (pa / width) > (pleft / width) ) {
      ++*found;
      i = (*(rb + *found) = pleft % width) -
          (*(lb + *found) = pa    % width);   
      j = (i=MIN(i >> 1, width >> 2)) % storepix;
      return i-j; /* sichern, da grnes Byte wieder getroffen wird! */
   }
   return 0;
}

static int WeightedMiddle(int *middle, int abw, int *data, int len)
/* Ermittelt den Mittelwert von Daten, die hchstens um einen gewissen Betrag vom
 * vorberechneten Mittelwert abweichen
 * --- Parameter: ---
 * int *middle : vorgegebener Mittelwert
 * int  abw    : alle Werte mit abs(*middle - Wert) < abw werden gemittelt
 * int *data   : Werte
 * int  len    : Anzahl der Werte
 * --- R"uckgabe: ---
 * int *middle : neuer, "gewichteter" Mittelwert
 * int WeightedMiddle(): != 0, falls zu wenig Werte einbezogen werden konnten
 *                       ( < 25% )
 */
{
   register int *ap, orimiddle = *middle, oriabw = abw, sum = 0, nsum = 0, *fip;
   
   for ( fip = (ap = data) + len; ap < fip; ap++ ) {
      if ( abs(*ap - orimiddle) < oriabw ) {
	 sum += *ap;
         nsum++;
      }
   }
   if ( nsum < (len >> 2) ) return nsum;
   *middle = sum / nsum;
   return 0;
}

static void CountThreshCol(PICTURE *bild, int col, int testcols, 
                    int *nthresh, int *nterr, int thresh, int terr)
/* Zhlt alle Werte der Umgebung einer Spalte die Stufe thresh und
 * Fehler terr berschreiten
 * --- Parameter: ---
 * PICTURE *bild       : Bild
 * int      col        : zu testende Spalte
 * int      testcols   : Anzahl der zu untersuchenden Spalten
 * int     *nthresh    : Feld fr Schwellwertauftreten
 * int     *nterr      : Feld fr Fehlerauftreten
 * unsigned char thresh: Schwellwert fr "dunkel"
 * unsigned char terr  : Toleranz fr einzelne Werte
 * --- Rueckgabe: ---
 * int     *nthresh    : Feld mit Schwellwertauftreten
 * int     *nterr      : Feld mit Fehlerauftreten
 * int CountThreshCol(): neue Spalte, im Fehlerfall -1
 */
{
   register int *ip, *jp, *fip, storepix = bild->storepix;
   register unsigned char *ap, *fap;
   int      step = bild->W;

   step = storepix*(step - testcols); 
        /* bild->storepix*testcols Byte werden in innerer Schleife behandelt! */
   
   for ( fip = (ip = nthresh) + testcols, jp = nterr; 
         ip < fip; ip++, jp++ ) 
      *ip = *jp = 0;

   for ( fap = bild->DATA + storepix*bild->size, 
         ap  = bild->DATA + storepix*(col - (testcols>>1)) + (storepix == 3 ? 1 : 0);  
                                                             /* +1 wegen GRN!! */
         ap < fap; ap += step ) {
      for ( fip = (ip = nthresh) + testcols, jp = nterr; 
            ip < fip; ip++, jp++, ap += storepix ) {         /* +3 wegen GRN!! */
	 if ( *ap > thresh ) {
            ++*ip;
            if ( *ap > terr ) ++*jp;
	 }
      }
   }
}

static void CountThreshRow(PICTURE *bild, int row, int testrows, 
                    int *nthresh, int *nterr, int thresh, int terr)
/* Zhlt alle Werte der Umgebung einer Zeile die Stufe thresh und
 * Fehler terr berschreiten
 * --- Parameter: ---
 * PICTURE *bild       : Bild
 * int      row        : zu testende Zeile
 * int      testrows   : Anzahl der zu untersuchenden Zeilen
 * int     *nthresh    : Feld fr Schwellwertauftreten
 * int     *nterr      : Feld fr Fehlerauftreten
 * unsigned char thresh: Schwellwert fr "dunkel"
 * unsigned char terr  : Toleranz fr einzelne Werte
 * --- Rueckgabe: ---
 * int     *nthresh    : Feld mit Schwellwertauftreten
 * int     *nterr      : Feld mit Fehlerauftreten
 * int CountThreshRow(): neue Zeile, im Fehlerfall -1
 */
{
   register int           *ip, *jp, storepix = bild->storepix;
   register unsigned char *bp, *fbp, *ap, *fap;
   int                    *fip;

   for ( fip = (ip = nthresh) + testrows, jp = nterr; 
         ip < fip; ip++, jp++ ) 
      *ip = *jp = 0;

   for ( ip = nthresh, jp = nterr, ap = bild->DATA + storepix*(row - (testrows>>1)),
         fap = bild->DATA + storepix*bild->size; 
         ip < fip; ip++, jp++ ) 

   ap = bild->DATA + storepix*(row - (testrows>>1)) + (storepix == 3 ? 1 : 0); /*+1 wegen GRN!!*/
   for ( fap = ap + storepix*testrows*bild->W, ip = nthresh, jp = nterr; 
         ap < fap; ap = fbp, ip++, jp++ ) {
      for ( fbp = (bp = ap) + storepix * bild->W; bp < fbp; bp += storepix ) { /*+3 wegen GRN!!*/
	 if ( *bp > thresh ) {
            ++*ip;
            if ( *bp > terr ) ++*jp;
	 }
      }
   }
}

static int CheckBorderColumn(PICTURE *bild, int *col, unsigned char thresh, 
                             unsigned char terr, int dir, int flag)
/* Testet die Umgebung einer Bezugsspalte auf "Dunkelheit"
 * FALLS bild->storepix == 3 WIRD NUR GRN GETESTET!!!!!!!!!!!!!!!!!!!
 * --- Parameter: ---
 * PICTURE *bild          : Bild
 * int     *col           : zu testende Spalte
 * unsigned char thresh   : Schwellwert fr "dunkel"
 * unsigned char terr     : Toleranz fr einzelne Werte
 * int      dir           : Richtung, in der Rand zu suchen ist 
 *                          (wenn *col = *left ist dir = -1)
 * int      flag          : wie oft wurde CheckBorderColumn() schon iterativ
 *                          aufgerufen
 * --- Rueckgabe: ---
 * int     *col           : gegebenenfalls neue Spalte, die Rand besser trifft
 * int CheckBorderColumn(): neue Spalte, im Fehlerfall -1
 */
{
#define TESTCOLS 11        /* wieviel Spalten auf einmal testen? */
#define NITER    10        /* wie tief rekursive Aufrufe?        */
   int                    nthresh[TESTCOLS], nterr[TESTCOLS], 
                          tolthr, tolerr, neucol;
   register int           *ip, *jp, *fip, i;

   if ( flag > NITER ) return -1;
      /* Nach mehr als NITER Iterationen konnte kein sinnvoller Rand gefunden werden! */

   if ( *col <  TESTCOLS>>1           ) *col = TESTCOLS>>1;             /* linken Rand exakt erwischen  */
   if ( *col >= bild->W-(TESTCOLS>>1) ) *col = bild->W-(TESTCOLS>>1)-1; /* rechten Rand exakt erwischen */
   CountThreshCol(bild, *col, TESTCOLS, nthresh, nterr, thresh, terr);

   /*   tolthr = bild->W >> 5;   1/32 darf drber liegen     */
   /*   tolerr = tolthr >> 3;    1/256 darf ber Fehler sein */
   tolthr = bild->W >> 4;  /* 1/16 darf drber liegen     */
   tolerr = tolthr >> 3;   /* 1/128 darf ber Fehler sein */
   if ( tolthr < 4 ) tolthr = 4;
   if ( tolerr < 1 ) tolerr = 1;
   /* ... damit es bei kleinen Bildern auch klappt!! */
    
   neucol = -1;
   for ( ip = nthresh, jp = nterr, 
         fip = nthresh + TESTCOLS, i = *col - (TESTCOLS>>1);
         ip < fip; ip++, jp++, i++ ) {
      if ( *ip < tolthr && *jp < tolerr ) {
         if ( dir > 0 ) {
            if ( ip == nthresh )  /* schon am Anfang nicht im Bild => gehe nach links */
               i = (neucol = TESTCOLS) + 1;
            else
               neucol = i;
            break;
	 } else {  /* linker Rand */
            neucol = i;
	 }
      }
   }

   if ( neucol == --i || neucol < 0 ) {
      neucol = (neucol < 0 ? *col + dir*(TESTCOLS-1) : *col - dir*(TESTCOLS-1));
      if ( CheckBorderColumn(bild, &neucol, thresh, terr, dir, ++flag) < 0 ) return -1;
   }

   return *col = neucol;
#undef TESTCOLS
#undef NITER
}

static int NewBorder(PICTURE *bild, int oldwd, int neuwd, int *left, int *right, 
                     unsigned char thresh, unsigned char terr, CountThreshFunc ctf)
/* Linker/oberer und rechter/unterer Rand eines Bildes werden noch etwas korrigiert,
 * damit bereinstimmung mit anderen Bildern erzielt wird
 * --- Parameter: ---
 * PICTURE         *bild       : Bild
 * int             oldwd       : alte Gesamtbreite/-hhe
 * int             neuwd       : Vorgabe der zu erreichenden Breite/Hhe
 * int             *left       : linke/obere Grenze
 * int             *right      : rechte(untere Grenze
 * unsigned char   thresh      : Schwellwert fr "dunkel"
 * unsigned char   terr        : Toleranz fr einzelne Werte
 * CountThreshFunc *ctf        : Funktion zur Berechnung der Anzahl von Fehlern in Spalten/Zeilen
 * --- Rueckgabe: ---
 * int             *left       : neue linke/obere Grenze
 * int             *right      : neue rechte/untere Grenze
 * int             NewBorder() : 0 fr OK
 */
{
#define NEXTCOL(bor, a, e, z)  { bor += z; a += z; e += z; } 

   register int  *al, *ar, *el, *er, zw;
   int      delta, *nthl, *ntel, *nthr, *nter, width;
   
   if ( (delta = abs(neuwd - *right + *left)) > (oldwd/20) || 
        *left >= *right || delta == 0 ) /* delta kann nicht == 0, aber Vorsicht */
      return delta;                     /* hat noch nie geschadet               */

   if ( (delta & 1) == 0 ) delta++;  /* ungeradzahlig geht besser! */

   /* delta+2, weil links und rechts von delta gesucht wird!!!! */
   assert ( (nthl = calloc((delta+2)<<2, sizeof(int))) != NULL );
   nter = (nthr = (ntel = nthl + delta) + delta) + delta;
   
   if ( (zw = *left - (delta/2) ) < 0 ) {
      zw = delta/2 + 1;
      al = nthl + *left;
      el = ntel + *left;
   } else {
      al = nthl + (delta/2) + 1;
      el = ntel + (delta/2) + 1;
   }
   (*ctf)(bild, zw, delta, nthl, ntel, thresh, terr);

   if ( (zw = *right + (delta>>1) ) > (width = oldwd) ) {
      zw = width - (delta>>1) - 1;
      ar = nthr + delta - width + *right;
      er = nter + delta - width + *right;
   } else {
      ar = nthr + (delta>>1) + 1;
      er = nter + (delta>>1) + 1;
   }
   (*ctf)(bild, zw, delta, nthr, nter, thresh, terr);

   if ( neuwd - *right + *left < 0 ) zw = -1;  /* Bild verkleinern */
   else                              zw = +1;  /* Bild vergrern  */
   
   width--;  /* Im Folgenden, mu mit width-1 verglichen werden! */
   do {
      if ( *al == *ar ) {
	 if ( *el <= *er ) { /* in beide Richtungen gleich ==>   *
                              * arbeite vorzugsweise oBdA. links */
	    if ( zw == +1 ) {
               if ( *left > 0 ) {
                  NEXTCOL(*left,  al, el, -1);
	       } else {
		  NEXTCOL(*right, ar, er, +1);
	       }
	    } else {
               NEXTCOL(*left, al, er, +1);
	    }
	 } else {  /* ( *el > *er ) */
            if ( zw == +1 ) {
               if ( *right < width ) {
                  NEXTCOL(*right, ar, er, +1);
               } else {
		  NEXTCOL(*left,  al, el, -1);
	       }
	    } else {
               NEXTCOL(*right, ar, er, -1);
	    }
	 }
      } else {
	 if ( *al < *ar ) {
	    if ( zw == +1 ) {
               if ( *left > 0 ) {
                  NEXTCOL(*left,  al, el, -1);
	       } else {
		  NEXTCOL(*right, ar, er, +1);
	       }
	    } else {
               NEXTCOL(*left, al, er, +1);
	    }
	 } else {  /* ( *al > *ar ) */
            if ( zw == +1 ) {
               if ( *right < width ) {
                  NEXTCOL(*right, ar, er, +1);
               } else {
		  NEXTCOL(*left,  al, el, -1);
	       }
	    } else {
               NEXTCOL(*right, ar, er, -1);
	    }
	 }
      }
   } while ( *right - *left != neuwd );
   free(nthl);
   return 0;

#undef NEXTCOL   
}


static int GetBorder(PICTURE *bild, int *left, int *right, unsigned char thresh, 
                   unsigned char terr)
/* Rechter und linker Rand eines Bildes werden abgeschnitten.
 * Als "Rand" werden "gleichmig dunkle" rechteckige Gebiete links und rechts
 * einer tatschlich Information enthaltenden Flche bezeichnet.
 * Dabei heit "dunkel", da die Pixel kleiner sind als die Schwelle thresh.
 * Dabei drfen "einzelne" Werte auch einen Wert bis zu terr erreichen.
 * --- Parameter: ---
 * PICTURE *bild        : Bild
 * int      *left       : Zeiger auf linke Grenze (erste Spalte innerhalb des Bildes)
 * int      *right      : Zeiger auf rechte Grenze (erste Spalte auerhalb des Bildes)
 * unsigned char thresh : Schwellwert fr "dunkel"
 * unsigned char terr   : Toleranz fr einzelne Werte
 * --- Rueckgabe: ---
 * int      *left       : linke Grenze
 * int      *right      : rechte Grenze
 * int      GetBorder() : 0 fr OK
 */
{
   register unsigned char *ap, *fip;
   register int flag = 0, bad = 0, *lbp, *rbp, storepix;
   unsigned char *tleft, *start = bild->DATA;
   int      width = bild->W, *lb, *rb, makesense, dobad, foundlines = 0, *fop;
   char     *nonsense = "Es kann kein sinnvoller Bildrand mit diesen Parametern gefunden werden (%s).\n";

   storepix = bild->storepix;
   *left    = *right = 0;
   tleft    = start;
   assert ( (lb = calloc(2*bild->H, sizeof(int))) != NULL );
   rb = lb + (makesense = bild->H);
   makesense *= 100;
   
   if ( storepix == 3 ) 
      ap = start + (width << 1) - (width << 1)%3 + 1;
      /* Anfang erst bei 2/3 => man erhlt nur einen dunklen Bereich, *
       * damit wird auch gleich das grne Byte getroffen              */
   else
      ap = start + (width << 1);

   dobad = (width > 1024 ? 1 : 0);
      /* Wenn die Bilder gro sind, ist auch die Auflsung der Fehler hher! *
       * Daher wird noch ein Pixel weiter geschaut                           */

   for ( fip = bild->DATA + storepix*(bild->size - dobad); 
         ap < fip; ap += storepix ) {
      if ( flag ) {             /* Verdacht auf Punkt im Randstreifen */
         if ( *ap >= thresh ) { /* Fehler im Rand oder Bildpunkt???   */
            if ( bad ) {        /* vorangegangener Punkt auch schon   */
	       if ( dobad )     /* Falls groes Bild, betrachte auch  */
                  if ( *(ap += storepix) < terr ) { /* ... nchsten Punkt noch    */
                     if ( *ap < thresh ) {
                        bad = 0;   /* War doch nix mit Bildanfang     */
                        continue;  /* Also weiter im Rand             */
		     }
		     if ( ++bad < 4 ) {  /* HIER DREHEN !!!!!!!!!!!!   */
                        continue;  /* mal sehen                       */
		     }
		  }

	       ap += (flag = TestLeftRight(lb, rb, width, ap-start, tleft-start, storepix, &foundlines));
	       if ( !flag )     /* Wenn Verdacht auf Rand nicht besttigt ... */
                  if ( !--makesense ) break;  /* merken bzw. Abbruch  */
	       flag = 0;        /* In jedem Fall jetzt wieder im Bild */
	       bad  = 0;        /* ... und kein Verdacht mehr         */
	    } else {            /* erster Punkt der Serie >= thresh   */
	       if ( *ap < terr ) bad = 1;   /* Knnte Fehler sein     */
	       else              flag = 0;  /* Ist doch im Bild       */
	    }
	 }
      } else {                  /* innerhalb des eigentlichen Bildes  */
         if ( *ap < thresh ) {
            flag  = 1;
            tleft = ap;
	 }
      }
   }
   if ( !makesense ) {
      fprintf(stderr, nonsense, "erster Scan fehlgeschlagen");
      free(lb);
      return -1;
   }
   if ( bild->H < 3*foundlines ) {
      flag = bad = 0;  /* Registervariablen ausnutzen zum Summen speichern. */   
      for ( fop = (lbp = lb) + foundlines, rbp = rb; lbp < rb; lbp++, rbp++ ) {
         flag += *lbp; 
         bad  += *rbp;
      }
      *left  = flag / foundlines;
      *right = bad  / foundlines;
      if ( !*right || (!*left && *right == width) ) {
         fprintf(stderr, nonsense, "Rand unsinnig");
         free(lb);
         return -1;
      }

      /* Noch einmal Durchschnitt ermitteln, aber Ausreier herauslassen! */
      flag = MIN(*left, width/20);
      flag = MIN(width-*right, flag);  /* alles auerhalb flag ist Abweichung */
      if ( flag < width/25 ) flag = width / 20;  /* Zahlen sind rein empirisch!!! */
   
      if ( WeightedMiddle(left,  flag, lb, foundlines) )
         WeightedMiddle(left,  flag <<= 1, lb, foundlines);
      if ( WeightedMiddle(right, flag, rb, foundlines) ) 
         WeightedMiddle(right, flag <<=1, rb, foundlines);

      flag >>= 1;  /* und noch einmal mit der halben Abweichung!  Das bringt was!!   *
                    * Ein Fehlertest ist nun aber nicht mehr ntig (eher schdlich!) */
      WeightedMiddle(left,  flag, lb, foundlines);
      WeightedMiddle(right, flag, rb, foundlines);
   } else {
      if ( width > 20 ) {
         *left  = 5;
	 *right = width - 5;
      } else {
         fprintf(stderr, nonsense, "zu wenig gltige Zeilen");
         free(lb);
         return -1;
      }
   }
   
   /* Nun noch die Randspalten testen */
   flag = *left; bad = *right;

   if ( CheckBorderColumn(bild, left, thresh, terr, -1, 0) < 0 ||
        CheckBorderColumn(bild, right, thresh, terr, 1, 0) < 0 ) 
      return -1;

   if ( *left >= *right ) {
      fprintf(stderr, nonsense, "senkrechter Feinscan");
      free(lb);
      *left = flag; *right = bad;
      return -1;
   }
   ++*left;    /* Linker Rand soll innerhalb des Bildes sein!! */
   free(lb);
   return 0;
}

int CreateCheckLine(PICTURE *bild, int *lb, int *rb)
/* erzeugt eine invertierte Linie an dem neu ermittelten Rand und
 * erweitert beide Rnder um fnf Spalten
 * --- Parameter: ---
 * PICTURE *bild        : Bild
 * int      *lb         : Zeiger auf linke Grenze
 * int      *rb         : Zeiger auf rechte Grenze
 * --- Rueckgabe: ---
 * int      *lb         : linke Grenze
 * int      *rb         : rechte Grenze
 * int CreateCheckLine(): neue Breite
 */
{
#define REST 10
   register unsigned char *lp, *rp, *fip;
   register int w = (bild->storepix)*bild->W;

   for ( fip = bild->DATA + bild->storepix * bild->size, 
         lp  = bild->DATA + bild->storepix * *lb,
         rp  = bild->DATA + bild->storepix * *rb;
         lp < fip; lp += w, rp += w ) {
      *lp     = ~(*lp);
      *(lp+1) = ~(*(lp+1));
      *(lp+2) = ~(*(lp+2));      
      *rp     = ~(*rp);
      *(rp+1) = ~(*(rp+1));
      *(rp+2) = ~(*(rp+2));      
   }
   if ( *lb > REST ) *lb -= REST;
   else              *lb  = 0;
   if ( *rb < bild->W - REST ) *rb += REST;
   else                        *rb  = bild->W;
   return *rb - *lb;
#undef REST
}

static int GetTopBottom(PICTURE *bild, int *top, int *bot, unsigned char thresh, 
                   unsigned char terr)
/* Oberer und unterer Rand eines Bildes werden abgeschnitten.
 * Als "Rand" werden "gleichmig dunkle" rechteckige Gebiete links und rechts
 * einer tatschlich Information enthaltenden Flche bezeichnet.
 * Dabei heit "dunkel", da die Pixel kleiner sind als die Schwelle thresh.
 * Dabei drfen "einzelne" Werte auch einen Wert bis zu terr erreichen.
 * --- Parameter: ---
 * PICTURE  *bild          : Bild
 * int      *top           : Zeiger auf obere Grenze (erste Zeile innerhalb des Bildes)
 * int      *bot           : Zeiger auf untere Grenze (erste Zeile auerhalb des Bildes)
 * unsigned char thresh    : Schwellwert fr "dunkel"
 * unsigned char terr      : Toleranz fr einzelne Werte
 * --- Rueckgabe: ---
 * int      *top           : obere Grenze
 * int      *bot           : untere Grenze
 * int      GetTopBottom() : 0 fr OK
 */
{
   register unsigned char *bp, *fbp;
   unsigned char     *ap, *fip;
   register int      storepix;
   int               ip, jp, tolthr, tolerr, dobad;

   storepix = bild->storepix;
   /*   tolthr = bild->W >> 5;   1/32 darf drber liegen     */
   /*   tolerr = tolthr >> 3;    1/256 darf ber Fehler sein */
   tolthr = bild->W >> 4;  /* 1/16 darf drber liegen     */
   tolerr = tolthr >> 3;   /* 1/128 darf ber Fehler sein */
   if ( tolthr < 4 ) tolthr = 4;
   if ( tolerr < 1 ) tolerr = 1;
   /* ... damit es bei kleinen Bildern auch klappt!! */

   dobad = 0;
   if ( bild->H > 512 ) {
      if ( bild->H > 1024 ) dobad = 2;
      else                  dobad = 1;
   }
      /* Wenn die Bilder gro sind, ist auch die Auflsung der Fehler hher! *
       * Daher wird noch ein Pixel weiter geschaut                           */
   
   /* oben */
   for ( fip = (ap = bp = bild->DATA + (storepix == 3 ? 1 : 0)) + (bild->storepix * bild->size); 
         ap < fip; ap = fbp) {
      for ( fbp = ( bp = ap ) + storepix*bild->width, ip = jp = 0; bp < fbp; bp += storepix ) 
         if ( *bp > thresh ) {
            if ( dobad ) {
               if ( ((bp += dobad * storepix) < fbp) && (*bp > thresh) ) {
                  ++ip;
                  if ( *bp > terr ) ++jp;
	       }
	    } else {
               ++ip;
               if ( *bp > terr ) ++jp;
	    }
         }
      if ( ip > tolthr || jp > tolerr ) break;
   }
   if ( (*top = ((ap-bild->DATA)/storepix)/bild->W) ) ++*top; /* +1 da innerhalb des Bildes gesucht */

   /* unten */
   for ( ap = bp = (fip = bild->DATA) - (storepix == 3 ? 2 : 1) + (bild->storepix * bild->size); 
         ap > fip; ap = fbp) {
      for ( fbp = ( bp = ap ) - storepix*bild->width, ip = jp = 0; bp > fbp; bp -= storepix ) 
         if ( *bp > thresh ) {
            if ( dobad ) {
               if ( ((bp -= dobad * storepix) > fbp) && (*bp > thresh) ) {
                  ++ip;
                  if ( *bp > terr ) ++jp;
	       }
	    } else {
               ++ip;
               if ( *bp > terr ) ++jp;
	    }
         }
      if ( ip > tolthr || jp > tolerr ) break;
   }
   *bot = ((ap-bild->DATA+2)/storepix)/bild->W;
   if ( *top == 0 && *bot == bild->H ) return -1;
   return 0;
}

int SelectPictureWithoutBorder(PAUL *p)
/* cut border of images
 * "Borders" are defined as "permanent dark" rectangular areas left and right as well
 * as top and bottom of a information containing rectangular area.
 * "Dark" means all pixels are lass than the threshold "thresh" but single
 * pixels can have values up to the error threshold "terr"
 * --- parameters: ---
 * PAUL *p                           : list of images, options
 *                                     used options:
 *                                     thresh : threshold for dark pixels
 *                                     terr   : threshold for single (noisy) pixels
 *                                     flag   : used, if only left and right border should be cut
 * --- R"uckgabe: ---
 * int   SelectPictureWithoutBorder(): if OK 0, else -1
 */
{
   PICTURE        *bild;
   register GList *pl, *piclist;
   int            *lb, *rb, *wd, *testlr, *tb, *bb, *testtb, i, h, neuwd, neuht = 0;
   register       unsigned char *ap, *np;
   unsigned       char *buf, *fap;
   char           *msg = "%s remains unchanged.\n", desc[512];
   
   piclist = p->piclist;
   if ( !NBILDER(piclist) ) return 0;

   assert ( (lb = calloc(7*(i = NBILDER(piclist)), sizeof(int))) != NULL );
   testtb = (testlr = ( wd = ( bb = ( tb = (rb = lb + i)+i)+i)+i)+i)+i;

   /* seitlichen Rand abschneiden */   
   h = neuwd = 0;

   for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
      if ( (*(testlr+i) = 
            GetBorder(bild, lb+i, rb+i, p->opt->thresh, p->opt->terr)) == 0 ) {
         neuwd += (*(wd+i) = *(rb+i) - *(lb+i));
         h++;
      }
   }
   if ( h > 1 )   /* Durchschnitt lieber etwas mehr */
      neuwd = (int)( neuwd + ((3*h)>>2) ) / h;   
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next)) 
      neuwd = MIN(neuwd, bild->W);
   
   for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
      if ( *(testlr+i) ) continue;
      if ( *(wd+i) != neuwd ) {
	 if ( NewBorder(bild, bild->W, neuwd, lb+i, rb+i, p->opt->thresh, p->opt->terr, CountThreshCol) ) {
            fprintf(stderr, "Width of %s cant be changed to %i.\n", bild->file, neuwd);
	    continue; 
	 }
      }

      *(wd+i) = neuwd;  /* OHNE CHECKLINE!!!! */
      /*      *(wd+i) = CreateCheckLine(bild, lb+i, rb+i);  */

      assert ( (buf = malloc(bild->storepix * *(wd+i) * bild->H)) != NULL );
      for ( fap = (ap = bild->DATA + bild->storepix * *(lb+i)) + bild->storepix*bild->size,
            np  = buf;
            ap < fap; 
            ap += bild->storepix * bild->W,
            np += bild->storepix * *(wd+i) ) {
         memcpy(np, ap, bild->storepix * *(wd+i));
      }
      NewImage(bild, *(wd+i), bild->H, buf);
      bild->size = bild->W * bild->H;
   }

   if ( !OnlyLeftRight(p->opt->f) ) {
      /* Rand oben und unten abschneiden */
      h = neuht = 0;
      for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
            if ( (*(testtb+i) = 
               GetTopBottom(bild, tb+i, bb+i, p->opt->thresh, p->opt->terr)) == 0 ) {
            neuht += (*(wd+i) = *(bb+i) - *(tb+i));
            h++;
         }
      }
      if ( h > 1 )   /* Durchschnitt lieber etwas mehr */
         neuht = (int)( neuht + ((3*h)>>2) ) / h;   
      for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
         neuht = MIN(neuht, bild->H);
         if ( *(testtb+i) && (neuht < bild->H) && NBILDER(piclist) > 1 ) {
            fprintf(stderr, "No change of height for all images because %s can't be cut vertically.\n",
                    bild->file);
            neuht = 0;
            break;
	 }
      }

      if ( neuht ) {
         for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
            if ( *(wd+i) != neuht ) {
               if ( NewBorder(bild, bild->H, neuht, tb+i, bb+i, p->opt->thresh, p->opt->terr, CountThreshRow) ) {
                  fprintf(stderr, "Height of %s cant be changed to %i.\n", 
                          bild->file, neuht);
                  continue; 
               }
            }
            *(wd+i) = neuht;  /* OHNE CHECKLINE!!!! */
            /*      *(wd+i) = CreateVerticalCheckLine(bild, tb+i, bb+i);  */
            assert ( (buf = malloc(bild->storepix * *(wd+i) * bild->W)) != NULL );
            bild->size = ( bild->H = *(wd+i) ) * bild->W;
            memcpy(buf, bild->DATA + *(tb+i)*bild->W*bild->storepix, bild->storepix*bild->size);
            NewImage(bild, bild->W, *(wd+i), buf);
            bild->size = bild->W * bild->H;
	 }
      }
   }

   /* set spezification chunks in the end */
   for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
      if ( *(testlr+i) && !neuht ) {   /* unterschiedliche Testbedingungen ... mal sehn, ob das gut ist*/
         printf(msg, bild->file);
	 continue;
      }
      if ( DelScanBorder(p->opt->f) ) {
         NewFileName(bild, APPDELSCANBORDER);
         CopySpec(bild->spec, ChunkNameTyp, TypDelScanRand);
         strcpy(desc, "Scanner border (white) deleted:");
         p->opt->f  &= ~DELSCANBORDER;
	 bild->flag |= DELSCANBORDER;
      } else {
         char typ[256];
	 
         sprintf(typ, "%s (+%i-%i)", TypDelRand, p->opt->thresh, p->opt->terr);
         NewFileName(bild, APPDELBORDER);
         CopySpec(bild->spec, ChunkNameTyp, typ);
         strcpy(desc, "New border:");
         p->opt->f  &= ~DELBORDER;
	 bild->flag |= DELBORDER;
      }
      if ( !*(testlr+i) )
         sprintf(desc + strlen(desc), " (left = %i, right = %i)", *(lb+i), *(rb+i));
      if ( !*(testtb+i) )
         sprintf(desc + strlen(desc), " (top = %i, bottom = %i)", *(tb+i), *(bb+i));
      CreateDescription(bild, desc);
   }
   free(lb);
   return 0;
}

