/*
 *  ppmtoagafb -- Display PPM or PGM images on the AGA frame buffer
 *
 *   Copyright 1996 by Geert Uytterhoeven
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING in the main directory of the Linux
 *  distribution for more details.
 */

/*  Known bug:
 *    - ppmtoagafb must be setuid root to access the ttys it needs.
 *      However, no disk I/O is done as root unless the uid and euid are root. 
 *
 *  To do:
 *    - Perhaps a config option to support other fb organizations
 *      (i.e. Atari, Cyber64 [cfb8] -- maybe a separate ppmtoatafb?)
 */

/* Enhanced to use Gnther Rhrich's more sophisticated HAM8 encoding
 * from his jpegAGA program.
 *
 * Gnther's implementation (in 680x0 assembler) is freely redistributable.
 *
 * My modifications (converting it to C and using the P?M interface)
 * are copyrighted by me and subject to the GPL.
 * - Chris Lawrence <lawrencc@debian.org>
 *
 * [Release 0.8]
 */

#define VERSION "Release 0.8 - 2 August 1998"
#define COPYING "/usr/doc/copyright/GPL" /* Change for non-Debian systems */

#include <stdio.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/fb.h>
#include <linux/kd.h>
#include <linux/vt.h>
#include <signal.h>
#include <ppm.h>
#include <ppmcmap.h>
#include <errno.h>

    /*
     *  Command Line Options
     */

static const char *ProgramName;
static char **Files;
static int NumFiles = 0;
static int epsilon, pref_epsilon = 0, delay = 5, center = 0;
static int grayscale = 0, verbose = 0;


    /*
     *  Console and Frame Buffer
     */

static int ActiveVT = -1;
static int ConsoleFd = 0;
static const char *FrameBufferName = NULL;
static int FrameBufferFD = -1;
static caddr_t FrameBufferBits = (caddr_t) - 1;
static size_t FrameBufferSize;
u_int16_t DisplayWidth, DisplayHeight;
u_int32_t NextLine, NextPlane;


    /*
     *  Color Maps
     */

#define CACHESIZE 262145

static u_int16_t Grey[256];
static struct fb_cmap GreyMap =
{
  0, 256, Grey, Grey, Grey
};

static u_int16_t Red[64];
static u_int16_t Green[64];
static u_int16_t Blue[64];
static struct fb_cmap ColorMap =
{
  0, 64, Red, Green, Blue
};

static u_int16_t PlanarRed[256];
static u_int16_t PlanarGreen[256];
static u_int16_t PlanarBlue[256];
static struct fb_cmap PlanarColorMap =
{
  0, 256, PlanarRed, PlanarGreen, PlanarBlue
};

static u_int32_t PlanarLookup[256];
static u_int16_t PlanarColors;


#if 0
/* The color table from jpegAGA; seems rather bogus to me.
 */
static u_char FixedColorTable[64 * 3] =
{0, 0, 0, 4, 4, 4, 8, 8, 8, 12, 12, 12,
 16, 16, 16, 20, 20, 20, 24, 24, 24, 28, 28, 28,	/* 16 colors */
 32, 32, 32, 36, 36, 36, 41, 41, 41, 46, 46, 46,
 51, 51, 51, 55, 55, 55, 59, 59, 59, 63, 63, 63,


 17, 17, 39, 17, 17, 55,	/* 13 colors */
 17, 29, 17, 17, 29, 39, 17, 29, 55,
 17, 39, 17, 17, 39, 29, 17, 39, 39, 17, 39, 55,
 17, 55, 17, 17, 55, 39, 17, 55, 39, 17, 55, 55,


 29, 17, 29, 29, 17, 39, 29, 17, 55,	/* 11 colors */
 29, 29, 55,
 29, 39, 17, 29, 39, 29, 29, 39, 55,
 29, 55, 17, 29, 55, 29, 29, 55, 39, 29, 55, 55,


 39, 17, 17, 39, 17, 29, 39, 17, 39, 39, 17, 55,	/* 12 colors */
 39, 29, 17, 39, 29, 29, 39, 29, 55,
 39, 39, 17, 39, 39, 29,
 39, 55, 17, 39, 55, 29,


 55, 17, 17, 55, 17, 29, 55, 17, 39, 55, 17, 55,	/* 13 colors */
 55, 29, 27, 55, 29, 29, 55, 29, 39, 55, 29, 55,
 55, 39, 17, 55, 39, 29, 55, 39, 39,
 55, 55, 17, 55, 55, 29
};

#else
static u_char FixedColorTable[64 * 3];
#endif

u_char ColorTable[64 * 3], *ColorCache = NULL;
u_int16_t Mult_Table[64];

#define LOOKUP(p) (PPM_GETR(p) << 16) | (PPM_GETG(p) << 8) | PPM_GETB(p)

    /*
     *  Function Prototypes
     */

static void Die(const char *fmt,...) __attribute__((noreturn));
static void Warn(const char *fmt,...);
static void SigHandler(int signo) __attribute__((noreturn));
static void VTRequest(int signo) __attribute__((noreturn));
static void Usage(void) __attribute__((noreturn));
static void Version(void);
static void OpenFB(void);
static void SetFBMode(int color);
static void CloseFB(void);
static void Chunky2Planar(u_char chunky[32], u_int32_t * fb);
int ReadImage(FILE * fp, u_int32_t * buffer, int do_histogram);
int main(int argc, char *argv[]);

extern void EncodeHAM8(pixel * pixrow, pixval maxval, u_char * yham,
                       u_int16_t xsize, int do_histogram);

    /*
     *  Print an Error Message and Exit
     */

static void Die(const char *fmt,...)
{
  va_list ap;

  fflush(stdout);
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);

  CloseFB();

  exit(1);
}

    /*
     *  Print a Warning Message
     */

static void Warn(const char *fmt,...)
{
  va_list ap;

  fflush(stdout);
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
}


    /*
     *  Signal Handler
     */

static void SigHandler(int signo)
{
  signal(signo, SIG_IGN);
  Die("Caught signal %d. Exiting\n", signo);
}


    /*
     *  Handler for the Virtual Terminal Request
     */

static void VTRequest(int signo)
{
  Die("VTRequest: Exiting\n");
}


    /*
     *  Open the Frame Buffer
     */

static void OpenFB(void)
{
  int i, fd, vtno;
  u_char r, g, b;
  struct fb_fix_screeninfo fix;
  struct fb_var_screeninfo var;
  char vtname[11];
  struct vt_stat vts;
  struct vt_mode VT;

  if (geteuid())
    Die("%s must be suid root\n", ProgramName);
  if ((fd = open("/dev/tty0", O_WRONLY, 0)) < 0)
    Die("Cannot open /dev/tty0: %s\n", strerror(errno));
  if (ioctl(fd, VT_OPENQRY, &vtno) < 0 || vtno == -1)
    Die("Cannot find a free VT\n");
  close(fd);
  sprintf(vtname, "/dev/tty%d", vtno);	/* /dev/tty1-64 */
  if ((ConsoleFd = open(vtname, O_RDWR | O_NDELAY, 0)) < 0)
    Die("Cannot open %s: %s\n", vtname, strerror(errno));
  /*
   * Linux doesn't switch to an active vt after the last close of a vt,
   * so we do this ourselves by remembering which is active now.
   */
  if (ioctl(ConsoleFd, VT_GETSTATE, &vts) == 0)
    ActiveVT = vts.v_active;
  /*
   * Detach from the controlling tty to avoid char loss
   */
  if ((i = open("/dev/tty", O_RDWR)) >= 0)
  {
    ioctl(i, TIOCNOTTY, 0);
    close(i);
  }

  /*
   * now get the VT
   */
  if (ioctl(ConsoleFd, VT_ACTIVATE, vtno) != 0)
    Warn("ioctl VT_ACTIVATE: %s\n", strerror(errno));
  if (ioctl(ConsoleFd, VT_WAITACTIVE, vtno) != 0)
    Warn("ioctl VT_WAITACTIVE: %s\n", strerror(errno));

  if (ioctl(ConsoleFd, VT_GETMODE, &VT) < 0)
    Die("ioctl VT_GETMODE: %s\n", strerror(errno));
  signal(SIGUSR1, VTRequest);
  VT.mode = VT_PROCESS;
  VT.relsig = SIGUSR1;
  VT.acqsig = SIGUSR1;
  if (ioctl(ConsoleFd, VT_SETMODE, &VT) < 0)
    Die("ioctl VT_SETMODE: %s\n", strerror(errno));

  /*
   *  Switch to Graphics Mode and Open the Frame Buffer Device
   */

  if (ioctl(ConsoleFd, KDSETMODE, KD_GRAPHICS) < 0)
    Die("ioctl KDSETMODE KD_GRAPHICS: %s\n", strerror(errno));
  if ((FrameBufferFD = open(FrameBufferName, O_RDWR)) < 0)
    Die("Cannot open %s: %s\n", FrameBufferName, strerror(errno));
  if (ioctl(FrameBufferFD, FBIOGET_VSCREENINFO, &var))
    Die("ioctl FBIOGET_VSCREENINFO: %s\n", strerror(errno));

  for (i = 0; i < 256; i++)
    Grey[i] = i * (65535/255);

  i = 0;
  for (r = 0; r < 0x40; r += (r == 0 ? 0xf : 0x10))
    for (g = 0; g < 0x40; g += (g == 0 ? 0xf : 0x10))
      for (b = 0; b < 0x40; b += (b == 0 ? 0xf : 0x10))
      {
	FixedColorTable[i++] = b;
	FixedColorTable[i++] = r;
	FixedColorTable[i++] = g;
      }

  ColorCache = malloc(CACHESIZE);
  if (!ColorCache)
    Die("colorcache: %s\n", strerror(errno));

  for (i = 0; i < 64; i++)
    Mult_Table[i] = (u_int16_t) i * i;

  var.xres_virtual = var.xres;
  var.yres_virtual = var.yres;
  var.xoffset = 0;
  var.yoffset = 0;
  var.bits_per_pixel = 8;
  var.vmode &= ~FB_VMODE_YWRAP;
  if (ioctl(FrameBufferFD, FBIOPUT_VSCREENINFO, &var))
    Die("ioctl FBIOPUT_VSCREENINFO: %s\n", strerror(errno));
  if (ioctl(FrameBufferFD, FBIOGET_FSCREENINFO, &fix))
    Die("ioctl FBIOGET_FSCREENINFO: %s\n", strerror(errno));
  DisplayWidth = var.xres;
  DisplayHeight = var.yres;
  switch (fix.type)
  {
  case FB_TYPE_PACKED_PIXELS:
    Die("Packed pixels are not supported\n");
    break;
  case FB_TYPE_PLANES:
    if (fix.line_length)
      NextLine = fix.line_length;
    else
      NextLine = var.xres_virtual >> 3;
    NextPlane = NextLine * var.yres_virtual;
    break;
  case FB_TYPE_INTERLEAVED_PLANES:
    if (fix.line_length)
      NextLine = fix.line_length;
    else
      NextLine = fix.type_aux;
    NextPlane = NextLine / var.bits_per_pixel;
    break;
  default:
    Die("Unknown frame buffer type %s\n", fix.type);
    break;
  }
  FrameBufferSize = fix.smem_len;
  FrameBufferBits = (caddr_t) mmap(0, FrameBufferSize, PROT_READ | PROT_WRITE,
				   MAP_SHARED, FrameBufferFD, 0);
  if (FrameBufferBits == (caddr_t) - 1)
    Die("mmap: %s\n", strerror(errno));
}


    /*
     *  Set the Color Mode of the Frame Buffer
     */

static void SetFBMode(int color)
{
  struct fb_var_screeninfo var;
  struct fb_cmap *cmap;

  if (ioctl(FrameBufferFD, FBIOGET_VSCREENINFO, &var))
    Die("ioctl FBIOGET_VSCREENINFO: %s\n", strerror(errno));
  if (color == 1 && !grayscale)
  {
    var.nonstd = FB_NONSTD_HAM;
    cmap = &ColorMap;
  }
  else if (color == 2)
  {
    var.nonstd = 0;
    cmap = &PlanarColorMap;
  }
  else
  {
    var.nonstd = 0;
    cmap = &GreyMap;
  }
  if (ioctl(FrameBufferFD, FBIOPUT_VSCREENINFO, &var))
    Die("ioctl FBIOPUT_VSCREENINFO: %s\n", strerror(errno));
  if (ioctl(FrameBufferFD, FBIOPUTCMAP, cmap))
    Die("ioctl FBIOPUTCMAP: %s\n", strerror(errno));
}


    /*
     *  Close the Frame Buffer
     */

static void CloseFB(void)
{
  struct vt_mode VT;

  if (FrameBufferBits != (caddr_t) - 1)
  {
    munmap(FrameBufferBits, FrameBufferSize);
    FrameBufferBits = (caddr_t) - 1;
  }
  if (FrameBufferFD != -1)
  {
    close(FrameBufferFD);
    FrameBufferFD = -1;
  }
  if (ConsoleFd)
  {
    ioctl(ConsoleFd, KDSETMODE, KD_TEXT);
    if (ioctl(ConsoleFd, VT_GETMODE, &VT) != -1)
    {
      VT.mode = VT_AUTO;
      ioctl(ConsoleFd, VT_SETMODE, &VT);
    }
    if (ActiveVT >= 0)
    {
      ioctl(ConsoleFd, VT_ACTIVATE, ActiveVT);
      ActiveVT = -1;
    }
    close(ConsoleFd);
    ConsoleFd = 0;
  }
}


void ConvertPGMRow(pixel * pixelrow, pixval maxval, int cols, int shift,
		   u_int32_t * fb, int dummy)
{
  pixel p;
  int i = 0, j;
  u_char chunky[32];

  for (; i < shift; i++)
  {
    chunky[i % 32] = 0;
    if (i % 32 == 31)
      Chunky2Planar(chunky, fb++);
  }

  for (; i < cols + shift; i++)
  {
    PPM_DEPTH(p, pixelrow[i - shift], maxval, 255);

    /* Note that these both return the same thing for a true grayscale
     * image, but the latter is faster...
     */
    if(grayscale)
      chunky[i % 32] = ((PPM_GETR(p)*299 + PPM_GETG(p)*587 +
                         PPM_GETB(p)*114) / 1000);
    else
      chunky[i % 32] = PPM_GETR(p);

    if (i % 32 == 31)
      Chunky2Planar(chunky, fb++);
  }
  if (i % 32)
  {
    while (i % 32)
    {
      chunky[i % 32] = 0;
      i++;
    }
    Chunky2Planar(chunky, fb++);
  }
  while (i < DisplayWidth)
  {
    u_int32_t *fb2 = fb;

    for (j = 0; j < 8; j++)
    {
      *fb2 = 0;
      fb2 += NextPlane / sizeof(u_int32_t);
    }
    i += 32;
    fb++;
  }
}


void ConvertPlanarPPMRow(pixel * pixelrow, pixval maxval, int cols, int shift,
                         u_int32_t * fb, int dummy)
{
  pixel p, prevp;
  int i = 0, j = -1;
  u_int32_t look_for;
  u_char chunky[32];

  for (; i < shift; i++)
  {
    chunky[i % 32] = 0;
    if (i % 32 == 31)
      Chunky2Planar(chunky, fb++);
  }

  for (; i < cols + shift; i++)
  {
    PPM_DEPTH(p, pixelrow[i - shift], maxval, 255);
    if(j < 0 || !PPM_EQUAL(prevp,p))
    {
      look_for = LOOKUP(p);
      
      for(j=0; j < PlanarColors; j++)
        if(PlanarLookup[j] == look_for) break;

      if(j == PlanarColors)
        Die("Weirdness in color lookup\n");

      prevp = p;
    }

    chunky[i%32] = j;

    if (i % 32 == 31)
      Chunky2Planar(chunky, fb++);
  }
  if (i % 32)
  {
    while (i % 32)
    {
      chunky[i % 32] = 0;
      i++;
    }
    Chunky2Planar(chunky, fb++);
  }
  while (i < DisplayWidth)
  {
    u_int32_t *fb2 = fb;

    for (j = 0; j < 8; j++)
    {
      *fb2 = 0;
      fb2 += NextPlane / sizeof(u_int32_t);
    }
    i += 32;
    fb++;
  }
}


void ConvertPPMRow(pixel * pixelrow, pixval maxval, int cols, int shift,
		   u_int32_t * fb, int do_histogram)
{
  int i, j;
  u_char *chunky;

  chunky = calloc((cols + shift + 31) / 32, 32);
  if (!chunky)
    Die("chunky: %s\n", strerror(errno));

  EncodeHAM8(pixelrow, maxval, &chunky[shift], cols, do_histogram);

  for (i = 0; i < cols + shift; i += 32)
  {
    Chunky2Planar(&chunky[i], fb);
    fb++;
  }

  free(chunky);
  chunky = 0;

  while (i < DisplayWidth)
  {
    u_int32_t *fb2 = fb;

    for (j = 0; j < 8; j++)
    {
      *fb2 = 0;
      fb2 += NextPlane / sizeof(u_int32_t);
    }
    i += 32;
    fb++;
  }
}


void FillBlackRow(u_int32_t * fb)
{
  int i;

  for (i = 0; i < 8; i++)
  {
    memset(fb, 0, DisplayWidth / 8);
    fb += NextPlane / sizeof(u_int32_t);
  }
}


void DisplayImage(u_char * buffer, int color)
{
  u_char *fb = (u_char *) FrameBufferBits;
  u_int i, j;

  SetFBMode(color);
  for (i = 0; i < DisplayHeight; i++)
  {
    u_char *src = buffer;
    u_char *dst = fb;

    for (j = 0; j < 8; j++)
    {
      memcpy(dst, src, DisplayWidth / 8);
      src += NextPlane;
      dst += NextPlane;
    }
    buffer += NextLine;
    fb += NextLine;
  }
}


void Chunky2Planar(u_char chunky[32], u_int32_t * fb)
{
  int i;
  u_char mask;
  u_int32_t val, planemask;

  for (mask = 1; mask; mask <<= 1)
  {
    val = 0;
    for (planemask = 0x80000000, i = 0; planemask; planemask >>= 1, i++)
    {
      if (chunky[i] & mask)
	val |= planemask;
    }
    *fb = val;
    fb += NextPlane / sizeof(u_int32_t);
  }
}


static int countcompare(const void *x, const void *y)
{
  colorhist_vector ch1 = (colorhist_vector) x,
    ch2 = (colorhist_vector) y;

  return ch2->value - ch1->value;
}

static inline int is_close(pixel x, u_char tab[3])
{
  u_int16_t err;

#ifdef OLD_CLOSENESS
  err = Mult_Table[abs(PPM_GETB(x) - tab[0])];
  err += Mult_Table[abs(PPM_GETR(x) - tab[1])];
  err += Mult_Table[abs(PPM_GETG(x) - tab[2])];
#else
  err  = abs(PPM_GETB(x) - tab[0]);
  err += abs(PPM_GETR(x) - tab[1]);
  err += abs(PPM_GETG(x) - tab[2]);
#endif

  return (err < epsilon);
}

static int in_colortable(pixel x, int count)
{
  int i;

  for (i = 0; i < count; i++)
  {
    if (ColorTable[(i * 3) + 0] == PPM_GETB(x) &&
	ColorTable[(i * 3) + 1] == PPM_GETR(x))
      return 1;

    if (ColorTable[(i * 3) + 0] == PPM_GETB(x) &&
	ColorTable[(i * 3) + 2] == PPM_GETG(x))
      return 1;

    if (ColorTable[(i * 3) + 1] == PPM_GETR(x) &&
	ColorTable[(i * 3) + 2] == PPM_GETG(x))
      return 1;

    if (epsilon > 1 && is_close(x, &ColorTable[i * 3]))
      return 1;
  }

  return 0;
}

    /*
     *  Read an Image
     */

int ReadImage(FILE * fp, u_int32_t * buffer, int do_histogram)
{
  int cols, rows, format, imcols, imrows, i, has_color;
  int vertshift, horizshift, use_planar_color = 0;
  pixel **pixels;
  pixval maxval;
  void (*func) (pixel *, pixval, int, int, u_int32_t *, int);

  memcpy(ColorTable, FixedColorTable, 64 * sizeof(ColorTable[0]));

  ppm_readppminit(fp, &cols, &rows, &maxval, &format);

  if (verbose)
    fprintf(stderr, "Image size: %d x %d\n", cols, rows);

  pixels = ppm_allocarray( cols, rows );
  if(!pixels)
    Die("ppm_allocarray: %s\n", strerror(errno));
    
  for (i = 0; i < rows; i++)
  {
    ppm_readppmrow(fp, pixels[i], cols, maxval, format);
  }

  has_color = (PPM_FORMAT_TYPE(format) != PGM_TYPE);

  if (do_histogram && has_color && !grayscale)
  {
    colorhist_vector chv;
    int colors;
    pixel x;

    chv = ppm_computecolorhist(pixels, cols, rows, 1024 * 1024, &colors);

    if (chv && verbose)
      fprintf(stderr, "Image contains %d colors.\n", colors);

    if (!chv)
    {
      Warn("Unable to histogram: too many colors (%d)\n", colors);
    }
    else if (colors <= 256)
    {
      int offset_color = (colors < 256 ? 1 : 0);
      
      use_planar_color = 1;

      /* Black background for < 256 colors */
      if(offset_color)
      {
        PlanarRed[0] = PlanarGreen[0] = PlanarBlue[0] = 0;
      }

      memset(PlanarLookup, '\0', 256 * sizeof(PlanarLookup[0]));

      for (i = 0; i < colors; i++)
      {
	PPM_DEPTH(x, chv[i].color, maxval, 255);

        PlanarRed[i+offset_color]   = PPM_GETR(x) * (65535/255);
        PlanarGreen[i+offset_color] = PPM_GETG(x) * (65535/255);
        PlanarBlue[i+offset_color]  = PPM_GETB(x) * (65535/255);

        PlanarLookup[i+offset_color] = LOOKUP(x);
      }

      PlanarColors = colors + offset_color;
    }
    else
    {
      int pos = 1;

      qsort(chv, colors, sizeof(struct colorhist_item), countcompare);

      if (!pref_epsilon)
        epsilon = 5;
      else
	epsilon = pref_epsilon;

      if(verbose)
        fprintf(stderr, "Using epsilon: %d\n", epsilon);

      ColorTable[0] = ColorTable[1] = ColorTable[2] = 0;

      while (pos < 64)
      {
	for (i = 0; pos < 64 && i < colors; i++)
	{
	  PPM_DEPTH(x, chv[i].color, maxval, 63);

	  if (in_colortable(x, pos - 1))
	    continue;

	  ColorTable[(pos * 3) + 0] = PPM_GETB(x);
	  ColorTable[(pos * 3) + 1] = PPM_GETR(x);
	  ColorTable[(pos * 3) + 2] = PPM_GETG(x);
	  pos++;
	}
        
        if(verbose)
          fprintf(stderr, "Pass complete: scanned %d colors, allocated %d "
                  "colormap entries.\n", i, pos);

	epsilon--;
      }
    }

    if(chv) ppm_freecolorhist(chv);
  }

  /* Convert it to planar */
  if(!grayscale && !use_planar_color)
  {
    memset(ColorCache, '\0', CACHESIZE);
    for (i = 0; i < 64 * 3; i += 3)
    {
      Blue[i / 3] = ColorTable[i] * (65535 / 63);
      Red[i / 3] = ColorTable[i + 1] * (65535 / 63);
      Green[i / 3] = ColorTable[i + 2] * (65535 / 63);
    }
  }

  imcols = DisplayWidth < cols ? DisplayWidth : cols;
  imrows = DisplayHeight < rows ? DisplayHeight : rows;

  if( grayscale || !has_color )
    func = ConvertPGMRow;
  else if( use_planar_color )
    func = ConvertPlanarPPMRow;
  else
    func = ConvertPPMRow;

  if (center)
    horizshift = (DisplayWidth - imcols) / 2;
  else
    horizshift = 0;

  if (center && (imrows < DisplayHeight))
    vertshift = (DisplayHeight - imrows) / 2;
  else
    vertshift = 0;

  for (i = 0; i < vertshift; i++)
  {
    FillBlackRow(buffer);
    buffer += NextLine / sizeof(u_int32_t);
  }
  
  for (i = 0; i < imrows; i++)
  {
    func(pixels[i], maxval, imcols, horizshift, buffer, do_histogram);
    buffer += NextLine / sizeof(u_int32_t);
  }

  for (i = imrows + vertshift; i < DisplayHeight; i++)
  {
    FillBlackRow(buffer);
    buffer += NextLine / sizeof(u_int32_t);
  }

  if(verbose)
    fprintf(stderr, "Final buffer location: %p\n", buffer);

  ppm_freearray(pixels, rows);

  return (use_planar_color ? 2 : (has_color ? 1 : -1));
}


    /*
     *  Print the Usage Template and Exit
     */

static void Usage(void)
{
  Version();
  Die("\nUsage: %s [options] <filenames | ->\n\n"
      "Valid options are:\n"
      "    -h, --help             : Display this usage information and exit\n"
      "    -V, --version          : Display the version and exit\n"
      "    -v, --verbose          : Display information about the image(s)\n"
      "    -c, --center, --centre : Center image within display\n"
      "    -d, --delay            : Set how long to wait between images\n"
      "    -g, --grayscale,       : Render as grayscale (even color images)\n"
      "        --greyscale\n"
      "    -f, --frame-buffer     : Framebuffer device to use\n\n"
      "Advanced Options:\n"
      "    --no-histogram         : Use the default palette instead of\n"
      "                             calculating one.\n"
      "    --exact-colors,        : Disable close matching of colors in\n"
      "        --exact-colours      favor of exact colors.\n"
      "    --epsilon              : Determines how close two colors have\n"
      "                             to be before being the 'same'.\n"
      "                             (defaults to 5)\n"
      "                             [--epsilon 1 == --exact-colors]\n",
      ProgramName);
}


static void Version(void)
{
  fprintf(stderr, "ppmtoagafb "VERSION"\n"
          "(C) 1996 Geert Uytterhoeven and (C) 1996-98 Chris Lawrence\n"
          "Maintained by Chris Lawrence <lawrencc@debian.org>\n"
          "Read "COPYING" for your license and lack of warranty.\n");
}


int main(int argc, char *argv[])
{
  FILE *fp = 0;
  int res, first = 1, do_histogram = 1, use_stdin = 0, allocwidth;
  time_t start = 0;
  u_int32_t *buffer;
  int c = 0;

  ProgramName = argv[0];
  Files = argv + 1;
  while (--argc > 0)
  {
    argv++;
    if (!strcmp(argv[0], "-h") || !strcmp(argv[0], "--help"))
      Usage();
    else if (!strcmp(argv[0], "-V") || !strcmp(argv[0], "--version"))
    {
      Version();
      return 1;
    }
    else if (!strcmp(argv[0], "-f") || !strcmp(argv[0], "--frame-buffer"))
    {
      if (argc-- > 1 && !FrameBufferName)
      {
	FrameBufferName = argv[1];
	argv++;
      }
      else
	Usage();
    }
    else if (!strcmp(argv[0], "--no-histogram"))
      do_histogram = 0;
    else if (!strcmp(argv[0], "--exact-colors") ||
             !strcmp(argv[0], "--exact-colours"))
      pref_epsilon = 1;
    else if (!strcmp(argv[0], "--center") || !strcmp(argv[0], "--centre") || 
             !strcmp(argv[0], "-c"))
      center = 1;
    else if (!strcmp(argv[0], "--verbose") || !strcmp(argv[0], "-v"))
      verbose = 1;
    else if (!strcmp(argv[0], "--epsilon"))
    {
      if (argc-- > 1)
      {
	pref_epsilon = atoi(argv[1]);
	if (pref_epsilon < 1)
	  pref_epsilon = 1;
	argv++;
      }
      else
	Usage();
    }
    else if (!strcmp(argv[0], "--delay") || !strcmp(argv[0], "-d"))
    {
      if (argc-- > 1)
      {
	delay = atoi(argv[1]);
	if (delay < 0)
	  delay = 0;
	argv++;
      }
      else
	Usage();
    }
    else if (!strcmp(argv[0], "--grayscale") ||
             !strcmp(argv[0], "--greyscale") || !strcmp(argv[0], "-g"))
      grayscale = 1;
    else if (!strcmp(argv[0], "-"))
      use_stdin = 1;
    else
      Files[NumFiles++] = argv[0];
  }
  if (!FrameBufferName && !(FrameBufferName = getenv("FRAMEBUFFER")))
    FrameBufferName = "/dev/fb0current";

  if (!NumFiles && !use_stdin)
  {
    Usage();
    return 1;
  }

  signal(SIGSEGV, SigHandler);
  signal(SIGILL, SigHandler);
  signal(SIGFPE, SigHandler);
  signal(SIGBUS, SigHandler);
  signal(SIGXCPU, SigHandler);
  signal(SIGXFSZ, SigHandler);
  signal(SIGALRM, SigHandler);

  OpenFB();

  if (use_stdin)
    fp = stdin;

  /* Allocate memory in 64-byte chunks.  This seems to be Important for
   * some reason.
   */
  allocwidth = DisplayWidth + (64 - DisplayWidth) % 64;

  buffer = malloc(allocwidth * DisplayHeight);
  if(!buffer)
    Die("buffer: %s\n", strerror(errno));

  if (use_stdin)
    c = fgetc(fp);

  for (; (use_stdin ? c != EOF : NumFiles--); (use_stdin ? Files : Files++))
  {
    if (use_stdin)
      ungetc(c, fp);
    else
    {
      /* This setreuid business supposedly restricts non-superusers from
       * accessing files not their own. [see setreuid(3)]
       */
      setreuid(geteuid(), getuid());
      fp = fopen(*Files, "rb");
      setreuid(geteuid(), getuid());
      
      if(!fp)
      {
        Warn("Can't open file `%s'\n", *Files);
        continue;
      }
    }

    if(verbose && !use_stdin)
      fprintf(stderr, "Reading and displaying '%s'\n", *Files);

    res = ReadImage(fp, buffer, do_histogram);
    if (res)
    {
      if (first)
	first = 0;
      else
      {
	/* Wait any remaining number of seconds in our delay period */
	long tmp;

	if (start > 0)
	{
	  tmp = (delay - (time(0) - start));
	  if (tmp < 0)
	    tmp = 0;
	}
	else
	  tmp = delay;

	if (tmp > 0)
	{
          if(verbose)
            fprintf(stderr, "Sleeping %ld seconds.\n", tmp);
	  sleep(tmp);
	}
      }

      DisplayImage((u_char *) buffer, res);
      start = time(0);
    }

    if (use_stdin)
      c = fgetc(fp);
    else
    {
      fclose(fp);
      fp = 0;
    }
  }
  sleep(delay);
  CloseFB();

  return (0);
}
