/* This file is part of PyApache.
 * Copyright (C) 1996,97,98 by Lele Gaifax.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
 
/* 
 * RCSfile:  mod_pyapache.c,v
 * Revision: 4.14
 * Date:     1998/07/27 19:56:57
 *
 * Created Fri May  3 15:12:20 1996.
 *
 * Mon Jun 10 1996: This was formerly known as ``mod_python.c''. It
 * supercedes a previous work by Dave Mitchell <davem@magnet.com>, a
 * patch to Apache's ``mod_cgi.c'' to make it recognize scripts
 * written in Python and handling them consequently; it was
 * distribuited as tar file named ``pyapache.tar''. To avoid confusion
 * between the names, he kindly suggested to rename my module using
 * that name. I also find that ``mod_pyapache.c'' makes easier guess
 * its content, so I happily applied the suggestion. Thanx Dave.  */

/* This module supports the Native Execution of Python Scripts within
   the Apache HTTPD Server. It does not add any new capability to the
   server, but lets it execute Python Scripts without the need of the
   external Python Interpreter. */

/* See the accompaning README.PyApache for installation instruction
   and other information */

#include "mod_pyapache.h"

/* Global version, initialized ad module's init time. */
PyObject *PyApache_version;

/* This define is used to test whether the ApacheXXX instances get
   released at each cycle. Normally set it to 0 (or #undef it) */
/* #define RELEASE_TEST 1 */


/**********************************************************************
                         ApacheBuff Object

  This is probably the smarter (?) object in this module: it is an
  approximation of a Python fileobject, in the sense that it
  implements just the very basic methods of such an object: read,
  write and a do-nothing flush; they are enough to hook an instance of
  it in place of the standard fileobject sys.stdin and sys.stdout.

  The read method will return a string composed of bytes coming from
  the client, thus giving the script access to the arguments of a M_POST
  or a M_PUT request.

  The write method needs additional machinery: until it does not find
  the end-of-headers condition it will collect them, updating the
  internal tables of the server, like scan_script_headers() does. To
  complicate things, we have to parse headers `incrementally', since
  we aren't getting them from a stream, but directly as the script
  writes them on its output.  Once the headers come at end, the
  remaining stuff is sent to the client, as usual.
  This step does not happen for NPH scripts: the output of the script
  is sent directly to the client.

  The flush method is there for compatibility: it lets you use your `old'
  scripts, where you used to add a "sys.stdout.flush()" to sync the
  script output.
***********************************************************************/

#define FUNDESCR(s) s "<BR>"
#define FUNSYN(s) "Synopsis:\t" s "<BR>"
#define FUNARGS(s) "Arguments:\t" s "<BR>"
#define FUNRET(s) "Returns:\t" s "<BR>"
#define FUNDOC(d,s,a,r) FUNDESCR(d) FUNSYN(s) FUNARGS(a) FUNRET(r)

#if BUFSIZ < 8192
#define SMALLCHUNK 8192
#else
#define SMALLCHUNK BUFSIZ
#endif

typedef struct _ApacheBuff
{
  PyObject_HEAD

  /* The REQUEST_REC. */
  request_rec *r;

  /* Read buffer: we allocate SMALLCHUNK bytes for it when needed, and will
     use it to rebuffer data coming from the client. */
  unsigned char *read_buffer;

  /* Current cursor on read_buffer. */
  unsigned char *read_buffer_ptr;

  /* How much read-ahead we have in read_buffer if >= 0, otherwise
     signals the EOF condition with -1 and an error condition with
     -2. */
  int read_buffer_remain;

  /* Total amount of data read so far. */
  size_t read_so_far;

  /* FALSE until we parse and validate all the headers coming from
     the script, TRUE after that. TRUE all the time for NPH scripts. */
  int headers_sent;

  /* TRUE if we found a malformed header. */
  int malformed_header;

  /* If not NULL, contains yet-to-be-parsed headers, that is headers
     coming from previous calls to parse_headers(): since it works
     parsing line by line, we eventually store here the last non
     terminated line for the next time. */
  char *headers;

  /* Flag used by Python's `print' command */
  int f_softspace;
} ApacheBuff;

staticforward PyTypeObject ApacheBuffType;

#define BUF(v) PyString_AS_STRING((PyStringObject *)v)

#define AB_ERROR(ab) ((ab)->read_buffer_remain<=-2)
#define AB_EOF(ab) ((ab)->read_buffer_remain<0)
#define AB_EMPTY(ab) ((ab)->read_buffer_remain==0)


/* Apache uses an hidden mechanism to handle input from the client and
   sending it back to the CGI. It exposes the machinery through
   ap_get_client_block(): here we need to rebuffer the input, to make
   the readline implementation a little faster. Moreover, it would be
   very difficult to use that function to get a character at a time. */

/* Fill the read-ahead buffer, reading data from the client: return -1
   on EOF or errors, the first char fetched advancing the cursor
   otherwise. */
static int
ab_fill_read_buffer (ApacheBuff *self)
{
  if (AB_ERROR(self))
    return -1;

  if (self->read_buffer == NULL)
    {
      self->read_buffer = PyMem_NEW (char, SMALLCHUNK);
      if (self->read_buffer == NULL)
        return -1;              /* XXX what else? */
    }


  self->read_buffer_remain = ap_get_client_block (self->r,
                                                  self->read_buffer,
                                                  SMALLCHUNK);
  /* ap_get_client_block() returns -1 on errors, 0 on EOF */
  if (self->read_buffer_remain <= 0)
    {
      /* Set the EOF/Error condition:
          0 => -1 on EOF
         -1 => -2 on Error */
      self->read_buffer_remain--;

      return -1;
    }
  else
    {
      self->read_so_far += self->read_buffer_remain;
      self->read_buffer_ptr = self->read_buffer;
      self->read_buffer_remain--;

      return (int) *self->read_buffer_ptr++;
    }
}

/* Return the next character from the client, filling the buffer if
   necessary. Return -1 on EOF or errors */
#define ab_getc(self) ((--self->read_buffer_remain >= 0) ?      \
                       (int) *self->read_buffer_ptr++ :         \
                       ab_fill_read_buffer (self))

/* Return the position within the input stream */
#define ab_tell(self) ((self->read_buffer_remain > 0) ?               \
                       self->read_so_far - self->read_buffer_remain : \
                       self->read_so_far)

/* Simple implementation of fread().  Return 0 on EOF or errors: to
   discriminate, see read_buffer_remain, that signals EOF when <=
   0. Normally return the actual number of bytes read. */
static size_t
ab_fread (ApacheBuff *self, register char *ptr,  register size_t size)
{
  size_t ret = 0;

  while (size>0 && !AB_EOF(self))
    {
      register size_t count;
      register int c;

      if (AB_EMPTY (self))
        {
          if ((c = ab_fill_read_buffer (self)) < 0)
            break;

          *ptr++ = c;
          size--;
          ret++;
        }

      count = (size < self->read_buffer_remain ? size : self->read_buffer_remain);
      memcpy (ptr, self->read_buffer_ptr, count);
      ptr += count;
      self->read_buffer_ptr += count;
      self->read_buffer_remain -= count;
      size -= count;
      ret += count;
    }

  return ret;
}

static char ApacheBuff_read_doc[] =
FUNDOC("Read data from the client",
       ".read (size)",
       "an optional count of bytes to read, like standard file objects",
       "a string object");

static PyObject *
ApacheBuff_read (ApacheBuff *self, PyObject *args)
{
  long bytesrequested = -1;
  size_t bytesread, buffersize, chunksize;
  PyObject *v;

  if (!PyArg_ParseTuple (args, "|l", &bytesrequested))
    return NULL;
  if (bytesrequested < 0)
    buffersize = SMALLCHUNK;
  else
    buffersize = bytesrequested;
  v = PyString_FromStringAndSize ((char *)NULL, buffersize);
  if (v == NULL)
    return NULL;
  bytesread = 0;
  for (;;)
    {
      Py_BEGIN_ALLOW_THREADS
      chunksize = ab_fread (self, BUF(v) + bytesread, buffersize - bytesread);
      Py_END_ALLOW_THREADS
      if (chunksize == 0)
        {
          if (AB_EOF (self))
            break;

          /* Otherwise we are in error */

          PyErr_SetString (PyExc_IOError, "Cannot read from client");

          Py_DECREF(v);
          return NULL;
        }
      bytesread += chunksize;
      if (bytesrequested < 0)   /* if we wanna read the entire file */
        {
          buffersize = bytesread + SMALLCHUNK;
          if (_PyString_Resize (&v, buffersize) < 0)
            return NULL;
        }
      else if (bytesread >= bytesrequested) /* it cannot be greater, but... */
        break;
    }
  if (bytesread != buffersize)
    _PyString_Resize (&v, bytesread);
  return v;
}

/* Code adapted from fileobject.c:1.5.1 */

/* Internal routine to get a line.
   Size argument interpretation:
   > 0: max length;
   = 0: read arbitrary line;
   < 0: strip trailing '\n', raise EOFError if EOF reached immediately
*/

static PyObject *
getline (ApacheBuff *self, long size)
{
  register char *buf, *end;
  size_t total, rsize;
  PyObject *v;

  rsize = size > 0 ? size : 100;
  v = PyString_FromStringAndSize ((char *) NULL, rsize);
  if (v == NULL)
    return NULL;

  buf = BUF(v);
  end = buf + rsize;

  Py_BEGIN_ALLOW_THREADS
  for (;;)
    {
      register int c = ab_getc (self);

      if (c == -1)
        {
          if (AB_ERROR(self))
            {
              PyErr_SetString (PyExc_IOError, "Cannot read from client");
              Py_DECREF(v);
              return NULL;
            }

          /* EOF */
          Py_BLOCK_THREADS
          if (PyErr_CheckSignals())
            {
              Py_DECREF(v);
              return NULL;
            }
          if (size < 0 && buf == BUF(v))
            {
              Py_DECREF(v);
              PyErr_SetString (PyExc_EOFError, "EOF when reading a line");
              return NULL;
            }
          Py_UNBLOCK_THREADS
          break;
        }

      if ((*buf++ = c) == '\n')
        {
          if (size < 0)
            buf--;
          break;
        }

      if (buf == end)
        {
          if (size > 0)
            break;
          total = rsize;
          rsize += 1000;
          Py_BLOCK_THREADS
          if (_PyString_Resize(&v, rsize) < 0)
            return NULL;
          Py_UNBLOCK_THREADS
            buf = BUF(v) + total;
          end = BUF(v) + rsize;
        }
    }
  Py_END_ALLOW_THREADS

  total = buf - BUF(v);
  if (total != rsize)
    _PyString_Resize (&v, total);
  return v;
}

static char ApacheBuff_readlines_doc[] =
FUNDOC("Read an entire file from the client",
       ".readlines (size)",
       "an optional count of chars to read, like standard .readlines() methods",
       "a list of strings");

static PyObject *
ApacheBuff_readlines (ApacheBuff *self, PyObject *args)
{
  long sizehint = 0;
  PyObject *list;
  PyObject *line;
  char small_buffer[SMALLCHUNK];
  char *buffer = small_buffer;
  size_t buffersize = SMALLCHUNK;
  PyObject *big_buffer = NULL;
  size_t nfilled = 0;
  size_t nread;
  size_t totalread = 0;
  char *p, *q, *end;
  int err;

  if (!PyArg_ParseTuple (args, "|l", &sizehint))
    return NULL;

  if ((list = PyList_New(0)) == NULL)
    return NULL;

  for (;;)
    {
      Py_BEGIN_ALLOW_THREADS
      nread = ab_fread (self, buffer+nfilled, buffersize-nfilled);
      Py_END_ALLOW_THREADS
      if (nread == 0)
        {
          sizehint = 0;
          if (AB_EOF(self))
            break;

        error:
          Py_DECREF(list);
          list = NULL;
          goto cleanup;
        }
      totalread += nread;
      p = memchr (buffer+nfilled, '\n', nread);
      if (p == NULL)
        {
          /* Need a larger buffer to fit this line */
          nfilled += nread;
          buffersize *= 2;
          if (big_buffer == NULL)
            {
              /* Create the big buffer */
              big_buffer = PyString_FromStringAndSize (NULL, buffersize);
              if (big_buffer == NULL)
                goto error;
              buffer = PyString_AS_STRING (big_buffer);
              memcpy (buffer, small_buffer, nfilled);
            }
          else
            {
              /* Grow the big buffer */
              _PyString_Resize (&big_buffer, buffersize);
              buffer = PyString_AS_STRING (big_buffer);
            }
          continue;
        }
      end = buffer+nfilled+nread;
      q = buffer;
      do
        {
          /* Process complete lines */
          p++;
          line = PyString_FromStringAndSize (q, p-q);
          if (line == NULL)
            goto error;
          err = PyList_Append (list, line);
          Py_DECREF(line);
          if (err != 0)
            goto error;
          q = p;
          p = memchr (q, '\n', end-q);
        } while (p != NULL);
      /* Move the remaining incomplete line to the start */
      nfilled = end-q;
      memmove (buffer, q, nfilled);
      if (sizehint > 0)
        if (totalread >= (size_t)sizehint)
          break;
    }
  if (nfilled != 0)
    {
      /* Partial last line */
      line = PyString_FromStringAndSize(buffer, nfilled);
      if (line == NULL)
        goto error;
      if (sizehint > 0)
        {
          /* Need to complete the last line */
          PyObject *rest = getline (self, 0);
          if (rest == NULL)
            {
              Py_DECREF(line);
              goto error;
            }
          PyString_Concat (&line, rest);
          Py_DECREF(rest);
          if (line == NULL)
            goto error;
        }
      err = PyList_Append (list, line);
      Py_DECREF(line);
      if (err != 0)
        goto error;
    }
 cleanup:
  if (big_buffer)
    {
      Py_DECREF(big_buffer);
    }
  return list;
}

static char ApacheBuff_readline_doc[] =
FUNDOC("Read an entire line from the client",
       ".readline (size)",
       "an optional count of chars to read, like standard .readline() methods",
       "a string object");

static PyObject *
ApacheBuff_readline (ApacheBuff *self, PyObject *args)
{
  long n = -1;

  if (!PyArg_ParseTuple (args, "|l;SIZE", &n))
    return NULL;

  if (n == 0)
    return PyString_FromString ("");

  if (n < 0)
    n = 0;

  return getline (self, n);
}

static char ApacheBuff_tell_doc[] =
FUNDOC("Return the current position in the input stream",
       ".tell()",
       "none",
       "an integer value");

static PyObject *
ApacheBuff_tell (ApacheBuff *self, PyObject *args)
{
  if (PyArg_ParseTuple (args, ""))
    return PyInt_FromLong (ab_tell(self));
  else
    return NULL;
}

static void
log_malformed (const char *m, int lentoshow, ApacheBuff *self)
{
#define MALFORMED_HEADER_LENGTH_TO_SHOW 30

  if (lentoshow > MALFORMED_HEADER_LENGTH_TO_SHOW)
    lentoshow = MALFORMED_HEADER_LENGTH_TO_SHOW;

  ap_log_error (APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, self->r->server,
                "access to %s failed for %s, reason: malformed header from script ('%.*s')",
                self->r->filename,
                ap_get_remote_host (self->r->connection,
                                    self->r->per_dir_config, REMOTE_NAME),
                lentoshow, m);
}

static void
fatal_error (const char *reason, request_rec *r, int traceback)
{
  ap_log_error (APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
                "access to %s failed for %s, reason: %s%s",
                r->filename,
                ap_get_remote_host (r->connection, r->per_dir_config, REMOTE_NAME),
                reason, (traceback ? ". Script's traceback follows:" : ""));
  if (traceback)
    PyErr_Print();
}

/* Handle the header. Keep this up-to-date with Apache's
   scan_script_header(). Returns -1 on error, 0 OK. */
static int
handle_header (ApacheBuff *self, char *header, int size)
{
  char *colon, *hend=header+size;

  for (colon = header; colon < hend && *colon && *colon != ':'; colon++)
    /* do nothing */;

  if (*colon != ':')
    {
      log_malformed (header, size, self);
      self->malformed_header++;

      return -1;
    }
  else
    {
      char *arg;

      *colon = '\0';
      arg = colon+1;
      while (arg < hend && *arg && isspace (*arg))
        arg++;

      if (! strcasecmp (header, "Content-type"))
        {
          /* Nuke trailing whitespace */

          char *argendp = hend-1;

          while (argendp > arg && isspace (*argendp))
            *argendp-- = '\0';

          self->r->content_type = ap_pstrdup (self->r->pool, arg);
        }
      else if (! strcasecmp (header, "Status"))
        {
          sscanf (arg, "%d", &self->r->status);
          self->r->status_line = ap_pstrdup (self->r->pool, arg);
        }
      else if (! strcasecmp (header, "Location") ||
               ! strcasecmp (header, "Content-Length") ||
               ! strcasecmp (header, "Transfer-Encoding"))
        {
          ap_table_set (self->r->headers_out, header, arg);
        }
      /*
       * If the script gave us a Last-Modified header, we can't just
       * pass it on blindly because of restrictions on future values.
       */
      else if (! strcasecmp (header, "Last-Modified"))
        {
          time_t mtime = ap_parseHTTPdate (arg);

          ap_update_mtime (self->r, mtime);
          ap_set_last_modified (self->r);
        }
      else
        /* The HTTP specification says that it is legal to
         * merge duplicate headers into one.  Some browsers
         * that support Cookies don't like merged headers and
         * prefer that each Set-Cookie header is sent
         * separately.  Lets humour those browsers.  */
        if (! strcasecmp (header, "Set-Cookie"))
          {
            ap_table_add (self->r->err_headers_out, header, arg);
          }
        else
          {
            ap_table_merge (self->r->err_headers_out, header, arg);
          }

      return 0;
    }
}

static const char *
is_redirected (ApacheBuff *self)
{
  const char *location = ap_table_get (self->r->headers_out, "Location");
  
  return (location && location[0] == '/' && self->r->status == HTTP_OK
          ? location
          : NULL);
}

static void
redirect (ApacheBuff *self, const char *location)
{
  /* Soak up all the script output */
  while (ab_getc (self) != -1)
    /* do nothing */;
  
  /* This redirect needs to be a GET no matter what the original
   * method was.
   */
  self->r->method = ap_pstrdup(self->r->pool, "GET");
  self->r->method_number = M_GET;
  
  /* We already read the message body (if any), so don't allow
   * the redirected request to think it has one.  We can ignore 
   * Transfer-Encoding, since we used REQUEST_CHUNKED_ERROR.
   */
  ap_table_unset (self->r->headers_in, "Content-Length");

  ap_internal_redirect_handler (location, self->r);
}

/* This function gets called from the ApacheBuff_write method until
   the end-of-headers (two consecutive new line characters) condition is met.

   Splitting the input STR into '\n' separated lines, recognises the
   header and updates the various tables, like scan_script_headers() does.

   Then returns the number of characters consumed, -1 on errors, -2
   when there is a valid Location header. */
static int
parse_headers (ApacheBuff *self, const char *str, int size)
{
  char *lbeg, *lend, *buff;
  int consumed = 0;

  /* Shortcut for common Python usage: when it executes "print
     'blabla'", it does actually write the string 'blabla' to the
     output stream, then it writes a single newline char. If so, maybe
     we can save a realloc and an empty scan... */
  if (*str == '\n' || (size>1 && *str == '\015' && *(str+1) == '\n'))
    {
      if (self->headers)
        {
          /* If there is an incomplete header, consider it done. */

          if (handle_header (self, self->headers, strlen (self->headers)))
            return -1;
          self->headers = NULL;

          /* If there is just the newline... */
          if (size == 1 || (size == 2 && *str == '\015'))
            return size;

          /* ...no, there is something beyond. */

          consumed = (*str == '\n' ? 1 : 2);
          str += consumed;
          size -= consumed;
        }
      else
        {
          /* Otherwise we are done! */

          const char *location;
          
          if ((location = is_redirected (self)) != NULL)
            {
              redirect (self, location);
              return size;
            }
          
          ap_send_http_header (self->r);
          self->headers_sent++;

          /* our caller will eventually write the rest of the string */
          return (*str == '\n' ? 1 : 2);
        }
    }


  /* First of all, check if previous call left something to do */
  if (self->headers)
    {
      int len = strlen (self->headers);

      buff = ap_palloc (self->r->pool, len+size+1);
      strncpy (buff, self->headers, len);
      strncpy (buff+len, str, size);
      buff[len+size] = '\0';
    }
  else
    {
      buff = ap_palloc (self->r->pool, size+1);

      strncpy (buff, str, size);
      buff[size] = '\0';
    }

  lbeg = buff;
  lend = strchr (buff, '\n');

  while (lend)
    {
      int llen = lend - lbeg;

      /* Ok, we found a eoln, but isn't it a MessyDos CR-NL pair instead? */
      if (llen)
        {
          if (llen > 1 && *(lend-1) == '\015')
            {
              llen -= 2;
              *(lend-1) = '\0';
            }
          else
            {
              llen--;
              *lend = '\0';
            }
        }

      if (llen == 0)
        {
          /* Yeah, no more headers! Now, first of all send them
             back, then finish writing this string to the client. */

          const char *location;
          
          if ((location = is_redirected (self)) != NULL)
            {
              redirect (self, location);
              return size;
            }
          
          ap_send_http_header (self->r);
          self->headers_sent++;
          lbeg = lend+1;

          break;
        }
      else
        if (handle_header (self, lbeg, llen))
          return -1;

      lbeg = lend+1;
      lend = strchr (lbeg, '\n');
    }

  if (lend == NULL)
    {
      /* We can't complete the operation since the (last) header is not
         terminated by an eoln character. So record the tail of the string
         for the next call. */

      if (*lbeg)
        self->headers = ap_pstrdup (self->r->pool, lbeg);
      else
        self->headers = NULL;

      return size;
    }
  else
    self->headers = NULL;

  if (self->malformed_header || ((lbeg - buff) != size && !self->headers_sent))
    {
      log_malformed (buff, lbeg - buff, self);
      self->malformed_header++;

      return -1;
    }
  else
    {
      consumed += lbeg - buff;

      return consumed;
    }
}
static char ApacheBuff_write_doc[] =
FUNDOC("Write a string to the client",
       ".write (string)",
       "the string to be written",
       "None");

static PyObject *
ApacheBuff_write (ApacheBuff *self, PyObject *args)
{
  char *str;
  int size;

  if (! self->malformed_header)
    {
      if (PyArg_ParseTuple (args, "s#;STRING", &str, &size))
        {
          int put;

          ap_hard_timeout ("send script output", self->r);

          /* Did we parse and send the headers? Or maybe we belong to an
             NPH script? */
          if (!self->headers_sent)
            {
              int consumed = parse_headers (self, str, size);

              if (consumed < 0)
                {
                  ap_kill_timeout (self->r);

                  PyErr_SetString (PyExc_IOError, "Script generated malformed headers");
                  return NULL;
                }

              size -= consumed;
              str += consumed;
            }

          if (size)
            {
              Py_BEGIN_ALLOW_THREADS
                put = ap_bwrite (self->r->connection->client, str, size);
              Py_END_ALLOW_THREADS

              if (put != size)
                {
                  ap_kill_timeout (self->r);

                  PyErr_SetFromErrno (PyExc_IOError);
                  return NULL;
                }
            }

          ap_kill_timeout (self->r);

          Py_INCREF(Py_None);
          return Py_None;
        }
      else
        return NULL;
    }
  else
    {
      PyErr_SetString (PyExc_IOError, "Script generated malformed headers");
      return NULL;
    }
}

/* This was suggested by Lance Ellinghaus <lance@deserTelcom.com>: it's a
   no-op method, just to let you say ``sys.stdout.flush()'' in the scripts. */

static char ApacheBuff_flush_doc[] =
FUNDOC("No-op method, for compatibility",
       ".flush()",
       "none",
       "None");

static PyObject *
ApacheBuff_flush (ApacheBuff *self, PyObject *args)
{
  if (PyArg_ParseTuple (args, ""))
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
  return NULL;
}

/* For the very same reason */

static char ApacheBuff_close_doc[] =
FUNDOC("No-op method, for compatibility",
       ".write()",
       "none",
       "None");

static PyObject *
ApacheBuff_close (ApacheBuff *self, PyObject *args)
{
  if (PyArg_ParseTuple (args, ""))
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
  return NULL;
}

static PyMethodDef ApacheBuff_methods[] =
{
  { "read",     (PyCFunction) ApacheBuff_read,     METH_VARARGS,   ApacheBuff_read_doc },
  { "readline", (PyCFunction) ApacheBuff_readline, METH_VARARGS,   ApacheBuff_readline_doc },
  { "readlines",(PyCFunction) ApacheBuff_readlines,METH_VARARGS,   ApacheBuff_readlines_doc },
  { "tell",     (PyCFunction) ApacheBuff_tell,     METH_VARARGS,   ApacheBuff_tell_doc },
  { "write",    (PyCFunction) ApacheBuff_write,    METH_VARARGS,   ApacheBuff_write_doc },
  { "flush",    (PyCFunction) ApacheBuff_flush,    METH_VARARGS,   ApacheBuff_flush_doc },
  { "close",    (PyCFunction) ApacheBuff_close,    METH_VARARGS,   ApacheBuff_close_doc },
  { 0, 0, 0, 0 }
};

#define AB_OFF(f) offsetof(ApacheBuff, f)

static struct memberlist ApacheBuff_memberlist[] =
{
  { "softspace",        T_INT,  AB_OFF(f_softspace) },
  { 0, 0, 0, 0 }
};

static PyObject *
ApacheBuff_new (request_rec *r, int nph)
{
  ApacheBuff *self = PyObject_NEW (ApacheBuff, &ApacheBuffType);

  if (self == NULL)
    return NULL;

  self->r = r;

  self->read_buffer = NULL;
  self->read_buffer_remain = 0;
  self->read_so_far = 0;

  self->malformed_header = 0;

  /* If we belong to an NPH script consider done the headers
     parse/validation step. */
  self->headers_sent = nph;

  self->headers = NULL;

  self->f_softspace = 0;

  return (PyObject *) self;
}

static void
ApacheBuff_dealloc (ApacheBuff *self)
{
#if RELEASE_TEST
  fprintf (stderr, "%s: destroying instance at %lx\n", __FUNCTION__, self);
#endif
  if (self->read_buffer)
    PyMem_DEL (self->read_buffer);

  PyMem_DEL (self);
}

static char ApacheBuff_doc[] = "\
ApacheBuff is a Python wrapper around Apache's `BUFF' object.\
";

static PyObject *
ApacheBuff_getattr (ApacheBuff *self, char *name)
{
  PyObject *m;

  if (! strcmp (name, "__doc__"))
    return PyString_FromString (ApacheBuff_doc);

  m = PyMember_Get ((char *) self, ApacheBuff_memberlist, name);
  if (m)
    return m;

  PyErr_Clear();
  return Py_FindMethod (ApacheBuff_methods, (PyObject *) self, name);
}

static int
ApacheBuff_setattr (ApacheBuff *self, char *name, PyObject *value)
{
  if (value == NULL)
    {
      PyErr_SetString (PyExc_AttributeError, "can't delete ApacheBuff attributes");
      return -1;
    }
  else
    return PyMember_Set ((char *) self, ApacheBuff_memberlist, name, value);
}

static PyTypeObject ApacheBuffType =
{
  PyObject_HEAD_INIT(&PyType_Type)
  0,                                          /*ob_size*/
  "ApacheBuff",                               /*tp_name*/
  sizeof(ApacheBuff),                         /*tp_basicsize*/
  0,                                          /*tp_itemsize*/

  /* methods */
  (destructor) ApacheBuff_dealloc,            /*tp_dealloc*/
  (printfunc) 0,                              /*tp_print*/
  (getattrfunc) ApacheBuff_getattr,           /*tp_getattr*/
  (setattrfunc) ApacheBuff_setattr,           /*tp_setattr*/
  (cmpfunc) 0,                                /*tp_compare*/
  (reprfunc) 0,                               /*tp_repr*/
  0,                                          /*tp_as_number*/
  0,                                          /*tp_as_sequence*/
  0,                                          /*tp_as_mapping*/
  (hashfunc) 0,                               /*tp_hash*/
  (ternaryfunc) 0,                            /*tp_call*/
  (reprfunc) 0,                               /*tp_str*/
  (getattrofunc) 0,                           /*tp_getattro*/
  (setattrofunc) 0,                           /*tp_setattro*/

  /* Space for future expansion */
  0L,0L,

  ApacheBuff_doc                              /* Documentation string */
};


/**********************************************************************
                       CONFIGURATION DIRECTIVES
**********************************************************************/

/* Create and initialize the per-directory config-structure.
   This is called by the server at initialization time. */
static void *
create_pyapache_dir_config (pool *p, char *dummy)
{
  pyapache_dir_config *conf = (pyapache_dir_config *) ap_palloc (p, sizeof (*conf));

  conf->path = NULL;
  conf->debug = 0;
  conf->verbose = 0;
  
  return conf;
}

/* Create and initialize the per-server config-structure.
   This is called by the server at initialization time. */
static void *
create_pyapache_server_config (pool *p, server_rec *s)
{
  pyapache_server_config *conf = (pyapache_server_config *) ap_palloc (p, sizeof (*conf));

  conf->startup = NULL;
  conf->path = NULL;
  conf->server_dict = NULL;

  return conf;
}

/* This function gets called at configuration time when a
   PyApacheStartupScript directive is found. */
static const char *
set_pyapache_startup_script (cmd_parms *parms, void *dummy, char *arg)
{
  server_rec *s = parms->server;
  pyapache_server_config *conf = (pyapache_server_config *) ap_get_module_config (s->module_config, &PyApache_module);

  conf->startup = ap_pstrdup (parms->pool, arg);

  return NULL;
}

/* This function gets called at configuration time when a
   PyApachePath directive is found. */
static const char *
set_pyapache_path (cmd_parms *parms, void *dummy, char *arg)
{
  server_rec *s = parms->server;
  pyapache_server_config *conf = (pyapache_server_config *) ap_get_module_config (s->module_config, &PyApache_module);

  conf->path = ap_pstrdup (parms->pool, arg);

  return NULL;
}

static command_rec pyapache_cmds[] =
{
  { "PythonPath",           ap_set_string_slot,
    (void*)XtOffsetOf(pyapache_dir_config, path),
    ACCESS_CONF|RSRC_CONF|OR_ALL, TAKE1,
    "the path to be used for Python Scripts" },
  { "PythonVerbose",        ap_set_flag_slot,
    (void*)XtOffsetOf(pyapache_dir_config, verbose),
    ACCESS_CONF|RSRC_CONF|OR_ALL, FLAG,
    "Set Python verbosity" },
  { "PythonDebug",          ap_set_flag_slot,
    (void*)XtOffsetOf(pyapache_dir_config, debug),
    ACCESS_CONF|RSRC_CONF|OR_ALL, FLAG,
    "Set Python debug level" },

  { "PyApacheStartupScript",  set_pyapache_startup_script,
    NULL,
    RSRC_CONF, TAKE1,
    "a Python Script to be executed at PyApache server startup" },
  { "PyApachePath",  set_pyapache_path,
    NULL,
    RSRC_CONF, TAKE1,
    "the path to be used by the PyApache server" },

  { NULL }
};


/**********************************************************************
                               HANDLERS
**********************************************************************/

typedef struct pyapache_req_stuff
{
  request_rec *r;
  pid_t pid;
  char *argv0;                  /* This is r->filename with the path stripped off */
  FILE *script_in;              /* This gets attached to the script's stdin */
  FILE *script_out;             /* and this to its stdout, if not NPH. */
  FILE *script_err;             /* This gets attached to the script's stderr. */
  int nph;                      /* If != 0 this is an NPH script */
} pyapache_req_stuff;

/*XXX I do not understand very well the logic here! Is this needed?*/
static int
is_scriptaliased (request_rec *r)
{
  const char *t = ap_table_get (r->notes, "alias-forced-type");
  return t && (!strcmp(t, "cgi-script"));
}

/* Validates the script, initializing the `argv0' and `nph' slots of
   PRSP. Returns OK or the error code. */
static int
check (pyapache_req_stuff *prsp)
{
  request_rec *r = prsp->r;

  if ((prsp->argv0 = strrchr (prsp->r->filename, '/')) != NULL)
    prsp->argv0++;
  else
    prsp->argv0 = prsp->r->filename;

  /* If the name of the script begins with "nph-" then we are not going
     to parse the headers coming from it, but we will pass the script output
     "as is" to the client. */
  prsp->nph = !(strncmp (prsp->argv0, "nph-", 4));

  if (!(ap_allow_options (r) & OPT_EXECCGI) && !is_scriptaliased (r))
    {
      fatal_error ("option ExecCGI is off in this directory", r, 0);
      return FORBIDDEN;
    }

  if (prsp->nph && !strcmp (r->protocol, "INCLUDED"))
    {
      fatal_error ("attempt to include NPH CGI script", r, 0);
      return FORBIDDEN;
    }

  if (S_ISDIR(r->finfo.st_mode))
    {
      fatal_error ("attempt to invoke directory as script", r, 0);
      return FORBIDDEN;
    }

  if (r->finfo.st_mode == 0)
    {
      ap_log_reason ("script not found or unable to stat", r->filename, r);
      return NOT_FOUND;
    }

  if (! ap_can_exec (&r->finfo))
    {
      ap_log_reason ("file permissions deny server execution", r->filename, r);
      return FORBIDDEN;
    }

  return ap_setup_client_block (r, REQUEST_CHUNKED_ERROR);
}

/* This is the equivalent of util_script.c::create_argv(). Since about
   1.2b6 it has been removed from the public API, and this forced me to
   either patch the server' sources or to clone it here... */
static void
set_cmdline_args (pyapache_req_stuff *prsp)
{
  const char *args = prsp->r->args;

  /* If there are no params in the query and this is not FORM
     submission, just set the name of the script as the first
     argument. */
  if (!args || !args[0] || (ap_ind (args, '=') >= 0))
    PySys_SetArgv (1, &prsp->argv0);
  else
    {
      int idx, argc;
      char **argv;
      pool *p = prsp->r->pool;

      /* The query info is split into separate arguments, where
         "+" is the separator between keyword arguments. */

      for (idx=0, argc=1; args[idx]; idx++)
        if (args[idx] == '+')
          argc++;

      /* Truncate args to prevent overrun */
      if (argc > APACHE_ARG_MAX - 5)
        argc = APACHE_ARG_MAX - 5;

      /* adjust for argv0 */
      argc++;

      argv = ap_palloc (p, (argc+1) * sizeof (char *));

      argv[0] = prsp->argv0;
      for (idx=1; idx < argc; idx++)
        {
          char *arg = ap_getword_nulls (p, &args, '+');

          ap_unescape_url (arg);
          argv[idx] = ap_escape_shell_cmd (p, arg);
        }

      argv[idx] = NULL;

      PySys_SetArgv (argc, argv);
    }
}

static int
setup_environ (pyapache_req_stuff *prsp)
{
  PyObject *os_module = PyImport_ImportModule ("os");

  if (!os_module)
    {
      fatal_error ("cannot import ``os'' module", prsp->r, 1);
      return SERVER_ERROR;
    }
  else
    {
      int i, status = OK;
      PyObject *module_dict;
      PyObject *environ_dict;
#if APACHE_RELEASE>1030000
      array_header *env_arr = ap_table_elts (prsp->r->subprocess_env);
#else
      array_header *env_arr = table_elts (prsp->r->subprocess_env);
#endif
      table_entry *elts = (table_entry *) env_arr->elts;
      char *tz = getenv ("TZ");

      module_dict = PyModule_GetDict (os_module);

      /* extract the current environ dictionary from the OS module,
        make it empty and then fill it with our variables; in this
        way we keep `os.environ' and `posix.environ' in sync: since
        they actually are the same object, inserting a new dict in
        one of the two modules would break the link. */

      environ_dict = PyDict_GetItemString (module_dict, "environ");

      /* Python 1.4' os.py replaces, where possible, the environ
        dictionary member with a subclass of UserDict that calls
        putenv() in its __setitem__ method, to keep the real
        environment in sync. */
      if (! PyDict_Check (environ_dict))
        {
          PyObject *data = PyObject_GetAttrString (environ_dict, "data");

          if (data && PyDict_Check (data))
            PyDict_Clear (data);
          else
            {
              fatal_error ("cannot update script environment, unhandled environ type", prsp->r, 0);
              status = SERVER_ERROR;
            }
        }
      else
        PyDict_Clear (environ_dict);

      if (status == OK)
        {
          if (tz != NULL)
            {
              PyObject *ptz = PyString_FromString (tz);

              PyMapping_SetItemString (environ_dict, "TZ", ptz);
              Py_DECREF (ptz);
            }

          PyMapping_SetItemString (environ_dict, "PyApache_VERSION",
                                   PyApache_version);

          i = env_arr->nelts;
          while (i--)
            {
              if (elts[i].key)
                {
                  PyObject *value = PyString_FromString (elts[i].val);

                  PyMapping_SetItemString (environ_dict, elts[i].key, value);
                  Py_DECREF (value);
                }
            }
        }
      Py_DECREF (os_module);

      return status;
    }
}

/* Checks the existence of a valid compiled script. If it is there, returns
   an opened file on it, otherwise NULL.
   Pass 0 as MTIME to disable the check on the timestamp, for example when
   you do not have the source, but only the pyc. */
static FILE *
check_compiled_module (const char *cpathname, long mtime)
{
  FILE *fp;
  long magic;
  long pyc_mtime;

  fp = fopen (cpathname, "rb");
  if (fp == NULL)
    return NULL;

  magic = PyMarshal_ReadLongFromFile (fp);
  if (magic != PyImport_GetMagicNumber())
    {
      fclose(fp);
      return NULL;
    }

  pyc_mtime = PyMarshal_ReadLongFromFile (fp);
  if (mtime && pyc_mtime != mtime)
    {
      fclose(fp);
      return NULL;
    }

  return fp;
}

/* Actually execute a Python script given the name: if it's a source file
   (the name doesn't end in 'c'), a check is made for a valid compiled
   version. If it doesn't exist try to write out the code after the
   compilation.
   Returns a string with the error or NULL if all right. */
static const char *
exec_the_script (PyObject *globals, const char *script, long mtime, pool *p)
{
  char *compiled = NULL;
  unsigned int slen;
  FILE *fp;
  PyCodeObject *code;
  PyObject *main, *maindict, *result, *error, *builtins;

  /* Running the script with a simpler PyRun_SimpleFile() has one major
     disadvantage: it does not look for the compiled script (.pyc). On
     the other hand, importing the script as a module and then executing
     it wouldn't perform exacty what we want: for example the common
     usage of doing something like 'if __name__ == "__main__": main()' to
     let the script be used both as an executable script as well as a
     module will fail, because in that way the script in not executed in
     the '__main__' namespace...

     So here I had to study the Python implementation and, cutting&pasting,
     write another way, merging their best qualities. */

  main = PyImport_AddModule ("__main__");
  if (!main)
    return "Cannot add __main__ module";

  maindict = PyModule_GetDict (main);
  builtins = PyEval_GetBuiltins();
  
  if (PyDict_SetItemString (builtins, "__persistdict__", globals))
    return "Cannot add __persistdict__ to __builtins__ dict";
  
  slen = strlen (script);

  fp = NULL;

  /* Maybe the resource points to a pyc file already */

  if (script[slen-1] == 'c' ||
      script[slen-1] == 'o')
    fp = check_compiled_module (script, 0L);

  if (!fp)
    {
      /* No, let's try adding a 'o' to the name */

      compiled = ap_palloc (p, slen+2);
      strcpy (compiled, script);

      compiled[slen] = 'o';
      compiled[slen+1] = '\0';

      /* Does the optimized version exists ? */
      fp = check_compiled_module (compiled, mtime);

      if (!fp)
        {
          /* No, let's try with the pyc */

          compiled[slen] = 'c';

          fp = check_compiled_module (compiled, mtime);
        }
    }

  if (fp)
    {
      /* Ok, a valid compiled script exist. */

      code = (PyCodeObject *) PyMarshal_ReadObjectFromFile (fp);

      fclose (fp);

      if (! code)
        return "cannot load compiled codeobject";
    }
  else
    {
      if (script[slen-1] == 'c' ||
          script[slen-1] == 'o')
        /* XXX The resource refers to a pyc file. Should we
          try to load from the source name computed by
          stripping the ending 'c'? */

        return "PYC/PYO file corrupted or out of date";
      else
        {
          /* No, there is no valid compiled code there: parse and compile the
            source then try to dump the compiled code. */

          struct _node *n;
          struct stat cinfo;

          fp = fopen (script, "r");

          if (! fp)
            return "cannot open the Python Script";

          n = PyParser_SimpleParseFile (fp, (char*) script, Py_file_input);

          fclose (fp);

          if (! n)
            return "cannot parse the Python Script";

          code = PyNode_Compile (n, (char*) script);
          PyNode_Free (n);

          if (! code)
            return "cannot compile the Python Script";

          /* Be sure the compiled filename does not exist: it it does, maybe
            some other process was in the process of writing it. */
          if (stat (compiled, &cinfo))
            {
              FILE *compfile;

              /* Write out the compiled version */
              compfile = fopen (compiled, "wb");
              if (compfile)
                {
                  PyMarshal_WriteLongToFile (PyImport_GetMagicNumber(), compfile);

                  /* First write a 0 for mtime */
                  PyMarshal_WriteLongToFile (0L, compfile);

                  PyMarshal_WriteObjectToFile ((PyObject *) code, compfile);

                  if (ferror (compfile))
                    {
                      /* Don't keep partial file */
                      fclose (compfile);
                      (void) unlink (compiled);
                    }
                  else
                    {
                      /* Now write the true mtime */
                      fseek (compfile, 4L, 0);
                      PyMarshal_WriteLongToFile (mtime,
                                                 compfile);

                      fflush (compfile);
                      fclose (compfile);
                    }
                }
            }
        }
    }

  PyErr_Clear();
  
  result = PyEval_EvalCode (code, maindict, maindict);
  Py_DECREF (code);

  Py_XDECREF (result);

  if ((error = PyErr_Occurred()))
    if (! PyErr_GivenExceptionMatches (error, PyExc_SystemExit))
      return "the script raised an unhandled exception";

  return NULL;                  /* that means OK! */
}

/* Execute the script in the same process space of the server but in a
   new separated subinterpreter.
   Hook an ApacheBuff instance to the script's stdin & stdout: in this
   way the script will read incoming data directly from the server, and
   will write its output directly to the client.
   Once done, destroy the subinterpreter. */
static int
execute_python_script (pyapache_req_stuff *prsp)
{
  PyThreadState *new_tstate, *old_tstate;
  int status = OK;

#ifdef WITH_THREAD
  PyEval_AcquireLock();
#endif
  old_tstate = PyThreadState_Swap (NULL);
  
  new_tstate = Py_NewInterpreter();
  if (new_tstate == NULL)
    {
      fatal_error ("Sorry -- can't create an interpreter", prsp->r, 1);
      status = SERVER_ERROR;
    }
  else
    {
      pyapache_dir_config *conf =
        (pyapache_dir_config *) ap_get_module_config (prsp->r->per_dir_config,
                                                      &PyApache_module);

      if (conf->path)
        {
          PyObject *path = PySys_GetObject ("path");
          PyObject *dirpath = PyString_FromString (conf->path);

          PyList_Insert (path, 0, dirpath);
          Py_DECREF(dirpath);
        }
      
      /* Set the command line arguments list */
      set_cmdline_args (prsp);

      /* Reinitialize os.environ. */
      status = setup_environ (prsp);

      if (status == OK)
        {
          /* Setup Input/Output channels */
          const char *error;
          PyObject *err, *buff;
          pyapache_server_config *sconf = ap_get_module_config (prsp->r->server->module_config,
                                                                &PyApache_module);
          
          /* Hook up the ApacheBuff to both stdin and stdout file objects of
            the script. In this way the script can comunicate directly with
            our client, without other intervention by us. */

          buff = ApacheBuff_new (prsp->r, prsp->nph);

          PySys_SetObject ("stdin", buff);
          PySys_SetObject ("stdout", buff);

          Py_DECREF (buff);

          /* Hook up stderr to server's error log */

          err = PyFile_FromFile (prsp->r->server->error_log, "<stderr>", "w", NULL);
          PySys_SetObject ("stderr", err);

          Py_DECREF (err);

          error = exec_the_script (sconf->server_dict,
                                   prsp->r->filename,
                                   prsp->r->finfo.st_mtime,
                                   prsp->r->pool);
          if (error)
            {
              fatal_error (error, prsp->r, 1);
              status = SERVER_ERROR;
            }
          else
            status = OK;

          fflush (prsp->r->server->error_log);
        }
    }

  /* Since we are processing all errors via Apache's logs, do not
     propagate any exception raised by the execution. */
  PyErr_Clear();

  Py_EndInterpreter (new_tstate);

  PyThreadState_Swap (old_tstate);
  
#ifdef WITH_THREAD
  PyEval_ReleaseLock();
#endif

  if (status == OK)
    status = ap_meets_conditions (prsp->r);

  return status;
}

/* This is the entry point: this function is called by the server to
   handle our preferite script type. */
static int
python_handler (request_rec *r)
{
  int ret;
  pyapache_req_stuff prs;
  pyapache_dir_config *conf = (pyapache_dir_config *) ap_get_module_config (r->per_dir_config, &PyApache_module);

  prs.r = r;

  /* These other members of prs get initialized by check():
     prs.argv0, prs.nph */

  if ((ret = check (&prs)) == OK)
    {
      ap_chdir_file (r->filename);

      ap_add_common_vars (r);
      ap_add_cgi_vars (r);

      Py_VerboseFlag = conf->verbose;
      Py_DebugFlag = conf->debug;

      ret = execute_python_script (&prs);
    }

  return ret;

}

static void
pyapache_init (server_rec *s, pool *p)
{
  const char *revision = "4.14";
  pyapache_server_config *conf = (pyapache_server_config *) ap_get_module_config (s->module_config, &PyApache_module);

  /* Did we exported the sources with `-kv'? */
  if (*revision == '$')
    {
      const char *start, *end;

      start = revision + sizeof ("$Revision:");
      for (end = start; *end && *end != ' '; end++)
        /* do nothing */;

      PyApache_version = PyString_FromStringAndSize ((char *) start, end-start);
    }
  else
    PyApache_version = PyString_FromString ((char *) revision);

  ap_error_log2stderr (s);

  /* This initialize the Python environment and creates the main
     interpreter, the only one that will never be freed. This of
     course for each single server child. */
  Py_Initialize();
#ifdef WITH_THREAD
  PyEval_InitThreads();
  PyEval_ReleaseLock();
#endif
  
  /* Setup the server wide name space: the scripts will receive it as
     their global name space. */
  conf->server_dict = PyDict_New();

  if (conf->startup)
    {
      struct stat sinfo;

      if (!stat (conf->startup, &sinfo))
        {
          const char *error;

          if (conf->path)
            {
              PyObject *path = PySys_GetObject ("path");
              PyObject *sp = PyString_FromString (conf->path);

              PyList_Insert (path, 0, sp);
              Py_DECREF(sp);
            }
          
          error = exec_the_script (conf->server_dict,
                                   conf->startup, sinfo.st_mtime, p);

          if (error)
            {
              fprintf (s->error_log, "Error executing startup script: %s", error);
              PyErr_Print();
            }
        }
    }
}

static handler_rec pyapache_handlers[] = {
  { PYTHON_MAGIC_TYPE, python_handler },
  { ALT_PYTHON_MAGIC_TYPE, python_handler },
  { NULL }
};

module PyApache_module = {
   STANDARD_MODULE_STUFF,
   pyapache_init,               /* initializer */
   create_pyapache_dir_config,  /* dir config creater */
   NULL,                        /* dir merger --- default is to override */
   create_pyapache_server_config, /* server config */
   NULL,                        /* merge server config */
   pyapache_cmds,               /* command table */
   pyapache_handlers,           /* handlers */
   NULL,                        /* filename translation */
   NULL,                        /* check_user_id */
   NULL,                        /* check auth */
   NULL,                        /* check access */
   NULL,                        /* type_checker */
   NULL,                        /* fixups */
   NULL,                        /* logger */
   NULL                         /* header parser */
};

/*
 * MODULE-DEFINITION-START
Name: PyApache
ConfigStart
PyVERSION=`python1.5 -c "import sys; print sys.version[:3]"`
PyEXEC_INSTALLDIR=`python1.5 -c "import sys; print sys.exec_prefix"`
PyLIBP=${PyEXEC_INSTALLDIR}/lib/python${PyVERSION}
PyLIBPL=${PyLIBP}/config
PyLIBS=`grep "^LIB[SMC]=" ${PyLIBPL}/Makefile | cut -f2 -d= | tr '\011\012\015' '   '`
PyPYTHONLIBS=${PyLIBPL}/libpython${PyVERSION}.a
LIBS="${LIBS} ${PyPYTHONLIBS} ${PyLIBS}"
ConfigEnd
 * MODULE-DEFINITION-END
 */

/*
** Local Variables:
** change-log-default-name: "ChangeLog"
** eval: (add-hook 'write-contents-hooks '(lambda () (untabify 0 (point-max)) nil) t nil)
** End:
*/
