/*
 * te_demo.c 
 *
 * written by Andrew Apted.  <andrew@ggi-project.org>
 *
 * This is a demonstration of LibGGI's functions and can be used as a
 * reference programming example.
 *
 *   This program is placed in the public domain and can be used freely
 *   for any purpose. It comes with absolutely NO WARRANTY, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   USE IT AT YOUR OWN RISK. The author is not responsible for any
 *   damage or consequences raised by use or inability to use this
 *   program.
 *
 * Main use is to demonstrate the TrueEmu display target.
 *
 * USAGE:  te_demo  [ --mode <mode> ]  [ --target <target> ]
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <math.h>

#include <ggi/ggi.h>


#ifndef MAX
#define MAX(a,b)  ((a) > (b) ? (a) : (b))
#endif 

#ifndef MIN
#define MIN(a,b)  ((a) < (b) ? (a) : (b))
#endif 


static void usage(void)
{
	fprintf(stderr, "\nUSAGE:  te_demo  [ --mode <mode> ] "
		"[ --target <target> ]\n\n");
}

#define CMP_OPT(short, long, extra)  \
	((argc >= (extra+1)) && \
	 ((strcmp(*argv, short) == 0) ||  \
	  (strcmp(*argv, long) == 0)))

int main(int argc, char **argv)
{ 
	ggi_visual_t vis;

	int x, y, w, h;
	int quit=0;

	ggi_mode mode;
	
	char *prog;
	char *target_str = NULL;

	
	/* initialize */
	
	ggiParseMode("", &mode);


	/* handle args */

	prog = *argv++; argc--;

	while (argc > 0) {

		if (CMP_OPT("-h", "--help", 0)) {

			usage();
			exit(1);
		}

		if (CMP_OPT("-m", "--mode", 1)) {

			ggiParseMode(argv[1], &mode);
			argv += 2; argc -= 2;
			continue;
		}

		if (CMP_OPT("-t", "--target", 1)) {

			target_str = argv[1];
			argv += 2; argc -= 2;
			continue;
		}
			
		fprintf(stderr, "te_demo: Unknown option '%s'.\n",
			*argv);
		usage();
		exit(1);
	}


	/* setup graphics mode */

	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, "
			"exiting.\n", argv[0]);
		exit(1);
	}


	vis = ggiOpen(target_str, NULL);

	if (vis == NULL) {
		fprintf(stderr, "te_demo: Failed to open visual.\n");
		ggiExit();
		exit(1);
	}

	if (GT_SCHEME(mode.graphtype) == GT_AUTO) {
		mode.graphtype |= GT_TRUECOLOR;
	}
	
	ggiCheckMode(vis, &mode);

	if (ggiSetMode(vis, &mode) < 0) {
		fprintf(stderr, "te_demo: Could not set mode.\n");
		ggiClose(vis);
		ggiExit();
		exit(1);
	}

	if (GT_SCHEME(mode.graphtype) != GT_TRUECOLOR) {
		fprintf(stderr, "te_demo: Display is not truecolor !\n");
		ggiClose(vis);
		ggiExit();
		exit(1);
	}
	
	ggiSetFlags(vis, GGIFLAG_ASYNC);

	fprintf(stderr, "te_demo: Mode is: ");
	ggiFPrintMode(stderr, &mode);
	fprintf(stderr, "\n");

	w = mode.virt.x;
	h = mode.virt.y;

	/* draw a colorful pattern */

	for (y=0; y < h/2; y++)
	for (x=0; x < w/2; x++) {

		int xi = (w/2-x-1) * 0x10000 / (w/2);
		int yi = (h/2-y-1) * 0x10000 / (h/2);
		int zi = MAX(xi, yi);

		ggi_color R_G;
		ggi_color R_B;
		ggi_color G_B;
		ggi_color B_W;


		R_G.r = yi;  R_G.g = xi;  R_G.b = 0;
		R_B.r = yi;  R_B.g = 0;   R_B.b = xi;
		G_B.r = 0;   G_B.g = xi;  G_B.b = yi;
		B_W.r = zi;  B_W.g = zi;  B_W.b = zi;

		ggiPutPixel(vis, x,     y,     ggiMapColor(vis, &R_G));
		ggiPutPixel(vis, w-x-1, y,     ggiMapColor(vis, &R_B));
		ggiPutPixel(vis, x,     h-y-1, ggiMapColor(vis, &G_B));
		ggiPutPixel(vis, w-x-1, h-y-1, ggiMapColor(vis, &B_W));
	}

	ggiFlush(vis);

	while (! quit) {

		int sym = ggiGetc(vis);

		switch (sym) {
			case 'q': case 'Q':
			case 'x': case 'X':
			case '\033':
				quit = 1;
				break;

			default:
				printf("Key sym=0x%04x.\n", sym);
		}
	}
		
	ggiClose(vis);
	ggiExit();

	exit(0);
}
