/*
   LibGGI GLIDE target - event management

   Copyright (C) 1998  Andrew Apted  [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>
#include "glidevisual.h"

#define WANT_TRANSLATE_SHIFT  1

#include "../common/evqueue.inc"
#include "../Linux_common/vtswitch.inc"
#include "../Linux_common/key_trans.inc"
#include "../Linux_common/keyboard.inc"
#include "../Linux_common/mouse.inc"


/* ---------------------------------------------------------------------- */


ggi_event_mask
GGI_glide_eventpoll(ggi_visual_t vis, ggi_event_mask mask,
		    struct timeval *t)
{
	ggi_event_mask evmask;

	int err;
	
	DPRINT_EVENTS("display-glide: eventpoll mask=0x%08x ", mask);

	if (t == NULL) {
		DPRINT_EVENTS("timeout=NULL\n");
	} else {
		DPRINT_EVENTS("timeout=(%d,%d)\n", t->tv_sec, t->tv_usec);
	}


	for (evmask = _ggiEvQueueSeen(vis, mask);
	     evmask == 0;
	     evmask = _ggiEvQueueSeen(vis, mask)) {

		fd_set fds;
		int fdmax;

		FD_ZERO(&fds);
		
		FD_SET(KEYBOARD_FD(vis), &fds);
		fdmax = KEYBOARD_FD(vis)+1;

		if (MOUSE_FD(vis) >= 0) {
			FD_SET(MOUSE_FD(vis), &fds);
			if (MOUSE_FD(vis) > fdmax) {
				fdmax = MOUSE_FD(vis)+1;
			}
		}

		/* NB: The following EINTR handling assumes the Linux
		 * behaviour of select(), which modifies the timeout to
		 * indicate the time not slept.
		 */

		err = select(fdmax+1, &fds, NULL, NULL, t);

		if ((err < 0) && (errno == EINTR)) {
			continue;
		
		} else if (err < 0) {
			DPRINT_EVENTS("display-glide: select error.\n");
			return 0;

		} if (err == 0) {   /* select timed-out */
			DPRINT_EVENTS("display-glide: eventpoll timed out.\n");
			return 0;
		}


		if (FD_ISSET(KEYBOARD_FD(vis), &fds)) {
			keyboard_handle_data(vis);

		} else if ((MOUSE_FD(vis) >= 0) &&
		           FD_ISSET(MOUSE_FD(vis), &fds)) {
			mouse_handle_data(vis);
		}
	}

	return evmask;
}


int GGI_glide_eventread(ggi_visual_t vis, ggi_event *ev, ggi_event_mask mask)
{
	/* Block if we don't have anything queued... */
	
	GGI_glide_eventpoll(vis, mask, NULL);

	return _ggiEvQueueRelease(vis, ev, mask);
}
