/* $Id: palemu.c,v 1.9 1998/10/13 00:30:37 marcus Exp $
***************************************************************************

   Display-palemu: palette emulation on true-color modes

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>

#include "palemu.h"


/**************************************************
 ***
 ***  Internal functions 
 ***
 **************************************************/


static void blitter_1(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8 *s = (uint8 *) src;
	uint8 *d = (uint8 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}

static void blitter_2(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8  *s = (uint8  *) src;
	uint16 *d = (uint16 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}

static void blitter_3(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8 *s = (uint8 *) src;
	uint8 *d = (uint8 *) dest;

	for (; w > 0; w--) {
		ggi_pixel pix = ph->lookup[*s++];

		*d++ = pix; pix >>= 8;
		*d++ = pix; pix >>= 8;
		*d++ = pix;
	}
}

static void blitter_4(PalemuHook *ph, void *dest, void *src, int w)
{
	uint8  *s = (uint8  *) src;
	uint32 *d = (uint32 *) dest;

	for (; w > 0; w--) {
		*d++ = ph->lookup[*s++];
	}
}


/**************************************************
 ***
 ***  Exported functions 
 ***
 **************************************************/


static uint8 src_buf[8192];
static uint8 dest_buf[8192];


/* !!! flesh out all four possibilities: (a) direct access to source
 * (Y/N), and (b) direct access to destination (Y/N).
 */
		
int _ggi_palemu_Transfer(ggi_visual *vis, int x, int y, int w, int h)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int old_r_frame = vis->r_frame_num;

	ph->mem_opdraw->setreadframe(vis, vis->d_frame_num);

	
	/* do transfer */

	for (; h > 0; h--, y++) {

		ggiGetHLine(vis, x, y, w, src_buf);
			
		(* ph->do_blit)(ph, dest_buf, src_buf, w);

		ggiPutHLine(ph->parent, x, y, w, dest_buf);
	}

	ph->mem_opdraw->setreadframe(vis, old_r_frame);

	return 0;
}

int _ggi_palemu_Flush(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int sx = ph->dirty_tl.x; int sy = ph->dirty_tl.y;
	int ex = ph->dirty_br.x; int ey = ph->dirty_br.y;


	/* clear the `dirty region' */

	ph->dirty_tl.x = LIBGGI_VIRTX(vis);
	ph->dirty_tl.y = LIBGGI_VIRTY(vis);
	ph->dirty_br.x = 0;
	ph->dirty_br.y = 0;


	/* When write_frame != display_frame, then there is no need to
	 * update the parent since the affected area(s) are not visible.
	 */
	 
	if ((vis->w_frame_num == vis->d_frame_num) && 
	    (sx < ex) && (sy < ey)) {

		return _ggi_palemu_Transfer(vis, sx, sy, ex-sx, ey-sy);
	}

	return 0;
}

int _ggi_palemu_Open(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);


	DPRINT("display-palemu: Open %dx%d#%dx%d\n", LIBGGI_X(vis), 
		LIBGGI_Y(vis), LIBGGI_VIRTX(vis), LIBGGI_VIRTY(vis));


	/* set the parent mode */
	
	if (ggiSetMode(ph->parent, &ph->mode) < 0) {

		DPRINT("display-palemu: Couldn't set parent mode.\n");
		return -1;
	}

	DPRINT("display-palemu: parent is %d/%d\n",
		GT_DEPTH(ph->mode.graphtype), 
		GT_SIZE(ph->mode.graphtype));
	

	/* setup tables and choose blitter function */

	switch ((GT_SIZE(ph->mode.graphtype) + 7) / 8) {

	case 1: ph->do_blit = &blitter_1;
		break;
		
	case 2: ph->do_blit = &blitter_2;
		break;
		
	case 3: ph->do_blit = &blitter_3;
		break;
		
	case 4: ph->do_blit = &blitter_4;
		break;
		
	default:
		DPRINT("Unsupported pixel size '%d'.\n",
			GT_SIZE(ph->mode.graphtype));
		return -1;
	}


	ph->palette = _ggi_malloc(256 * sizeof(ggi_color));
	ph->lookup  = _ggi_malloc(256 * sizeof(ggi_pixel));

	ph->red_gamma = ph->green_gamma = ph->blue_gamma = 1.0;

	
	/* clear the 'dirty region' */

	ph->dirty_tl.x = LIBGGI_VIRTX(vis);
	ph->dirty_tl.y = LIBGGI_VIRTY(vis);
	ph->dirty_br.x = 0;
	ph->dirty_br.y = 0;

	return 0;
}

int _ggi_palemu_Close(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	if (ph->palette != NULL) {
		free(ph->palette);
	}
	if (ph->lookup != NULL) {
		free(ph->lookup);
	}

	return 0;
}
