/* 
 *  polygon drawing
 *
 *  Copyright (C) 1998 Thomas Tanner. See CREDITS for details.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include <malloc.h>
#include "lib.h"
#include "util.h"

/*----------------------------------------------------------------------*/

void	GGIEXP(trapezoid)(ggi_visual_t vis, ggi_sint xl1, ggi_sint xr1, ggi_sint y1, ggi_sint xl2, ggi_sint xr2, ggi_sint y2)
{
	ggi_sint	y, ye;
	linecontext a, b;
	ggi_sint	dxl, dxr, dy;
	void 	(*scanline)(ggi_visual_t, ggi_sint, ggi_sint, ggi_sint) 
		= GGI2D_FUNC(vis)->scanline;

	if (y1 > y2) {
		swap(xl1, xl2);
		swap(xr1, xr2);
		swap(y1, y2);
	};
	dxl = xl2 - xl1; 
	dxr = xr2 - xr1; 
	dy  = y2 - y1;
	lineInit(a, xl1, dxl, dy);
	lineInit(b, xr1, dxr, dy);
	y = y1; 
	ye = (y2 < GGI2D_GC_CLIP_Y2(vis)) ? y2 : GGI2D_GC_CLIP_Y2(vis)-1;
	while (y < ye) {
		scanline(vis, a.x, b.x, y);
		lineNext(a); lineNext(b); y++;
	}
	scanline(vis, a.x, b.x, y);
}

void	GGIEXP(triangle)(ggi_visual_t vis, ggi_sint x1, ggi_sint y1, ggi_sint x2, ggi_sint y2, ggi_sint x3, ggi_sint y3)
{
	ggi_sint	y, ye, tri;
	linecontext a, b;
	ggi_sint	dxa, dya, dxb, dyb;
	void 	(*scanline)(ggi_visual_t, ggi_sint, ggi_sint, ggi_sint) 
		= GGI2D_FUNC(vis)->scanline;

	if (y1 > y2) {
		swap(x1, x2);
		swap(y1, y2);
	}
	if (y2 > y3) {
		swap(x2, x3);
		swap(y2, y3);
		if (y1 > y2) {
			swap(x1, x2);
			swap(y1, y2);
		}
	}
	dxa = x3 - x1; dya = y3 - y1;
	dxb = x2 - x1; dyb = y2 - y1;
	lineInit(a, x1, dxa, dya);
	lineInit(b, x1, dxb, dyb);
	y = y1; 
	ye = (y2 < GGI2D_GC_CLIP_Y2(vis)) ? y2 : GGI2D_GC_CLIP_Y2(vis)-1;
	for (tri = 0; tri <= 1; tri++) {
		while (y < ye) {
			scanline(vis, a.x, b.x, y);
			lineNext(a); lineNext(b); y++;
		}
		scanline(vis, a.x, b.x, y);
		if (!tri) {
			if (y2 >= GGI2D_GC_CLIP_Y2(vis)) return;
			dxb = x3 - x2; dyb = y3 - y2;
			lineInit(b, x2, dxb, dyb);
			ye = (y3 < GGI2D_GC_CLIP_Y2(vis)) ? 
				y3 : GGI2D_GC_CLIP_Y2(vis)-1;
		}
	}
}

static void	drawPoly(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count)
{
	ggi_sint	i, lx, ly;
	void 	(*line)(ggi_visual_t, ggi_sint, ggi_sint, ggi_sint, ggi_sint) 
		= GGI2D_FUNC(vis)->line;
	
	lx = coords[0].x; ly = coords[0].y;
	if (count == 1) {
		GGI2D_FUNC(vis)->drawPixel(vis, lx, ly);
		return;
	}
	for (i = 1; i < count; i++) {
		line(vis, lx, ly, coords[i].x, coords[i].y);
		lx = coords[i].x; ly = coords[i].y;
	}
}

static void	drawPoly_st(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count)
{
	int	append = GGI2D_GC_APPEND(vis);
	if (!append)	ggi2dSetAppendMode(vis, 1);
	drawPoly(vis, coords, count);
	if (!append)	ggi2dSetAppendMode(vis, 0);
}

#define MAX_XP  50

static void	sortXP(ggi_sint *x, ggi_sint *dir, ggi_sint l, ggi_sint r)
{
	ggi_sint	i, j, v;

	i = l; j = r; v = x[(l+r) >> 1];
	do {
		while (x[i] < v) i++;
		while (v < x[j]) j--;
		if (i <= j) {
			swap(x[i],x[j]);
			swap(dir[i],dir[j]);
			i++; j--;
		}
	} while (i <= j);
	if (l < j) sortXP(x, dir, l, j);
	if (i < r) sortXP(x, dir, i, r);
}

static void	fillAreas(ggi_visual_t vis, polyline lines[], ggi_uint count,
			      ggi_uint miny, ggi_uint maxy)
{
/* FIXME: generalize+optimize (single poggi_ss, lines) */

	ggi_uint	xpcount, y, i;
	ggi_sint	xp[MAX_XP], dir[MAX_XP];
	void 	(*scanline)(ggi_visual_t, ggi_sint, ggi_sint, ggi_sint) 
		= GGI2D_FUNC(vis)->scanline;

	if (miny >= GGI2D_GC_CLIP_Y2(vis) || maxy < GGI2D_GC_CLIP_Y1(vis)) 
		return;
	if (miny <  GGI2D_GC_CLIP_Y1(vis)) 
		miny = GGI2D_GC_CLIP_Y1(vis);
	if (maxy >= GGI2D_GC_CLIP_Y2(vis)) 
		maxy = GGI2D_GC_CLIP_Y2(vis)-1;
	for (y = miny; y < maxy; y++) {
		xpcount = 0;
		for (i = 0; i < count; i++) {
			if ((lines[i].y <= y) && (y < lines[i].ye)) {
				ggi_sint n = y - lines[i].y;
				lines[i].y += n;
				while (n--) lineNext(lines[i].context);
			}
			if (lines[i].y == y) {
				xp[xpcount] = lines[i].context.x;
				dir[xpcount] = lines[i].dir;
     				xpcount++;
    			}
   		}
		sortXP(xp, dir, 0, xpcount-1);
		if (GGI2D_GC_POLYMODE(vis) == GGI2D_POLY_WINDING) {
			ggi_sint	d = 0;
			for (i = 0; i < xpcount; i++) {
				if (d != 0)
					scanline(vis, xp[i-1], xp[i], y);
				d += dir[i];
			}
		} else {	/* GGI2D_POLY_EVENODD */
			i = 0;
			while (i < xpcount) {
				scanline(vis, xp[i], xp[i+1], y);
				i += 2;
			}
		}
	}
}

static ggi_uint	addPoly(ggi2d_coord coords[], ggi_uint count,	polyline *polylines, 
			ggi_sint *miny, ggi_sint *maxy)
{
	ggi_sint	i, x1, y1, x2, y2;
	ggi_uint 	lcount = count - 1;

	for (i = 0; i < count; i++) {
		x1 = coords[i].x;   
		y1 = coords[i].y;
		x2 = coords[i+1].x; 
		y2 = coords[i+1].y;
		if (y1 < y2) {
			if (y1 < *miny) *miny = y1;
			if (y2 > *maxy) *maxy = y2;
		} else {
			if (y2 < *miny) *miny = y2;
			if (y1 > *maxy) *maxy = y1;
		}
		if (y1 != y2) 
			polylineInit(polylines[i]);
	}
	if (coords[0].x != coords[count-1].x ||
	    coords[0].y != coords[count-1].y) {
		lcount++;
		x1 = coords[count-1].x; 
		y1 = coords[count-1].y;
		x2 = coords[0].x;       
		y2 = coords[0].y;
		polylineInit(polylines[count-1]);
	} 
	return lcount;
}

void	GGIEXP(fillPoly)(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count)
{
	polyline *polylines;
	ggi_uint	lcount;
	ggi_sint	miny, maxy;

	if (!count) return;
	polylines = malloc(count * sizeof(polyline));
	miny = coords[0].y; maxy = coords[0].y;
	lcount = addPoly(coords, count, polylines, &miny, &maxy);
	fillAreas(vis, polylines, lcount, miny, maxy);
	free(polylines);
}

void	GGIEXP(fillPolys)(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint counts[], ggi_uint count)
{
	polyline *polylines;
	ggi_uint	ccount, lcount;
	ggi_sint	i, miny, maxy;

	if (!count) return;
	lcount = 0;
	for (i = 0; i < count; i++) 
		lcount += counts[i];
	polylines = malloc(lcount * sizeof(polyline));
	ccount = lcount = 0;
	miny = coords[0].y; maxy = coords[0].y;
	for (i = 0; i < count; i++) {
		lcount += addPoly(&coords[ccount], counts[i], 
				  &polylines[lcount], &miny, &maxy);
		ccount += counts[i];
	}
	fillAreas(vis, polylines, lcount, miny, maxy);
	free(polylines);
}

/*----------------------------------------------------------------------*/

void	(*GGIEXP(drawPoly)[2])(ggi_visual_t vis, ggi2d_coord coords[], ggi_uint count) =
{
	drawPoly,
	drawPoly_st
};

/*----------------------------------------------------------------------*/
