/* $Id: events.h,v 1.6 1998/10/29 21:37:03 becka Exp $
***************************************************************************

   LibGII event definitions

   1995-1997	Steffen Seeger	[seeger@ggi.project.org]
   1998		Andrew Apted	[andrew@ggi-project.org]
  
***************************************************************************
*/

#ifndef _GII_EVENTS_H
#define	_GII_EVENTS_H

#include <ggi/system.h>
#include <ggi/keyboard.h>

typedef enum gii_input_type
{
	GII_IC_NULL = 0,

	GII_IC_KEYBOARD = 0,		/* character input devices	*/
		GII_IT_KEYBOARD,	/* a simple keyboard		*/
		GII_IT_VOICE,		/* voice recognition 		*/

	GII_IC_2DPOINTER = 0x10000000,	/* simple 2D pointing devices	*/
		GII_IT_MOUSE,		/* all kinds of mice		*/
		GII_IT_TRACKBALL,	/* a mouse on it's back ;-)	*/
		GII_IT_DIGITIZER,	/* digitizer boards with pen	*/
		GII_IT_LIGHTPEN,	/* are there any left?		*/
		GII_IT_JOYSTICK,	/* the thing used for gaming	*/

	GII_IC_3DPOINTER = 0x20000000,	/* 3D pointing devices	 	*/
		GII_IT_3DMOUSE,

	GII_IC_CONTROLLER = 0x30000000,	/* any other stuff		*/
		GII_IT_DATAGLOVE,
		GII_IT_VR_HEAD,

	GII_IC_KEYPAD			/* character input but no kbd	*/

} gii_input_type;

#define	GII_IC_MASK	0xF0000000	/* avoid ANSI warning -- core	*/

typedef enum gii_event_type {

	evNothing = 0,	/* event is not valid. (must be zero)	*/

	evCommand,	/* report command/do action		*/
	evInformation,	/* notification of new information	*/

	evExpose,	/* exposure event			*/	
	/* empty slot */

	evKeyPress=5,	/* key has been pressed			*/
	evKeyRelease,	/* key has been released		*/
	evKeyRepeat,	/* automatically repeated keypress	*/
	evKeyState,	/* resynchronize keys state		*/

	evPtrRelative,	/* pointer movements reported relative	*/
	evPtrAbsolute,	/* pointer movements reported absolute	*/
	evPtrButtonPress,	/* pointer button pressed	*/
	evPtrButtonRelease,	/* pointer button released	*/
	evPtrState,	/* resynchronize pointer state		*/

	evValRelative,	/* valuator change (reported relative)	*/
	evValAbsolute,	/* valuator change (reported absolute)	*/
	evValState,	/* resynchronize valuator state		*/

	evLast		/* must be less than 33			*/

}  gii_event_type;

#define	EVMASK(x)	em ## x = (1 << ev ## x)
typedef enum gii_event_mask {

	EVMASK(Nothing),

	EVMASK(Command),
	EVMASK(Information),
	EVMASK(Expose),

	EVMASK(KeyPress),
	EVMASK(KeyRelease),
	EVMASK(KeyRepeat),
	EVMASK(KeyState),
	emKey		= emKeyPress | emKeyRelease | emKeyRepeat,
	emKeyboard	= emKey | emKeyState,

	EVMASK(PtrRelative),
	EVMASK(PtrAbsolute),
	EVMASK(PtrButtonPress),
	EVMASK(PtrButtonRelease),
	EVMASK(PtrState),
	emPtrMove	= emPtrRelative | emPtrAbsolute,
	emPtrButton	= emPtrButtonPress | emPtrButtonRelease,
	emPointer	= emPtrMove | emPtrButton | emPtrState,

	EVMASK(ValRelative),
	EVMASK(ValAbsolute),
	EVMASK(ValState),
	emValuator	= emValRelative | emValAbsolute | emValState,

	emZero  = 0,
	emAll	= ((1 << evLast) - 1) & ~emNothing

} gii_event_mask;
#undef EVMASK


#define EV_ORIGIN_NONE		0x00000000	/* Anonymous */
#define EV_ORIGIN_SENDEVENT	0x80000000	/* Event was due to SendEvent call */
#define EV_ORIGIN(stackid)	(stackid)	/* Otherwise, it's a stackid */
				
#define EV_TARGET_NONE		0x00000000	/* Ignore this event */
#define EV_TARGET_ISSPECIAL	0x80000000
#define EV_TARGET_ISVT		0x80020000	/* Sent to a VT/Head combo 
						** Low byte is VT, next to low
						** byte is head 
						*/
	/* Specific VT */
#define EV_VT_SPECIAL			0x80
#define EV_TARGET_VT(head,vt)		(EV_TARGET_ISVT | (((head)&0x7F)<<8) \
							| ((vt) & 0x7F))	
	/* Get the VT */
#define EV_TARGET_GET_VT(targ)		((targ)&0xFF)
	/* Get the Head */
#define EV_TARGET_GET_HEAD(targ)	(((targ)&0xFF00)>>8)

	/* All VTs on all heads */
#define EV_VT_ALL			0x81
#define EV_TARGET_VT_ALL		(EV_TARGET_ISVT|0x81)
	/* Current VT on target head */
#define EV_VT_HEAD_CURRENT		0x82
#define EV_TARGET_VT_HEAD_CURRENT(hdid)	(EV_TARGET_ISVT|(((hdid)&0xFF)<<8)|0x82)
	/* Current VTs on all heads */
#define EV_VT_ALL_CURRENT		0x83
#define EV_TARGET_VT_ALL_CURRENT	(EV_TARGET_ISVT|0x83)
	/* All VTs on target head */
#define EV_VT_HEAD_ALL			0x84
#define EV_TARGET_VT_HEAD_ALL(hdid)	(EV_TARGET_ISVT|(((hdid)&0xFF)<<8)|0x84)


#define	COMMON_DATA  \
	uint8	size;		/* size of event in bytes	*/\
	uint8	type;		/* type of this event		*/\
	uint32	origin;		/* Origin stack/device		*/\
	uint32	target;		/* Target VT/head/stack		*/\
	struct timeval	time	/* timestamp			*/

#define	MODIFIER_DATA  \
	uint16	effect		/* current modifiers in effect	*/


/*	this information is reported with all events. Use the <any> field
**	in a gii_event structure to access these fields.
*/ 
typedef struct gii_any_event {

	COMMON_DATA;

} gii_any_event;

/*	These are used internally either to the application or the
**	kernel. The recipient must not store references to the data.
**	If the data information is needed afterwards, copy it!
*/
typedef struct gii_cmd_nodata_event {

	COMMON_DATA;

	uint32	code;		/* command/request code		*/

} gii_cmd_nodata_event;

#define GII_CMD_DATA_MAX  (255-sizeof(gii_cmd_nodata_event))

/*	These are used internally either to the application or the
 *	kernel. The same event is used for both Command and Information
 *	events. The recipient must not store references to the data. If
 *	the data information is needed afterwards, copy it!
 */
typedef struct gii_cmd_event {

	COMMON_DATA;

	uint32	code;			/* command/request code */
	char	data[GII_CMD_DATA_MAX];	/* command related data */ 

} gii_cmd_event;

/* This event is sent/received to require/get the capabilities of a device
 * as specified in target/origin.
 */
#define GII_CMDCODE_GETDEVINFO	(0x00004449)
typedef struct gii_cmddata_getdevinfo {
	
	char		longname[75];
	char		shortname[5];
	
	gii_event_mask	can_generate;

	int 		key_maxcode;
	int		val_maxaxis;

} gii_cmddata_getdevinfo;

typedef struct gii_valrange {
	int		mix, center, max;
} gii_valrange;

typedef enum gii_physical_quantity_measured { 
	GII_PT_UNKNOWN,			/* unknown */
	GII_PT_TIME, 			/* base unit s */
	GII_PT_FREQUENCY,		/* base unit 1/s (Hz) */
	GII_PT_LENGTH, 			/* base unit m */
	GII_PT_VELOCITY, 		/* base unit m/s */
	GII_PT_ACCELERATION, 		/* base unit m/s^2 */
	GII_PT_ANGLE,			/* base unit radian */
	GII_PT_ANGVELOCITY,		/* base unit radian/s */
	GII_PT_ANGACCELERATION,		/* base unit radian/s^2 */
	GII_PT_AREA, 			/* base unit m^2 */
	GII_PT_VOLUME, 			/* base unit m^3 */
	GII_PT_PRESSURE, 		/* base unit N/m^2 (Pa) */
	GII_PT_FORCE, 			/* base unit N (kg*m/s^2) */
	GII_PT_TEMPERATURE,		/* base unit K */
	GII_PT_CURRENT,			/* base unit A */
	GII_PT_VOLTAGE,			/* base unit V (kg*m^2/(As^3)) */
	GII_PT_RESISTANCE, 		/* base unit V/A (Ohm) */
	GII_PT_CAPACITY, 		/* base unit As/V (Farad) */
	GII_PT_INDUCTIVITY, 		/* base unit Vs/A (Henry) */
	GGI_PT_LAST
} gii_phystype;

/* This event is sent/received to require/get detailed data about valuator
 * axis.
 */
#define GII_CMDCODE_GETVALINFO	(0x00005649)
typedef struct gii_cmddata_getvalinfo {
	
	char		longname[75];
	char		shortname[5];
	
	gii_valrange	range;
	gii_phystype	phystype;

/*	"double" factor_from_event_units_to_common_SI_unit; 
	int add, mul, div, shift; ???
*/	

} gii_cmddata_getvalinfo;

/*	Exposure events give rectangles that need to be refreshed.
*/
typedef struct gii_expose_event {

	COMMON_DATA;

	uint32	x,y;
	uint32	h,w;

} gii_expose_event;


/*	key events should be used to report events obtained from keys and
**	other switches. 
*/
typedef struct gii_key_event {

	COMMON_DATA;
	MODIFIER_DATA;		/* current state of modfiers	*/

	uint32	sym;		/* meaning of key	*/
	uint32  label;		/* label on key		*/
	uint32  button;		/* button number	*/

} gii_key_event;

/*	This is used to report change of pointer position. 
**	Depending on the event type, the values are either absolute
**	or relative.
*/
typedef struct gii_pmove_event {

	COMMON_DATA;
	MODIFIER_DATA;	/* DEPRECATED */

	sint32	x, y;		/* absolute/relative position	*/
	sint32  z, wheel;

} gii_pmove_event;

/*	Button events are sent to report a change in pointer button
**	state.  Depending on the event type, the button is either being
**	pressed or released.
*/
typedef struct gii_pbutton_event {

	COMMON_DATA;
	MODIFIER_DATA;	/* DEPRECATED */

	uint32	button;		/* button number	*/
	uint32	state;		/* DEPRECATED */

} gii_pbutton_event;

/*	Valuator events report a change of up to 32 of the
**	input device valuators. Only a range of 32 valuators beginning
**	from first can be reported with one event.
*/
typedef struct gii_val_event {

	COMMON_DATA;

	uint32	first;		/* first valuator reported	*/
	uint32	count;		/* number reported		*/
	sint32	value[32];	/* absolute/relative values	*/

} gii_val_event;

typedef union gii_event {

	uint8			size;		/* size of this event	*/

	gii_any_event		any;		/* access COMMON_DATA	*/
	gii_cmd_event		cmd;		/* command/information	*/
	gii_expose_event	expose;		/* exposure event	*/
	gii_val_event		val;		/* valuator change	*/
	gii_key_event		key;		/* key press/release	*/
	gii_pmove_event		pmove;		/* pointer move		*/
	gii_pbutton_event	pbutton;	/* pointer buttons	*/

} gii_event;

#endif	/* #ifdef _GII_EVENTS_H */
