/* $Id: input.c,v 1.3 1998/10/30 05:18:00 ajapted Exp $
***************************************************************************

   Linux_kbd: input

   Copyright (C) 1998 Andrew Apted     [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <linux/kd.h>
#include <linux/vt.h>
#include <linux/keyboard.h>

#include <ggi/internal/gii.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#define WANT_TRANSLATE_SHIFT  1
#include "../../../libggi/display/Linux_common/key_trans.inc"


typedef struct keyboard_hook
{
	int fd;

	int old_mode;
	struct termios old_termios;
	
	unsigned char keydown_buf[128];

} KeyboardHook;

#define KEYBOARD_HOOK(inp)	((KeyboardHook *) inp->priv)


/* ---------------------------------------------------------------------- */


static int GII_keyboard_init(gii_input *inp, char *filename)
{
	struct termios new;
	KeyboardHook *kk;

	/* allocate keyboard hook */

	kk = inp->priv = _gii_malloc(sizeof(KeyboardHook));

	/* open the tty */

	kk->fd = open(filename, O_RDWR);

	if (kk->fd < 0) {
		perror("Linux_kbd: Couldn't open TTY");
		return -1;
	}

	/* put tty into "straight through" mode.
	 */

	if (tcgetattr(kk->fd, &kk->old_termios) < 0) {
		perror("Linux_kbd: tcgetattr failed");
	}

	new = kk->old_termios;

	new.c_lflag &= ~(ICANON | ECHO  | ISIG);
	new.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
	new.c_cc[VMIN]  = 0;
	new.c_cc[VTIME] = 0;

	if (tcsetattr(kk->fd, TCSANOW, &new) < 0) {
		perror("Linux_kbd: tcsetattr failed");
	}

	/* Put the keyboard into mediumraw mode.  Despite the name, this
	 * is really "mostly raw", with the kernel just folding long
	 * scancode sequences (e.g. E0 XX) onto single keycodes.
	 */

	if (ioctl(kk->fd, KDGKBMODE, &kk->old_mode) < 0) {
		perror("Linux_kbd: couldn't get mode");
		kk->old_mode = K_XLATE;
	}

	if (ioctl(kk->fd, KDSKBMODE, K_MEDIUMRAW) < 0) {
		perror("Linux_kbd: couldn't set raw mode");
		tcsetattr(kk->fd, TCSANOW, &(kk->old_termios));
		return -1;
	}
	
	DPRINT("Linux_kbd: init OK.\n");

	return 0;
}

static void GII_keyboard_exit(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);

	if (ioctl(kk->fd, KDSKBMODE, kk->old_mode) < 0) {
		perror("Linux_kbd: couldn't set mode");
	}

	if (tcsetattr(kk->fd, TCSANOW, &kk->old_termios) < 0) {
		perror("Linux_kbd: tcsetattr failed");
	}

	close(kk->fd);
	kk->fd = -1;

	free(kk);
	inp->priv = NULL;

	DPRINT("Linux_kbd: exit OK.\n");
}

static gii_event_mask GII_keyboard_handle_data(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);

	gii_event ev;

	struct kbentry entry;

	unsigned char buf[32];

	int code, shift;


	_giiEventBlank(&ev);

	/* first get the keycode */

	if (read(kk->fd, buf, 1) < 1) {
		perror("Linux_kbd: Error reading keyboard");
		return 0;
	}

	code = buf[0];

	if (code & 0x80) {
		code &= 0x7f;
		ev.key.type = evKeyRelease;
		kk->keydown_buf[code] = 0;

	} else if (kk->keydown_buf[code] == 0) {
		ev.key.type = evKeyPress;
		kk->keydown_buf[code] = 1;

	} else {
		ev.key.type = evKeyRepeat;
	}

	ev.key.button = code;
	
	
	/* Get the kernel's shift state.  This way is easier than
	 * keeping track of it ourselves.
	 */

	buf[0] = 6;

	if (ioctl(kk->fd, TIOCLINUX, buf) < 0) {
		perror("Linux_kbd: Couldn't read shift state");
		entry.kb_table = 0;
	}

	shift = buf[0];

	ev.key.effect = translate_shift(shift);
	

	/* Look up the keysym without modifiers, which will give us
	 * the key label (more or less).
	 */

	entry.kb_table = 0;
	entry.kb_index = code;
	
	if (ioctl(kk->fd, KDGKBENT, &entry) < 0) {
		DPRINT("Linux_kbd: ioctl(KDGKBENT) failed.\n");
		return 0;
	}

	ev.key.label = translate_label(entry.kb_value, shift);


	/* Now look up the full keysym in the kernel's table */

	entry.kb_table = shift;
	entry.kb_index = code;
	
	if (ioctl(kk->fd, KDGKBENT, &entry) < 0) {
		DPRINT("Linux_kbd: ioctl(KDGKBENT) failed.\n");
		return 0;
	}

	switch (entry.kb_value) {
		case K_NOSUCHMAP: case K_HOLE:
			return 0;
	}

	ev.key.sym = translate_sym(entry.kb_value, shift);


	DPRINT_EVENTS("KEY-%s button=0x%02x effect=0x%02x "
		"sym=0x%04x label=0x%04x\n",
		(ev.key.type == evKeyRelease) ? "UP" : "DN",
		ev.key.button, ev.key.effect, 
		ev.key.sym,  ev.key.label);


	/* Check for console switch.  Unfortunately, the kernel doesn't
	 * recognize KT_CONS when the keyboard is in RAW or MEDIUMRAW
	 * mode, so _we_ have to.  Sigh.
	 */
	
	if (KTYP(entry.kb_value) == KT_CONS) {

		int new_cons = 1+KVAL(entry.kb_value);

		DPRINT_MISC("Switching to console %d.\n", new_cons);
		
		if (ioctl(kk->fd, VT_ACTIVATE, new_cons) < 0) {
			  perror("ioctl(VT_ACTIVATE)");
		}
		
		/* No need to wait till it's active, the vt_process
		 * signal handler does that for us.
		 */
		 
		return 0;
	}
	

	/* finally queue the key event */

	ev.key.size   = sizeof(gii_key_event);
	ev.key.origin = EV_ORIGIN_NONE;
	ev.key.target = EV_TARGET_NONE;

	EV_TIMESTAMP(&ev);

	_giiEvQueueAdd(inp, &ev);

	return (1 << ev.any.type);
}

static gii_event_mask GII_keyboard_poll(gii_input *inp)
{
	KeyboardHook *kk = KEYBOARD_HOOK(inp);
	
	gii_event_mask result = 0;


	DPRINT_MISC("linux_kbd: poll(%p)\n", inp);
	
	for (;;) {

		fd_set readset;

		struct timeval t = {0,0};
		int rc;

		FD_ZERO(&readset);
		FD_SET(kk->fd, &readset);

		/* FIXME !!! doesn't handle -EINTR */
		rc = select(inp->maxfd, &readset, NULL, NULL, &t);

		if (rc <= 0)
			break;

		result |= GII_keyboard_handle_data(inp);
	}

	return result;
}


/* ---------------------------------------------------------------------- */


static int GII_linux_kbd_close(gii_input *inp)
{
	DPRINT_MISC("Linux_kbd cleanup\n");

	if (KEYBOARD_HOOK(inp)) {
		GII_keyboard_exit(inp);
	}

	return 0;
}

int GIIdlinit(void *vis, const char *args)  /* !!! */
{
	gii_input *inp = (gii_input *) vis;   /* FIXME ! HACKHACKHACK !*/
	char *filename = "/dev/tty";

	
	DPRINT_MISC("linux_kbd starting.\n");

	/* Initialize */

	if (args && *args) {
		filename = (char *) args;
	}

	if (GII_keyboard_init(inp, filename) < 0) {
		return -1;
	}
	
	/* We leave these on the default handlers
	 *	inp->GIIseteventmask = _GIIstdseteventmask;
	 *	inp->GIIgeteventmask = _GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset = _GIIstdgetselectfd;
	 */
	 
	inp->GIIeventpoll = GII_keyboard_poll;
	inp->GIIclose = GII_linux_kbd_close;

	inp->targetcan = emKey;
	inp->GIIseteventmask(inp, emKey);

	inp->maxfd = KEYBOARD_HOOK(inp)->fd + 1;
	FD_SET(KEYBOARD_HOOK(inp)->fd, &inp->fdset);

	DPRINT_MISC("linux_kbd fully up\n");

	return 0;
}
