#
# $Id: Menu.w,v 1.3 1997-01-15 14:59:34+01 mho Exp $
#
# Purpose: Menu widget based on the Board widget
#
# Authors: Markus Holzem
#
# Copyright: (C) 1996, GNU (Markus)
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Additionally everyone using this library has to announce it with:
#
#   This software uses the wxWindows-Xt GUI library
#   (C) Markus Holzem, available via
#       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
#

@class XfwfMenu (XfwfBoard) @file=Menu

@utilities

@ The standard tolower function is not compatible with 16-bit values.
I have to check first, if the keysym is an ascii-value.

	@def mytolower(key) = (isascii(key)&&isupper(key) ? tolower(key) : key)

@exports

@ A menu can consit of the following items:

\item{-} |MENU_TEXT|: text item, not selectable
\item{-} |MENU_BUTTON|: button item, selectable
\item{-} |MENU_RADIO|: button item with an diamond indicator (to select on of many)
\item{-} |MENU_TOGGLE|: button item with square indicator ( to switch on and off)
\item{-} |MENU_CASCADE|: item with a submenu
\item{-} |MENU_SEPARATOR|: separator line item
\item{-} |MENU_PUSHRIGHT|: item to push right the rest of a menubar

	@type menu_item_type = enum {
	    MENU_TEXT,
	    MENU_BUTTON,
	    MENU_RADIO,
	    MENU_TOGGLE,
	    MENU_CASCADE,
	    MENU_SEPARATOR,
	    MENU_PUSHRIGHT
	}

@ One menu item is specified by the following structure. |label| is the label of the
item, |key_binding| an optional hotkey (not maintained by the menu widget), |help_text|
an assignable help, |ID| an optional identifier. The item needs its type, which is one
of |menu_item_type|. Selectable items can be |enabled| and switchable items can be
|set|. If the menu item is of the type |MENU_CASCADE|, contents is the list with the
submenu. |next| is used to chain the menu items of one menu together. Optionally a
|user_data| can be assigned with a menu item. |start| and |end| are needed to speed
up internal processing.

	@type menu_item = struct _menu_item {
	    /* public data */
	    char              *label;
	    char	      *underline;
	    char              *key_binding;
	    char              *help_text;
	    int               ID;
	    menu_item_type    type;
	    Boolean           enabled;
	    Boolean           set;
	    struct _menu_item *contents;
	    struct _menu_item *next;
	    void              *user_data;
	    /* private data */
	    Position          start, end;
	}

@ Usually the menu would not accept the focus if it is used as a menubar.
Nevertheless it takes the focus, if it's |accept_focus| method is called.
But the most desired method to be called is to use one of the menubar
hotkeys. From outside the |XfwfHandleHotKey| function may try to give the
focus to the menu widget.

@ If the given key event is not a hotkey of the menubar the function will
try to find a registered hotkey in a menu_item.

@ The function returns |True|, if the event was handled by the menubar.

@proc Boolean XfwfHandleHotKey($, XKeyEvent* event)
{
    menu_state *ms;
    menu_item  *item;
    Boolean    alt  = event->state & Mod3Mask;
    Boolean    ctrl = event->state & ControlMask;
    Boolean    meta = event->state & Mod1Mask;
    KeySym     key;

    (void)XLookupString(event, NULL, 0, &key, NULL);
    key = mytolower(key);

    if (! XtIsSubclass($, xfwfMenuWidgetClass))
	XtError("XfwfHandleHotKey called with incorrect widget type");

    if ($state->prev)
	return False; /* menu is popped up, keys will be handled by translations */

    /* try menubar hotkeys */
    if (meta || alt) {
	ms = $state;
	for (item = $state->menu; item; item = item->next)
	    if (item->enabled && item->type != MENU_TEXT
	    &&  item->type != MENU_SEPARATOR && item->type != MENU_PUSHRIGHT
	    &&  item->underline != NULL && mytolower(*(item->underline)) == key) {
		if (item->type == MENU_CASCADE) {
		    /* start menu */
		    Time time = CurrentTime;
		    XtGrabPointer($, FALSE,
				  (ButtonMotionMask | PointerMotionHintMask |
				   ButtonReleaseMask | ButtonPressMask),
				  GrabModeAsync, GrabModeAsync,
				  None, $cursor, time);
		    XtCallAcceptFocus($, &time);
		    XtTranslateCoords($, 0, 0, &($state->x), &($state->y));
		    HighlightItem($, ms, item, True);
		    /* $accept_focus($, &(event->time)); */
		    $mouse_selection = FALSE;
		    $pointer_grabbed = TRUE;
		} else {
		    /* execute button, toggle, or radio */
		    XtCallCallbackList($, $onSelect, (XtPointer)item);
		}
		return True;
	    }
    }
    /* handle submenu hotkeys */
    if ( (item = FindHotKey($state->menu, ctrl, alt, meta, key)) ) {
	/* execute button, toggle, or radio */
	XtCallCallbackList($, $onSelect, (XtPointer)item);
	return True;
    }
    /* hotkey not handled by menu */
    return False;
}

@ |XfwfSetMenu| is used to set a new menu tree. The benefit to 
|XtVaSetValues| is, that it will only redraw the menu and not the border
and tries to resize, if the calling program specifies |True| for the
|shrink| parameter.

@proc void XfwfSetMenu($, menu_item *top, Boolean shrink)
{
    Position x, y; Dimension w, h;

    if (! XtIsSubclass($, xfwfMenuWidgetClass))
	XtError("XfwfSetMenu called with incorrect widget type");

    /* set new menu */
    $menu = top; $state->menu = top;
    ComputeMenuSize($, $state);

    if (shrink) {
	/* resize if shrink is true */
	if ($width != $state->w || $height != $state->h) {
	    XtVaSetValues($,
			  XtNwidth,  (Dimension)$state->w,
			  XtNheight, (Dimension)$state->h,
			  NULL);
	    return;
	}
    }
    $state->w = $width; $state->h = $height;
    if (XtIsRealized($)) {
	/* redraw menu via $expose */
	$compute_inside($, &x, &y, &w, &h);
	XClearArea(XtDisplay($), XtWindow($), x, y, w, h, True);
    }
}

@public

@ For menus usually the right pointer is used.

        @var cursor = <String> "right_ptr"

@ The |XtNframeType| determines how the border looks.

        @var frameType = XfwfRaised

@ |XtNframeWidth| is set initially to |2|. It is the frame width, when
the menu widget is used as a menubar (|XtNhorizontal| = TRUE).

        @var frameWidth = 2
	@var innerVOffset = 2
	@var innerHOffset = 2

@ |XtNmenuFrameWidth| is used as the shadowWidth of the menus. This may be
different from |XtNframeWidth|.

	@var int menuFrameWidth = 2

@ Highlighting is done using the shadows.

	@var highlightThickness = 0

@ The text is drawn in the font which is given as the |font| resource.

	@var <FontStruct> XFontStruct *font = <String> XtDefaultFont

@ The foreground color is the color used to draw the text.

	@var Pixel foreground = <String> XtDefaultForeground

@ If a menu item is a toggle, an indicator is drawn (square or diamond).
|indicatorSize| specifies the size of the indicator.

	@var Dimension indicatorSize = 0

@ |XtNhMargin| is the margin left and right of a menu item.

	@var Dimension hMargin = 2

@ |XtNvMargin| is the margin above and below of a menu item.

	@var Dimension vMargin = 2

@ |XtNspacing| is the space between label and key, and between label and
indicator.

	@var Dimension spacing = 4

@ Direction of the toplevel menu is specified by the |XtNhorizontal|
resource.

	@var Boolean horizontal = True

@ The menu widget may be used as a choice widget. This is like a menubar with
only one item in the bar.

	@var Boolean usedAsChoice = False

@ The |XtNonSelect| callback is called, when the selecting of a menu item is
done. The selected item is given as the |call_data|. If now item was selected
it is |NULL|.

	@var <Callback> XtCallbackList onSelect = NULL

@ |XtNonChange| is provided for helps on menu items. It is called whenever the
menu item changes. The |client_data| is the same as of |XtNonSelect|.

	@var <Callback> XtCallbackList onChange = NULL

@ The |XtNreturnFocus| specifies the Widget, that should recieve focus, after
the selection is finished. If the resource is |0| the focus will remain on the
menu widget.

	@var Widget returnFocus = 0

@ The menu is stored in the |XtNmenu| resource.

	@var <Pointer> menu_item* menu = NULL

@classvars

@ The Core variable |compress_exposure| is OR'ed with
|XtExposeGraphicsExpose|, in order to get graphics expose events delivered
to the |expose| method.

@var compress_exposure = XtExposeCompressMultiple | XtExposeGraphicsExpose

@private

@ The |menu_state| structure is needed to store the current state of the menu. The
state is kept in a stack (|$state|) with the last opened submenu on the top of the
stack.

|menu| is the (sub)menu on this level and |selected| the currently highlighted item.
|win| is the X window on which this (sub)menu is drawn. |(x,y)| is the root
coordoninate of the window and |(w,h)| it's size. |wLeft| and |wMiddle| are needed
for the layout of the menu items. |prev| is the previous in the menu state stack.

	@type menu_state = struct _menu_state {
	    menu_item          *menu;
	    menu_item          *selected;
	    Window	       win;
	    Position	       x;
	    Position	       y;
	    Dimension	       w;
	    Dimension	       h;
	    Dimension	       wLeft;
	    Dimension	       wMiddle;
	    struct _menu_state *prev;
	}

@ |$state| holds the stack of the menu state.

	@var menu_state *state

@ |gc| and |graygc| are used to draw enabled and disabled menu items.

	@var GC gc
	@var GC graygc

@ |moused_out| and |was_dragged| are needed to allow stay up menus.

	@var Boolean mouse_selection
	@var Boolean popped_up
	@var Boolean moused_out
	@var Boolean was_dragged

@ The pointer should only be grabbed once and only ungrabbed once. To
prevent infinite loops, I use |pointer_grabbed| to keep the "grab level".

	@var Boolean pointer_grabbed

@methods

@ |initialize| creates the GCs and computes the initial state and the
initial size of the menu

@proc initialize
{
    CreateGCs($);

    if (!$indicatorSize || $indicatorSize > $font->ascent)
	$indicatorSize = $font->ascent;

    $popped_up       = FALSE;
    $mouse_selection = FALSE;
    $pointer_grabbed = FALSE;
    $state           = (menu_state*)XtMalloc(sizeof(menu_state));
    $state->menu     = $menu;
    $state->selected = NULL;
    $state->prev     = NULL;
    $state->x        = 0;
    $state->y        = 0;

    ComputeMenuSize($, $state);
    $width  = $state->w;
    $height = $state->h;
}

@ |realize| sets the |save_under| window attribute to speed up the menu
handling. The size of the toplevel window and it's window id are stored too.

@proc realize
{
    *mask |= CWSaveUnder;
    attributes->save_under = True;
    #realize($, mask, attributes);
    /* initialize top level state structure */
    $state->win = XtWindow($);
    $state->w   = $width;
    $state->h   = $height;
}

@ |_expose| redraws all visible menus.

@proc _expose
{
    menu_state *ms;

    if (!XtIsRealized($))
	return;

    /* redisplay all visible menus */
    for (ms=$state; ms; ms=ms->prev)
	DisplayMenu($, ms);
    /* redraw focus border */
    if ($traversal_focus && $highlightThickness)
	$highlight_border($);
}

@ |destroy| has to ungrab the pointer, destroy the text GCs, and to
destroy the additional popped up windows.

@proc destroy
{
    menu_state *ms;

    if ($pointer_grabbed)
	XtUngrabPointer($, CurrentTime);
    if ($popped_up)
	XtUngrabKeyboard($, CurrentTime);
    ReleaseGCs($);

    /* unhighlight all submenus and free structures */
    for (ms=$state; ms->prev!=NULL; ms=ms->prev)
	; /* find top level menu state */
    UnhighlightItem($, ms, ms->selected);
    ms->selected = NULL;
    /* free menu_state of widget's window */
    XtFree((char*)ms);
}

@ |set_values| is usually used to store a new menu structure. So the new sizes
of the menu have to be computed and it is necessary to redraw the menu.

@proc set_values
{
    if ($menu != $old$menu)
	$state->menu = $menu;
    /* To be sure */
    ComputeMenuSize($, $state);
    $state->w = $width;
    $state->h = $height;

    if ($foreground != $old$foreground || $background_pixel != $old$background_pixel) {
	ReleaseGCs($);
	CreateGCs($);
    }

    return True;
}

@ |resize| is used to update the size of the toplevel window inside the menu_state
structure.

@proc resize
{
    $state->w = $width;
    $state->h = $height;
}

@ |highlight_border| is called, when the focus is given to the menubar. Additionaly
to the highlighting the pointer and the keyboard are grabbed to allow keyboard and
pointer handling of the menu simulatously.

@proc highlight_border
{
    #highlight_border($);
}

@ |unhighlight_border| is called, when the focus is given away. It only has to
change the look of the menu, because the focus is given away by now.

@proc unhighlight_border
{
    menu_state *ms;

    for (ms=$state; ms->prev!=NULL; ms=ms->prev)
	; /* find top level menu state */
    UnhighlightItem($, ms, ms->selected);
    ms->selected = NULL;

    #unhighlight_border($);
}

@ The menu widget does not like to get the keyboard focus that easy if it
is used as a menubar. It wants to get it offered by a hotkey, e.g by ALT+F
to open the file menu.

@proc would_accept_focus
{
    return (#would_accept_focus($) && $usedAsChoice);
}

@translations

@ A button down starts the menu selection, dragging continues the menu
selection and button up stops it. If the mouse is moved out of a window,
the current menu won't be immediately popped down. It will stay up, until
an item is selected or it was clicked outside of the menu.

	@trans <BtnDown>:	 traverseCurrent() start()
	@trans Button1 <Motion>: drag()
	@trans Button2 <Motion>: drag()
	@trans Button3 <Motion>: drag()
	@trans <BtnUp>:		 traverseCurrent() finish()

@ There is another set of keyboard translations, that is used to handle keyboard
selections. |Escape| traverses out of the menu.

	@trans <Key>: handle_key()

@actions

@ |handle_key| is called whenever a cursor key is pressed to move around in the
menu tree. At the beginning the menu items next to the current selected item are
computed - disabled and non-selectable items are skipped. After this the action
is selected depending on the pressed key. The meaning of the keys vary if the current
or parent menu is a menubar.

@proc handle_key
{
    menu_state *ms, *top;
    menu_item  *item;
    Boolean    is_top, in_menubar, prev_menubar;
    menu_item  *selected, *prev, *next, *up, *down, *topprev, *topnext;
    KeySym     keysym; (void)XLookupString(&(event->xkey), NULL, 0, &keysym, NULL);

    /* nothing to do when the mouse currently does the work */
    if ($mouse_selection)
	return;
    next_menus($, &ms, &is_top, &in_menubar, &prev_menubar, &selected,
	       &prev, &next, &up, &down, &top, &topprev, &topnext);
    /* handle motion in the menu tree */
    switch (keysym) {
    case XK_Up:
	if      (!in_menubar && prev)                HighlightItem($, ms, prev, True);
	else if (is_top && !$state->prev)	     $traverse($, TraverseUp, $, &event->xkey.time);
	break;
    case XK_Down:
	if      (!in_menubar && next)		     HighlightItem($, ms, next, True);
	else if (is_top && !$state->prev)	     $traverse($, TraverseDown, $, &event->xkey.time);
	break;
    case XK_Left:
	if      (!in_menubar && up && !prev_menubar) UnhighlightItem($, ms, selected);
	else if (is_top && !$state->prev)	     $traverse($, TraverseLeft, $, &event->xkey.time);
	else if (topprev)			     HighlightItem($, top, topprev, True);
	break;
    case XK_Right:
	if      (!in_menubar && down)		     HighlightItem($, $state, down, True);
	else if (is_top && !$state->prev)	     $traverse($, TraverseRight, $, &event->xkey.time);
	else if (topnext)			     HighlightItem($, top, topnext, True);
	break;
    case XK_Escape: case XK_Home: /* unhighlight menu */
	$state->selected = NULL;
	DoSelect($, CurrentTime, True);
	break;
    case XK_Tab: /* traverse to next and previous */
	if (is_top && !$state->prev) {
	    if (!event->xkey.state)		    $traverse($, TraverseNext, $, &event->xkey.time);
	    else if (event->xkey.state & ShiftMask) $traverse($, TraversePrev, $, &event->xkey.time);
	}
	break;
    default:
	item = NULL;
	/* search for a hotkey (all visible menus, all enabled items) */
	for (ms = $state; ms; ms = ms->prev) {
	    for (item = ms->menu; item; item = item->next) {
		if (item->enabled && item->underline && mytolower(keysym) == mytolower(*(item->underline)))
		    break;
	    }
	    if (item) break;
	}
	/* hotkey found: highlight and activate if button */
	if (ms && item)
	    HighlightItem($, ms, item, False);
	if ( !item ||
	     !(item->type == MENU_BUTTON ||
	       item->type == MENU_TOGGLE ||
	       item->type == MENU_RADIO) )
	    break;
    case XK_Return:
	if (!$pointer_grabbed) { /* grab pointer and keyboard */
	    XtTranslateCoords($, 0, 0, &($state->x), &($state->y));
	    HighlightItem($, $state, $state->menu, True);
	    XtGrabPointer($, FALSE,
			  (ButtonMotionMask | PointerMotionHintMask |
			   ButtonReleaseMask | ButtonPressMask),
			  GrabModeAsync, GrabModeAsync,
			  None, $cursor, CurrentTime);
	    $pointer_grabbed = TRUE;
	} else
	    DoSelect($, CurrentTime/*event->xkey.time*/, True);
	break;
    }
}

@ |leave_focus| is used to give the focus away, when |Escape| or |Home|
is pressed.

@proc leave_focus
{
    /* nothing to do when the mouse currently does the work */
    if ($mouse_selection) return;
}

@ |start| is called when a button is pressed. This action grabs pointer
and keyboard and all incoming events are now delivered to the menu widget.

@proc start
{
    $mouse_selection = True; /* start of mouse selection */

    if ($popped_up)
	$was_dragged = True;

    if (!$state->prev) {
	$state->x = event->xbutton.x_root - event->xbutton.x;
	$state->y = event->xbutton.y_root - event->xbutton.y;
	$moused_out = False;
	/* grab pointer and keyboard */
	if (!$pointer_grabbed) {
	    XtGrabPointer($, FALSE,
			  (ButtonMotionMask | PointerMotionHintMask |
			   ButtonReleaseMask | ButtonPressMask),
			  GrabModeAsync, GrabModeAsync,
			  None, $cursor, CurrentTime);
	    $pointer_grabbed = TRUE;
	}
    } else if ($state->selected) {
	UnhighlightItem($, $state, $state->selected);
	$state->selected = NULL;
    }
    /* check if pointer was clicked outside the menu */
    if (!HandleMotionEvent($, event->xbutton.x_root, event->xbutton.y_root))
	if ($moused_out) /* set by HandleMotionEvent */
	    DoSelect($, CurrentTime/*event->xbutton.time*/, True);
}

@ |drag| is called when the mouse pointer is moved.

@proc drag
{
    XMotionEvent  *ev   = &event->xmotion;
    int           x     = ev->x_root;
    int           y     = ev->y_root;
    int           state = ev->state;

    if (!$mouse_selection) return; /* avoid spurious drag events after finish() */

    if ($popped_up) $was_dragged = True;

    HandleMotionEvent($, x, y);
    XSync(XtDisplay($), FALSE);
    /* allow motion events to be generated again */
    if (ev->is_hint
    && XQueryPointer(XtDisplay($), ev->window,
		     &ev->root, &ev->subwindow,
		     &ev->x_root, &ev->y_root,
		     &ev->x, &ev->y, &ev->state)
    && ev->state == state
    && (ev->x_root != x || ev->y_root != y)) {
	HandleMotionEvent($, ev->x_root, ev->y_root);
	XSync(XtDisplay($), FALSE);
    }
}

@ |finish| checks, if a selection was done.

@proc finish
{
    XButtonEvent *ev = &event->xbutton;
    int force;

    $mouse_selection = False; /* end of mouse selection */

    force = !HandleMotionEvent($, ev->x_root, ev->y_root) || $moused_out;
  
    if (($popped_up && $was_dragged) || (!$popped_up))
	DoSelect($, CurrentTime/*ev->time*/, force);
}

@utilities

@ DoSelect does the real selection of the menu. If the menu was popped up,
the the parent (a override shell widget) is popped down. The |onSelect|
callback is called with the selected item, if any. At the end the Xt keyboard
focus is given away.

@proc DoSelect($, Time time, Boolean force)
{
    menu_item   *selected_item = $state->selected;
    menu_state  *ms;

    $mouse_selection = False; /* end of mouse selection */

    if (!force && !selected_item) {
	/* ms is the previous menu state */
	if ($usedAsChoice && $state->prev && !$state->prev->prev) {
	    menu_item *subitem = FindItem($state, $state->prev->menu->label);
	    HighlightItem($, $state, subitem, False);
	} else {
	    HighlightFirstItem($, $state);
	}
	return;
    }
    $moused_out = False;
    /* ungrab pointer and keyboard */
    if ($pointer_grabbed) {
	XtUngrabPointer($, time);
	$pointer_grabbed = FALSE;
    }
    /* unhighlight all subwindows */
    for (ms=$state; ms->prev!=NULL; ms=ms->prev)
	;
    UnhighlightItem($, ms, ms->selected);
    ms->selected = NULL;
    /* popdown, if this is a popup menu */
    if ($popped_up) {
	$popped_up = FALSE;
	$was_dragged = False;
	XtUngrabKeyboard($, time);
	XtPopdown(XtParent($));
    }
    /* update display */
    XFlush(XtDisplay($));
    /* give keyboard focus away */
    if ($returnFocus)
	XtCallAcceptFocus($returnFocus, &time);
    /* call callback for selection, if valid item is selected */
    if (selected_item && selected_item->enabled
    && selected_item->type != MENU_TEXT
    && selected_item->type != MENU_SEPARATOR
    && selected_item->type != MENU_PUSHRIGHT)
	XtCallCallbackList($, $onSelect, (XtPointer)selected_item);
}

@ |next_menus| is a utility to find the surrounding menus to the current
menu state, i.e. the next and the previous menu item, the parent of the
selected menu and the child of the menu item, if it is a cascade item.

@proc void next_menus($, menu_state **ms,
		      Boolean *is_top, Boolean *in_menubar, Boolean *prev_menubar,
		      menu_item **selected, menu_item **prev, menu_item **next,
		      menu_item **up, menu_item **down,
		      menu_state **top, menu_item **topprev, menu_item **topnext)
{
    menu_item *before;

    /* find menu state with the top level highlighted item */
    *ms = ($state->selected ? $state : ($state->prev ? $state->prev : NULL));
    if (*ms) {
	/* menubar or not */
	*is_top       = (!(*ms)->prev);
	*in_menubar   = (($horizontal || $usedAsChoice) && !(*ms)->prev);
	*prev_menubar = (($horizontal || $usedAsChoice) && (*ms)->prev && !(*ms)->prev->prev);
	/* previous and next menu item, up and down menu */
	*selected = (*ms)->selected;
	before    = ((*ms)->menu == (*selected) ? NULL : (*ms)->menu);
	*prev     = NULL;
	*next     = ((*selected) ? (*selected)->next : NULL);
	*up       = ((*ms)->prev ? (*ms)->prev->selected : NULL);
	*down     = ((*selected)->type == MENU_CASCADE ? (*selected)->contents : NULL);
	/* search for selected's previous, next, and down selectable item */
	while (before) {
	    if (before->enabled && before->type != MENU_TEXT
	    &&  before->type != MENU_SEPARATOR && before->type != MENU_PUSHRIGHT)
		*prev = before; /* item is before selected and selectable */
	    if ( (before = before->next) == (*selected))
		break;
	}
	while ((*next) && !((*next)->enabled && (*next)->type != MENU_TEXT &&
	       (*next)->type != MENU_SEPARATOR && (*next)->type != MENU_PUSHRIGHT))
	    (*next) = (*next)->next;
	while ((*down) && !((*down)->enabled && (*down)->type != MENU_TEXT &&
	       (*down)->type != MENU_SEPARATOR && (*down)->type != MENU_PUSHRIGHT))
	    (*down) = (*down)->next;
    } else {
	*is_top = TRUE;
	*in_menubar = *prev_menubar = FALSE;
	*selected = *prev = *next = *up = *down = NULL;
    }
    /* top of menu tree */
    for (*top = $state; (*top)->prev != NULL; *top = (*top)->prev)
	;
    /* next previous items in top of menu tree */
    *topnext = ((*top)->selected ? (*top)->selected->next : NULL);
    *topprev = NULL;
    before   = ((*top)->menu == (*top)->selected ? NULL : (*top)->menu);
    while ((*topnext) && !((*topnext)->enabled && (*topnext)->type != MENU_TEXT &&
			   (*topnext)->type != MENU_SEPARATOR && (*topnext)->type != MENU_PUSHRIGHT))
	(*topnext) = (*topnext)->next;
    while (before) {
	if (before->enabled && before->type != MENU_TEXT
	&&  before->type != MENU_SEPARATOR && before->type != MENU_PUSHRIGHT)
	    *topprev = before; /* item is before selected and selectable */
	if ( (before = before->next) == (*top)->selected )
	    break;
    }
}

@ Create and destroy the GCs for menu item text drawing.

@proc CreateGCs($)
{
    XtGCMask  mask = GCFont | GCBackground | GCForeground;
    XGCValues values;

    /* gc is used to draw the enabled menu items */
    values.background = $background_pixel;
    values.foreground = $foreground;
    values.font       = $font->fid;
    $gc               = XtGetGC($, mask, &values);

    /* graygc is used to draw the disabled menu items */
    values.fill_style = FillStippled;
    values.stipple    = $gray;
    mask             |= GCStipple | GCFillStyle;
    $graygc           = XtGetGC($, mask, &values);
}

@proc ReleaseGCs($)
{
    XtReleaseGC($, $gc);
    XtReleaseGC($, $graygc);
}

@ It should be possible to override the label, the helptext, and the hotkey
by a resource. The current label is used for the subresource name - turned in
something suitable for a resource name, e.g. "Kill Buffer" -> "killBuffer".
"Kill buffer" is the label gived in the menu item.

	@type Subresource = enum {
	    SUBRESOURCE_LABEL, SUBRESOURCE_HELP, SUBRESOURCE_KEY }

@proc void GetResourceName(char *in, char *out)
{
    char *first = out;

    while (*in)
	if (isalnum((unsigned char)*in) || (*in)=='_')
	    *out++ = *in++;
	else
	    in++;
    *first = mytolower(*first);
    *out   = '\0';
}

@proc char *ResourcedText($, menu_item *item, Subresource type)
{
    static XtResource labelResource[] = {
	{ "label", "Label", XtRString, sizeof(String), 0, XtRImmediate, 0 },
	{ "help",  "Help",  XtRString, sizeof(String), 0, XtRImmediate, 0 },
	{ "key",   "Key",   XtRString, sizeof(String), 0, XtRImmediate, 0 },
    };

    char resource_name[1024];
    char *resourced_text=NULL;

    if (!item || !item->label)
	return NULL;

    GetResourceName(item->label, resource_name);
    XtGetSubresources($, (XtPointer)(&resourced_text),
		      resource_name, resource_name, &labelResource[type],
		      1, NULL, 0);
    if (!resourced_text)
	switch (type) {
	case SUBRESOURCE_LABEL:  return item->label;
	case SUBRESOURCE_HELP:   return item->help_text;
	case SUBRESOURCE_KEY:    return item->key_binding;
	}
    return (resourced_text);
}

@ Easier acces to the width of a string.

@proc unsigned StringWidth($, char *s)
{
    XCharStruct xcs;
    int         dummy;

    XTextExtents($font, s, strlen(s), &dummy, &dummy, &dummy, &xcs);
    return xcs.width;
}

@ The different types of menu items have different sizes. The following utility
functions are the "compute size" methods for the different items. They are
collected in an array to call the appropriate function by an index: the type
of the menu item.

@proc void MenuTextSize($, menu_item *item, Boolean in_menubar,
		        unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    *h      = $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth;
    *l = *r = $hMargin + $menuFrameWidth;
    *m      = StringWidth($, ResourcedText($, item, SUBRESOURCE_LABEL));
}

@proc void MenuButtonSize($, menu_item *item, Boolean in_menubar,
			  unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    MenuTextSize($, item, in_menubar, l, m, r, h);
    if ( !(*h) )
	*h = $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth;
    if (!in_menubar && item->key_binding)
	*r += StringWidth($, ResourcedText($, item, SUBRESOURCE_KEY)) + 2*$spacing;
}

@proc void MenuToggleSize($, menu_item *item, Boolean in_menubar,
			  unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    MenuButtonSize($, item, in_menubar, l, m, r, h);
    if ( !(*h) )
	*h = $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth;
    *l += $indicatorSize + $spacing;
}

@proc void MenuCascadeSize($, menu_item *item, Boolean in_menubar,
			   unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    MenuTextSize($, item, in_menubar, l, m, r, h);
    if ( !(*h) )
	*h = $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth;
    if (!in_menubar || $usedAsChoice)
	*r += $indicatorSize + $spacing;
}

@proc void MenuPushrightSize($, menu_item *item, Boolean in_menubar,
			     unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    *l = *m = *r = *h = 0;
}

@proc void MenuSeparatorSize($, menu_item *item, Boolean in_menubar,
			     unsigned *l, unsigned *m, unsigned *r, unsigned *h)
{
    *l = *m = *r = *h = 0;
    if (!in_menubar) {
	*h = max(2, $menuFrameWidth);
	*m = 1;
    }
}

@ The size functions are stored inside an array for easier access. This array is
defined static since there is no use in overriding it by an ancestor.

	@var void* SizeFunctionList[] = {
	    MenuTextSize,
	    MenuButtonSize,
	    MenuToggleSize,
	    MenuToggleSize,
	    MenuCascadeSize,
	    MenuSeparatorSize,
	    MenuPushrightSize
	}

@ ComputeMenuSize computes the size specified by the current menu state. Since menus
are drawn one after the other it is usually only necessary to compute and draw a newly
visible menu.

@def SET_MAX_VALUE(val) = {
    if (val > max_####val)
	max_####val = val;	
}

@proc void ComputeMenuSize($, menu_state *ms)
{
    typedef void (*SizeFunction)(Widget, menu_item*, Boolean, unsigned*, unsigned*, unsigned*, unsigned*);

    unsigned  left_width, label_width, right_width, height;
    unsigned  max_left_width, max_label_width, max_right_width, max_height;
    Boolean   in_menubar = ($horizontal && !ms->prev);
    Boolean   is_choice = ($usedAsChoice && !ms->prev);
    menu_item *item, *pushright_item = NULL;

    max_left_width = max_label_width = max_right_width = max_height = 0;
    for (item=ms->menu; item; item=item->next) {
	((SizeFunction)SizeFunctionList[item->type])
	    ($, item, in_menubar, &left_width, &label_width, &right_width, &height);
	if (in_menubar || is_choice) {
	    if (!pushright_item && item->type == MENU_PUSHRIGHT)
		pushright_item = item;
	    if (item != ms->menu)
		max_label_width += $spacing;
	    item->start      = max_label_width + $frameWidth + $innerHOffset;
	    max_label_width += left_width + label_width + right_width;
	    item->end        = max_label_width + $frameWidth + $innerHOffset;
	    SET_MAX_VALUE(height);
	} else {
	    SET_MAX_VALUE(left_width);
	    SET_MAX_VALUE(label_width);
	    SET_MAX_VALUE(right_width);
	    item->start  = max_height + $menuFrameWidth;
	    max_height  += height;
	    item->end    = max_height + $menuFrameWidth;
	}
    }
    ms->wLeft   = max_left_width;
    ms->wMiddle = max_label_width;
    if (in_menubar || is_choice) {
	ms->w = max_left_width + max_label_width + max_right_width + 2 * ($frameWidth + $innerHOffset);
	ms->h = max_height + 2 * ($frameWidth + $innerVOffset);
	if (max_height < $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth)
	    ms->h += $font->ascent + $font->descent + 2*$vMargin + 2*$menuFrameWidth;
	if (pushright_item)
	    pushright_item->end = ms->w - pushright_item->end;
	ms->wLeft = $hMargin + $menuFrameWidth;
    } else {
	ms->w = max_left_width + max_label_width + max_right_width + 2 * $menuFrameWidth;
	ms->h = max_height + 2*$menuFrameWidth;
    }
}

@ The different types of menu items have different looks. The following utility
functions are the "draw menu item" methods for the different items. As for the size
functions these are collected in another array for drawing.

@proc void DrawTextItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    Boolean        top        = (!ms->prev && ($horizontal || $usedAsChoice));
    Dimension      extra_x    = 0;
    char*          label      = ResourcedText($, item, SUBRESOURCE_LABEL);
    Dimension      wd         = (top ? item->end - item->start : ms->w - 2*$menuFrameWidth);
    Dimension      ht         = (top ? ms->h - 2*($frameWidth+$innerVOffset) : item->end - item->start);
    XfwfShadowType frame_type = (ms->selected==item) ? XfwfRaised : XfwfBackground;

    /* extra space for toggle in menubar */
    if (top && (item->type == MENU_TOGGLE || item->type == MENU_RADIO))
	extra_x = $indicatorSize + $spacing;
    /* draw label */
    if (label)
	XfwfDrawString(XtDisplay($), ms->win,
		       item->enabled || item->type==MENU_TEXT ? $gc : $graygc, $font,
		       (x + ms->wLeft + extra_x), (y + $menuFrameWidth + $vMargin + $font->ascent),
		       label, strlen(label), NULL, item->underline);
    if (item->enabled && item->type != MENU_TEXT)
	if (item->type == MENU_CASCADE && !ms->prev && $usedAsChoice) {
	    /* used as a choice item */
	    /* --> the job is done by DrawCascadeItem */
	} else {
	    XfwfDrawRectangle($, ms->win, $lightgc, $darkgc, $bordergc, $backgroundgc,
			      x, y, wd, ht, $menuFrameWidth, frame_type);
	}
}

@proc void DrawButtonItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    char *key = ResourcedText($, item, SUBRESOURCE_KEY);

    /* draw label and frame */
    DrawTextItem($, ms, item, x, y);
    /* draw key, if menu is vertical */
    if (key && (!$horizontal || ms->prev))
	XfwfDrawString(XtDisplay($), ms->win, item->enabled ? $gc : $graygc, $font,
		       (x + ms->wLeft + ms->wMiddle + 2*$spacing),
		       (y + $menuFrameWidth + $vMargin + $font->ascent),
		       key, strlen(key), NULL, NULL);
}

@proc void DrawRadioItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    Position ix = x + $menuFrameWidth + $hMargin;
    Position iy = y + $menuFrameWidth + $vMargin + ($font->ascent + $font->descent - $indicatorSize) / 2;

    /* draw button */
    DrawButtonItem($, ms, item, x, y);
    /* draw diamond */
    XfwfDrawDiamond($, ms->win, $lightgc, $darkgc, $sunkengc, $backgroundgc, $bordergc,
		    ix, iy, $indicatorSize, $menuFrameWidth, 0, item->set);
}

@proc void DrawToggleItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    Position ix = x + $menuFrameWidth + $hMargin;
    Position iy = y + $menuFrameWidth + $vMargin + ($font->ascent + $font->descent - $indicatorSize) / 2;

    /* draw button */
    DrawButtonItem($, ms, item, x, y);
    /* draw diamond */
    XfwfDrawSquare($, ms->win, $lightgc, $darkgc, $sunkengc, $backgroundgc, $bordergc,
		   ix, iy, $indicatorSize, $menuFrameWidth, 0, item->set);
}

@proc void DrawCascadeItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    Position ix = x + ms->w - (3 * $menuFrameWidth + $hMargin + $indicatorSize);
    Position iy = y + $menuFrameWidth + $vMargin + ($font->ascent + $font->descent - $indicatorSize) / 2;

    /* draw text (no key) */
    DrawTextItem($, ms, item, x, y);
    
    if (!ms->prev && $usedAsChoice) {
	/* choice item */
	XfwfDrawRectangle($, ms->win, $lightgc, $darkgc, $sunkengc, $backgroundgc,
			  ix-$highlightThickness-$frameWidth-$outerOffset-$innerHOffset,
			  iy+$indicatorSize/4,
			  $indicatorSize,
			  $indicatorSize/2,
			  $menuFrameWidth, XfwfRaised);
    } else if (!$horizontal || ms->prev) {
	/* submenu item */
	XfwfDrawArrow($, ms->win, $lightgc, $darkgc, $sunkengc, $backgroundgc,
		      ix, iy, $indicatorSize, $indicatorSize, $menuFrameWidth,
		      XfwfArrowRight, (item->enabled && ms->selected==item));
    }
}

@proc void DrawSeparatorItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
    if (!$horizontal ||ms->prev)
	XfwfDrawLine($, ms->win,  $lightgc, $darkgc, $sunkengc, $backgroundgc,
		     x, y, ms->w, max(2, $menuFrameWidth), FALSE,
		     XfwfChiseled, FALSE, FALSE);
}

@proc void DrawPushrightItem($, menu_state *ms, menu_item *item, unsigned x, unsigned y)
{
}

@ The draw functions are stored inside an array for easier access. This array is
defined static since there is no use in overriding it by an ancestor.

	@var void* DrawFunctionList[] = {
	    DrawTextItem,
	    DrawButtonItem,
	    DrawRadioItem,
	    DrawToggleItem,
	    DrawCascadeItem,
	    DrawSeparatorItem,
	    DrawPushrightItem
	}

@ DisplayMenu draws the menu specified by the current menu state. Since menus are drawn
one after the other it is usually only necessary to compute and draw a newly visible menu.

@proc void DisplayMenu($, menu_state *ms)
{
    typedef void (*DrawFunction)(Widget, menu_state*, menu_item*, unsigned, unsigned);

    menu_item *item;
    unsigned  x, y;
    Boolean   top = !ms->prev && ($horizontal || $usedAsChoice);

    if (top) {
	x = $frameWidth + $innerHOffset;
	y = $frameWidth + $innerVOffset;
    } else
	x = y = $menuFrameWidth;
    
    if (!ms->menu && ms == $state && $usedAsChoice) {
	/* draw choice item, even if there is no menu */
	menu_item choice = { "", NULL, NULL, NULL, -1, MENU_CASCADE, TRUE, FALSE,
			     NULL, NULL, NULL };
	choice.start = ms->x;
	choice.end = ms->x+ms->w;
	DrawCascadeItem($, $state, &choice, x, y);
    }
    for (item = ms->menu; item; item = item->next) {
	((DrawFunction)DrawFunctionList[item->type])($, ms, item, x, y);
	if (top) {
	    if (item->type == MENU_PUSHRIGHT) {
		if (x + item->end <= ms->w)
		    x = ms->w - item->end;
	    } else
		x = item->end + $spacing;
	} else {
	    y = item->end;
	}
    }
    if ($popped_up || ms->prev)
	XfwfDrawRectangle($, ms->win, $lightgc, $darkgc, $bordergc, $backgroundgc,
			  0, 0, ms->w, ms->h, $menuFrameWidth, XfwfRaised);
    else
	XfwfDrawRectangle($, ms->win, $lightgc, $darkgc, $bordergc, $backgroundgc,
			  $highlightThickness, $highlightThickness,
			  ms->w-2*$highlightThickness, ms->h-2*$highlightThickness,
			  $frameWidth, XfwfRaised);
}

@ The following code handles the highlighting and unhighlighting of menu items.
If a cascade menu item is highlighted, the submenu is immediately popped up, and,
if it is unhighlighted, popped down.

@ |ComputeItemPos| computes the |(x, y)| coordinate of the item in the given
menu specified by the menu state.

@proc void ComputeItemPos($, menu_state *ms, menu_item *item, unsigned *x, unsigned *y)
{
    if (!ms->prev && ($horizontal || $usedAsChoice)) { /* in menubar ? */
	menu_item  *i;
	Dimension  pushright = 0;
	for (i = ms->menu; i && i != item; i = i->next)
	    if (!pushright && i->type==MENU_PUSHRIGHT)
		pushright = (ms->w - $frameWidth - $innerHOffset) - i->end - i->start;
	*x = item->start + pushright;
	*y = $frameWidth + $innerVOffset;
    } else {
	*x = $menuFrameWidth;
	*y = item->start;
    }
}

@ |MakeNewMenuWindow| is called, when a submenu has to be popped up. The menu
item for which the submenu has to be pooped up is specified by the pair |prev|,
|item|. |(x, y)| is the position of this item. For the new menu the coordinates
have to be computed.

@proc void MakeNewMenuWindow($, menu_state *prev, menu_item *item, unsigned x, unsigned y)
{
    int        scr_width  = WidthOfScreen(XtScreen($));
    int        scr_height = HeightOfScreen(XtScreen($));
    menu_state *new       = (menu_state*)XtMalloc(sizeof(menu_state));
    int        mask;
    XSetWindowAttributes xswa;

    /* Create new menu_state, initialize it and compute menu size */
    new->menu      = item->contents;
    new->selected  = NULL;
    new->prev      = prev;
    $state = new;
    ComputeMenuSize($, new);

    /* position window on screen */
    if (!prev->prev && $usedAsChoice) {
	menu_item *subitem = FindItem(new, item->label);
	int       option_wd = prev->w
			      - (  3 * $menuFrameWidth
				 + 2 * $hMargin 
				 +     $indicatorSize
				 +     $spacing
				 + 2 * $frameWidth
				 + 2 * $highlightThickness);
	new->x = prev->x + x;
	new->y = prev->y + y - (subitem ? subitem->start : 0);
	if (new->y < 0)
	    new->y = 0;
	if (new->y && (new->y + new->h > scr_height))
	    new->y = scr_height - new->h;
	if (new->w < option_wd)
	    new->w = option_wd;
    } else if ($horizontal && !prev->prev) { /* item in menubar? */
	new->x = prev->x + x;
	if (new->x + new->w > scr_width)
	    new->x = scr_width -  new->w;
	new->y = prev->y + prev->h - $menuFrameWidth;
	if (new->y + new->h > scr_height) /* menu doesn't below menubar -> */
	    if (new->y - new->h >= 0)     /* is enough place above the menubar ?*/
		new->y = prev->y - new->h +$menuFrameWidth;
	    else			  /* show as much as possible */
		new->y = scr_height - new->h;
    } else {
	if (prev->x + prev->w + new->w < scr_width) /* place right of menu? */
	    new->x = prev->x + prev->w;
	else if (prev->x - new->w > 0)              /* place left of menu? */
	    new->x = prev->x - new->w;
	else			                    /* place on screen border*/
	    new->x = scr_width - new->w;

	new->y = prev->y + y - $menuFrameWidth;
	if (new->y + new->h > scr_height)
	    new->y = scr_height - new->h;
    }

    /* Create new window */
    xswa.save_under        = TRUE;
    xswa.override_redirect = TRUE;
    xswa.background_pixel  = $background_pixel;
    xswa.event_mask        = ExposureMask | ButtonMotionMask | 
	                     PointerMotionHintMask | ButtonReleaseMask |
	                     ButtonPressMask;
    xswa.cursor            = $cursor;
    mask                   = CWSaveUnder | CWOverrideRedirect | CWBackPixel |
	                     CWEventMask | CWCursor;
    new->win
	= XCreateWindow(XtDisplay($), RootWindowOfScreen(DefaultScreenOfDisplay(XtDisplay($))),
			new->x, new->y, new->w, new->h,
			0, 0, CopyFromParent, CopyFromParent, mask, &xswa);
}

@ |HighlightItem| draws the highlight border and pops up submenus if any.

@proc void HighlightItem($, menu_state *ms, menu_item *item, Boolean select_child)
{
    typedef void (*DrawFunction)(Widget, menu_state*, menu_item*, unsigned, unsigned);

    unsigned x, y;
    Time     time = CurrentTime;

    if (!item) /* nothing to highlight */
	return;

    /* unhighlight old selected item */
    if (ms->selected)
	UnhighlightItem($, ms, ms->selected);
    /* new item: select and popup eventually submenu */
    ms->selected = item;
    ComputeItemPos($, ms, item, &x, &y);
    ((DrawFunction)DrawFunctionList[item->type])($, ms, item, x, y);
    if (item->type == MENU_CASCADE && item->enabled) {
	CreateItemMenu($, ms, item, x, y);
	if (select_child)
	    /* ms is the previous menu state */
	    if ($usedAsChoice && !ms->prev) {
		menu_item *subitem = FindItem($state, ms->menu->label);
		HighlightItem($, $state, subitem, False);
	    } else {
		HighlightFirstItem($, $state);
	    }
    }
    /* call onChange callback */
    for (ms = $state; ms; ms=ms->prev)
	if (ms->selected)
	    break;
    if (ms && item->enabled
    && item->type != MENU_TEXT
    && item->type != MENU_SEPARATOR
    && item->type != MENU_PUSHRIGHT)
	XtCallCallbackList($, $onChange, (XtPointer)ms->selected);

    $traversal_focus = FALSE;
    XSync(XtDisplay($), FALSE);
    XtCallAcceptFocus($, &time);
}

@proc HighlightFirstItem($, menu_state* ms)
{
    menu_item *item;

    /* only for second level menus */
    if ( !(ms && ms->prev && !ms->prev->prev) ) return;
    /* nothing to do, if something is selected */
    if (ms->selected) return;
    /* only for choice items or menubars */
    if ( !($usedAsChoice || $horizontal) ) return;
    /* find first selectable item in menu */
    item = ms->menu;
    while (item && !(item->enabled && item->type != MENU_TEXT &&
		     item->type != MENU_SEPARATOR && item->type != MENU_PUSHRIGHT))
	item = item->next;
    /* highlight item */
    HighlightItem($, ms, item, False);
}

@ |UnhighlightItem| undraws the highlight border and pops down submenus if any.
This may happen to a complete submenu tree if the mouse has moved far away to
another menu item e.g. in the menubar.

@proc void UnhighlightItem($, menu_state *ms, menu_item *item)
{
    typedef void (*DrawFunction)(Widget, menu_state*, menu_item*, unsigned, unsigned);

    unsigned   x, y;

    if (!item) /* nothing to unhighlight */
	return;
    ms->selected = NULL;
    ComputeItemPos($, ms, item, &x, &y);
    ((DrawFunction)DrawFunctionList[item->type])($, ms, item, x, y);
    if (item->type == MENU_CASCADE && item->enabled)
	DestroyItemMenu($, ms, item);

    /* call onChange callback */
    for (ms = ms->prev; ms; ms=ms->prev)
	if (ms->selected)
	    break;
    if (ms && item->enabled
    && item->type != MENU_TEXT
    && item->type != MENU_SEPARATOR
    && item->type != MENU_PUSHRIGHT)
	XtCallCallbackList($, $onChange, (XtPointer)ms->selected);
}

@ |CreateItemMenu| and |DestroyItemMenu| create or destroy submenus to the 
specified item. They are used only by |HighlightItem| and |UnhighlightItem|.

@proc void CreateItemMenu($, menu_state *ms, menu_item *item, int x, int y)
{
    MakeNewMenuWindow($, ms, item, x, y);
    XClearWindow(XtDisplay($), $state->win);
    XMapRaised(XtDisplay($), $state->win);
    DisplayMenu($, $state);
}

@proc void DestroyItemMenu($, menu_state *ms, menu_item *item)
{
    menu_state *state = $state;

    while (state != ms) {
	menu_state *last = state;

	XDestroyWindow(XtDisplay($), state->win);
	state = state->prev;
	XtFree((char*)last);
    }
    $state = ms;
}

@ |FindSelectedChoice| is used to find an item in menu. It returns the offset
of the found menu item.

@proc menu_item* FindItem(menu_state *ms, char *item_label)
{
    menu_item *i = NULL;

    if (item_label) {
	for (i = ms->menu; i; i = i->next)
	    if (strcmp(i->label, item_label) == 0)
		break;
    }
    return (i);
}

@proc menu_item* FindHotKey(menu_item *item, Boolean ctrl, Boolean alt, Boolean meta, KeySym key)
{
    for ( ; item; item = item->next) {
	/* skip inactive items and submenus */
	if (!item->enabled) continue;
	/* recurse on submenus */
	if (item->type == MENU_CASCADE) {
	    menu_item *found = FindHotKey(item->contents, ctrl, alt, meta, key);
	    if (found)
		return found; /* FOUND */
	}
	/* check buttons, toggles, radios for hotkey */
	else if (item->type == MENU_BUTTON || item->type == MENU_TOGGLE || item->type == MENU_RADIO) {
	    char *hotkey;
	    if ( (hotkey = item->key_binding) ) {
		/* first check if modifiers are conform: ctrl, alt, meta */
		if (strstr(hotkey, "Ctrl-") || strstr(hotkey, "C-")
		||  strstr(hotkey, "Ctrl+") || strstr(hotkey, "C+")) {
		    if (!ctrl) continue;
		} else {
		    if (ctrl) continue;
		}
		if (strstr(hotkey, "Alt-") || strstr(hotkey, "A-")
		||  strstr(hotkey, "Alt+") || strstr(hotkey, "A+")) {
		    if (!alt) continue;
		} else {
		    if (alt) continue;
		}
		if (strstr(hotkey, "Meta-") || strstr(hotkey, "M-")
		||  strstr(hotkey, "Meta+") || strstr(hotkey, "M+")) {
		    if (!meta) continue;
		} else {
		    if (meta) continue;
		}
		/* modifiers are conform: check key */
		while (*hotkey)
		    ++hotkey;
		--hotkey;
		if (mytolower(*hotkey) == key)
		    return item; /* FOUND */
	    }
	}
    }
    return NULL; /* NOT FOUND */
}

@ |HandleMotionEvent| finds the item corresponding to the root window coordinates
|(mx, my)|. It checks, if the mouse has moved outside the window or if the item
did not change. If the item changed, the old item is unhighlighted and the new is
highlighted. The |XtNonChange| callback is called, when the item changed.

@proc int HandleMotionEvent($, int mx, int my)
{
    menu_state *ms;
    menu_item  *item = NULL;
    Dimension  pushright = 0;
    Boolean    foundone = 0;

    /* find menu_state belonging to event */
    for (ms = $state; ms; ms = ms->prev) {
	if (ms->x <= mx && mx <= ms->x + ms->w && ms->y <= my && my <= ms->y + ms->h) {
	    foundone = 1;
	    /* find menu_item belonging to event */
	    item = ms->menu;
	    if ( item && !(item->next) && (item->type == MENU_CASCADE) && ($usedAsChoice) )
		 break; /* only one cascade item with indicator -> HIT */
	    for ( ; item; item = item->next)
		if ($horizontal && !ms->prev) {
		    if (!pushright && item->type == MENU_PUSHRIGHT)
			pushright = ms->w - item->end - item->start;
		    else if (ms->x + pushright + item->start <= mx && mx < ms->x + pushright + item->end)
			break;
		} else if (ms->y + item->start <= my && my < ms->y + item->end) {
		    break;
		}
	    break;
	}
    }
    
    if (!foundone)
	$moused_out = 1;
    if (!item) { /* if pointer not on menu_item unhighlight last selected */
	UnhighlightItem($, $state, $state->selected);
	return 0;
    }
    if (item == ms->selected)
	return 1;
    /* highlight item: unhighlight old item on same level (ms!), popup submenu */
    HighlightItem($, ms, item, False);

    return 1;
}

@exports

@ |XfwfPopupMenu| and |XfwfPopupMenuAtPos| are used to popup a menu at the
pointer position or at a specified position (x, y).

@proc void XfwfPopupMenu($, Widget calling_widget)
{
    XButtonPressedEvent ev;

    if (! XtIsSubclass($, xfwfMenuWidgetClass))
	XtError("XfwfPopupMenu called with incorrect widget type");

    /* get position of pointer in calling widget */
    ev.type = ButtonPress;
    ev.serial = 0;
    ev.send_event = 0;
    ev.display = XtDisplay(calling_widget);
    ev.window = XtWindow(calling_widget);
    ev.time = CurrentTime;
    ev.button = 0;
    XQueryPointer(ev.display, ev.window, &ev.root,
                  &ev.subwindow, &ev.x_root, &ev.y_root,
                  &ev.x, &ev.y, &ev.state);
    XfwfPopupMenuAtPos($, ev.x_root, ev.y_root);
}

@proc void XfwfPopupMenuAtPos($, int x, int y)
{
    Time    time         = CurrentTime;
    Screen  *scr         = XtScreen($);
    Widget  popup_shell  = XtParent($);
    int     border_width = popup_shell->core.border_width;
    int     w, h;

    if (! XtIsSubclass($, xfwfMenuWidgetClass))
	XtError("XfwfPopupMenuAtPos called with incorrect widget type");

    /* compute size and position of popup menu */
    $popped_up = TRUE;
    $was_dragged = False;
    $horizontal = FALSE;
    ComputeMenuSize($, $state);
    w = $state->w;
    h = $state->h;
    if (x + w > WidthOfScreen(scr))
	x = WidthOfScreen(scr) - w - 2*border_width;
    if (y + h > HeightOfScreen(scr))
	y = HeightOfScreen(scr) - h - 2*border_width;
    x = (x > border_width ? x - border_width : border_width);
    y = (y > border_width ? y - border_width : border_width);
    XtConfigureWidget(popup_shell, x, y, w, h, border_width);
    /* popup, display and handle menu */
    XtPopup(popup_shell, XtGrabNone);
    DisplayMenu($, $state);
    XtGrabPointer($, FALSE,
		  (ButtonMotionMask | PointerMotionHintMask |
		  ButtonReleaseMask | ButtonPressMask),
		  GrabModeAsync, GrabModeAsync,
		  None, $cursor, time);
    XtGrabKeyboard($, FALSE, GrabModeAsync, GrabModeAsync, CurrentTime);
    $state->x = x+border_width;
    $state->y = y+border_width;
    /* init first motion event */
    $mouse_selection = True; /* start of mouse selection */
    $moused_out = False;
    $pointer_grabbed = True;
    HandleMotionEvent($, x, y);
}

@imports

@incl <stdio.h>
@incl <stdlib.h>
@incl <string.h>
@incl <ctype.h>
@incl <XfwfDraw3D.h>
@incl <X11/keysym.h>
@incl <TabString.h>
