
/*
 * seq_read.c -- read the .mh_sequence file and
 *            -- initialize sequence information
 *
 * $Id$
 */

#include <h/mh.h>

/*
 * static prototypes
 */
static int seq_init (struct msgs *, char *, char *);


/*
 * Get the sequence information for this folder from
 * .mh_sequence (or equivalent specified in .mh_profile)
 * or context file (for private sequences).
 */

void
seq_read (struct msgs *mp)
{
    int alen, i, j, plen, state;
    char *cp, seqfile[PATH_MAX];
    char name[NAMESZ], field[BUFSIZ];
    struct node *np;
    FILE *fp;

    /* sanity check - check that context has been read */
    if (defpath == NULL)
	adios (NULL, "oops, context hasn't been read yet");

    /* Always add the "cur" sequence */
    mp->msgattrs[i = 0] = getcpy (current);
    mp->msgattrs[++i] = NULL;
    make_all_public (mp);	/* initially, all public */

    /*
     * If nmh has been compiled with NOPUBLICSEQ (so that
     * mh_seq == NULL), or the user has defined the
     * "mh-sequences" profile entry, but left it empty
     * (*mh_seq == '\0'), then make all sequences private.
     */
    if (mh_seq && *mh_seq != '\0')
	snprintf (seqfile, sizeof(seqfile), "%s/%s", mp->foldpath, mh_seq);
    else
	goto private_only;

    /*
     * read folder's sequences file for public sequences
     */
    if ((fp = fopen (seqfile, "r"))) {
	for (state = FLD;;) {
	    switch (state = m_getfld (state, name, field, sizeof(field), fp)) {
		case FLD: 
		case FLDPLUS:
		case FLDEOF: 
		    if (state == FLDPLUS) {
			cp = getcpy (field);
			while (state == FLDPLUS) {
			    state = m_getfld (state, name, field, sizeof(field), fp);
			    cp = add (field, cp);
			}
			seq_init (mp, getcpy (name), trimcpy (cp));
			free (cp);
		    } else {
			seq_init (mp, getcpy (name), trimcpy (field));
		    }
		    if (state == FLDEOF)
			break;
		    continue;

		case BODY: 
		case BODYEOF: 
		    adios (NULL, "no blank lines are permitted in %s", seqfile);
		    /* fall */

		case FILEEOF:
		    break;

		default: 
		    adios (NULL, "%s is poorly formatted", seqfile);
	    }
	    break;
	}
	fclose (fp);
    }

private_only:

    /*
     * Scan profile/context list for private sequences.
     *
     * We search the context list for all keys that look like
     * "atr-seqname-folderpath", and add them as private sequences.
     */
    alen = strlen ("atr-");
    plen = strlen (mp->foldpath) + 1;

    for (np = m_defs; np; np = np->n_next) {
	if (ssequal ("atr-", np->n_name)
		&& (j = strlen (np->n_name) - plen) > alen
		&& *(np->n_name + j) == '-'
		&& strcmp (mp->foldpath, np->n_name + j + 1) == 0) {
	    cp = getcpy (np->n_name + alen);
	    *(cp + j - alen) = '\0';
	    if ((i = seq_init (mp, cp, getcpy (np->n_field))) != -1)
		make_seq_private (mp, i);
	}
    }
}


/*
 * Set the various message bits for this sequence
 */

static int
seq_init (struct msgs *mp, char *name, char *field)
{
    int i, j, k, is_current;
    char *cp, **ap;

    /*
     * Check if this is "cur" sequence,
     * so we can do some special things.
     */
    is_current = !strcmp (current, name);

    /*
     * If we've seen this sequence before, then
     * clear the bit for this sequence from
     * all the mesages in this folder.
     */
    for (i = 0; mp->msgattrs[i]; i++) {
	if (!strcmp (mp->msgattrs[i], name)) {
	    for (j = mp->lowmsg; j <= mp->hghmsg; j++)
		clear_sequence (mp, i, j);
	    break;
	}
    }

    if (i >= NUMATTRS) {
	free (name);
	free (field);
	return -1;
    }

    if (mp->msgattrs[i]) {
	free (name);
    } else {
	mp->msgattrs[i] = name;
	mp->msgattrs[i + 1] = NULL;
    }

    /*
     * split up the different message ranges at whitespace
     */
    for (ap = brkstring (field, " ", "\n"); *ap; ap++) {
	if ((cp = strchr(*ap, '-')))
	    *cp++ = '\0';
	if ((j = m_atoi (*ap)) > 0) {
	    k = cp ? m_atoi (cp) : j;
#if 0
	    if (is_current && j >= mp->lowmsg &&
		j <= mp->hghmsg && does_exist(mp, j))
		mp->curmsg = j;
#endif
	    if (is_current)
		mp->curmsg = j;
	    /*
	     * We iterate through messages in this range
	     * and flip on bit for this sequence.
	     */
	    for (; j <= k; j++) {
		if (j >= mp->lowmsg && j <= mp->hghmsg && does_exist(mp, j))
		    add_sequence (mp, i, j);
	    }
	}
    }
    free (field);
    return i;
}
