# preferences.tcl --
#
# This file controls the preferences window. There is just one preferences
# window, if the user requests a new preferences window when the window
# already exists, it is just unmapped and raised (if possible).
#
#
#  TkRat software and its included text is Copyright 1996,1997,1998
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.

# The last wisited window
set lastPref General

# Preferences --
#
# Make sure the user sees an preferences window.
#
# Arguments:

proc Preferences {} {
    if ![winfo exists .pref] {
	BuildPreferences
    } else {
	wm deiconify .pref
    }
}


# BuildPreferences --
#
# Builds the preferences window.
#
# Arguments:

proc BuildPreferences {} {
    global t b pref lastPref

    # Initialize defaults
    set pref(options) {}

    # Initialize data table
    foreach l [GetLanguages] {
	if ![string length [GetFont [lindex $l 2] 0 {}]] {
	    continue
	}
	lappend lang [lrange $l 0 1]
    }

    # variable label type type_args
    #		      entry unit_label
    #                 option options
    #		      special procedure
    set pref(General) "\
	{language language option [list $lang]} \
	{color_set color_scheme special SetupColor} \
	{iconic startup_mode option \
		{{1 {$t(iconic)}} {0 {$t(normal)}}}} \
	{info_changes show_changes option \
		{{1 {$t(show)}} {0 {$t(dont_show)}}}} \
	{mail_steal check_stolen_mail option \
		{{1 {$t(check)}} {0 {$t(dont_check)}}}} \
	{print_command print_command entry} \
	{print_header print_headers option \
		{{all {$t(all)}} {selected {$t(selected)}} {none {$t(none)}}}} \
	{expunge_on_close expunge_on_close option \
		{{1 {$t(do)}} {0 {$t(do_not)}}}} \
	{tmp tmp_dir entry} \
	{permissions file_permissions entry} \
	{userproc userproc_file entry} \
	{hold_dir hold_dir entry} \
	{fontsize character_size option \
		{{8 8} {10 10} {12 12} {14 14} {18 18} {24 24}}} \
	{keep_pos remember_pos option \
		{{1 {$t(do_remember)}} {0 {$t(dont_remember)}}}} \
	{mailcap_path mailcap_path entry} \
	{terminal terminal_command entry} \
    "
    set pref(Folder) "\
	{main_window_name window_name entry} \
	{icon_name icon_name entry} \
	{icon icon_bitmap option \
		{{normal {$t(normal_bitmap)}} \
		 {small {$t(small_bitmap)}} \
		 {none {$t(none)}}}} \
	{message_llength messages_shown entry} \
	{list_format list_format entry} \
	{show_header show_headers option \
		{{all {$t(show_all_headers)}} \
		 {selected {$t(show_selected_headers)}} \
		 {none {$t(show_no_headers)}}}} \
	{show_header_selection show_header_selection entry} \
	{browse default_browse_mode option \
		{{normal {$t(no_browse)}} \
		 {browse {$t(do_browse)}} \
		 {folder {$t(use_folder_default)}}}} \
	{folder_sort sort_order option \
		{{subject {$t(subject_by_date)}} \
		 {subjectonly {$t(subjectonly)}} \
		 {sender {$t(bysender)}} \
		 {folder {$t(natural_order)}} \
		 {reverseFolder {$t(reverse_natural_order)}} \
		 {date {$t(by_date)}} \
		 {reverseDate {$t(reverse_date)}}}} \
	{start_selection start_selection option \
		{{first {$t(first_message)}} \
		 {last {$t(last_message)}} \
		 {first_new {$t(first_new_message)}} \
		 {before_new {$t(before_first_new_message)}}}} \
	{log_timeout log_timeout entry_unit {$t(seconds)}} \
	{cache_passwd cache_passwd option \
		{{1 {$t(do_cache)}} {0 {$t(do_not_cache)}}}} \
	{cache_passwd_timeout cache_timeout entry_unit {$t(seconds)}} \
	{cache_conn cache_conn option \
		{{1 {$t(do_cache)}} {0 {$t(do_not_cache)}}}} \
	{cache_conn_timeout cache_timeout entry_unit {$t(seconds)}} \
	{urlprot url_protocols entry} \
	{url_color url_color entry} \
	{url_viewer url_viewer option \
		{{RatUP {$t(userproc): RatUP_ShowURL}} \
		 {netscape Netscape}}} \
	{netscape netscape_cmd entry} \
	{netscape_args netscape_args entry} \
    "
    set pref(Dbase) "\
	{dbase_dir dbase_dir entry} \
	{def_extype extype option \
		{{none {$t(none)}} \
		 {remove {$t(remove)}} \
		 {incoming {$t(add_incoming)}} \
		 {backup {$t(backup)}}}} \
	{def_exdate exdate entry_unit {$t(days)}} \
	{dbase_backup dbase_backup entry} \
	{chunksize chunksize entry} \
	{expire_interval expire_interval entry_unit {$t(days)}} \
    "
    set pref(Watcher) "\
	{watcher_enable watcher_enable option \
		{{1 {$t(enabled)}} {0 {$t(disabled)}}}} \
	{watcher_time intervals special SetupWatcherTime} \
	{watcher_name window_name entry} \
	{watcher_max_height max_height entry_unit {$t(lines)}} \
	{watcher_font font entry} \
	{watcher_show show_messages option \
		{{new {$t(new)}} {all {$t(all)}}}} \
	{watcher_format list_format entry} \
	{watcher_bell bell_ringings entry} \
    "
    set pref(Compose) "\
	{reply_to default_reply_to entry} \
	{compose_headers headers entry} \
	{editor external_editor entry} \
	{attribution attribution entry} \
	{forwarded_message forwarded_label entry} \
	{reply_lead reply_lead entry} \
	{skip_sig on_reply option \
		{{true {$t(skip_sig)}} {false {$t(keep_sig)}}}} \
	{no_subject no_subject entry} \
	{signature signature_file entry} \
	{lookup_name lookup_name option \
		{{true {$t(do_lookup)}} {false {$t(dont_lookup)}}}} \
	{copy_attached copy_attached_files option \
		{{1 {$t(true)}} {0 {$t(false)}}}} \
    "
    set pref(Sending) "\
	{from use_from_address entry} \
	{masquerade_as masquerade_host_as entry} \
	{domain domain entry} \
	{delivery_mode delivery_mode option \
		{{direct {$t(direct)}} {deferred {$t(deferred)}}}} \
	{sendprot sendprot option \
		{{smtp {$t(smtp)}} {prog {$t(user_program)}}}} \
	{smtp_hosts smtp_hosts entry} \
	{smtp_timeout smtp_timeout entry_unit {$t(seconds)}} \
	{smtp_reuse smtp_reuse option \
		{{1 {$t(true)}} {0 {$t(false)}}}} \
	{smtp_verbose smtpv option \
		{{0 {$t(none)}} \
		 {1 {$t(terse)}} \
		 {2 {$t(normal)}} \
		 {3 {$t(verbose)}}}} \
	{sendprog sendprog entry} \
	{sendprog_8bit progin option \
		{{true {$t(eightbit)}} {false {$t(sevenbit)}}}} \
	{charset charset entry} \
	{force_send force_send option \
		{{1 {$t(force)}} {0 {$t(no_force)}}}} \
    "
    set pref(Notification) "\
	{dsn_directory dsn_directory entry} \
	{dsn_request default_action option \
		{{0 {$t(not_request_dsn)}} {1 {$t(request_dsn)}}}} \
	{dsn_snarf_reports folderwindow option \
		{{0 {$t(not_snarf_dsn)}} {1 {$t(snarf_dsn)}}}} \
	{dsn_expiration dsn_expiration entry_unit {$t(days)}} \
	{dsn_verbose report_level special SetupDSNVerbose} \
    "
    set pref(PGP) "\
	{pgp_enable pgp_handling option \
		{{1 {$t(enabled)}} {0 {$t(disabled)}}}} \
	{pgp_prog pgp_program entry} \
	{pgp_args pgp_extra_args entry} \
	{cache_pgp cache_passwd option \
		{{1 {$t(do_cache)}} {0 {$t(do_not_cache)}}}} \
	{cache_pgp_timeout cache_timeout entry_unit {$t(seconds)}} \
	{pgp_sign sign_outgoing option \
		{{1 {$t(true)}} {0 {$t(false)}}}} \
	{pgp_encrypt encrypt_outgoing option \
		{{1 {$t(true)}} {0 {$t(false)}}}} \
    "

    # Create toplevel
    set w .pref
    toplevel $w
    wm title $w $t(preferences)

    # The buttons
    frame $w.buttons
    frame $w.buttons.def -relief sunken -bd 1
    button $w.buttons.def.ok -text $t(apply) -command PrefApply
    pack $w.buttons.def.ok -padx 4 -pady 4
    button $w.buttons.cancel -text $t(reset) \
	    -command {
			global pref
			set pref(options) {}
		     }
    button $w.buttons.dismiss -text $t(dismiss) \
	    -command "PrefCheck; \
		      RecordSize $w.f.canvas prefCanvas;
		      RecordPos $w preferences;
		      destroy $w"
    pack $w.buttons.def \
	 $w.buttons.cancel \
         $w.buttons.dismiss -side left -expand 1
    bind $w <Return> PrefApply
    set b($w.buttons.def.ok) apply_prefs
    set b($w.buttons.cancel) cancel
    set b($w.buttons.dismiss) dismiss

    # Prepare for the tabs
    canvas $w.page -highlightthickness 0
    after idle [list PrefDrawTabs $w.page]
    set pref(tab) $w.page

    # The canvas
    frame $w.f -relief sunken -bd 1
    scrollbar $w.f.scroll \
	    -relief sunken \
	    -bd 1 \
	    -command "$w.f.canvas yview" \
	    -highlightthickness 0
    canvas $w.f.canvas \
	    -yscrollcommand "$w.f.scroll set" \
	    -highlightthickness 0
    Size $w.f.canvas prefCanvas
    pack $w.f.scroll -side right -fill y
    pack $w.f.canvas -side left -expand 1 -fill both
    set pref(frame) [frame $w.f.canvas.f]
    set pref(cid) [$w.f.canvas create window 0 0 \
	    -anchor nw \
	    -window $w.f.canvas.f]
    set pref(canvas) $w.f.canvas

    pack $w.page -side top -fill x
    pack $w.buttons -side bottom -fill x -pady 5
    pack $w.f -side top -pady 10 -expand 1 -fill both

    Place $w preferences

    PrefShow $lastPref
}

# PrefDrawTabs -
#
# Draw the tabs at the top of the page
#
# Arguments:
# w - Canvas to draw in

proc PrefDrawTabs {w} {
    global t lastPref

    set normalColor [$w cget -selectbackground]
    set activeColor [$w cget -background]
    set margin 6
    set x 2
    set maxh 0

    foreach name {General Folder Dbase Watcher Compose Sending Notification
	    PGP} {
	set id [$w create text [expr $x+$margin+2] [expr -0.5*$margin] \
		-anchor sw -text $t([string tolower $name]) -tags $name]
	set bbox [$w bbox $id]
	set wd [expr [lindex $bbox 2]-[lindex $bbox 0]]
	set ht [expr [lindex $bbox 3]-[lindex $bbox 1]]
	if {$ht > $maxh} {
	    set maxh $ht
	}

	$w create polygon 0 0  $x 0 \
		[expr $x+$margin] [expr -$ht-$margin] \
		[expr $x+$margin+$wd] [expr -$ht-$margin] \
		[expr $x+$wd+2*$margin] 0 \
		2000 0  2000 10  0 10 \
		-outline black -fill $normalColor \
		-tags [list $name tab tab-$name]
	$w raise $id
	$w bind $name <ButtonPress-1> "PrefCheck; PrefShow $name"

	incr x [expr $wd+2*$margin]
    }
    set height [expr $maxh+2*$margin]
    $w move all 0 $height
    $w configure -width $x -height [expr $height+4]

    $w itemconfigure tab-$lastPref -fill $activeColor
    $w raise $lastPref
}

# PrefShow --
#
# Show a preferences window
#
# Arguments:
# pane - Which pane of preferences to show

proc PrefShow {pane} {
    global t b option pref lastPref

    set w $pref(tab)
    $w itemconfigure tab -fill [$w cget -selectbackground]
    $w itemconfigure tab-$pane -fill [$w cget -background]
    $w raise $pane

    set f $pref(frame)
    set s [grid slaves $f]
    if [llength $s] {
	eval "destroy $s"
    }
    foreach bn [array names b $f.*] {unset b($bn)}
    set row 0

    set pref(options) {}
    foreach p $pref($pane) {
	set var [lindex $p 0]
	lappend pref(options) $var
	set pref(opt,$var) $option($var)

	label $f.r${row}_lab -text $t([lindex $p 1]):
	grid $f.r${row}_lab -row $row -sticky ne -pady 5
	switch [lindex $p 2] {
	    entry {
		    entry $f.r${row}_item -textvariable pref(opt,$var) \
			    -width 60
		    grid $f.r${row}_item - -row $row -column 1 -sticky we
		    set b($f.r${row}_item) pref_$var
		}
	    entry_unit {
		    entry $f.r${row}_item -textvariable pref(opt,$var)
		    label $f.r${row}_unit -text ([lindex $p 3])
		    grid $f.r${row}_item -row $row -column 1 -sticky we
		    grid $f.r${row}_unit -row $row -column 2 -sticky we
		    set b($f.r${row}_item) pref_$var
		    set b($f.r${row}_unit) unit_pref
		}
	    option {
    		    OptionMenu $f.r${row}_item pref(opt,$var) \
			    pref(text,$var) [lindex $p 3]
		    grid $f.r${row}_item - -row $row -column 1 -sticky w
		    set b($f.r${row}_item) pref_$var
		}
	    special {
		    [lindex $p 3] $f.r${row}_item
		    grid $f.r${row}_item - -row $row -column 1 -sticky w
		}
	    default {puts "Internal error <$p>"}
	}
	if [RatIsLocked option([lindex $p 0])] {
	    $f.r${row}_item configure -state disabled -relief flat
	}

	incr row
    }

    grid columnconfigure $f 1 -weight 1

    set lastPref $pane

    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}

# PrefApply --
#
# Applies any changes to the preferences made in the current window.
#
# Arguments:

proc PrefApply {} {
    global option pref t folderWindowList lastPref

    switch $lastPref {
    general {
	    if [string compare $option(mailcap_path) $pref(opt,mailcap_path)] {
		RatMailcapReload
	    }
	}
    Watcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    Notification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    Compose {
	    if {[string compare $option(signature) $pref(opt,signature)]
		    && 1 == [llength [info commands RatUP_Signature]]} {
		Popup $t(sig_cmd_takes_precedence)
	    }
	}
    Sending {
	    if {[string compare $option(sendprog) $pref(opt,sendprog)]
		    && ![file executable [lindex $pref(opt,sendprog) 0]]} {
		Popup $t(warning_sendprog)
	    }
	}
    Folder {
	    if [string compare $option(icon) $pref(opt,icon)] {
		SetIcon . $pref(opt,icon)
	    }
	    if [string compare $option(message_llength) \
			       $pref(opt,message_llength)] {
		foreach fw $folderWindowList {
		    upvar #0 $fw fh
		    $fh(message_list) configure \
			    -height $pref(opt,message_llength)
		}
	    }
	    if [string compare $option(url_color) $pref(opt,url_color)] {
		foreach fw $folderWindowList {
		    upvar #0 $fw fh
		    $fh(text) tag configure URL -foreground $pref(opt,url_color)
		}
	    }
	}
    PGP {
	    if {[string compare $option(pgp_prog) $pref(opt,pgp_prog)]
		    && ![file executable [lindex $pref(opt,pgp_prog) 0]]} {
		Popup $t(warning_pgp_prog)
	    }
	}
    }
    set hasChanged 0
    set needRestart 0
    set newSendArgs 0
    foreach opt $pref(options) {
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set option($opt) $pref(opt,$opt)
	    set hasChanged 1
	    if { -1 != [lsearch -exact {language charset fontsize
		    			main_window_name icon_name
					default_folder watcher_name pgp_enable}\
				$opt]} {
		set needRestart 1
	    }
	    if { -1 != [lsearch -exact {charset masquerade_as domain sendprot
					smtp_hosts sendprog sendprog_8bit
					charset from smtp_verbose
					delivery_mode smtp_timeout force_send
					pgp_prog pgp_args} $opt]} {
		set newSendArgs 1
	    }
	}
    }

    if $hasChanged {
	switch $lastPref {
	Folder {
		foreach f $folderWindowList {
		    Sync $f 0
		}
	    }
	}

	SaveOptions
    }
    if $needRestart {
	Popup $t(need_restart)
    }
    if $newSendArgs {
	RatSend kill
    }
}


# PrefCheck --
#
# Checks if there are any unapplied changes and if there is the user is
# queried if he wants to apply them.
#
# Arguments:

proc PrefCheck {} {
    global option pref t lastPref

    switch $lastPref {
    Watcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    Notification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    }
    foreach opt $pref(options) {
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set value [RatDialog $t(unapplied_changes_title) \
		    $t(unapplied_changes) {} 0 $t(apply) $t(discard)]
	    if { 0 == $value } {
		PrefApply
	    }
	    return
	}
    }
}


# OptionMenu --
#
# Generates an option menu. The generated menu will have window name "w"
# and will set the "varName" variable. The different options are
# controlled by the value arguments. Each value argument is a list of
# two elemnts. The first is the value to set "varName" to and the second
# is the text to show. The menubutton will use "textVar" as the textvariable.
#
# Arguments:
# w	  -	Name of menubutton to create
# varName -	Variable to set to value
# textVar -	Variable to use for the text we show
# values  -	A list of lists which describes the values of this button

proc OptionMenu {w varName textVar values} {
    upvar #0 $varName var
    upvar #0 $textVar text

    set width 20
    menubutton $w -textvariable $textVar -indicatoron 1 \
		  -relief raised -menu $w.m
    menu $w.m -tearoff 0
    foreach elem $values {
	if ![string compare [lindex $elem 0] $var] {
	    set text [lindex $elem 1]
	}
	$w.m add command -label [lindex $elem 1] \
		-command "set $varName [list [lindex $elem 0]]; \
			  set $textVar [list [lindex $elem 1]]"
	if { $width < [string length [lindex $elem 1]]} {
	    set width [string length [lindex $elem 1]]
	}
    }
    $w configure -width $width
}



proc SetupColor {w} {
    global t option pref tk_version

    if {$tk_version < 8.0} {
	button $w -width 21 -text $t(out_of_order) \
		-command [list Popup $t(colorbug)]
	return
    }
    menubutton $w -textvariable pref(text,color_set) \
	    -indicatoron 1 -relief raised -menu $w.m
    menu $w.m -tearoff 0
    set width 20
    foreach c { {gray85 black} {PeachPuff2 black} {bisque black}
		{SlateBlue1 black} {SteelBlue4 white} {SkyBlue1 black}
		{aquamarine2 black} {SpringGreen4 black}} {
	set name $t([lindex $c 0])
	if ![string compare $c $option(color_set)] {
	    set pref(text,color_set) $name
	}
	$w.m add command -label $name \
		-command "set pref(opt,color_set) [list $c]; \
			  set pref(text,color_set) [list $name]; \
			  SetColor $c" \
		-background [lindex $c 0] -foreground [lindex $c 1]
	if { $width < [string length $name]} {
	    set width [string length $name]
	}
    }
    $w configure -width $width
}

proc SetupWatcherTime {w} {
    global t option pref

    frame $w
    label $w.flabel -text $t(stdfolders):
    entry $w.fentry -textvariable pref(opt,finterval)
    label $w.funit -text ($t(seconds))
    if [RatIsLocked option(watcher_time)] {
	$w.fentry configure -state disabled -relief flat
    }
    label $w.dlabel -text $t(dbase_folders):
    entry $w.dentry -textvariable pref(opt,dinterval)
    label $w.dunit -text ($t(seconds))
    if [RatIsLocked option(watcher_time)] {
	$w.dentry configure -state disabled -relief flat
    }
    foreach elem $option(watcher_time) {
	if ![string compare [lindex $elem 0] std] {
	    set pref(opt,finterval) [lindex $elem 1]
	}
	if ![string compare [lindex $elem 0] dbase] {
	    set pref(opt,dinterval) [lindex $elem 1]
	}
    }
    grid $w.flabel -row 0 -column 0 -sticky e
    grid $w.fentry -row 0 -column 1 -sticky we
    grid $w.funit -row 0 -column 2 -sticky we
    grid $w.dlabel -row 1 -column 0 -sticky e
    grid $w.dentry -row 1 -column 1 -sticky we
    grid $w.dunit -row 1 -column 2 -sticky we
    grid columnconfigure $w 1 -weight 1
}

proc SetupDSNVerbose {w} {
    global option t pref

    foreach elem $option(dsn_verbose) {
	set pref(opt,[lindex $elem 0]) [lindex $elem 1]
    }
    frame $w
    set irow 0
    foreach cat {failed delayed delivered relayed expanded} {
	set sf $w.$cat
	label ${sf}_l -text $t($cat): -width 12 -anchor e
	OptionMenu ${sf}_mbut pref(opt,$cat) pref(text,$cat) \
		[list [list none $t(rl_none)] \
		      [list status $t(rl_status)] \
		      [list notify $t(rl_notify)]]
	if [RatIsLocked option(dsn_verbose)] {
	    ${sf}_mbut configure -state disabled
	}
	grid ${sf}_l -row $irow -column 0 -sticky e
	grid ${sf}_mbut -row $irow -column 1 -sticky w
	incr irow
    }
}

# PrefProbeDSN --
#
# Probe the current SMTP servers for DSN support.
#
# Arguments:

proc PrefProbeDSN {} {
    global idCnt option t defaultFont

    # Create identifier
    set id probeWin[incr idCnt]
    upvar #0 $id hd
    set w .$id
    set hd(w) $w

    # Create toplevel
    toplevel $w
    wm title $w $t(probe)

    if [string compare $option(sendprot) smtp] {
	message $w.message -aspect 600 -text $t(dsn_need_smtp)
	button $w.button -text $t(dismiss) \
		-command "RecordPos $w prefProbeDSN; destroy $w; unset $id"
	pack $w.message \
	     $w.button -side top -padx 5 -pady 5
	return
    }

    set row 0
    foreach h $option(smtp_hosts) {
	label $w.l$row -text $h -width 32 -anchor e
	label $w.a$row -textvariable ${id}($h) -font $defaultFont -width 32 \
		-anchor w
	grid $w.l$row -row $row -column 0 -sticky e
	grid $w.a$row -row $row -column 1 -sticky w
	incr row
    }
    button $w.button -text $t(dismiss) -command "destroy $w; unset $id" \
	    -state disabled
    grid $w.button -row $row -column 0 -columnspan 2
    Place $w prefProbeDSN

    foreach h $option(smtp_hosts) {
        set hd($h) $t(probing)...
	update idletasks
	if [RatSMTPSupportDSN $h] {
	    set hd($h) $t(supports_dsn)
	} else {
	    set hd($h) $t(no_dsn_support)
	}
    }
    $w.button configure -state normal
}
