/*
 * Copyright 1997 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: saml1.c,v 1.20 1997/04/18 16:52:52 bousch Exp $
 *
 * A vanilla SAML interface for python.
 */

#include <stdio.h>
#include <stdlib.h>
#include "Python.h"
#include "saml.h"
#include "saml-errno.h"
#include "saml-util.h"
#include "saml-parse.h"
#include "mnode.h"

#define is_mathnode(v)	((v)->ob_type == &Mathnode_Type)
#define i_mnode(v)	(((mathnode_ob*)(v))->m)

typedef struct {
	PyObject_HEAD
	s_mnode *m;
} mathnode_ob;

static PyObject* SamlErrors[SE_MAX_ERROR+1];
static PyTypeObject Mathnode_Type;

static inline void debug_mnode (s_mnode *m)
{
	fprintf(stderr, "mnode_debug: addr=%p, type=%d, refs=%d\n",
		m, m->type, m->refs);
}

static PyObject* new_mobj (s_mnode *value)
{
	mathnode_ob *result;
	int reason;
	const char *errmsg, *where;
	char *message;

	reason = saml_errno(value);
	if (reason) {
		/* Error. Raise an exception */
		if ((unsigned)reason > SE_MAX_ERROR)
			reason = 0;
		errmsg = saml_strerror(reason);
		where = ((void_mnode*)value)->where;
		message = alloca(strlen(errmsg)+strlen(where)+5);
		strcpy(message, errmsg);
		if (where[0]) {
			strcat(message, " in ");
			strcat(message, where);
		}
		PyErr_SetString(SamlErrors[reason], message);
		unlink_mnode(value);
		return NULL;
	}
	result = PyObject_NEW(mathnode_ob, &Mathnode_Type);
	result->m = value;
	return (PyObject*) result;
}

/*
 * Methods
 */

static PyObject* mathnode_create (PyObject *self, PyObject *args)
{
	int mt;
	char *string;
	
	if (!PyArg_ParseTuple(args, "is", &mt, &string))
		return NULL;
	return new_mobj(mnode_build(mt,string));
}

static void mathnode_free (PyObject *self)
{
	unlink_mnode(i_mnode(self));
	PyMem_DEL(self);
}

static PyObject* ret_counters (PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return Py_BuildValue("iii", nb_mnodes_allocated,
		nb_mnodes_reserved, nb_mnodes_freed);
}

static int mathnode_print (mathnode_ob *self, FILE *fp, int flags)
{
	gr_string *grs;

	grs = mnode_stringify(self->m);
	grs = grs_append1(grs, 0);
	fputs(grs->s, fp);
	free(grs);
	return 0;
}

static PyObject* mathnode_repr (mathnode_ob *self)
{
	gr_string *grs;
	PyObject *result;

	grs = mnode_stringify(self->m);
	grs = grs_append1(grs, 0);
	result = Py_BuildValue("s", grs->s);
	free(grs);
	return result;
}

static int mathnode_cmp (mathnode_ob *U1, mathnode_ob *U2)
{
	/*
	 * WARNING: this assumes that mnode_differ() and mnode_lessthan()
	 * don't return -1. For incomparable mathnodes, the result will be
	 * bogus, and it's not possible to raise an exception here.
	 * For proper error reporting, you should use methods `ne0', `lt0',
	 * `differ' and `lessthan' instead of the comparison operator.
	 */
	if (!mnode_differ(U1->m,U2->m))
		return 0;
	if (mnode_lessthan(U1->m,U2->m))
		return -1;
	else return 1;
}

static PyObject* mathnode_array (PyObject *self, PyObject *args)
{
	int mtype, length, i;
	std_mnode *m;
	PyObject *arglist, *arg;
	
	if (!PyArg_ParseTuple(args, "iO!", &mtype, &PyTuple_Type, &arglist))
		return NULL;
	if (!MTYPE_STD(mtype)) {
		PyErr_SetString(SamlErrors[SE_NOT_ARRAY],
			"mnarray: not a standard type");
		return NULL;
	}
	length = PyTuple_Size(arglist);
	m = mstd_alloc(mtype, length);
	for (i = 0; i < length; i++) {
		arg = PyTuple_GetItem(arglist, i);
		if (!is_mathnode(arg)) {
			/* We're screwed. Free everything */
			while (--i > 0)
				unlink_mnode(m->x[i]);
			free(m);
			PyErr_SetString(SamlErrors[SE_WR_TYPE],
				"mnarray: expected a tuple of Mathnodes");
			return NULL;
		}
		m->x[i] = copy_mnode(i_mnode(arg));
	}
	return new_mobj((s_mnode*)m);
}

static PyObject* mathnode_parse (PyObject *self, PyObject *args)
{
	char *string;
	mref_t mr;
	s_mnode *r;
	
	if (!PyArg_ParseTuple(args, "s", &string))
		return NULL;
	saml_init_lexer_mem(string, strlen(string));
	/*
	 * saml_parse(result, model) permits aliasing, so we need only
	 * one mref, for both input and output.
	 */
	mr = mnode_to_mref(i_mnode(self));
	saml_parse(mr, mr);
	r = mref_to_mnode(mr);
	mref_free(mr);

	return new_mobj(r);
}

static PyObject* mathnode_cast (PyObject *self, PyObject *args)
{
	int newtype;
	s_mnode *m = i_mnode(self);

	if (!PyArg_ParseTuple(args, "i", &newtype))
		return NULL;
	m = mnode_cast(m, newtype);
	return new_mobj(m);
}

static PyObject* mathnode_notzero (PyObject *self, PyObject *args)
{
	int ret;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	ret = mnode_notzero(i_mnode(self));
	if (ret < 0) {
		PyErr_SetString(SamlErrors[SE_ONSUPP],
			"notzero: cannot compare Mathnode with zero");
		return NULL;
	}
	return Py_BuildValue("i", ret);
}

static PyObject* mathnode_isneg (PyObject *self, PyObject *args)
{
	int ret;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	ret = mnode_isneg(i_mnode(self));
	if (ret < 0) {
		PyErr_SetString(SamlErrors[SE_ONSUPP],
			"isneg: cannot compare Mathnode with zero");
		return NULL;
	}
	return Py_BuildValue("i", ret);
}

static PyObject* mathnode_power (PyObject *self, PyObject *args)
{
	s_mnode *m = i_mnode(self);
	int exponent;

	if (!PyArg_ParseTuple(args, "i", &exponent))
		return NULL;
	m = mnode_power(m, exponent);
	return new_mobj(m);
}

static PyObject* mathnode_info (PyObject *self, PyObject *args)
{
	s_mnode *m = i_mnode(self);
	int what;

	if (!PyArg_ParseTuple(args, "i", &what))
		return NULL;
	return Py_BuildValue("i", mnode_info(m,what));
}

static PyObject* mathnode_differ(PyObject *self, PyObject *args)
{
	int ret;
	PyObject *arg;

	if (!PyArg_ParseTuple(args, "O!", &Mathnode_Type, &arg))
		return NULL;
	ret = mnode_differ(i_mnode(self), i_mnode(arg));
	if (ret < 0) {
		PyErr_SetString(SamlErrors[SE_ONSUPP],
			"differ: cannot compare Mathnodes");
		return NULL;
	}
	return Py_BuildValue("i", ret);
}

static PyObject* mathnode_lessthan(PyObject *self, PyObject *args)
{
	int ret;
	PyObject *arg;

	if (!PyArg_ParseTuple(args, "O!", &Mathnode_Type, &arg))
		return NULL;
	ret = mnode_lessthan(i_mnode(self), i_mnode(arg));
	if (ret < 0) {
		PyErr_SetString(SamlErrors[SE_ONSUPP],
			"lessthan: cannot compare Mathnodes");
		return NULL;
	}
	return Py_BuildValue("i", ret);
}

static inline PyObject* mathnode_uop (PyObject *self, PyObject *args,
	s_mnode* (*op)(s_mnode*))
{
	s_mnode *m = i_mnode(self);

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	m = (*op)(m);
	return new_mobj(m);
}

static inline PyObject* mathnode_bop (PyObject *self, PyObject *args,
	s_mnode* (*op)(s_mnode*,s_mnode*))
{
	PyObject *arg;
	s_mnode *m;

	if (!PyArg_ParseTuple(args, "O!", &Mathnode_Type, &arg))
		return NULL;
	m = (*op)(i_mnode(self),i_mnode(arg));
	return new_mobj(m);
}

static PyObject* mathnode_promote (PyObject *self, PyObject *args)
{ return mathnode_bop(self, args, mnode_promote); }

static PyObject* mathnode_gcd (PyObject *self, PyObject *args)
{ return mathnode_bop(self, args, mnode_gcd); }

static PyObject* mathnode_diff (PyObject *self, PyObject *args)
{ return mathnode_bop(self, args, mnode_diff); }

static PyObject* mathnode_zero (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_zero); }

static PyObject* mathnode_one (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_one); }

static PyObject* mathnode_negate (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_negate); }

static PyObject* mathnode_invert (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_invert); }

static PyObject* mathnode_sqrt (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_sqrt); }

static PyObject* mathnode_det (PyObject *self, PyObject *args)
{ return mathnode_uop(self, args, mnode_det); }

static PyObject* mathnode_subs (PyObject *self, PyObject *args)
{
	s_mnode *m;
	PyObject *a2, *a3;

	if (!PyArg_ParseTuple(args, "O!O!", &Mathnode_Type, &a2,
	    &Mathnode_Type, &a3))
		return NULL;
	m = mnode_subs(i_mnode(self), i_mnode(a2), i_mnode(a3));
	return new_mobj(m);
}

static PyObject* mathnode_reslt (PyObject *self, PyObject *args)
{
	s_mnode *m;
	PyObject *a1, *a2, *a3;

	if (!PyArg_ParseTuple(args, "O!O!O!", &Mathnode_Type, &a1,
	    &Mathnode_Type, &a2, &Mathnode_Type, &a3))
		return NULL;
	m = mnode_elim(i_mnode(a1), i_mnode(a2), i_mnode(a3));
	return new_mobj(m);
}

static PyObject* mathnode_movelit (PyObject *self, PyObject *args)
{
	char *lit1, *lit2;
	s_mnode *m, *m1, *m2;
	
	if (!PyArg_ParseTuple(args, "ss", &lit1, &lit2))
		return NULL;
	m1 = mnode_build(ST_LITERAL, lit1);
	m2 = mnode_build(ST_LITERAL, lit2);
	m = mnode_move_lit(i_mnode(self), m1, m2);
	unlink_mnode(m1); unlink_mnode(m2);
	return new_mobj(m);
}

static PyObject* mathnode_slen (PyObject *self, PyObject *args)
{
	s_mnode *arr = i_mnode(self);
	int length;
	
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (!MTYPE_STD(arr->type)) {
		PyErr_SetString(SamlErrors[SE_NOT_ARRAY], "Not an array");
		return NULL;
	}
	length = ((std_mnode*)arr)->length;
	return Py_BuildValue("i", length);
}

static PyObject* mathnode_getitem (PyObject *self, PyObject *args)
{
	int index;
	s_mnode *arr = i_mnode(self);

	if (!PyArg_ParseTuple(args, "i", &index))
		return NULL;
	if (!MTYPE_STD(arr->type)) {
		PyErr_SetString(SamlErrors[SE_NOT_ARRAY], "Not an array");
		return NULL;
	}
	if ((unsigned)index >= ((smn_ptr)arr)->length) {
		PyErr_SetString(SamlErrors[SE_INDEX], "Index out of range");
		return NULL;
	}
	arr = ((smn_ptr)arr)->x[index];
	return new_mobj(copy_mnode(arr));
}

static struct PyMethodDef mathnode_methods[] = {
	{ "promote",	mathnode_promote, 1 },
	{ "cast",	mathnode_cast, 1 },
	{ "parse",	mathnode_parse, 1 },
	{ "gcd",	mathnode_gcd, 1 },
	{ "zero",	mathnode_zero, 1 },
	{ "negate",	mathnode_negate, 1 },
	{ "one",	mathnode_one, 1 },
	{ "invert",	mathnode_invert, 1 },
	{ "sqrt",	mathnode_sqrt, 1 },
	{ "det",	mathnode_det, 1 },
	{ "ne0",	mathnode_notzero, 1 },
	{ "lt0",	mathnode_isneg, 1 },
	{ "pow",	mathnode_power, 1 },
	{ "mvl",	mathnode_movelit, 1 },
	{ "info",	mathnode_info, 1 },
	{ "differ",	mathnode_differ, 1 },
	{ "lessthan",	mathnode_lessthan, 1 },
	{ "diff",	mathnode_diff, 1 },
	{ "subs",	mathnode_subs, 1 },
	{ "len",	mathnode_slen, 1 },
	{ "item",	mathnode_getitem, 1 },
	{ NULL, NULL }
};

static PyObject* mathnode_getattr (mathnode_ob *self, char *name)
{
	switch(name[0]) {
	case 'r':
	    if (!strcmp(name, "refs"))
		return Py_BuildValue("i", self->m->refs);
	    break;
	case 's':
	    if (!strcmp(name, "stype"))
		return Py_BuildValue("s", mtype_table[self->m->type]->name);
	    break;
	case 't':
	    if (!strcmp(name, "type"))
		return Py_BuildValue("i", self->m->type);
	    break;
	case '_':
	    if (!strcmp(name, "__members__"))
		return Py_BuildValue("[sss]", "refs", "type", "stype");
	    break;
	}
	return Py_FindMethod(mathnode_methods, (PyObject*)self, name);
}

/*
 * Arithmetic operations
 */

static inline PyObject* mnn_bop (PyObject *O1, PyObject *O2,
	s_mnode* (*op)(s_mnode*,s_mnode*))
{
	s_mnode *result = (*op)(i_mnode(O1),i_mnode(O2));
	return new_mobj(result);
}

static PyObject* mnn_add (PyObject *U, PyObject *V)
{ return mnn_bop(U,V,mnode_add); }

static PyObject* mnn_substract (PyObject *U, PyObject *V)
{ return mnn_bop(U,V,mnode_sub); }

static PyObject* mnn_multiply (PyObject *U, PyObject *V)
{ return mnn_bop(U,V,mnode_mul); }

static PyObject* mnn_divide (PyObject *U, PyObject *V)
{ return mnn_bop(U,V,mnode_div); }

static PyObject* mnn_remainder (PyObject *U, PyObject *V)
{ return mnn_bop(U,V,mnode_mod); }

static PyObject* mnn_divmod (PyObject *U, PyObject *V)
{
	s_mnode *mu, *mv, *quot, *t1, *rem;

	mu = i_mnode(U);
	mv = i_mnode(V);
	quot = mnode_div(mu, mv);
	t1 = mnode_mul(quot, mv);
	rem = mnode_sub(mu, t1);
	unlink_mnode(t1);
	return Py_BuildValue("OO", new_mobj(quot), new_mobj(rem));
}

static PyObject* mnn_negative (PyObject *U)
{
	s_mnode *result = mnode_negate(i_mnode(U));
	return new_mobj(result);
}

static PyObject* mnn_positive (PyObject *U)
{
	Py_INCREF(U);
	return U;
}

static PyObject* mnn_absolute (PyObject *U)
{
	if (mnode_isneg(i_mnode(U)))
		return mnn_negative(U);
	else return mnn_positive(U);
}

static int mnn_nonzero (PyObject *U)
{
	return mnode_notzero(i_mnode(U));
}

static int mnn_coerce (PyObject **p_self, PyObject **p_arg)
{
	s_mnode *m, *m1, *m2;
	int number;
	char nb[30];
	PyObject *arg2;

	if (is_mathnode(*p_self) && (*p_arg)->ob_type == &PyInt_Type) {
		m = i_mnode(*p_self);
		number = ((PyIntObject*)(*p_arg))->ob_ival;
		/* Optimize the common cases */
		if (number == 0)
			m2 = mnode_zero(m);
		else if (number == 1)
			m2 = mnode_one(m);
		else {
			sprintf(nb, "%d", number);
			m1 = mnode_build(ST_INTEGER, nb);
			m2 = mnode_promote(m1,m);
			unlink_mnode(m1);
		}
		if (saml_errno(m2)) {
			/* oops, something went wrong */
			unlink_mnode(m2);
			return 1;
		}
		arg2 = new_mobj(m2);
		Py_INCREF(*p_self);
		*p_arg = arg2;
		return 0;
	}
	/* failure */
	return 1;
}

static PyNumberMethods mathnode_as_number = {
	mnn_add,
	mnn_substract,
	mnn_multiply,
	mnn_divide,
	mnn_remainder,
	mnn_divmod,
	0,	/* power */
	mnn_negative,
	mnn_positive,
	mnn_absolute,
	mnn_nonzero,
	0,	/* invert */
	0,	/* lshift */
	0,	/* rshift */
	0,	/* and */
	0,	/* xor */
	0,	/* or */
	mnn_coerce,
	0,	/* int */
	0,	/* long */
	0,	/* float */
	0,	/* oct */
	0,	/* hex */
};

/*
 * Class registration
 */

static PyTypeObject Mathnode_Type = {
	/* header */
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"Mathnode",
	sizeof(mathnode_ob),
	0,

	/* standard methods */
	(destructor)	mathnode_free,
	(printfunc)	mathnode_print,
	(getattrfunc)	mathnode_getattr,
	(setattrfunc)	0,
	(cmpfunc)	mathnode_cmp,
	(reprfunc)	mathnode_repr,

	/* type categories */
	&mathnode_as_number,
	0,
	0,

	/* more methods */
	(hashfunc)	0,
	(ternaryfunc)	0,
	(reprfunc)	0,
};

/*
 * List of functions for this module
 */

static struct PyMethodDef saml1_methods[] = {
	{ "Mathnode",		mathnode_create,1 },
	{ "MnAllocStats",	ret_counters,	1 },
	{ "MnArray",		mathnode_array,	1 },
	{ "resultant",		mathnode_reslt,	1 },
	{ NULL, NULL }
};

/*
 * List of symbolic constants for this module
 */

typedef struct { const char *name; int c; } scst_entry;

static scst_entry saml1_csts[] = {
	{ "ST_VOID",	ST_VOID },
	{ "ST_LITERAL",	ST_LITERAL },
	{ "ST_MINT",	ST_MINT },
	{ "ST_INTEGER",	ST_INTEGER },
	{ "ST_CYCLIC",	ST_CYCLIC },
	{ "ST_FLOAT",	ST_FLOAT },
	{ "ST_RATIONAL", ST_RATIONAL },
	{ "ST_COMPLEX",	ST_COMPLEX },
	{ "ST_ALGEXT",	ST_ALGEXT },
	{ "ST_MONO",	ST_MONO },
	{ "ST_POLY",	ST_POLY },
	{ "ST_APOLY",	ST_APOLY },
	{ "ST_UPOLY",	ST_UPOLY },
	{ "ST_LINE",	ST_LINE },
	{ "ST_MATRIX",	ST_MATRIX },
	{ "ST_TENSOR",	ST_TENSOR },
	{ 0, 0 }
};

static const char *saml1_exlist[SE_MAX_ERROR+1] = {
	"SamlUnknownError",		/*  0 */
	"SamlError",			/*  1 */
	"SamlBadTypeError",		/*  2 */
	"SamlBadSizeError",		/*  3 */
	"SamlTypeConflictError",	/*  4 */
	"SamlSizeConflictError",	/*  5 */
	"SamlOpNotSuppError",		/*  6 */
	"SamlTableFullError",		/*  7 */
	"SamlZeroDivideError",		/*  8 */
	"SamlDomainError",		/*  9 */
	"SamlIndexError",		/* 10 */
	"SamlNoInverseError",		/* 11 */
	"SamlNoSuchTypeError",		/* 12 */
	"SamlNotArrayError",		/* 13 */
	"SamlTypeInUseError",		/* 14 */
	"SamlStringError",		/* 15 */
	"SamlConversionError",		/* 16 */
	"SamlNotImplementedError",	/* 17 */
	"SamlUninitializedError",	/* 18 */
	"SamlHeteroArgsError",		/* 19 */
	"SamlNotScalarError",		/* 20 */
	"SamlOverflowError",		/* 21 */
	"SamlTooSparseError",		/* 22 */
	"SamlModulusError",		/* 23 */
};

void initsaml1 (void)
{
	PyObject *m, *d, *cst;
	scst_entry *se;
	const char *errname;
	int i;
	
	saml_init();
	saml_error_handler = saml_silent_ehandler;

	m = Py_InitModule("saml1", saml1_methods);
	d = PyModule_GetDict(m);

	/* Insert the symbolic constants */
	for (se = saml1_csts; se->name; se++) {
		cst = PyInt_FromLong(se->c);
		PyDict_SetItemString(d, se->name, cst);
		Py_DECREF(cst);
	}

	/* Declare exception objects */
	for (i = 0; i <= SE_MAX_ERROR; i++) {
		errname = saml1_exlist[i];
		SamlErrors[i] = cst = Py_BuildValue("s", errname);
		PyDict_SetItemString(d, errname, cst);
	}
}
