/*
   Egon Animator
   Copyright (C) 1997-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
   Module name:    window.c

   This module handles windows: creating, destroying, printing on windows.
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/List.h>
#include <X11/xpm.h>

#include "types.h"
#include "../xcommon/Animator.h"
#include "egon.h"

#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../common/cmalloc.h"
#include "../xcommon/xcommon.h"
#include "../xcommon/Egon.h"
#include "../xcommon/plugin.h"

window *w_list;

int pr_scr_flag, pr_line_flag;

Display *display;

Window root;

Atom wm_delete_window;	/* Atom sent to destroy a window */
extern Atom target_atom;	/* used for selection */

void activate_window(window *w)
{
	char b[256];

	w_list = w;
	if (w) {
		strcpy(b, "Egon Animator: ");
		strncat(b, w->buf->name, 200);
		XtVaSetValues(topLevel, XtNtitle, b, NULL);
	}
}

static void object_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;
	char cmd[256];

	sprintf(cmd, "(select-object \"%s\")", string);
	execute(cmd);
	draw_buffer(display, w_list);
}

static void tick_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;
	char cmd[256];

	sprintf(cmd, "(select-tick %s)", string);
	execute(cmd);
	draw_buffer(display, w_list);
}

void draw_input(Display * display, char *text)
{
	char b[256];
	if (!w_list->object)
		sprintf(b, "No object selected");
	else
		sprintf(b, "[%s at %ld] [size = %dx%d] [duration=%ld delta=%ld]",
			w_list->object->name, w_list->script->time,
			w_list->buf->width, w_list->buf->height,
			w_list->buf->duration, w_list->buf->delta);
	label_set(label1, b);
	label_set(label3, "Egon");
}

void draw_status(Display * display, char *text)
{
	label_set(label2, text);
	XFlush(display);
}

/*
   void llpr(char *p)
   Prints the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed.
*/
void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(display, p);
}

#define FREE_LIST(l,n)\
	if (l) {\
		for (i = 0; i < (n); i++) cfree((l)[i]);\
		cfree(l);\
	}

#undef FREE_LIST

#define FREE_LIST(l,n)

void draw_buffer(Display *display, window *w)
{
	buffer *b = w->buf;
	ani_object *o;
	ani_script *s;
	char p[256];
	static String fallback[1] = {"<none>"};
	static int on = 0, sn = 0, pn = 0;
	static char **ol = NULL, **sl = NULL, **pl = NULL;

	FREE_LIST(ol, on)
	FREE_LIST(sl, sn)
	FREE_LIST(pl, pn)

	/* select the first object if none is selected */
	if (!w->object) {
		w->object = w->buf->cast;
		if (w->object) w->script = w->object->script;
	}

	on = 0;
	for (o = b->cast; o; o = o->next) on++;

	if (!on) {
		XawListChange(w->objl, fallback, 1, 0, True);
		XawListChange(w->tickl, fallback, 1, 0, True);
		XawListChange(w->propl, fallback, 1, 0, True);
		sn = pn = 0;
		goto The_end;
	}

	ol = (char **)cmalloc(on*sizeof(char *));
	on = 0;
	for (o = b->cast; o; o = o->next) {
		ol[on] = cstrdup(o->name);
		on++;
	}
	XawListChange(w->objl, ol, on, 0, True);

	o = w_list->object;
	sn = 0;
	for (s = o->script; s; s = s->next) sn++;
	sl = (char **)cmalloc(sn*sizeof(char *));
	sn = 0;
	for (s = o->script; s; s = s->next) {
		sprintf(p, "%ld", s->time);
		sl[sn] = cstrdup(p);
		sn++;
	}
	XawListChange(w->tickl, sl, sn, 0, True);

	s = w_list->script;
	pn = 0;
	pl = (char **)cmalloc(9*sizeof(char *));
	sprintf(p, "Position = (%d, %d)", s->x, s->y);
	pl[pn++] = cstrdup(p);
	sprintf(p, "Size = %dx%d", s->width, s->height);
	pl[pn++] = cstrdup(p);
	sprintf(p, "Visible = %s", s->visible?"True":"False");
	pl[pn++] = cstrdup(p);
	sprintf(p, "Color = %d", o->color);
	pl[pn++] = cstrdup(p);
	sprintf(p, "Font = %d", o->font);
	pl[pn++] = cstrdup(p);
	sprintf(p, "String = %s", o->string?o->string:"EMPTY");
	pl[pn++] = cstrdup(p);
	XawListChange(w->propl,
		pl, pn, 0, True);

The_end:
/*	XtUnmapWidget(topLevel);
*/
	XtVaSetValues(topLevel,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);
/*	XResizeWindow(XtDisplay(topLevel), XtWindow(topLevel),
		b->width, b->height);
*/
	XtVaSetValues(stage,
		XtNwidth, b->width,
		XtNheight, b->height,
		XtNanimatorCast, b->cast,
		XtNanimatorDelta, b->delta,
		XtNanimatorDuration, b->duration,
		XtNanimatorBgPixmap, b->bg,
		XtNanimatorNow, b->now,
		XtNanimatorMode, b->state,
		(char *)0);
	XtVaSetValues(topLevel,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);
/*	XtMapWidget(topLevel);
*/
	draw_input(display, NULL);
}

window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->viewport == wdg ||
			w->objv == wdg || w->objl == wdg ||
			w->tickv == wdg || w->tickl == wdg ||
			w->propv == wdg || w->propl == wdg)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

void free_window(window *w)
{
	window *pw;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	if (w->viewport != None)
		XtDestroyWidget(w->viewport);
	cfree(w);
}

/* When running as player, return a window structure without any widgets */
/* Slightly modified for Egon: only one window at a time */
window *new_window(buffer *b, window *prev)
{
	Dimension totalwidth, formheight;
	window *w;

	if (w_list) w = w_list;
	else w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;

	w->buf = b;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	w->object = NULL;	/* none selected yet */
	w->script = NULL;

	/* Figure out how big the new form should be. */
	/* The total width must be the width of the gridpane. */
	XtVaGetValues(gridpane,
		XtNwidth, &totalwidth, NULL);

	/* The form height is whatever we get, but if it is too small
	we cannot create the new window. */
	formheight = 100;
	w->viewport = XtVaCreateManagedWidget("viewport",
		egonWidgetClass, gridpane,
		XtNwidth, totalwidth,
		(char *)0);
	XtVaGetValues(w->viewport,
		XtNheight, &formheight,
		XtNwidth, &totalwidth,
		(char *)0);

	w->objv = XtVaCreateManagedWidget("objv",
		viewportWidgetClass, w->viewport,
		XtNwidth, 100,
		XtNheight, formheight/2,
		(char *)0);
	w->objl = XtVaCreateManagedWidget("objl",
		listWidgetClass, w->objv, NULL);
	XtAddCallback(w->objl, XtNcallback, object_select, NULL);
	w->tickv = XtVaCreateManagedWidget("tickv",
		viewportWidgetClass, w->viewport,
		XtNwidth, 100,
		XtNheight, formheight/2,
		(char *)0);
	w->tickl = XtVaCreateManagedWidget("tickl",
		listWidgetClass, w->tickv, NULL);
	XtAddCallback(w->tickl, XtNcallback, tick_select, NULL);
	w->propv = XtVaCreateManagedWidget("propv",
		viewportWidgetClass, w->viewport,
		XtNwidth, totalwidth-2*100,
		XtNheight, formheight/2,
		(char *)0);
	w->propl = XtVaCreateManagedWidget("propl",
		listWidgetClass, w->propv, (char *)0);

	XtVaSetValues(stage,
		XtNanimatorCast, b->cast,
		XtNanimatorNow, 0,
		XtNanimatorDelta, b->delta,
		XtNanimatorDuration, b->duration,
		XtNanimatorMode, ANI_STOP,
		XtNanimatorBgPixmap, b->bg,
		XtNwidth, b->width,
		XtNheight, b->height,
		(char *)0);
#if 0
	XtOverrideTranslations(stage,
		XtParseTranslationTable(
			"<Key>Q:        ani-done()"));
#endif

	return w;
}

int ani_ctl(int mode, unsigned long now)
{
	XtVaSetValues(stage,
		XtNanimatorMode, mode, (char *)0);
        return ANI_OK;
}

int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);

	if (w2 == NULL) return FALSE;
	return TRUE;
}

/* this does probably not belong here in window.c,
   because it doesn't depend on X */
static void save_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (savematrix(p, w_list->buf, NULL)) {
                        printf("501 Can't save %s\n", p);
                } else {
                        printf("250 Saved %s\n", p);
                }
        }
}

static void load_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (loadmatrix(p, w_list->buf, NULL)) {
                        printf("501 Can't load %s\n", p);
                } else {
                        printf("250 Loaded %s\n", p);
                }
        }
}

static void exec_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 Command missing\n");
        else {
                execute(p);
                printf("250 OK\n");
        }
}

static void help_plugin(char *p)
{
        printf("214 SAVE LOAD EXEC HELP NOOP QUIT PRNT\n");
}

static void noop_plugin(char *p)
{
        printf("250 OK\n");
}

static void win_plugin(char *p)
{
        printf("250 %lx\n", (unsigned long)XtWindow(topLevel));
}

static void quit_plugin(char *p)
{
        printf("221 Over and out\n");
        execute("(quit-egon)");
}

static void prnt_plugin(char *p)
{
        printf("502 Can't print yet\n");
}

static struct {
        char *verb;
        void (*cb)(char *);
} plugin_cmds[] = {
        {"SAVE", save_plugin},
        {"LOAD", load_plugin},
        {"EXEC", exec_plugin},
        {"HELP", help_plugin},
        {"NOOP", noop_plugin},
        {"WIN", win_plugin},
        {"QUIT", quit_plugin},
        {"PRNT", prnt_plugin},
        {NULL, NULL}
};

static void read_plugin_cmd(XtPointer client_data, int *fid, XtInputId *id)
{
        char b[1024], *p;
        int i, n;

        if ((n = read(*fid, b, 1020)) == -1) return;

        b[n] = '\0';
        if ((p = strchr(b, '\n')) == NULL) {
                printf("501 Incomplete command\n");
                fflush(stdout);
                return;
        }

        *p = '\0';
        for (i = 0; plugin_cmds[i].verb; i++) {
                if (!strncmp(b, plugin_cmds[i].verb,
                                strlen(plugin_cmds[i].verb)))
                        break;
        }
        if (plugin_cmds[i].verb)
                (*plugin_cmds[i].cb)(b+strlen(plugin_cmds[i].verb));
        else
                printf("500 What are you talking about\n");
        fflush(stdout);
}

/* handle spontaneous exit of plugins */
static void handle_plugin_exit(int ph)
{
        buffer *b = b_list;

        do {
                int n = buffer_plugin2index(b, ph);
                if (n != -1) {
                        cfree(b->plugin[n].name);
                        b->nplugin--;
                        for (; n < b->nplugin; n++)
                                b->plugin[n] = b->plugin[n+1];
                        b->change = pr_scr_flag = TRUE;
                }
                b = b->next;
        } while (b != b_list);
}

/*
   void init_windows(buffer *b)
   Sets up the whole initial window structure and initializes scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/
void init_windows(buffer * b, int argc, char **argv)
{
	XtRealizeWidget(topLevel);

        plugin_init(topLevel, handle_plugin_exit);

	display = XtDisplay(topLevel);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-egon)"));

	XSetWMProtocols(display, XtWindow(topLevel), &wm_delete_window, 1);

	root = DefaultRootWindow(display);

	init_color(display);

	draw_status(display, "");

	/* Set up selection */
	target_atom = XInternAtom(display, "EGON_BLOCK", False);
}

/*
   void exit_windows()
   Cleans up after Calc before exit.  All buffers and windows are freed.
*/
void exit_windows()
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
}

/*
   static void pr_scr()
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
*/
static void pr_scr()
{
	window *w;
	int i;

	draw_status(display, "");
	w = w_list;
	do {
		draw_buffer(display, w);
		for (i = 0; i < w->buf->nplugin; i++) {
                        if (!w->buf->plugin[i].displayed) {
                                plugin_show(w->buf->plugin[i].ph,
                                                stage);
                                w->buf->plugin[i].displayed = 1;
                        }
                }
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = pr_line_flag = FALSE;
}	/* pr_scr */

char *family2name(int fmt)
{
	switch (fmt & FONT_MASK) {
	case COURIER: return "Courier";
	case HELVETICA: return "Helvetica";
	case NEW_CENTURY: return "New Century Schoolbook";
	case TIMES: return "Times";
	default: return "ERROR";
	}
}

char *size2name(int fmt)
{
	switch (fmt & SIZE_MASK) {
	case SIZE_8:	return "8";
	case SIZE_10:	return "10";
	case SIZE_12:	return "12";
	case SIZE_14:	return "14";
	case SIZE_18:	return "18";
	case SIZE_24:	return "24";
	case SIZE_20:	return "20";
	case SIZE_30:	return "30";
	default:	return "XX";
	}
}

static char *types[] = {
	"None", "Line", "Rectangle", "Arc", "Ellipse", "Pixmap",
	"String", "Point", "Filled Rectangle", "Filled Arc",
	"Filled Ellipse"};

char *type2name(int fmt)
{
	if (!w_list->buf || !w_list->object || !w_list->script)
		return "None";
	return types[w_list->object->type];
}

char *color2name(int fmt)
{
	switch (fmt & COLOR_MASK) {
	case COLOR_0:		return "Black";
	case COLOR_1:		return "Red";
	case COLOR_2:		return "Green";
	case COLOR_3:		return "Blue";
	case COLOR_4:		return "Yellow";
	case COLOR_5:		return "Magenta";
	case COLOR_6:		return "Cyan";
	case COLOR_7:		return "White";
	default:		return "ERROR";
	}
}

void show_format()
{
	int fmt = ret_format(w_list->buf,
				w_list->object,
				w_list->script);
	int sty = STY_DEFAULT;
	int bold = fmt & BOLD;
	int italic = fmt & ITALIC;
	int hadj = HADJ_LEFT;
	int hadj_left = (hadj == HADJ_LEFT);
	int hadj_center = (hadj == HADJ_CENTER);
	int hadj_right = (hadj == HADJ_RIGHT);

	/* menus */
	label_set(btnFont, family2name(fmt));
	label_set(btnSize, size2name(fmt));
	label_set(btnStyle, type2name(sty));
	label_set(btnColor, color2name(fmt));

	/* toggle buttons */
	state_set(cmdBold, (bold?1:0));
	state_set(cmdItalic, (italic?1:0));
	state_set(cmdHLeft, (hadj_left?1:0));
	state_set(cmdHCenter, (hadj_center?1:0));
	state_set(cmdHRight, (hadj_right?1:0));
}

int cursor_visible = FALSE;

/*
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/
void show_cur(window *w)
{
	if (w) {
		if (pr_scr_flag) pr_scr();
		show_format();
	}
}	/* show_cur */

void hide_cur(window *w)
{
}

void mainloop(void)
{
        if (app_data.plugin) {
                /* control plugin from stdin */
                XtAppAddInput(XtWidgetToApplicationContext(topLevel),
                        fileno(stdin), (XtPointer)XtInputReadMask,
                        read_plugin_cmd, NULL);
                printf("220 %s\n", VERSION);
                fflush(stdout);
        }

        XtAppMainLoop(XtWidgetToApplicationContext(topLevel));

        exit(0);
}

