/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *      cmds.c
 *
 *      This rather bulky module contains all the functions that implement
 *      commands.  It also handles initialization of the interface to those
 *      functions.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>

#include "../common/cmalloc.h"
#include "pw.h"
#include "../siod/siod.h"
#include "../common/fonts.h"

extern int input_warp_pointer;	/* from input.c */

typedef struct {
	char *name;
	 LISP(*function) ();
} s_fn_table;

extern s_fn_table fn_table[];

/* moving around */

static char *testlist[] = {	"testitem0",
	"testitem1", "testitem2", "testitem3", "testitem4", "testitem5",
	"testitem6", "testitem7", "testitem8", "testitem9"
};

static LISP listsel_test(void)
{
	int i = select_from_list("Choose an item", testlist, 10);
	if (i < 0)
		printf("User clicked Cancel\n");
	else
		printf("User chose %d\n", i);
	return NIL;
}

static LISP what_cursor_position(void)
{
	char b[256];

	sprintf(b, "[%d,%d]", w_list->point_pos.row, w_list->point_pos.col);
	llpr(b);
	return NIL;
}

static LISP insert_line(void)
{
	downshift_text(w_list->buf, get_point(w_list).row);
	w_list->buf->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP remove_line(void)
{
	upshift_text(w_list->buf, get_point(w_list).row);
	w_list->buf->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP spawn(LISP command)
{
	char *argv[20], *p, cmd[1024];
	int argc = 0;

	strcpy(cmd, get_c_string(command));
	for (p = strtok(cmd, " \t\r\n");
		p && argc < 20;
		p = strtok(NULL, " \t\r\n")) {
		argv[argc++] = p;
	}
	argv[argc] = NULL;
	if (!fork()) {
		/* this is the child */
		execvp(argv[0], argv);
		exit(0);
	}
	return NIL;
}

/* Windows and buffers */

/*X
   static void load_buffer(void)

   Load a buffer from file.
   X */
static LISP load_buffer(void)
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	buffer *b;
	static int need_init = 1;

	if (need_init) {
		getcwd(path, 1024);
		name[0] = '\0';
		need_init = 0;
	}
	strcpy(name, "");
	fn[0] = '\0';
	if (select_file(path, name, patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);
		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

		if (loadmatrix(fn, b, fmt))
			llpr("New file");

		b->change = FALSE;
		w_list->buf = b;
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

/* Load using an external program

   1. Ask for the program to use
   2. Run the program and save output to file in /tmp
   3. Load the file using NULL format (i.e. ask for type)
*/
static LISP load_external(void)
{
	static int loaders = 0;
	static char *loadname[20], *loadprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	buffer *b;
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/pw/external.load", siaghome);
		if ((fp = fopen(fnl, "r")) == NULL) {
			llpr("Can't open header file");
			return NIL;
		}
		while (!feof(fp) && loaders < 20) {
			fgets(b, 250, fp);
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				loadname[loaders] = cstrdup(p);
				loadprog[loaders] = cstrdup(q);
				loaders++;
			}
		}
		fclose(fp);
		need_init = 0;
	}
	program[0] = param[0] = '\0';
	i = select_from_list("External Program:",
			loadname, loaders);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		sprintf(fn, "/tmp/pw%ld", (long)getpid());
		sprintf(cmd, "%s %s > %s", loadprog[i], param, fn);
		if (system(cmd)) {
			llpr("External program failed");
			return NIL;
		}

		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

		if (loadmatrix(fn, b, guess_file_format(fn)))
			llpr("New file");

		b->change = FALSE;
		w_list->buf = b;
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

static LISP
delete_window(void)
{
	if (!remove_window(w_list))
		llpr("Attempt to delete sole ordinary window");
	else
		pr_scr_flag = TRUE;
	activate_window(w_list);
	return NIL;
}

static LISP
delete_other_windows(void)
{
	while (remove_window(w_list->next));
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP
split_window_vertically(void)
{
	if (!split_window(w_list))
		llpr("This window is too small to split");
	else
		pr_scr_flag = TRUE;
	return NIL;
}

static LISP
other_window(void)
{
	activate_window(w_list->next);
	pr_scr_flag = TRUE;
	return NIL;
}

/*X
   static void save_buffer()

   Save the buffer in the currently active window to file.
   X */
static LISP save_buffer(void)
{
	llpr("Saving");

	if (savematrix(w_list->buf->path, w_list->buf, NULL))
		error_box("Couldn't save");
	else {
		w_list->buf->change = FALSE;
		pr_scr_flag = TRUE;
	}
	return NIL;
}

static LISP lsavematrix(LISP path, LISP bname, LISP format)
{
	char *p, *fmt;
	buffer *b;

	if (NULLP(bname)) b = w_list->buf;
	else b = find_buffer_by_name(get_c_string(bname));
	if (!b) {
		llpr("No such buffer");
		return NIL;
	}

	p = get_c_string(path);
	if (NULLP(format)) fmt = guess_file_format(p);
	else fmt = get_c_string(format);

	if (savematrix(p, b, fmt)) {
		llpr("File saved");
		return NIL;
	} else {
		return a_true_value();
	}
}

/*X
   static void save_buffer_as()

   Save the buffer in the currently active window to a named file.
   X */
static LISP save_buffer_as(void)
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	char *p;
	static int need_init = 1;

	if (need_init) {
		getcwd(path, 1024);
		name[0] = '\0';
		need_init = 0;
	}
	p = strrchr(w_list->buf->path, '/');
	if (p) strcpy(name, p+1);
	else strcpy(name, w_list->buf->path);
	fn[0] = '\0';
	if (select_file(path, name, patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);
		llpr("Saving");
		if (savematrix(fn, w_list->buf, fmt))
			error_box("Couldn't save");
		else {
			w_list->buf->change = FALSE;
			strcpy(w_list->buf->path, fn);
		}
		pr_scr_flag = TRUE;
	}
	return NIL;
}

/* Save using an external program

   1. Ask for the program to use
   2. Save to a file in /tmp using NULL format (i.e. ask for type)
   3. Run the program and read the file as input
*/
static LISP save_external(void)
{
	static int savers = 0;
	static char *savename[20], *saveprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/pw/external.save", siaghome);
		if ((fp = fopen(fnl, "r")) == NULL) {
			llpr("Can't open saver file");
			return NIL;
		}
		while (!feof(fp) && savers < 20) {
			fgets(b, 250, fp);
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				savename[savers] = cstrdup(p);
				saveprog[savers] = cstrdup(q);
				savers++;
			}
		}
		fclose(fp);
		need_init = 0;
	}

	program[0] = param[0] = '\0';
	i = select_from_list("External Program:", savename, savers);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		llpr("Saving");

		sprintf(fn, "/tmp/pw%ld", (long)getpid());
		if (savematrix(fn, w_list->buf, NULL)) {
			error_box("Couldn't save");
			return NIL;
		}
		sprintf(cmd, "%s %s < %s", program, param, fn);
		if (system(cmd)) {
			error_box("External program failed");
			return NIL;
		}
	}
	return NIL;
}

#if USE_COMPLETION
/*X
   static void complete_name(char *name)
   This function takes a partially completed buffer name
   and returns the first buffer name that matches it.
   X */
static int complete_name(name)
char *name;
{
	buffer *b;
	int len;

	b = w_list->buf;	/* start with the next buffer */
	do {
		b = b->next;
		if ((len = strlen(name)) == 0 
				|| !strncmp(b->name, name, len)) {
			strcpy(name, b->name);
			return FALSE;
		}
	} while (b != w_list->buf);
	return FALSE;
}
#endif

static LISP
switch_to_buffer(void)
{
	buffer *b;
	char *blist[100];
	int nblist = 0, n;

	b = w_list->buf;
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != w_list->buf);
	if ((n = select_from_list("Change Buffer:", blist, nblist)) >= 0)
		w_list->buf = find_buffer_by_name(blist[n]);
	activate_window(w_list);
	return NIL;
}

static LISP
kill_buffer(void)
{
	buffer *b, *next_b;
	window *w;
	char *blist[100];
	int nblist = 0, n;

	b = w_list->buf;
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != w_list->buf);
	if ((n = select_from_list("Kill Buffer:", blist, nblist)) >= 0) {
		if ((b = find_buffer_by_name(blist[n])) != NULL) {
			if (b != b->next) {
				next_b = free_buffer(b);
				w = w_list;
				do {
					if (w->buf == b)
						w->buf = next_b;
					w = w->next;
				} while (w != NULL && w != w_list);
				pr_scr_flag = TRUE;
			}
			else llpr("Couldn't kill last buffer");
		}
	}
	activate_window(w_list);
	return NIL;
}

extern char *psformat; /* in fileio_ps.c */

static LISP lpsformat(void)
{
	return strcons(strlen(psformat), psformat);
}

static LISP execute_interpreter_command(LISP intpr)
{
	char prompt[80];
	char b[256];
	char *intname = get_c_string(intpr);
	int intp = name2interpreter(intname);
	if (intp < 0) return NIL;
	sprintf(prompt, "%s command: ", intname);
	b[0] = '\0';
	if (ask_for_str(prompt, b))
		exec_expr(intp, b);
	return NIL;
}

static char *quit_buttons[] = {"Yes", "No", "Cancel"};

static LISP lalertbox(LISP prompt, LISP buttons)
{
	char *btext[10], *p = get_c_string(prompt);
	int bno = 0, n;

	while (bno < 10 && NNULLP(buttons)) {
		char *c = get_c_string(car(buttons));
		if (c) btext[bno] = cstrdup(c);
		else btext[bno] = cstrdup("button");
		buttons = cdr(buttons);
		bno++;
	}
	if (p == NULL) p = "prompt";
	n = alert_box(p, btext, bno);
	while (bno) cfree(btext[--bno]);
	return flocons(n);
}

static LISP quit_pw(void)
{
	char prompt[256], cmd[1024];
	buffer *b = b_list;
	do {
		if (b->change) {
			sprintf(prompt, "Save %s?", b->name);
			switch (alert_box(prompt, quit_buttons, 3)) {
			case 0:
				savematrix(b->path, b, NULL);
				break;
			case 2:
				return NIL;
			default:
				break;
			}
		}
		b = b->next;
	} while (b != b_list);
	/* remove all windows, buffers and plugins */
	exit_windows();
	/* remove the temp directory for this process */
	sprintf(cmd, "rm -rf %s/%ld", siag_basedir, (long)getpid());
	system(cmd);
	exit(0);
	return NIL;
}

static LISP go_to(void)
{
	char b[256];
	int tr = 0, tc = 0;

	b[0] = '\0';
	if (ask_for_str("Go to: ", b))
		sscanf(b, "%d %d", &tr, &tc);

	if ((tr > 0) && (tr <= max_lines)) {
		w_list->point_pos.row = tr;
		w_list->point_pos.col = col_last_used(w_list->buf, tr);
	}
	if ((tc > 0) && (tc <= w_list->point_pos.col))
		w_list->point_pos.col = tc;
	return NIL;
}

static LISP set_line_height(void)
{
	char b[256];
	int height = 0;

	sprintf(b, "%d", line_height(w_list->buf, w_list->point_pos.row));
	if (ask_for_str("Height: ", b))
		height = atoi(b);

	if (height > 5 && height < 200) {
		alloc_line(w_list->buf, w_list->point_pos.row);
		w_list->buf->text[w_list->point_pos.row].height = height;
		pr_scr_flag = 1;
	}
	return NIL;
}

static LISP set_segment_format(void)
{
	int format = 0, mask = 0;

	format = ret_format(w_list->buf, w_list->point_pos.row,
			    w_list->point_pos.col);
	if (font_input(&format, &mask)) {
		ins_format(w_list->buf, w_list->point_pos.row,
			   0, w_list->point_pos.col, format);
		pr_scr_flag = 1;
	}
	return NIL;
}

static long format_copy = HELVETICA | SIZE_10;
static long height_copy = 20;

static LISP copy_current_format(void)
{
	format_copy = w_list->current_fmt;
	height_copy = line_height(w_list->buf, w_list->point_pos.row);
	return NIL;
}

static LISP use_copied_format(void)
{
	w_list->current_fmt = format_copy;
	return NIL;
}

static LISP define_style(void)
{
	char b[256];
	long r = w_list->point_pos.row;
	int sty = ret_style(w_list->buf, r);
	styles[sty].format = w_list->current_fmt;
	styles[sty].height = line_height(w_list->buf, r);
	sprintf(b, "Style %s changed", styles[sty].name);
	return NIL;
}

static LISP cleanup_style(void)
{
	long r = w_list->point_pos.row;
	buffer *b = w_list->buf;
	int sty = ret_style(b, r);
	long fmt = styles[sty].format;
	int height = styles[sty].height;

	for (r = 1; r <= line_last_used(b); r++) {
		if (ret_style(b, r) == sty) {
			ins_format(b, r, 0, rc_strlen(b->text[r].p), fmt);
			b->text[r].height = height;
			b->change = TRUE;
			pr_scr_flag = TRUE;
		}
	}
	llpr("set every char of the current style to the default format");;
	return NIL;
}

/*
	Spelling checker

	This checker does the following for each line in the buffer:

	For each word in the line, feed it to ispell like this:
	^word

	The '^' tells ispell to spell check it and nothing else.

	Crucial to the proper operation of this interface is
	that PW and ispell agree on what a word is. For example,
	"shouldn't" shouldn't be an error, although it contains a
	non-alphabetic character. Also, the Swedish word "rksmrgs"
	(shrimp sandwich) should be treated as a single word.

	So here is my attempt to feed ispell words it will recognize as words.
	Note that the list of characters that ispell should understand is
	incomplete.

	0. Start ispell in a subprocess with the -a -w_-' options.

	1. Find the longest stretch of characters consisting of characters
	   entirely in the alphabetic set, or one of the characters in
	   this string:
	   _-

	2. Trim away leading and trailing characters in this string: _-'

	3. Feed the word as "^word\n" to ispell and read the response.

	4. If ispell prints a line beginning with '*' or '+', the word
	   is correct, otherwise a dialog with available options is
	   displayed.

	5. If the user decides to replace the word, this is done by
	   deleting the original, inserting the replacement and
	   forwarding point to directly after the newly inserted word.
	   Otherwise point is advanced to directly after the original
	   word.

	6. The checking sequence is then continued with step 1.

	Ispell doesn't have a mode in which every line read on stdin
	is treated as a single word, regardless of character set.
	I consider this to be a bug in ispell.

	Before running the checker, PW should offer to save the buffer.

	This is still only a test version. Error checking is mostly
	missing.
*/

int pfd[2], qfd[2];

/* write a line to ispell */
void spell_write(char *b)
{
        write(pfd[1], b, strlen(b));
}

/* read a line from ispell */
int spell_read(char *b)
{
        fd_set rfds;
        struct timeval tv;
        int retval, n;

        FD_ZERO(&rfds);
        FD_SET(qfd[0], &rfds);
        tv.tv_sec = 5;
        tv.tv_usec = 0;
        retval = select(qfd[0]+1, &rfds, NULL, NULL, &tv);
        if (!retval) return 0;
        n = read(qfd[0], b, 1024);
        b[n] = '\0';
        return n;
}

static char *letters = "";
static char *others = "_-'";

static int letter(int c)
{
	return isalpha(c) || strchr(letters, c);
}

static int other(int c)
{
	return strchr(others, c) != NULL;
}

static LISP dump_words(void)
{
	char *p, b[1024];
	int i, start = -1, end = -1;
	buffer *buf = w_list->buf;
	long row = w_list->point_pos.row;

	if (row > line_last_used(buf)) return NIL;

	p = (char *)rc_makeplain(buf->text[row].p);
	llpr(p);

	for (i = 0; p[i]; i++) {
		if (start == -1) {
			if (letter(p[i])) start = end = i;
		} else {
			if (letter(p[i])) end = i;
			else if (!other(p[i])) {
				strncpy(b, p+start, end-start+1);
				b[end-start+1] = '\0';
				fprintf(stderr, "%s\n", b);
				start = -1;
			}
		}
	}
	/* pick up slack */
	if (start != -1) {
		strncpy(b, p+start, end-start+1);
		b[end-start+1] = '\0';
		fprintf(stderr, "%s\n", b);
	}
	cfree(p);
	return NIL;
}

static void chomp(char *p)
{
	if ((p = strchr(p, '\n'))) *p = '\0';
}

/* asks ispell about the word, replacing it in the buffer if needed */
/* returns index of first char after checked/replaced word */
static int spell_word(buffer *buf, char *p, unsigned long r,
		int start, int end)
{
	char b[1024], d[1024], *q;
	int n, newend;
	unsigned long fmt;

	strncpy(b, p+start, end-start+1);
	b[end-start+1] = '\0';
	strcat(b, "\n");
	spell_write(b);
	if (!spell_read(d)) {
		fprintf(stderr, "ispell process probably hosed\n");
		return -1;
	}
	if (d[0] == '*' || d[0] == '+' || d[0] == '-')
		return end;	/* all is well */
	if (d[0] == '&') {	/* use first suggestion */
		q = strchr(d, ':');
		if (q == NULL) return -1;	/* bogus string */
		/* can't strcpy because the strings overlap */
		memmove(d, q+2, strlen(q));	/* skip one space */
		q = strchr(d, ',');
		if (q) *q = '\0';
	} else {		/* use incorrect word */
		strcpy(d, b);
	}
	n = spell_select(b, d);
	chomp(b);
	chomp(d);
	switch (n) {
	case SPELL_REPLACE:
		/* replace in the buffer */
		fmt = ret_format(buf, r, start);
		del_text(buf, make_position(r, start), end-start+1);
		ins_text(buf, make_position(r, start),
			(unsigned char *)d, fmt);
		/* replace in the plaintext string */
		newend = start+strlen(d)-1;
		memmove(p+newend+1, p+end+1, strlen(p+end+1)+1);
		memmove(p+start, d, strlen(d));
		return newend+1;
		break;
	case SPELL_ACCEPT:
		sprintf(d, "@%s\n", b);
		spell_write(d);
		break;
	case SPELL_INSERT:
		sprintf(d, "*%s\n", b);
		spell_write(d);
		break;
	case SPELL_SKIP:
		break;
	case SPELL_CANCEL:
		return -1;
		break;
	default:
		fprintf(stderr, "Spelling checker is broken\n");
		return -1;
	}
	return end+1;
}

/* returns 0 if we want to continue, otherwise 1 */
static int spell_line(buffer *buf, unsigned long r)
{
	char *p;
	char b[1024];
	int i, start = -1, end = -1;

	if (ret_style(buf, r) == STY_EMBED) return 0;
	p = (char *)peek_line(buf, r);
	if (!p) return 1;
	strncpy(b, p, 1020);
	b[1020] = '\0';
	cfree(p);
	llpr(b);	/* display it so we know where we are */

	for (i = 0; b[i]; i++) {
		if (start == -1) {
			if (letter(b[i])) start = end = i;
		} else {
			if (letter(b[i])) end = i;
			else if (!other(b[i])) {
				i = spell_word(buf, b, r, start, end);
				if (i == -1) {	/* abort */
					return 1;
				}
				start = end = -1;
			}
		}
	}
	/* pick up slack */
	if (start != -1) {
		i = spell_word(buf, b, r, start, end);
		if (i == -1) {	/* abort */
			return 1;
		}
	}
	return 0;
}

static LISP spell_buffer(void)
{
	buffer *b = w_list->buf;
	unsigned long row;
	int pid;

	if (b->change) {
		char prompt[1024];
		sprintf(prompt, "Save %s before checking spelling?", b->name);
		switch(alert_box(prompt, quit_buttons, 3)) {
		case 0:
			savematrix(b->path, b, NULL);
			break;
		case 1:
			break;
		default:
			return NIL;
		}
	}

        pipe(pfd);
        pipe(qfd);
        pid = fork();
        if (pid == -1) {        /* error */
                exit(EXIT_FAILURE);
        } else if (pid == 0) {  /* child */
                /* redir stdin */
                close(pfd[1]);
                dup2(pfd[0], 0);
                close(pfd[0]);
                /* redir stdout */
                close(qfd[0]);
                dup2(qfd[1], 1);
                close(qfd[1]);
                execlp("ispell", "ispell", "-a", "-w_-'", (char *)0);
                fprintf(stderr, "We don't want to be here\n");
                exit(EXIT_FAILURE);
        } else {                /* parent */
                char q[1024];
		char r[1024];
		close(pfd[0]);
		close(qfd[1]);
                if (!spell_read(r)) {
                        llpr("No response\n");
                        return NIL;
                }
                llpr(r);
                q[0] = '\0';
		for (row = 1; row <= line_last_used(b); row++)
			if (spell_line(b, row)) break;
		position_kludge2();
		spell_write("#\n");	/* save personal dictionary */
		close(pfd[1]);
		close(qfd[0]);

        }
        return NIL;
}

static LISP spell_test(void)
{
        int pid;

        pipe(pfd);
        pipe(qfd);
        pid = fork();
        if (pid == -1) {        /* error */
                exit(EXIT_FAILURE);
        } else if (pid == 0) {  /* child */
                /* redir stdin */
                close(pfd[1]);
                dup2(pfd[0], 0);
                close(pfd[0]);
                /* redir stdout */
                close(qfd[0]);
                dup2(qfd[1], 1);
                close(qfd[1]);
                execlp("ispell", "ispell", "-a", "-w_-'", (char *)0);
                fprintf(stderr, "We don't want to be here\n");
                exit(EXIT_FAILURE);
        } else {                /* parent */
                char b[1024];
		close(pfd[0]);
		close(qfd[1]);
                if (!spell_read(b)) {
                        llpr("No response\n");
                        return NIL;
                }
		llpr(b);
		b[0] = '\0';
                while (ask_for_str("Word:", b)) {
			strcat(b, "\n");
                        spell_write(b);
                        if (spell_read(b)) {
                                llpr(b);
                        }
                }
		close(pfd[1]);
		close(qfd[0]);
        }
        return NIL;
}

static LISP print_version(void)
{
	llpr(VERSION);
	return NIL;
}

static LISP buffer_changed(LISP bname)
{
	buffer *buf;

	if (NULLP(bname)) buf = w_list->buf;
	else buf = find_buffer_by_name(get_c_string(bname));

	if (buf) buf->change = TRUE;
	return NIL;
}

static LISP set_current_fmt(LISP font, LISP mask)
{
	long fo = get_c_long(font);
	long ma = get_c_long(mask);
	long of = w_list->current_fmt;
	w_list->current_fmt = (fo & ma) | (of & (~ma));
	return NIL;
}

static LISP set_format(LISP bname, LISP posu, LISP posl, LISP format)
{
	buffer *buf;
	long fmt;
	long ur, uc, lr, lc, r;

	if (NULLP(bname)) buf = w_list->buf;
	else buf = find_buffer_by_name(get_c_string(bname));

	if (!buf) return NIL;

	ur = POSITION_ROW(posu);
	uc = POSITION_COL(posu);
	lr = POSITION_ROW(posl);
	lc = POSITION_COL(posl);
	fmt = get_c_long(format);

	if (ur > lr || (ur == lr && uc > lc)) return NIL;

	alloc_line(buf, lr);	/* make sure we have all the lines */

	pr_scr_flag = TRUE;
	buf->change = TRUE;

	if (ur == lr) {		/* single line */
		ins_format(buf, ur, uc, lc, fmt);
		return NIL;
	}
	/* otherwise, several lines */
	/* first line */
	ins_format(buf, ur, uc, rc_strlen(buf->text[ur].p), fmt);
	/* middle lines */
	for (r = ur+1; r < lr; r++) {
		buf->text[r].sty = buf->text[ur].sty;
		ins_format(buf, r, 0, rc_strlen(buf->text[r].p), fmt);
	}
	/* last line */
	buf->text[lr].sty = buf->text[ur].sty;
	ins_format(buf, lr, 0, lc, fmt);

	/* work around some uglyness */
	if (w_list->buf == buf) {
		w_list->blku.row = ur;
		w_list->blku.col = uc;
		w_list->blkl.row = lr;
		w_list->blkl.col = lc;
	}
	return NIL;
}

static LISP get_format(LISP bname, LISP pos)
{
	buffer *buf;
	int r, c;
	long fmt;

	if (NULLP(bname)) buf = w_list->buf;
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	r = POSITION_ROW(pos);
	c = POSITION_COL(pos);
	if (r == w_list->point_pos.row && c == w_list->point_pos.col)
		fmt = w_list->current_fmt;
	else
		fmt = ret_format(buf, r, c);
	return flocons(fmt);
}

static LISP lset_style(LISP row, LISP sty)
{
	long r = get_c_long(row);
	long s = get_c_long(sty);
	buffer *buf = w_list->buf;
	w_list->current_fmt = styles[s].format;
	set_style(buf, r, s);
	buf->change = TRUE;
	ins_format(buf, r, 0, rc_strlen(buf->text[r].p),
					styles[s].format);
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP get_style(LISP row)
{
	int r = get_c_long(row);
	buffer *buf = w_list->buf;
	alloc_line(buf, r);
	return flocons(buf->text[r].sty);
}

static LISP set_hadjust(LISP row, LISP hadjust)
{
	int r = get_c_long(row);
	int hadj = get_c_long(hadjust);
	buffer *buf = w_list->buf;

	alloc_line(buf, r);
	buf->text[r].adj &= ~HADJ_MASK;
	buf->text[r].adj |= hadj;
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP get_hadjust(LISP row)
{
	return flocons(ret_hadj(w_list->buf, get_c_long(row)));
}

static LISP style_follower(LISP sty)
{
	return flocons(styles[get_c_long(sty)].follower);
}

/* These functions allow implementation of commands in Scheme
	rather than in C with Scheme wrappers */

static LISP insert_text(LISP text)
{
	if (TYPEP(text, tc_string)) {
		ins_text(w_list->buf, w_list->point_pos,
			(unsigned char *)text->storage_as.string.data,
			w_list->current_fmt);
	}
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP remove_text(LISP len)
{
	del_text(w_list->buf, w_list->point_pos, get_c_long(len));
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP join_lines_at(LISP row)
{
	int r = get_c_long(row);
	join_lines(w_list->buf, r);
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP split_line_at(LISP row, LISP col)
{
	int r = get_c_long(row), c = get_c_long(col);
	split_line(w_list->buf, r, c);
	w_list->buf->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP delete_lines(LISP row, LISP count)
{
	int r = get_c_long(row);
	int c = get_c_long(count);
	del_lines(w_list->buf, r, c);
	w_list->buf->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP lask_for_str(LISP prompt, LISP buf)
{
	char *p, b[256];

	strcpy(b, get_c_string(buf));
	p = get_c_string(prompt);
	if (ask_for_str(p, b))
		return strcons(strlen(b), b);
	else
		return NIL;
}

static LISP lexecute(LISP cmd)
{
	execute(get_c_string(cmd));
	return NIL;
}

static LISP linput_warp_pointer(LISP value)
{
	input_warp_pointer = get_c_long(value);
	return NIL;
}

#define MAGIC_MARKER ";;; Do not change or add anything below this line.\n"

static LISP save_preferences(void)
{
        char fn1[1024], fn2[1024];
        FILE *fp1, *fp2;
        char b[1024];

        sprintf(fn1, "%s/.siag/pw.scm", getenv("HOME"));
        sprintf(fn2, "%s.BAK", fn1);
        rename(fn1, fn2);
        fp2 = fopen(fn2, "r");
        fp1 = fopen(fn1, "w");
        if (!fp1) {
                rename(fn2, fn1);
                return NIL;
        }
        while (fp2 && fgets(b, sizeof b, fp2)) {
                if (!strcmp(b, MAGIC_MARKER)) break;
                fputs(b, fp1);
        }
        fputs(MAGIC_MARKER, fp1);
        fprintf(fp1, "(tooltip-mode %ld)\n",
                get_c_long(symbol_value(cintern("*tooltip-mode*"), NIL)));

        if (fp2) fclose(fp2);
        fclose(fp1);
        return NIL;
}


/* Set up the table of functions and names */

/* Commands that take no arguments */
s_fn_table fn_table[] =
{
	{"listsel-test", listsel_test},

	/* moving around */
	{"what-cursor-position", what_cursor_position},
	{"go-to", go_to},
	{"set-line-height", set_line_height},
	{"set-segment-format", set_segment_format},
	{"copy-current-format", copy_current_format},
	{"use-copied-format", use_copied_format},
	{"define-style", define_style},
	{"cleanup-style", cleanup_style},

	/* editing */
	{"insert-line", insert_line},
	{"remove-line", remove_line},
	/* block commands */
	{"psformat", lpsformat},

	/* new window */
	{"delete-window", delete_window},
	{"delete-other-windows", delete_other_windows},
	{"split-window-vertically", split_window_vertically},
	{"other-window", other_window},

	/* buffers and windows */
	{"switch-to-buffer", switch_to_buffer},
	{"kill-buffer", kill_buffer},
	{"load-buffer", load_buffer},
	{"save-buffer", save_buffer},
	{"save-buffer-as", save_buffer_as},
	{"load-external", load_external},
	{"save-external", save_external},

	/* help commands */
	{"print-version", print_version},

	/* screen layout */
	{"quit-pw", quit_pw},
	/* low level functions */
	{"insert-text", insert_text},

	/* misc */
	{"dump-words", dump_words},
	{"spell-buffer", spell_buffer},
	{"spell-test", spell_test},
	{"save-preferences", save_preferences},
	{NULL, NULL}
};

/* Commands that take 1 argument */

s_fn_table fn_table1[] = {
/*	{"auto-recalc", auto_recalc},*/
	/* low level functions */
	{"spawn", spawn},
	{"execute-interpreter-command", execute_interpreter_command},
	{"buffer-changed", buffer_changed},
	{"join-lines", join_lines_at},
	{"remove-text", remove_text},
	{"execute", lexecute},
	{"input-warp-pointer", linput_warp_pointer},
	{"get-style", get_style},
	{"get-hadjust", get_hadjust},
	{"style-follower", style_follower},
	{NULL, NULL}
};

/* Commands that take 2 arguments */
s_fn_table fn_table2[] = {
	{"split-line", split_line_at},
	{"delete-lines", delete_lines},
	{"ask-for-str", lask_for_str},
	{"alertbox", lalertbox},
	{"get-format", get_format},
	{"set-style", lset_style},
	{"set-hadjust", set_hadjust},
	{"set-current-fmt", set_current_fmt},
	{NULL, NULL}
};

void init_cmds(void)
{
	int i;

	for (i = 0; fn_table[i].name; i++)
		init_subr_0(fn_table[i].name, fn_table[i].function);
	for (i = 0; fn_table1[i].name; i++)
		init_subr_1(fn_table1[i].name, fn_table1[i].function);
	for (i = 0; fn_table2[i].name; i++)
		init_subr_2(fn_table2[i].name, fn_table2[i].function);
	init_subr_4("set-format", set_format);
	init_subr_3("savematrix", lsavematrix);
}

