#
#  'CBB' -- Check Book Balancer
#
#   file.tcl -- file management routines.
#
#  Written by Curtis Olson.  Started August 25, 1994.
#
#  Copyright (C) 1994 - 1999  Curtis L. Olson  - curt@me.umn.edu
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# $Id: file.tcl,v 1.2 2000/01/04 22:49:57 curt Exp $


#------------------------------------------------------------------------------
# File menu procedures
#------------------------------------------------------------------------------

# create a new account
proc acctNew {} {
    global cbb yesno argv0 eng newacct

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot make new accounts when running in 'safe' mode." 
	tkwait window .ok
        return
    }

    if { [acctIsDirty] } {
	if { $cbb(auto_save) } {
	    acctSave
	} else {
	    cbbWindow.yesno "You have not saved your current changes.  Would \
		    you like to save before making a new account?"
	    tkwait window .yesno

	    if { "$yesno(result)" == "yes" } {
		acctSave
	    } elseif { "$yesno(result)" == "no" } {
	    } elseif { "$yesno(result)" == "cancel" } {
		return
	    }
	}
    } 

    # use starting directory of current file if it exists
    if { [info exists cbb(cur_file)] && \
	    ("[file dirname $cbb(cur_file)]" != "") } {
	set newacct(name) "[file dirname $cbb(cur_file)]/"
    } else {
	set newacct(name) "[mypwd]/"
    }
    if { $cbb(debug) } { puts $newacct(name) }

    cbbWindows.newacct
    tkwait window .newacct

    if { "$newacct(name)" != "" } {
	acctNewClear
	
	if { $cbb(debug) } {
	    puts "Make Account $newacct(name) - $newacct(desc)"
	}

	if { "[file extension $newacct(name)]" == ".cbb" } {
	    set newacct(name) [file rootname $newacct(name)]
	}
	set cbb(cur_file) "$newacct(name).cbb"

	wm title . "[file tail $argv0] - [file tail $cbb(cur_file)]"
	wm iconname . "[file tail $argv0] - [file tail $cbb(cur_file)]"

	# load/create the category file first ... make_acct assumes
	# a category file is already open.
	if { $cbb(debug) } {
	    puts "Loading the category file \
		    [file dirname $cbb(cur_file)]/categories"
	}
	puts $eng "load_cats [file dirname $cbb(cur_file)]/categories"
	flush $eng

	if { $cbb(debug) } { puts "Reading result" }
	gets $eng result
	if { $cbb(debug) } { puts "Got result: $result" }
	    
	if { "$result" != "ok" } {
	    cbbWindow.ok "Error opening \
		    ``[file dirname $cbb(cur_file)]/categories'': \
		     $result.  An empty categories file will be created.  \
		     Please consider importing the default categories."
	    tkwait window .ok

            # initialize categories
	    puts $eng "init_cats"; flush $eng

	    if { $cbb(debug) } { puts "Reading result" }
	    gets $eng result 
	    if { $cbb(debug) } { puts "Got result: $result" }

	    # create an empty categories file 
	    puts $eng "save_cats [file dirname $cbb(cur_file)]/categories"
	    flush $eng
	    
	    if { $cbb(debug) } { puts "Reading result" }
	    gets $eng result
	    if { $cbb(debug) } { puts "Got result: $result" }
	}

	# tell engine to make the new account
	if { $cbb(debug) } { 
	    puts "make_acct $newacct(name) $newacct(desc)"
	}
	puts $eng "make_acct $newacct(name) $newacct(desc)"
	flush $eng
	gets $eng result; 
        if { $cbb(debug) } { 
	    puts "make_acct($newacct(name) $newacct(desc)): $result"
	}

	if { "$result" != "ok" } {
	    cbbWindow.ok "Error creating ``$cbb(cur_file)'':  $result."
	    tkwait window .ok
	}

	.status.line configure \
		-text "Account Created -- [file tail $cbb(cur_file)]"
	update
	
	acctLoadFile $cbb(cur_file)
	load_acct_listbox
	cbbcurSave
    }
}


# specify new account name & description
proc cbbWindows.newacct {} {
    global cbb newacct

    if {[winfo exists .newacct] == 1} {
	wm withdraw .newacct
        wm deiconify .newacct
	return                
    }

    set newacct(desc) ""

    toplevel .newacct

    option add *font $cbb(dialog_font)
    wm title .newacct "Make Account"
    wm iconname .newacct "Make Account"
    frame .newacct.frame -borderwidth 2 
    frame .newacct.frame.n
    frame .newacct.frame.d
    frame .newacct.frame.b

    label .newacct.frame.label -text \
	"Enter New Account Base Name (including directory) and Description"

    label .newacct.frame.n.label -text "New Acct"
    entry .newacct.frame.n.entry -textvariable newacct(name) -relief sunken \
	    -font $cbb(default_font)
    # .newacct.frame.n.entry icursor 0
    .newacct.frame.n.entry icursor end
    # tk_entrySeeCaret .newacct.frame.n.entry

    label .newacct.frame.d.label -text "Acct Description"
    entry .newacct.frame.d.entry -textvariable newacct(desc) -relief sunken \
	    -font $cbb(default_font)

    button .newacct.frame.b.create -text "Create Account" \
	    -font $cbb(button_font) -command { destroy .newacct }
    button .newacct.frame.b.cancel -text "Cancel" -font $cbb(button_font) \
	    -command { 
	set newacct(name) ""
	set newacct(desc) ""
	destroy .newacct
    }

    pack .newacct.frame -fill both -expand 1 -padx 4 -pady 4
    pack .newacct.frame.label -padx 4 -pady 2
    pack .newacct.frame.b -side bottom -fill x -padx 4 -pady 2
    pack .newacct.frame.n .newacct.frame.d \
	    -fill both -expand 1 -padx 4 -pady 2 
    pack .newacct.frame.n.label -side left
    pack .newacct.frame.n.entry -side left -fill x -expand 1
    pack .newacct.frame.d.label -side left
    pack .newacct.frame.d.entry -side left -fill x -expand 1
    pack .newacct.frame.b.create .newacct.frame.b.cancel -side left \
	    -fill x -expand 1 -padx 4 -pady 4

    focus .newacct.frame.n.entry 
}


proc acctNewClear {} {
    global cbb eng argv0

    if { $cbb(debug) } { puts "beginning of acctNewClear" }

    # clear our list box
### .trans.list delete 0 end
    .trans.list configure -state normal
    .trans.list delete 0.0 end
    .trans.list configure -state disabled

    # tell engine to clear transactions
    puts $eng "init_trans"; flush $eng
    gets $eng result
    if { $cbb(debug) } { puts "init_trans(): $result" }

    # tell engine to clear categories
    puts $eng "init_cats"; flush $eng
    gets $eng result
    if { $cbb(debug) } { puts "init_cats(): $result" }

    set cbb(cur_file) "noname.cbb"
    wm title . "[file tail $argv0] - [file tail $cbb(cur_file)]"
    wm iconname . "[file tail $argv0] - [file tail $cbb(cur_file)]"

    clear_entry_area
}


proc acctLoad {} {
    global cbb yesno

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot load accounts when running in 'safe' mode." 
	tkwait window .ok
        return
    }

    if { [acctIsDirty] } {
	if { $cbb(auto_save) } {
	    acctSave
	} else {
	    cbbWindow.yesno "You have not saved your current changes.  Would \
		    you like to save before loading a new account?"
	    tkwait window .yesno
	    
	    if { "$yesno(result)" == "yes" } {
		acctSave
	    } elseif { "$yesno(result)" == "no" } {
	    } elseif { "$yesno(result)" == "cancel" } {
		return
	    }
	}
    }

    option add *font $cbb(default_font)

    set select_result [ut:fsbox -prompt "Load Account:" \
	    -cancelvalue "noname.cbb" -grab 1 -master . -dir [mypwd] ]

    if { $cbb(debug) } { puts "After select: Load Account $select_result" }

    if { "$select_result" != "noname.cbb" } {
	acctLoadFile $select_result
	load_acct_listbox
        cbbcurSave
    }
}


proc acctLoadFile file {
    global cbb argv0 key eng yesno start_pos
    global date nicedate year month day check desc debit credit cat
    global nicecat com cleared total

    if { $cbb(debug) } { puts "ready to load $file" }

    set auto_save_file "[file dirname $file]/#[file tail $file]#"
    if { [file exists $auto_save_file] } {
	cbbWindow.yesno "CBB has detected an autosave file: \
		``[file tail $auto_save_file]''.\
		This indicates that CBB was killed or crashed after you\
		made changes to ``[file tail $file]'' but before you could\
		save the changes.  Would you like to load the auto saved\
		version instead?"
	tkwait window .yesno

        if { "$yesno(result)" == "yes" } {
	    if { [file exists $auto_save_file] } {
		# just in case we have a couple of these nested, lets not try
		# to do this more than once.
		exec mv $file "$file.bak"
		exec mv $auto_save_file $file
	    }
	    if { [file exists "$auto_save_file.bak"]} {
		exec rm "$auto_save_file.bak"
	    }
        } else {
	    cbbWindow.ok "Very well, but as a precaution,\
		    ``[file tail $file]'' will not be loaded as long as\
		    ``[file tail $auto_save_file]'' exists."
	    return
        }
    }

    acctNewClear

    set cbb(cur_file) $file

    wm title . "[file tail $argv0] - [file tail $file]"
    wm iconname . "[file tail $argv0] - [file tail $file]"
    . configure -cursor watch
    .trans.list configure -cursor watch
    .status.line configure -text "Loading transactions from [file tail $file]"
    update

    # load the transactions from a data file
    if { $cbb(debug) } { puts "Loading data file $file" }
    puts $eng "load_trans $file"; flush $eng

    if { $cbb(debug) } { puts "Reading result" }
    gets $eng result
    if { $cbb(debug) } { puts "Got result: $result" }

    if { "$result" != "ok" } {
        cbbWindow.ok "Error opening ``$file'':  $result."
        tkwait window .ok
	set cbb(cur_file) "noname.cbb"
	wm title . "[file tail $argv0] - [file tail $cbb(cur_file)]"
	wm iconname . "[file tail $argv0] - [file tail $cbb(cur_file)]"
	update
	return
    }

    .trans.list configure -state normal
    set start_pos 0
    set flag 0
    if { $cbb(use_sdate) } {
	puts $eng "part_trans $cbb(date_fmt) $cbb(int_sdate)"
    } else {
    	puts $eng "all_trans $cbb(date_fmt)"
    }
    flush $eng
    gets $eng line1
    while { $line1 != "none" } {
	gets $eng line2
	listAddTrans $line1 $line2

	set check [string trim [string range $line1 0 4]]
	if { "$check" != "" } {
	    set cbb(next_chk) $check
	}

        gets $eng line1
    }
    .trans.list configure -state disabled

    # set listbox view to start position
    goto $start_pos
    clear_entry_area

    . configure -cursor left_ptr
    .trans.list configure -cursor xterm

    # hash out the memorized transactions
    if { $cbb(debug) } { puts "Hashing memorized transactions" }
    puts $eng "rehash_mems"; flush $eng

    if { $cbb(debug) } { puts "Reading result" }
    gets $eng result
    if { $cbb(debug) } { puts "Got result: $result" }

    # load the categories
    if { $cbb(debug) } {
	puts "Loading the category file [file dirname $file]/categories"
    }
    puts $eng "load_cats [file dirname $file]/categories"; flush $eng

    if { $cbb(debug) } { puts "Reading result" }
    gets $eng result
    if { $cbb(debug) } { puts "Got result: $result" }

    if { "$result" != "ok" } {
        cbbWindow.ok "Error opening ``[file dirname $file]/categories'':  \
		$result."
        tkwait window .ok
    }

    acctSetClean
}


proc acctSave {} {
    global cbb argv0 eng

    wm title . "[file tail $argv0] - [file tail $cbb(cur_file)]"
    wm iconname . "[file tail $argv0] - [file tail $cbb(cur_file)]"

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot save when running in 'safe' mode." 
	tkwait window .ok
        return
    }

    . configure -cursor watch
    .trans.list configure -cursor watch
    .status.line configure \
	    -text "Saving transactions to [file tail $cbb(cur_file)]"
    update

    # tell engine to save transactions
    puts $eng "save_trans $cbb(cur_file)"; flush $eng
    gets $eng result
    if { $cbb(debug) } { puts "saving transactions ... $result" }

    . configure -cursor left_ptr
    .trans.list configure -cursor xterm

    if { "$result" != "ok" } {
	. configure -cursor left_ptr
	.trans.list configure -cursor xterm

        cbbWindow.ok "Error saving ``$cbb(cur_file)'':  $result.  You are \
		advised to use the ``Save As'' function to save to a different \
		file name."
        tkwait window .ok
    }

    # tell engine to save categories
    puts $eng "save_cats [file dirname $cbb(cur_file)]/categories"; flush $eng
    gets $eng result
    if { $cbb(debug) } { puts "saving categories ... $result" }

    if { "$result" != "ok" } {
	. configure -cursor left_ptr
	.trans.list configure -cursor xterm

        cbbWindow.ok "Error saving \
		``[file dirname $cbb(cur_file)]/categories'':  \
		$result.  Recent changes to your categories list may not \
		have been saved."
        tkwait window .ok
    }

    cbbcurSave

    # ok we are clean again
    acctSetClean

    . configure -cursor left_ptr
    .trans.list configure -cursor xterm
}


proc acctSaveAs {} {
    global cbb

    option add *font $cbb(default_font)

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot save as when running in 'safe' mode." 
        tkwait window .ok
        return
    }

    set select_result [ut:fsbox -prompt "Save Account As:" \
	    -cancelvalue "noname.cbb" -grab 1 -master . -dir [mypwd] ]

    if { $cbb(debug) } { puts "After select: Save Account As: $select_result" }

    if { "$select_result" != "noname.cbb" } {
	set cbb(cur_file) $select_result
	acctSave
    }
}


# Select a file to import
proc acctImport {} {
    global cbb import_type

    option add *font $cbb(default_font)

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot import when running in 'safe' mode." 
        tkwait window .ok
        return
    }

    if { $import_type == 0 } {
	set prompt "CBB Import:"
    } else { 
	set prompt "QIF Import:"
    }

    set select_result [ut:fsbox -prompt $prompt -cancelvalue "noname.cbb" \
	    -grab 1 -master . -dir [mypwd] ]

    if { $cbb(debug) } { puts "After select: Import [file tail $select_result]"}

    if { "$select_result" != "noname.cbb" } {
	acctImportFile $select_result $import_type
    }
}


# Do the actual import
proc acctImportFile args {
    global cbb eng key date nicedate year month day check desc debit credit cat
    global nicecat com cleared total

    set file [lindex $args 0]
    set type [lindex $args 1]

    # clear out the old ... not any more
    # acctNewClear
    # set cbb(cur_file) noname.cbb

    .status.line configure \
	    -text "Importing transactions from [file tail $file]."
    update

    if { $cbb(debug) } { puts "Importing data file $file" }

    if { $type == 0 } {
	puts $eng "load_cbb $file"; flush $eng

	if { $cbb(debug) } { puts "Reading result" }
	gets $eng result
	if { $cbb(debug) } { puts "Got result: $result" }
    } elseif { $type == 1 } {
	puts $eng "import_qif $file"; flush $eng

	if { $cbb(debug) } { puts "Reading result" }
	gets $eng result
	if { $cbb(debug) } { puts "Got result: $result" }
    }

    # clear our list box
    .trans.list configure -state normal
    .trans.list delete 0.0 end

    set flag 0
    puts $eng "all_trans $cbb(date_fmt)"; flush $eng
    gets $eng line1
    while { $line1 != "none" } {
	if { $cbb(debug) } { puts "line1 = $line1" }
	gets $eng line2
	listAddTrans $line1 $line2

        gets $eng line1
    }
    .trans.list configure -state disabled

    # set listbox view to end
    goto [listGetSize]
    clear_entry_area

    acctSetDirty
}


# export to a quicken ".qif" file
proc acctExportQIF {} {
    global cbb eng

    if { $cbb(safe_mode) == 1 } {
        cbbWindow.ok "Cannot export when running in 'safe' mode." 
	tkwait window .ok
        return
    }

    puts $eng "export_qif [file root $cbb(cur_file)].qif"; flush $eng
    gets $eng result

    cbbWindow.ok "File exported to ``[file root $cbb(cur_file)].qif''\n \
	     Result = ``$result''"
    tkwait window .ok
}


proc cbbQuit {} {
    global cbb yesno

    if { [acctIsDirty] } {
	if { $cbb(safe_mode) == 0 } {
	    if { $cbb(auto_save) } {
		acctSave
	    } else {
		cbbWindow.yesno "You have not saved your current changes. \
			Would you like to save before quitting?"
		tkwait window .yesno

		if { "$yesno(result)" == "yes" } {
		    acctSave
		} elseif { "$yesno(result)" == "no" } {
		} elseif { "$yesno(result)" == "cancel" } {
		    return
		}
	    }
	} else {
	    cbbWindow.ok "You have made changes to this file.  Normally you \
		    would be asked if you would like to save your changes.  \
		    However, you cannot save files in 'safe' mode so please \
		    click ``Dismis'' to continue without saving." 
	    tkwait window .ok
	}
    }

    if { $cbb(pref_chg) } {
	if { $cbb(safe_mode) == 0 } {
		cbbWindow.yesno "You have not saved your current preferences. \
			Would you like to save before quitting?"
		tkwait window .yesno

		if { "$yesno(result)" == "yes" } {
		    cbbrcSave
		} elseif { "$yesno(result)" == "no" } {
		} elseif { "$yesno(result)" == "cancel" } {
		    return
		}
	} else {
	    cbbWindow.ok "You have made changes to your preferences.  \
		    Normally you \
		    would be asked if you would like to save your changes.  \
		    However, you cannot save preferences in 'safe' mode so \
		    please click ``Dismis'' to continue without saving." 
	    tkwait window .ok
	}
    }

    exit
}


# write out current ~/.cbbcur.tcl file
proc cbbcurSave {} {
    global cbb

    if { $cbb(safe_mode) == 1 } {
	return
    }

    set now_today [get_sys_date]

    set rchandle [open $cbb(save_cur_file) w]
    puts $rchandle "# ~/.cbbcur.tcl"
    puts $rchandle "#"
    puts $rchandle "# Created by CBB $cbb(version)"
    puts $rchandle "# Created on $now_today"
    puts $rchandle "#"
    puts $rchandle "# This is machine generated tcl code.  Please use caution"
    puts $rchandle "# when hand editing!"
    puts $rchandle ""
    puts $rchandle "# File to load when CBB initially starts."
    puts $rchandle "set cbb(cur_file) $cbb(cur_file)"

    close $rchandle
}

# write out current ~/.cbbrc.tcl file
proc cbbrcSave {} {
    global cbb

    set cbb(pref_chg) 0

    if { $cbb(safe_mode) == 1 } {
	return
    }

    set now_today [get_sys_date]

    set rchandle [open $cbb(conf_file) w]
    puts $rchandle "# ~/.cbbrc.tcl"
    puts $rchandle "#"
    puts $rchandle "# Created by CBB $cbb(version)"
    puts $rchandle "# Created on $now_today"
    puts $rchandle "#"
    puts $rchandle "# This is machine generated tcl code.  Please use caution"
    puts $rchandle "# when hand editing!"
    puts $rchandle ""
    puts $rchandle "# Debugging:  0 = disabled, 1 = enabled"  
    puts $rchandle "set cbb(debug) $cbb(debug)"
    puts $rchandle ""
    puts $rchandle "# Enable Balloon Help:  0 = disabled, 1 = enabled"
    puts $rchandle "set cbb(balloon,on) $cbb(balloon,on)"
    puts $rchandle ""
    puts $rchandle "# Enable Splash Screen:  0 = disabled, 1 = enabled"
    puts $rchandle "set cbb(splash) $cbb(splash)"
    puts $rchandle ""
    puts $rchandle "# Auto-highlight entry fields:  0 = disabled, 1 = enabled"  
    puts $rchandle "set cbb(auto_hilite) $cbb(auto_hilite)"
    puts $rchandle ""
    puts $rchandle "# Use Memorized Transactions:  0 = disabled, 1 = enabled"
    puts $rchandle "set cbb(use_mems) $cbb(use_mems)"
    puts $rchandle ""
    puts $rchandle "# Date Format:  1=USA 2=International"
    puts $rchandle "set cbb(date_fmt) $cbb(date_fmt)"
    puts $rchandle ""
    puts $rchandle "# Auto Save Interval (in milleseconds)"
    puts $rchandle "set cbb(save_interval) $cbb(save_interval)"
    puts $rchandle "# Auto-save when switching files or quitting"
    puts $rchandle "set cbb(auto_save) $cbb(auto_save)"
    puts $rchandle ""
    puts $rchandle "# Main List Box Dimensions"
    puts $rchandle "set cbb(list_width) $cbb(list_width)"
    puts $rchandle "set cbb(list_height) $cbb(list_height)"
    puts $rchandle ""
    puts $rchandle "# Height of account list box"
    puts $rchandle "set cbb(acctlist_height) $cbb(acctlist_height)"
    puts $rchandle ""
    puts $rchandle "# Maximum splits per entry"
    puts $rchandle "set cbb(max_splits) $cbb(max_splits)"
    puts $rchandle ""
    puts $rchandle "# Startdate ..."
    puts $rchandle "set cbb(use_sdate) $cbb(use_sdate)"
    puts $rchandle "set cbb(sdate) \"$cbb(sdate)\""
    puts $rchandle "set cbb(report_sdate) \"$cbb(report_sdate)\""
    puts $rchandle ""
    puts $rchandle "# Account Cache ..."
    puts $rchandle "set cbb(cache) $cbb(cache)"
    puts $rchandle ""
    puts $rchandle "# Cryptography ..."
    puts $rchandle "set cbb(use_crypt) $cbb(use_crypt)"
    puts $rchandle "set cbb(encrypt) \"$cbb(encrypt)\""
    puts $rchandle "set cbb(decrypt) \"$cbb(decrypt)\""
    puts $rchandle ""
    puts $rchandle "# Duplicate Entries ..."
    puts $rchandle "set cbb(duplicate) $cbb(duplicate)"
    puts $rchandle "set cbb(cmpfields) $cbb(cmpfields)"
    puts $rchandle ""
    puts $rchandle "# Fonts ..."
    puts $rchandle "set cbb(msg_text_font) \"$cbb(msg_text_font)\""
    puts $rchandle "set cbb(button_font) \"$cbb(button_font)\""
    puts $rchandle "set cbb(fixed_header_font) \"$cbb(fixed_header_font)\""
    puts $rchandle "set cbb(fixed_font) \"$cbb(fixed_font)\""
    puts $rchandle "set cbb(status_line_font) \"$cbb(status_line_font)\""
    puts $rchandle "set cbb(menu_font) \"$cbb(menu_font)\""
    puts $rchandle "set cbb(dialog_font) \"$cbb(dialog_font)\""
    puts $rchandle "set cbb(default_font) \"$cbb(default_font)\""
    puts $rchandle ""
    puts $rchandle "# Colors ..."
    puts $rchandle "set cbb(head_color) \"$cbb(head_color)\""
    puts $rchandle "set cbb(hilite_color) \"$cbb(hilite_color)\""
    puts $rchandle "set cbb(list_line1_color) \"$cbb(list_line1_color)\""
    puts $rchandle "set cbb(list_line2_color) \"$cbb(list_line2_color)\""
    puts $rchandle "set cbb(trans_text_color) \"$cbb(trans_text_color)\""
    puts $rchandle "set cbb(trans_neg_color) \"$cbb(trans_neg_color)\""
    puts $rchandle "set cbb(future_color) \"$cbb(future_color)\""
    puts $rchandle "set cbb(future_neg_color) \"$cbb(future_neg_color)\""
    puts $rchandle ""
    puts $rchandle "# External web browser"
    puts $rchandle "set cbb(web_browser) \"$cbb(web_browser)\""

    close $rchandle
}


# save the current file at regular intervals in case the computer crashes
# or something
proc setup_auto_save {} {
    global cbb eng

    if { $cbb(safe_mode) == 1 } {
	return
    }

    after $cbb(save_interval) {
        if { "[file extension $cbb(cur_file)]" == ".cbb" } {
	    if { [acctIsDirty] } {
		set auto_save_file \
		    "[file dirname $cbb(cur_file)]/#[file tail $cbb(cur_file)]#"
		.status.line configure -text \
			 "Auto saving to [file tail $auto_save_file]"
		update
	        if { $cbb(debug) } {
		    puts "Auto saving to [file tail $auto_save_file]"
		}

                # tell engine to save transactions
                puts $eng "auto_save_trans $auto_save_file"
		flush $eng
	    }
	}

	# reset auto save
	setup_auto_save
    }
}
