/***

itrafmon.c - the IP traffic monitor module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <sys/time.h>
#include <sys/wait.h>
#include <errno.h>
#include "tcptable.h"
#include "othptab.h"
#include "packet.h"
#include "ifaces.h"
#include "ipcsum.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "othfilter.h"
#include "deskman.h"
#include "error.h"
#include "attrs.h"
#include "log.h"
#include "revname.h"
#include "rvnamed.h"
#include "dirs.h"
#include "timer.h"
#include "ipfrag.h"
#include "options.h"
#include "instances.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

/* Hot key indicators for the bottom line */

void ipmonhelp()
{
    move(LINES - 1, 1);
    printkeyhelp("Up/Dn/PgUp/PgDn", "-scrl actv win  ", stdscr);
    printkeyhelp("W", "-chg actv win  ", stdscr);
    printkeyhelp("M", "-more TCP info  ", stdscr);
    stdexitkeyhelp();
};

/* Mark general packet count indicators */

void prepare_ip_statwin(WINDOW * win)
{
    wmove(win, 0, 1);
    wprintw(win, "IP:");
    wmove(win, 0, 16);
    wprintw(win, "TCP:");
    wmove(win, 0, 31);
    wprintw(win, "UDP:");
    wmove(win, 0, 46);
    wprintw(win, "ICMP:");
    wmove(win, 0, 62);
    wprintw(win, "Non-IP:");
}

void markactive(int curwin, WINDOW * tw, WINDOW * ow)
{
    WINDOW *win1;
    WINDOW *win2;
    int x1, y1, x2, y2;

    if (!curwin) {
	win1 = tw;
	win2 = ow;
    } else {
	win1 = ow;
	win2 = tw;
    }

    getmaxyx(win1, y1, x1);
    getmaxyx(win2, y2, x2);

    wmove(win1, --y1, 68);
    wprintw(win1, " Active ");
    wmove(win2, --y2, 68);
    whline(win2, ACS_HLINE, 8);
}

void show_ip_stats(WINDOW * win, unsigned long long iptotal,
                   unsigned long long tcptotal,
		   unsigned long long udptotal, unsigned long long icmptotal,
		   unsigned long long noniptotal)
{
    wmove(win, 0, 6);
    printlargenum(iptotal, win);
    wmove(win, 0, 21);
    printlargenum(tcptotal, win);
    wmove(win, 0, 36);
    printlargenum(udptotal, win);
    wmove(win, 0, 52);
    printlargenum(icmptotal, win);
    wmove(win, 0, 70);
    printlargenum(noniptotal, win);
}


/* 
 * Scrolling and paging routines for the upper (TCP) window
 */

void scrollupperwin(struct tcptable *table, int direction, int *idx, int mode)
{
    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->tcpscreen, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->tcpscreen, table->imaxy - 1, 0);
	    scrollok(table->tcpscreen, 0);
	    wprintw(table->tcpscreen, "%78c", ' ');
	    scrollok(table->tcpscreen, 1);
	    printentry(table, table->lastvisible, *idx, mode);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->tcpscreen, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;
	    (*idx)--;
	    wmove(table->tcpscreen, 0, 0);
	    wprintw(table->tcpscreen, "%78c", ' ');
	    printentry(table, table->firstvisible, *idx, mode);
	}
    }
}

void pageupperwin(struct tcptable *table, int direction, int *idx, int mode)
{
    int i = 1;

    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	while ((i <= table->imaxy - 3) && (table->lastvisible != table->tail)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    } else {
	while ((i <= table->imaxy - 3) && (table->firstvisible != table->head)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    }
}

/*
 * Scrolling and paging routines for the lower (non-TCP) window.
 */

void scrolllowerwin(struct othptable *table, int direction)
{
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->othpwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;

	    if (table->htstat == HIND) {	/* Head indicator on? */
		wmove(table->borderwin, table->obmaxy - 1, 1);
		whline(table->borderwin, ACS_HLINE, 8);
		table->htstat = NOHTIND;
	    }
	    printothpentry(table, table->lastvisible, table->oimaxy - 1, 0, (FILE *) NULL);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->othpwin, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;

	    if (table->htstat == TIND) {	/* Tail indicator on? */
		wmove(table->borderwin, table->obmaxy - 1, 1);
		whline(table->borderwin, ACS_HLINE, 8);
		table->htstat = NOHTIND;
	    }
	    printothpentry(table, table->firstvisible, 0, 0, (FILE *) NULL);
	}
    }
}

void pagelowerwin(struct othptable *table, int direction)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= table->oimaxy - 2) && (table->lastvisible != table->tail)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    } else {
	while ((i <= table->oimaxy - 2) && (table->firstvisible != table->head)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    }
}

/*
 * Attempt to communicate with rvnamed, and if it doesn't respond, try
 * to start it.
 */

int checkrvnamed(void)
{
    int execstat = 0;
    int resp;
    pid_t cpid = 0;
    int cstat;
    extern int errno;

    indicate("Trying to communicate with reverse lookup server");
    if (!rvnamedactive()) {
	indicate("Starting reverse lookup server");

	if ((cpid = fork()) == 0) {
	    execstat = execl(RVNDFILE, NULL);

	    /* 
	     * execl() never returns, so if we reach this point, we have
	     * a problem.
	     */

	    _exit(1);
	} else if (cpid == -1) {
	    errbox("Can't spawn new process; lookups will block", ANYKEY_MSG, &resp);
	    return 0;
	} else {
	    while (waitpid(cpid, &cstat, 0) < 0)
		if (errno != EINTR)
		    break;

	    if (WEXITSTATUS(cstat) == 1) {
		errbox("Can't start rvnamed; lookups will block", ANYKEY_MSG, &resp);
		return 0;
	    } else {
		sleep(1);
		return 1;
	    }
	}
    }
    return 1;
}

/* 
 * The IP Traffic Monitor
 */

void ipmon(const struct OPTIONS *options, int filtered,
	   struct filterlist *fl, struct othpoptions *ofilter,
	   unsigned int tcptm, int facilitytime)
{
    int logging = options->logging;
    struct sockaddr_pkt fromaddr;	/* iface info */
    int fd;				/* raw socket */
    char tpacket[8192];			/* raw packet data */
    char *packet = NULL;		/* network packet ptr */
    struct iphdr *ippacket;
    struct tcphdr *transpacket;		/* IP-encapsulated packet */
    unsigned int sport = 0, dport = 0;	/* TCP/UDP port values */
    int firstin = 0;

    unsigned int screen_idx = 1;

    struct timeval tv;
    unsigned long starttime = 0;
    unsigned long now = 0;
    unsigned long timeint = 0;
    unsigned long updtime = 0;
    unsigned long long updtime_usec = 0;
    unsigned long long unow = 0;
    
    WINDOW *statwin;
    PANEL *statpanel;

    FILE *logfile = NULL;

    int curwin = 0;

    int isclosed;

    int readlen;
    char ifname[8];

    unsigned long long iptotal = 0;
    unsigned long long noniptotal = 0;
    unsigned long long tcptotal = 0;
    unsigned long long udptotal = 0;
    unsigned long long icmptotal = 0;
    unsigned long long othptotal = 0;

    unsigned int nbplen;
    unsigned int br;

    int hdrchk;			/* IP checksum values */
    int hdrcsum;

    unsigned int iphlen;
    unsigned int totalhlen;

    struct tcptable table;
    struct tcptableent *tcpentry;
    int mode = 0;

    struct othptable othptbl;
    struct othptabent *othpent;

    int p_sstat = 0, p_dstat = 0;	/* Reverse lookup statuses prior to */
    					/* reattempt in updateentry(). */
    int nifltok = 0;			/* Non-IP filter ok */

    int ch;
    int endloop = 0;
    char msgstring[82];
    int nomem = 0;

    int rvnfd = 0;

    /* 
     * Mark this instance of the traffic monitor
     */

    if (!facility_active(IPMONIDFILE))
        mark_facility(IPMONIDFILE, "IP traffic monitor");
    else {
        errbox("IP traffic monitor active in another process", ANYKEY_MSG, &ch);
        return;
    }
    
    init_tcp_table(&table);
    init_othp_table(&othptbl);

    if (options->revlook)
	if (checkrvnamed())
	    open_rvn_socket(&rvnfd);
	else
	    rvnfd = 0;

    if (options->servnames)
	setservent(1);

    ipmonhelp();
    statwin = newwin(1, 80, LINES - 2, 0);
    statpanel = new_panel(statwin);
    wattrset(statwin, FIELDATTR);
    wmove(statwin, 0, 0);
    wprintw(statwin, "%80c", ' ');
    prepare_ip_statwin(statwin);
    show_ip_stats(statwin, 0, 0, 0, 0, 0);
    markactive(curwin, table.borderwin, othptbl.borderwin);
    update_panels();
    doupdate();

    open_socket(&fd, "");

    if (fd < 0)
	return;

    /*
     * Try to open log file if logging activated.  Turn off logging
     * (for this session only) if an error was discovered in opening
     * the log file.  Configuration setting is kept.  Who knows, the
     * situation may be corrected later.
     */

    if (logging)
	opentlog(&logfile, IPMONLOG);

    if (logfile == NULL)
	logging = 0;

    writelog(logging, logfile, "******** IP traffic monitor started");

    gettimeofday(&tv, NULL);
    starttime = timeint = tv.tv_sec;

    while (!endloop) {
	getpacket(fd, tpacket, &fromaddr, &ch, &readlen, 5, table.tcpscreen);
	if (ch != ERR) {
	    switch (ch) {
	    case KEY_UP:
		if (!curwin)
		    scrollupperwin(&table, SCROLLDOWN, &screen_idx, mode);
		else
		    scrolllowerwin(&othptbl, SCROLLDOWN);
		break;
	    case KEY_DOWN:
		if (!curwin)
		    scrollupperwin(&table, SCROLLUP, &screen_idx, mode);
		else
		    scrolllowerwin(&othptbl, SCROLLUP);
		break;
	    case KEY_PPAGE:
	    case '-':
		if (!curwin)
		    pageupperwin(&table, SCROLLDOWN, &screen_idx, mode);
		else
		    pagelowerwin(&othptbl, SCROLLDOWN);
		break;
	    case KEY_NPAGE:
	    case ' ':
		if (!curwin)
		    pageupperwin(&table, SCROLLUP, &screen_idx, mode);
		else
		    pagelowerwin(&othptbl, SCROLLUP);
		break;
	    case KEY_F(6):
	    case 'w':
	    case 'W':
		curwin = ~curwin;
		markactive(curwin, table.borderwin, othptbl.borderwin);
		break;
	    case 'm':
	    case 'M':
		mode = ~mode;
		refreshtcpwin(&table, screen_idx, mode);
		break;
	    case 'Q':
	    case 'q':
	    case 'X':
	    case 'x':
	    case 24:
	    case 27:
		endloop = 1;
	    }
	}
	if (readlen > 0) {
	    strcpy(ifname, fromaddr.spkt_device);

	    if (!iface_supported(ifname))
		continue;

	    fromaddr.spkt_protocol = htons(fromaddr.spkt_protocol);

	    if (fromaddr.spkt_protocol != ETH_P_IP) {
		if ((fromaddr.spkt_protocol == ETH_P_ARP) ||
		    (fromaddr.spkt_protocol == ETH_P_RARP))
		    nifltok = othfilterok(ofilter, fromaddr.spkt_protocol,
					  0, 0, 0, 0);
		else
		    nifltok = 1;


		/*
		 * Display non-IP entry in lower window if filter ok,
		 * and nomem not set
		 */

		if ((nifltok) && (!nomem))
		    othpent = add_othp_entry(&othptbl, &table,
				    0, 0, NOT_IP, fromaddr.spkt_protocol,
		    fromaddr.spkt_family, (char *) tpacket, ifname, 0, 0,
			0, logging, logfile, options->servnames, &nomem);

		noniptotal++;
		continue;
	    }
	    /* 
	     * move past data link header, if present
	     */

	    adjustpacket(tpacket, fromaddr.spkt_family, &packet, &readlen);

	    if (packet == NULL)
		continue;

	    ippacket = (struct iphdr *) packet;
	    iphlen = ippacket->ihl * 4;
	    transpacket = (struct tcphdr *) (packet + iphlen);

	    /*
	     * Compute and verify IP header checksum
	     */

	    hdrcsum = ippacket->check;
	    ippacket->check = 0;
	    hdrchk = in_cksum((u_short *) ippacket, iphlen);

	    if (hdrcsum != hdrchk)
		continue;

	    nbplen = ntohs(ippacket->tot_len);

	    iptotal += nbplen;
	    if (ippacket->protocol == IPPROTO_TCP) {
		tcptotal += ntohs(ippacket->tot_len);

		/*
		 * TCP fragment handling policy:
		 *    IP fragment offset == 0 && !(more frags); continue
		 *    IP fragment offset != 0 {
		 *        pass fragment to fragment processor
		 *        if first fragment already in {
		 *            pass sport, dport, and bytes
		 *        }
		 *        if first frag not in; get next frame
		 *    }
		 *       DM
		 *      011 0 0000 0000 0000 
		 *      001 1 1111 1111 1111
		 *          3    F    F    F
		 */

		if ((ntohs(ippacket->frag_off) & 0x3fff) != 0) {
		    br = processfragment(ippacket, &sport, &dport, &firstin);
		    if (!firstin)
			continue;
		} else {
		    sport = transpacket->source;
		    dport = transpacket->dest;
		    br = nbplen;
		}

		/*
		 * Apply TCP filter, if active.
		 */

		if ((filtered) &&
		    (!(utfilter(fl, ippacket->saddr, ippacket->daddr,
				ntohs(sport), ntohs(dport))))) {
		    show_ip_stats(statwin, iptotal, tcptotal, udptotal, icmptotal, noniptotal);
		    goto cont;
		}
		tcpentry = in_table(&table, ippacket->saddr, ippacket->daddr,
				    ntohs(sport), ntohs(dport), ifname,
				    &isclosed, tcptm, logging, logfile,
				    &nomem);

		/* 
		 * Add a new entry if it doesn't exist, and if it is a 
		 * SYN packet or any packet with more than just headers,
		 * and, to reduce the chances of stales, not a FIN.
		 */

		if ((ntohs(ippacket->frag_off) & 0x3fff) == 0) {	/* first frag only */
		    totalhlen = iphlen + transpacket->doff * 4;

		    if ((tcpentry == NULL) && (!(transpacket->fin)) &&
			((totalhlen < ntohs(ippacket->tot_len)) ||
			 ((totalhlen == ntohs(ippacket->tot_len)) &&
			  (transpacket->syn)))) {

			/*
			 * Ok, so we have a packet.  Add it if this connection
			 * is not yet closed, or if it is a SYN packet.
			 */

			if ((!nomem) && ((!isclosed) || (isclosed && transpacket->syn))) {
			    tcpentry = addentry(&table,
					 (unsigned long) ippacket->saddr,
					 (unsigned long) ippacket->daddr,
				sport, dport, ippacket->protocol, ifname,
						options->revlook, rvnfd,
					     options->servnames, &nomem);

			    if ((tcpentry != NULL) && (logging)) {
				sprintf(msgstring, "TCP: %s:%s to %s:%s first packet received ",
				     tcpentry->s_fqdn, tcpentry->s_sname,
				    tcpentry->d_fqdn, tcpentry->d_sname);

				if (transpacket->syn)
				    strcat(msgstring, "(SYN)");

				writelog(logging, logfile, msgstring);
			    }
			    if (tcpentry != NULL)
				printentry(&table, tcpentry->oth_connection,
					   screen_idx, mode);
			}
		    }
		}
		/* 
		 * If we had an addentry() success, we should have no
		 * problem here.  Same thing if we had a table lookup
		 * success.
		 */

		if (tcpentry != NULL) {

		    /* 
		     * Don't bother updating the entry if the connection
		     * has been previously reset.  (Does this really
		     * happen in practice?)
		     */

		    if (!(tcpentry->stat & FLAG_RST)) {
			if (options->revlook) {
			    p_sstat = tcpentry->s_fstat;
			    p_dstat = tcpentry->d_fstat;
			}
			updateentry(&table, tcpentry, transpacket, nbplen,
			br, ippacket->frag_off, logging, options->revlook,
				    rvnfd, logfile, &nomem);

			if ((options->revlook) && (((p_sstat != RESOLVED) && (tcpentry->s_fstat == RESOLVED)) ||
						   ((p_dstat != RESOLVED) && (tcpentry->d_fstat == RESOLVED)))) {
			    clearaddr(&table, tcpentry, screen_idx);
			    clearaddr(&table, tcpentry->oth_connection, screen_idx);
			}
			printentry(&table, tcpentry, screen_idx, mode);

			/*
			 * Special cases: Update other direction if it's
			 * an ACK in response to a FIN. 
			 *
			 *         -- or --
			 *
			 * Addresses were just resolved for the other
			 * direction, so we should also do so here.
			 */

			if (((tcpentry->oth_connection->finsent == 2) &&	/* FINed and ACKed */
			     (ntohl(transpacket->seq) == tcpentry->oth_connection->finack)) ||
			    ((options->revlook) &&
			     (((p_sstat != RESOLVED) && (tcpentry->s_fstat == RESOLVED)) ||
			      ((p_dstat != RESOLVED) && (tcpentry->d_fstat == RESOLVED)))))
			    printentry(&table, tcpentry->oth_connection, screen_idx, mode);
		    }
		}
	    } else {		/* now for the other IP protocols */
		if ((ntohs(ippacket->frag_off) & 0x1fff) != 0)
		    continue;	/* Just interpret first fragment */

		if (ippacket->protocol == IPPROTO_UDP)
		    udptotal += nbplen;
		else if (ippacket->protocol == IPPROTO_ICMP) {

		    /*
		     * Cancel the corresponding TCP entry if an ICMP
		     * Destination Unreachable or TTL Exceeded message
		     * is received.
		     */

		    if (((struct icmphdr *) transpacket)->type == ICMP_DEST_UNREACH)
			process_dest_unreach(&table, (char *) transpacket,
			                     ifname, &nomem);

		    icmptotal += nbplen;
		} else
		    othptotal += nbplen;

		/*
		 * Add and display non-TCP packet entry if memory is ok,
		 * and if filter permits
		 */

		if ((!nomem) && (othfilterok(ofilter, ippacket->protocol,
					ippacket->saddr, ippacket->daddr,
			  ntohs(((struct udphdr *) transpacket)->source),
			ntohs(((struct udphdr *) transpacket)->dest)))) {
		    othpent = add_othp_entry(&othptbl, &table,
					ippacket->saddr, ippacket->daddr,
					     IS_IP, ippacket->protocol, 0,
					     (char *) transpacket, ifname,
			options->revlook, rvnfd, tcptm, logging, logfile,
					     options->servnames, &nomem);
		}
	    }

	  cont:
	    show_ip_stats(statwin, iptotal, tcptotal, udptotal, icmptotal, noniptotal);
	}
	gettimeofday(&tv, NULL);
	now = tv.tv_sec;
	unow = tv.tv_sec * 1e+06 + tv.tv_usec;
	
	if (now - timeint >= 5) {
	    printelapsedtime(starttime, now, othptbl.obmaxy - 1, 15, othptbl.borderwin);
	    timeint = now;
	}
	if (((options->updrate != 0) && (now - updtime >= options->updrate)) ||
	   ((options->updrate == 0) && (unow - updtime_usec >= DEFAULT_UPDATE_DELAY))) {
	    update_panels();
	    doupdate();
	    updtime = now;
	    updtime_usec = unow;
	}
	
	if ((facilitytime != 0) && (((now - starttime) / 60) >= facilitytime))
	    endloop = 1;
    }

    killrvnamed();

    if (options->servnames)
	endservent();

    close_rvn_socket(rvnfd);

    del_panel(table.tcppanel);
    del_panel(table.borderpanel);
    del_panel(othptbl.othppanel);
    del_panel(othptbl.borderpanel);
    del_panel(statpanel);
    update_panels();
    doupdate();
    delwin(table.tcpscreen);
    delwin(table.borderwin);
    delwin(othptbl.othpwin);
    delwin(othptbl.borderwin);
    delwin(statwin);
    close(fd);
    destroytcptable(&table);
    destroyothptable(&othptbl);
    destroyfraglist();
    writelog(logging, logfile, "******** IP traffic monitor stopped\n");
    if (logfile != NULL)
	fclose(logfile);
	
    unlink(IPMONIDFILE);
    return;
}
