!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  process.S
!* Purpose: Process management functions for DOS simulator
!* Entries: dos00, dos31, dos4C, dos4D, loadprog
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>
#include "./dospriv.inc"


!
!**************************************************************************
!
! Data segment
!
		.data

psp_tmpl:				! PSP template
		.byte	$CD, $20	! code for INT20h
psp_endmem:	.word	0		! first allocatable seg after program
		.byte	0		! reserved
		.byte	$9A		! far call OP-code to INT21h handler
		.word	call21		! pointer to INT21h handler
		.word	HIGHMEM		! segment of INT21h handler
psp_int22:	.long	0		! saved interrupt 22h vector
psp_int23:	.long	0		! saved interrupt 23h vector
psp_int24:	.long	0		! saved interrupt 24h vector
psp_parent:	.word	0		! PSP of parent process
psp_handles:	.byte	$FF, $FF, $FF	! reserved file handles
		.byte	$FF, $FF
		.space	20 - 5		! file handle table
psp_env:	.word	0		! segment of program environment
psp_oldstack:	.long	0		! stack pointer of parent
psp_numhndl:	.word	20		! number of file handles
psp_ofshndl:	.word	o_psp_handles	! offset of file handle table
psp_seghndl:	.word	0		! segment of file handle table
		.space	24		! reserved
		.byte	$CD, $21, $CB	! OP-codes for INT21h and RETF
		.space	9		! reserved
psp_fcb1:	.space	16		! first part of FCB 1
psp_fcb2:	.space	16		! first part of FCB 2
		.space	4		! reserved
psp_cmdlsize:	.byte	0		! length of command line
psp_cmdl:	.byte	0		! command line (the first of 127 bytes)

psp_tmpl_size	equ	129		! size of PSP template


!
!**************************************************************************
!
! BSS segment
!
	.bss

	extrn	dos_active		! DOS active flag
	extrn	first_mcb		! segment of first MCB

	.lcomm	ret_code,2		! program return code
	.comm	curr_psp,2		! current PSP


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	dos00
	public	dos31
	public	dos4C		! define entry poimts
	public	dos4C
	public	dos4D
	public	loadprog

	extrn	dos48
	extrn	dos4A
	extrn	freeall
	extrn	checkmcb
	extrn	dosfatal
	extrn	call21


!
!**************************************************************************
!
! Terminate program.
! Input:  none
! Output: none (function does not return)
! Registers changed: all
!
dos00:

	les	bx,old_stack[bp]
	seg	es
	mov	dx,[bx + 2]		! load callers code segment
	mov	curr_psp,dx
	xor	al,al
	jmp	dos4C


!
!**************************************************************************
!
! Terminate current process, but keep it resident in memory.
! Input:  AL  -  return code
!         DX  -  number of paragraphs to keep in memory
! Output: none (function does not return)
! Registers changed: all
!
dos31:

	mov	ah,#3
	mov	ret_code,ax		! set return code
	mov	ax,curr_psp
	or	ax,ax			! is there a program running at all?
	jnz	dos311
	mov	ax,#FATAL_NOPROC	! nope, cant continue
	jmp	dos315

dos311:	mov	bx,dx
	mov	dx,ax
	dec	ax			! compute MCB address
	mov	es,ax
	call	checkmcb		! check if MCB is valid
	jc	dos313

dos312:	seg	es
	mov	word ptr [mcb_owner],#$FFFF	! this saves the block from
	call	freeall				! being freed
	jnc	dos314
dos313:	mov	ax,#FATAL_MEMORY		! unable to continue
dos315:	jmp	near dosfatal

dos314:	seg	es
	mov	word ptr [mcb_owner],dx		! restore PSP address in MCB
	mov	es,dx
	call	dos4A				! resize the memory block
	jc	dos313
	jmp	terminate


!
!**************************************************************************
!
! Terminate current program.
! Input:  AL  -  return code
! Output: none (function does not return)
! Registers changed: all
!
dos4C:

	xor	ah,ah
	mov	ret_code,ax		! set return code
	mov	ax,curr_psp
	or	ax,ax			! is there a program running at all?
	jnz	dos4C1
	mov	ax,#FATAL_NOPROC	! nope, cant continue
	jmp	dos4C4

dos4C1:	mov	dx,ax
	dec	ax			! compute MCB address
	mov	es,ax
	call	checkmcb		! check if MCB is valid
	jnc	dos4C2
dos4C3:	mov	ax,#FATAL_MEMORY	! unable to continue
dos4C4:	jmp	near dosfatal

dos4C2:	call	freeall			! free all memory used by this process
	jc	dos4C3			! at least one block (PSP) has to be
	jcxz	dos4C3			! freed
	jmp	terminate


!
!**************************************************************************
!
! Get exit code.
! Input:  none
! Output: AL  -  exit code
!         AH  -  exit cause
! Registers changed: AX
!
dos4D:

	mov	ax,ret_code
	ret


!
!**************************************************************************
!
! Terminate current process. This routine does the actual job, except
! memory management.
! Input:  none
! Output: none (routine does not return)
! Registers changed: all
!
terminate:

! Restore the stack of the parent process, reset the PSP of the
! parent process, reset the DOS active flag, and return to the parent
! calling address.

	cli
	mov	es,curr_psp
	seg	es
	mov	ax,[o_psp_parent]	! reset current PSP
	mov	curr_psp,ax
	mov	byte ptr dos_active,#0	! reset DOS active flag
	seg	es
	mov	ss,word ptr [o_psp_oldstack+2]
	seg	es
	mov	sp,word ptr [o_psp_oldstack+0]
	retf				! return to loadprog


!
!**************************************************************************
!
! Start a new process. This routine gets called by the DOS initialization
! code, which passes a pointer to process image and to the command line.
! Note that both have to reside outside of the memory space which is used
! and maintained by the DOS simulator. Also note, that the simulator can
! only handle COM-type programs, not EXE-type programs.
! The program image must begin with a magic cookie followed by its
! size in paragraphs.
! Input:  ES:DI  -  pointer to program code image
!         ES:SI  -  pointer to command line
! Output: none
! Registers changed: AX, BX, CX, DX
!
loadprog:

	cld
	push	es
	push	si
	push	di

! First check that the program code image has the correct magic cookie
! at its beginning.

	seg	es
	cmp	word ptr [di],#COM_MAGIC
	mov	ax,#FATAL_PROG
	jne	load9

! Next allocate 64 kB of memory for the program, or as much as possible
! if 64 kB are not available. Then check if the program will fit into
! that memory space together with the PSP and some safety margin for
! the stack.

	mov	bx,#$1000		! try to acquire 64 kB of memory
	call	dos48
	jnc	load1
	cmp	ax,#ERR_NOMEM
	jne	load8			! couldnt get that much, so try
	call	near dos48		! to get as much as possible
	jc	load8
load1:	sub	bx,#COM_MIN_FREE
	jbe	load7			! check if the program will fit into
	seg	es
	cmp	COM_SIZE_OFS[di],bx	! the reserved memory area
	jb	load2
load7:	mov	ax,#FATAL_NOMEM
	jmp	load9
load8:	mov	ax,#FATAL_MEMORY
load9:	jmp	near dosfatal

! Copy the program code into the new memory block

load2:	push	si
	push	ds
	mov	bx,es			! preserve original ES in BX
	mov	es,ax
	mov	ds,bx
	mov	si,di
	mov	ax,COM_SIZE_OFS[si]	! size is in paragraphs, convert it
	shift	(shl,ax,3)		! to number of words
	mov	cx,ax
	lea	si,COM_CODE_OFS[si]	! copy program code image into new
	mov	di,#PSP_SIZE		! place
	rep
	movsw

	shift	(shr,di,4)		! compute segment of first usable
	inc	di			! address after the program end
	mov	dx,es
	add	dx,di

! Copy the PSP template into the new memory block and setup with some
! miscellaneous values.

	xor	ax,ax
	xor	di,di
	mov	cx,#PSP_SIZE/2		! first clear the whole PSP area
	rep
	stosw

	pop	ds
	mov	psp_seghndl,es
	mov	ax,curr_psp		! setup PSP template
	mov	psp_parent,ax
	mov	psp_endmem,dx
	mov	ax,first_mcb		! the first MCB contains an empty
	inc	ax			! memory block which we can use as
	mov	psp_env,ax		! a dummy environment

	xor	di,di
	mov	si,#psp_tmpl
	mov	cx,#psp_tmpl_size
	rep
	movsb				! copy PSP template into memory area

! Next copy the command line into the PSP. The FCBs remain empty because we
! dont use FCBs anyway.

	pop	si
	push	ds
	mov	ds,bx
	or	si,si
	jz	load4
#ifdef IS386
	movzx	cx,byte ptr [si]	! save size of command line
#else
	mov	cl,[si]			! save size of command line
	xor	ch,ch
#endif
	jcxz	load4
	cmp	cx,#PSP_MAXCMDL
	jbe	load3
	mov	cx,#PSP_MAXCMDL
load3:	seg	es
	mov	[o_psp_cmdlsize],cl
	inc	si
	mov	di,#o_psp_cmdl
	rep
	movsb				! save command line
	mov	al,#$0D			! terminate command line with CR
	stosb
	xor	al,al			! and with a zero byte
	stosb
load4:	pop	ds

! Now prepare the stack for the return from the called program. This involves
! pushing all relevant registers and a far return address.

	cli
	push	bp
	push	ds
	push	cs
#ifdef IS186
	push	#loadr			! push return address
#else
	mov	ax,#loadr
	push	ax			! push return address
#endif
	seg	es
	mov	word ptr [o_psp_oldstack + 0],sp
	seg	es
	mov	word ptr [o_psp_oldstack + 2],ss

! The PSP has been setup. Now set the owner of the programs memory
! block and switch the stack to the new program.

	mov	ax,es
	mov	curr_psp,ax		! save new PSP pointer
	dec	ax
	mov	ds,ax			! get size of memory block from
	mov	ax,[mcb_size]		! MCB
	mov	[mcb_owner],es		! set owner of memory block
	shift	(shl,ax,4)		! convert paragraph size into byte
	dec	ax			! offset for top of stack
	dec	ax
	mov	dx,es
	mov	ss,dx			! set new stack
	mov	sp,ax
	sti

! Finally make the new program current, and call it with the return pointer
! set correctly on the new stack.

	mov	ds,dx			! set data segment register correctly
	xor	ax,ax
	push	ax			! save near pointer to terminate fctn
	push	es
#ifdef IS186
	push	#PSP_SIZE
#else
	mov	ax,#PSP_SIZE		! save pointer to the new programs
	push	ax			! entry point
#endif
	retf				! call the new program

! The called program will return here. Simply pop all registers from the
! stack and return to the DOS initialization routine.

loadr:	pop	ds
	pop	bp
	pop	di
	pop	si
	pop	es
	sti
	ret


!
!**************************************************************************
!
	end

