/*
 * openrd.c  -  Open ramdisk image file
 *
 * Copyright (C) 1996-1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#include "nblib.h"
#include "mknbi.h"

#ifndef _MKNBI_H_DOS_
#error Included wrong header file
#endif

#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>



/*
 * List of possible output file formats
 */
static struct floppy {
	unsigned int size;		/* size of floppy in kB */
	unsigned int diskid;		/* disk ID byte */
	unsigned int clustsize;		/* size of one cluster in bytes */
	unsigned int maxcluster;	/* maximum cluster number */
	unsigned int fatsects;		/* number of sector per FAT */
	unsigned int rootentries;	/* number of entries in root dir */
	unsigned int spt;		/* sectors per track */
} floppylist[] = {
	/* This list has to be organized from lowest to highest size */
	{  320,	0xFF,	SECTSIZE*2,	 315,	 1,	112,	 8 },
	{  360, 0xFD,	SECTSIZE*2,	 354,	 2,	112,	 9 },
	{  720, 0xF9,	SECTSIZE*2,	 713,	 3,	112,	 9 },
	{ 1200, 0xF9,	SECTSIZE,	2371,	 7,	224,	15 },
	{ 1440, 0xF0,	SECTSIZE,	2847,	 9,	224,	18 },
	{ 2880,	0xF0,	SECTSIZE*2,	2863,	 9,	224,	36 },
	{    0,	   0,	       0,	   0,	 0,	  0,	 0 }
};

static struct floppy hdlist[] = {
	/*
	 * These values are the max available when simulating a hard disk. They
	 * have to be adjusted before the actual image is constructed. Only
	 * those values can be used which allow the construction of a 12 bit
	 * FAT - 16 bit FATs are not supported. The maximum cluster number
	 * is set to allow for approx. 10% free space in ramdisk.
	 */
	{ 8136,	0xF8,	SECTSIZE*4,	3648,	12,	512,	36 },
	{    0,	   0,	       0,	   0,	 0,	  0,	 0 }
};

static struct floppy *curformat;

#define DEFAULT_FORMAT	4		/* Default format if none specified */
#define MAX_CLUST_SIZE	SECTSIZE*4	/* Maximum cluster size in table */



/***************************************************************************

			Routines to handle directories

 ***************************************************************************/


/* Structure to hold file information from directory */
struct file_struct {
	char                 name[8];	/* file name */
	char                 ext[3];	/* extension name */
	char                 attrib;	/* attributes */
	unsigned short       cluster;	/* number of start cluster */
	unsigned short       clustnum;	/* number of clusters */
	unsigned long        size;	/* file size in bytes */
	struct file_struct  *next;	/* pointer to next file in dir */
	char                 path[1];	/* path name on server */
};


/* Structure to hold information about subdirectories */
struct dir_struct {
	char                 name[8];	/* directory name */
	char                 ext[3];	/* extension name */
	char                 attrib;	/* attributes */
	unsigned short       cluster;	/* number of start cluster */
	unsigned short       clustnum;	/* number of clusters */
	int                  subdirnum;	/* number of subdirectories */
	int                  filenum;	/* number of files in directory */
	struct dir_struct   *subdirs;	/* pointer to first subdir */
	struct file_struct  *files;	/* pointer to first file */
	struct dir_struct   *next;	/* pointer to next dir in cur subdir */
};


static struct dir_struct *root_dir = NULL;	/* root directory node */
static struct file_struct *io_sys = NULL;	/* pointer to IO.SYS file */



/*
 * Convert UNIX path name into DOS file name
 */
static void cvtname(path, name, ext)
char *path;
char *name;
char *ext;
{
  char *cp;
  int i;

  /* Isolate file name */
  if ((cp = strrchr(path, '/')) == NULL)
	cp = path;
  else
	cp++;

  /* Copy file name (max. 8 chars) */
  i = 0;
  while (*cp && *cp != '.') {
	if (i < 8) name[i++] = toupper(*cp);
	cp++;
  }
  while (i < 8)
	name[i++] = ' ';
  if (*cp == '.')
	cp++;

  /* Copy file extension (max. 3 chars) */
  i = 0;
  while (*cp) {
	if (i < 3) ext[i++] = toupper(*cp);
	cp++;
  }
  while (i < 3)
	ext[i++] = ' ';
}



/*
 * Find a file in the current directory
 */
static struct file_struct *findfile(dsp, name, ext)
struct dir_struct *dsp;
char *name;
char *ext;
{
  struct file_struct *fsp = dsp->files;

  while (fsp != NULL) {
	if (!strncmp(name, fsp->name, 8) && !strncmp(ext, fsp->ext, 3))
		return(fsp);
	fsp = fsp->next;
  }
  return(NULL);
}



/*
 * Read in the complete directory structure for the ramdisk image.
 */
static struct dir_struct *rddir(path)
char *path;
{
  DIR *dirp;
  char *cp;
  char tmppath[MAXNAMLEN + 1];
  struct stat sbuf;
  struct dirent *ent;
  struct dir_struct *dsp;
  struct dir_struct *tmpdsp;
  struct file_struct *fsp;

  /* Initialize directory structure */
  dsp = (struct dir_struct *)nbmalloc(sizeof(struct dir_struct));
  cvtname(path, dsp->name, dsp->ext);
  dsp->attrib = ATTR_DIR;
  dsp->subdirnum = 0;
  dsp->filenum = 0;
  dsp->subdirs = NULL;
  dsp->files = NULL;
  dsp->next = NULL;

  /* Open directory */
  if ((dirp = opendir(path)) == NULL) {
	perror(path);
	exit(EXIT_OPENDIR);
  }

  /* Read in whole directory */
  while ((ent = readdir(dirp)) != NULL) {
	if (!strcmp(ent->d_name, ".") || !strcmp(ent->d_name, ".."))
		continue;
	sprintf(tmppath, "%s/%s", path, ent->d_name);
	if (stat(tmppath, &sbuf) != 0) {
		perror(tmppath);
		exit(EXIT_STAT);
	}
	if ((sbuf.st_mode & S_IFMT) == S_IFREG) {
		fsp = (struct file_struct *)nbmalloc
				(sizeof(struct file_struct) + strlen(tmppath));
		fsp->attrib = 0;
		cp = ent->d_name;
		if (*cp == '.') {
			fsp->attrib |= ATTR_HIDDEN;
			cp++;
		}
		cvtname(cp, fsp->name, fsp->ext);
		if (findfile(dsp, fsp->name, fsp->ext) != NULL) {
			fprintf(stderr, "%s: more than one file %s [%s.%s]\n",
				progname, tmppath, fsp->name, fsp->ext);
			exit(EXIT_DOS_DOUBLEFILE);
		}
		if (!(sbuf.st_mode & S_IWRITE))
			fsp->attrib |= ATTR_READONLY;
		if (!strncmp(fsp->ext, "SYS", 3) &&
		    (!strncmp(fsp->name, "IO      ", 8) ||
		     !strncmp(fsp->name, "MSDOS   ", 8)))
			fsp->attrib |= ATTR_HIDDEN + ATTR_SYSTEM + ATTR_READONLY;
		if (!strncmp(fsp->ext, "COM", 3) &&
		    (!strncmp(fsp->name, "IBMBIO  ", 8) ||
		     !strncmp(fsp->name, "IBMDOS  ", 8)))
			fsp->attrib |= ATTR_HIDDEN + ATTR_SYSTEM + ATTR_READONLY;
		strcpy(fsp->path, tmppath);
		fsp->size = (unsigned long)sbuf.st_size;
		fsp->next = dsp->files;
		dsp->files = fsp;
		dsp->filenum++;
	} else if ((sbuf.st_mode & S_IFMT) == S_IFDIR) {
		tmpdsp = rddir(tmppath);
		tmpdsp->next = dsp->subdirs;
		dsp->subdirs = tmpdsp;
		dsp->subdirnum++;
	}
  }

  /* Close directory and return */
  closedir(dirp);
  return(dsp);
}



/*
 * Move the given file into the first position in the file list. This
 * is required for the MS-DOS system files.
 */
static void movefirst(dsp, fsp)
struct dir_struct *dsp;
struct file_struct *fsp;
{
  struct file_struct *cur = dsp->files;
  struct file_struct *prev = NULL;

  while (cur != NULL && cur != fsp) {
	prev = cur;
	cur = cur->next;
  }
  if (cur == fsp && prev != NULL) {
	prev->next = fsp->next;
	fsp->next = dsp->files;
	dsp->files = fsp;
  }
}



/*
 * Read in the root directory tree
 */
static void readroot(path)
char *path;
{
  struct file_struct *fsp;

  if (verbose > 0)
	printf("Scanning directory tree\n");

  /* Read the root directory recursively */
  root_dir = rddir(path);

  /* Check that the MSDOS system files are present */
  if ((fsp = findfile(root_dir, "MSDOS   ", "SYS")) == NULL &&
      (fsp = findfile(root_dir, "IBMDOS  ", "COM")) == NULL) {
	fprintf(stderr, "%s: Cant find msdos.sys or ibmdos.com in %s\n", progname, path);
	exit(EXIT_DOS_MSDOSSYS);
  }
  movefirst(root_dir, fsp);
  if ((fsp = findfile(root_dir, "IO      ", "SYS")) == NULL &&
      (fsp = findfile(root_dir, "IBMBIO  ", "COM")) == NULL) {
	fprintf(stderr, "%s: Cant find io.sys or ibmbio.com in %s\n", progname, path);
	exit(EXIT_DOS_IOSYS);
  }
  movefirst(root_dir, fsp);

  /* Set pointer to IO.SYS for later reference */
  io_sys = fsp;
}





/***************************************************************************

			Routines to generate the FAT

 ***************************************************************************/


static unsigned short cur_cluster = 2;	/* current cluster number */
static unsigned char *fat_buf;		/* FAT buffer */



/*
 * Generate 12 bit FAT entries for a given cluster range
 */
static void genclusters(start, length)
unsigned short start;
unsigned short length;
{
  unsigned short cur = start;
  unsigned short value;
  int i;

  while (cur < (start + length)) {
	if (cur > curformat->maxcluster) {
		fprintf(stderr, "%s: not enough space on ramdisk\n", progname);
		exit(EXIT_DOS_RDSPACE);
	}
	value = (cur == (start + length - 1)) ? 0xfff : cur + 1;
	i = (cur * 3) / 2;
	if ((cur * 3) % 2 == 0) {
		fat_buf[i]    = (unsigned char)(value & 0x0ff);
		fat_buf[i+1] |= (unsigned char)((value & 0xf00) >> 8);
	} else {
		fat_buf[i]   |= (unsigned char)((value & 0x00f) << 4);
		fat_buf[i+1]  = (unsigned char)((value & 0xff0) >> 4);
	}
	cur++;
  }
}



/*
 * Scan through the directory tree to generate the FAT
 */
static void genfatdir(dsp)
struct dir_struct *dsp;
{
  struct file_struct *fsp;

  /* First compute the number of clusters for current directory */
  if (dsp != root_dir) {
	dsp->clustnum = ((dsp->filenum + dsp->subdirnum + 2) *
			sizeof(struct dos_dir) + curformat->clustsize - 1) /
			curformat->clustsize;
	dsp->cluster = cur_cluster;
	cur_cluster += dsp->clustnum;
	genclusters(dsp->cluster, dsp->clustnum);
  } else {
	dsp->clustnum = (curformat->rootentries * sizeof(struct dos_dir) +
			(curformat->clustsize - 1)) / curformat->clustsize;
	dsp->cluster = 0;
  }

  /* Determine the number of clusters for each file in current directory */
  fsp = dsp->files;
  while (fsp != NULL) {
	fsp->clustnum = (fsp->size + curformat->clustsize - 1) /
							curformat->clustsize;
	fsp->cluster = cur_cluster;
	cur_cluster += fsp->clustnum;
	genclusters(fsp->cluster, fsp->clustnum);
	fsp = fsp->next;
  }

  /* Scan through all subdirectories recursively */
  dsp = dsp->subdirs;
  while (dsp != NULL) {
	genfatdir(dsp);
	dsp = dsp->next;
  }
}



/*
 * Generate the FAT for the files and directories in the tree.
 */
static void genfat(rdsize)
int rdsize;
{
  register unsigned int fatbytes = curformat->fatsects * SECTSIZE;
  struct floppy *newformat = NULL;
  unsigned int rootsects;
  unsigned int fatsects;
  unsigned int cylinders;
  unsigned long minrdsize;
  unsigned long l, m;

  if (verbose > 0)
	printf("Generating FATs\n");

  /* Setup FAT buffer */
  fat_buf = (unsigned char *)nbmalloc(fatbytes);
  fat_buf[0] = (unsigned char)(curformat->diskid & 0xff);
  fat_buf[1] = 0xff;					/* filler byte */
  fat_buf[2] = 0xff;

  /* Recursively scan through all directories */
  genfatdir(root_dir);

  /*
   * When generating a hard disk image we have to adjust the parameters
   * in the format table.
   */
  if (curformat == &hdlist[0]) {
	if (verbose > 0)
		printf("Adjusting image parameters\n");
	newformat = (struct floppy *)nbmalloc(sizeof(struct floppy));
	memcpy(newformat, curformat, sizeof(struct floppy));

	/*
	 * Determine minimum ramdisk size. The actual size will be deter-
	 * mined lateron, when the FAT gets rewritten in the main module.
	 */
	if (rdsize != 0)
		minrdsize = MIN_RDSIZE * (1024 / SECTSIZE);
	else
		minrdsize = floppylist[DEFAULT_FORMAT].size * (1024 / SECTSIZE);

	/*
	 * Determine new size of ramdisk image in sectors and add about
	 * 10% free space.
	 */
	rootsects = curformat->rootentries * sizeof(struct dos_dir);
	rootsects = (rootsects + SECTSIZE - 1) / SECTSIZE;
	l = ((cur_cluster * 10) / 9 - 2) * (curformat->clustsize / SECTSIZE);
	l += curformat->fatsects * 2 + rootsects + 1;
	cylinders = (l + curformat->spt * 2 - 1) / (curformat->spt * 2);
	if (cylinders > MAX_CYLS)
		cylinders = MAX_CYLS;
	l = cylinders * curformat->spt * 2;
	if (l < minrdsize)
		l = minrdsize;
	newformat->size = l / (1024 / SECTSIZE);

	/* Determine the number of sectors per FAT and maximum cluster number */
	l -= rootsects + 1;
	fatsects = 0;
	do {
		fatsects++;
		m = ((fatsects * SECTSIZE * 8) / 12 - 2) *
				(curformat->clustsize / SECTSIZE);
		m += fatsects * 2;
	} while (m < l);
	if (fatsects > curformat->fatsects) {
		fprintf(stderr, "%s: internal error: invalid number of sectors "
							"in FAT\n", progname);
		exit(EXIT_INTERNAL);
	}
	l -= fatsects * 2;
	newformat->fatsects = fatsects;
	newformat->maxcluster = (l + 2) / (curformat->clustsize / SECTSIZE);

	/* Make the new format active */
	curformat = newformat;
	if (verbose > 2) {
		printf("New image values:\n");
		printf("  Total size in kB:            %d\n", newformat->size);
		printf("  Cluster size in Bytes:       %d\n", newformat->clustsize);
		printf("  Maximum cluster number:      %d\n", newformat->maxcluster);
		printf("  Number of FAT sectors:       %d\n", newformat->fatsects);
		printf("  Number of sectors per track: %d\n", newformat->spt);
	}
  }
}





/***************************************************************************

		Write directory and files into output file

 ***************************************************************************/


/*
 * Get local time and convert it into MS-DOS format
 */
static void gettime(dirp)
struct dos_dir *dirp;
{
  struct tm *tmp;
  time_t cur_time;

  cur_time = time(NULL);
  tmp = localtime(&cur_time);
  assign(dirp->time, htot(tmp->tm_hour * 2048 + tmp->tm_min * 32 + tmp->tm_sec / 2));
  assign(dirp->date, htot((tmp->tm_year - 80) * 512 + (tmp->tm_mon + 1) * 32 + tmp->tm_mday));
}



/*
 * Write a DOS file into the ramdisk image file.
 */
static void writefile(handle, fsp)
int handle;
struct file_struct *fsp;
{
  static unsigned char buf[MAX_CLUST_SIZE];
  unsigned short num;
  int eof = 0;
  int readsize;
  int infile;

  if ((infile = open(fsp->path, O_RDONLY | O_BINARY)) < 0) {
	perror(fsp->path);
	exit(EXIT_OPEN);
  }

  for (num = 0; num < fsp->clustnum; num++) {
	memset(buf, 0, curformat->clustsize);
	if (!eof) {
		readsize = nbread(buf, curformat->clustsize, infile);
		eof = (readsize < curformat->clustsize);
	}
	(void)nbwrite(buf, curformat->clustsize, handle);
  }
  close(infile);
}



/*
 * Recursively write each subdirectory
 */
static void writedir(handle, dsp, parent)
int handle;
struct dir_struct *dsp;
struct dir_struct *parent;
{
  struct dos_dir *dirp;
  struct dir_struct *subdsp;
  struct file_struct *fsp;
  unsigned char *buf;
  size_t size;

  /* Allocate memory for directory */
  if (parent == NULL)
	size = curformat->rootentries * sizeof(struct dos_dir);
  else
	size = dsp->clustnum * curformat->clustsize;
  buf = (unsigned char *)nbmalloc(size);

  /* Generate the entries for current and parent directories */
  dirp = (struct dos_dir *)buf;
  if (parent != NULL) {
	memcpy(dirp->name, ".       ", sizeof(dirp->name));
	memcpy(dirp->ext, "   ", sizeof(dirp->ext));
	gettime(dirp);
	dirp->attrib = ATTR_DIR;
	assign(dirp->cluster, htot(dsp->cluster));
	assign(dirp->size.low, htot(0));
	assign(dirp->size.high, htot(0));
	dirp++;
	memcpy(dirp->name, "..      ", sizeof(dirp->name));
	memcpy(dirp->ext, "   ", sizeof(dirp->ext));
	gettime(dirp);
	dirp->attrib = ATTR_DIR;
	assign(dirp->cluster, htot(parent->cluster));
	assign(dirp->size.low, htot(0));
	assign(dirp->size.high, htot(0));
	dirp++;
  }

  /* Write the file entries into directory. The files have to come first! */
  fsp = dsp->files;
  while (fsp != NULL) {
	memcpy(dirp->name, fsp->name, sizeof(dirp->name));
	memcpy(dirp->ext, fsp->ext, sizeof(dirp->ext));
	gettime(dirp);
	dirp->attrib = fsp->attrib;
	assign(dirp->cluster, htot(fsp->cluster));
	assign(dirp->size.low, htot(low_word(fsp->size)));
	assign(dirp->size.high, htot(high_word(fsp->size)));
	dirp++;
	fsp = fsp->next;
  }

  /* Write the subdir entries into directory */
  subdsp = dsp->subdirs;
  while (subdsp != NULL) {
	memcpy(dirp->name, subdsp->name, sizeof(dirp->name));
	memcpy(dirp->ext, subdsp->ext, sizeof(dirp->ext));
	gettime(dirp);
	dirp->attrib = subdsp->attrib;
	assign(dirp->cluster, htot(subdsp->cluster));
	assign(dirp->size.low, htot(0));
	assign(dirp->size.high, htot(0));
	dirp++;
	subdsp = subdsp->next;
  }

  /* Write the directory into output file and free directory memory */
  (void)nbwrite(buf, size, handle);
  free(buf);

  /* Write all files in the same order as setup in the FAT */
  fsp = dsp->files;
  while (fsp != NULL) {
	writefile(handle, fsp);
	fsp = fsp->next;
  }

  /* Finally write all subdirectories into the output file */
  subdsp = dsp->subdirs;
  while (subdsp != NULL) {
	writedir(handle, subdsp, dsp);
	subdsp = subdsp->next;
  }
}





/***************************************************************************

		Main routine to open/create ramdisk image file

 ***************************************************************************/


/*
 * Open ram disk image file. If it's not a file but a directory, generate
 * an image file using this directory.
 */
int openrd(name, rdsize)
char *name;
int rdsize;
{
  struct boot_record *bp;
  struct stat sbuf;
  FILE *tmprd;
  int rdfile, i;

  /*
   * First check if the given file name is valid, and specifies either
   * a regular file or a directory. In the first case we can just open
   * the file.
   */
  if (stat(name, &sbuf) != 0) {
	perror(name);
	exit(EXIT_DOS_RDSTAT);
  }
  if (S_ISREG(sbuf.st_mode) || S_ISBLK(sbuf.st_mode)) {
	if ((rdfile = open(name, O_RDONLY | O_BINARY)) < 0) {
		perror(name);
		exit(EXIT_DOS_RDOPEN);
	}
	return(rdfile);
  } else if (!S_ISDIR(sbuf.st_mode)) {
	fprintf(stderr, "%s: not a file or directory\n", name);
	exit(EXIT_DOS_INVRD);
  }
  if (verbose > 0)
	printf("Using directory %s to create ramdisk image\n", name);

  /*
   * Determine the size of the image in the temporary file. For this, select
   * the next lowest available size.
   */
  curformat = &floppylist[DEFAULT_FORMAT];
  if (usehd) {
	curformat = &hdlist[0];
	if (verbose > 0)
		printf("Generating HD image from directory\n");
  } else if (rdsize > 0) {
	curformat = NULL;
	for (i = 0; floppylist[i].size != 0; i++)
		if (floppylist[i].size == rdsize) {
			curformat = &floppylist[i];
			break;
		}
	if (curformat == NULL) {
		fprintf(stderr, "%s: Invalid ramdisk size given\n", progname);
		exit(EXIT_DOS_RDSIZE);
	}
	if (verbose > 0)
		printf("Generating %d kB image from directory\n",
							curformat->size);
  }

  /*
   * Read the root directory, setup the directory tree and generate the FAT.
   */
  readroot(name);
  genfat(rdsize);

  /*
   * Create a temporary file to receive the ramdisk image.
   */
  if ((tmprd = tmpfile()) == NULL) {
	fprintf(stderr, "%s: cannot create temporary file\n", progname);
	exit(EXIT_CREATE);
  }
  rdfile = fileno(tmprd);

  /*
   * Setup the floppy boot sector. All values not set here have to be
   * preset in the file boot.S!
   */
  bp = (struct boot_record *)boot_data;
  assign(bp->bytes_per_sect, htot(SECTSIZE));
  assign(bp->dir_num, htot(curformat->rootentries));
  assign(bp->sect_num, htot(curformat->size * (1024 / SECTSIZE)));
  assign(bp->sect_per_fat, htot(curformat->fatsects));
  assign(bp->sect_per_track, htot(curformat->spt));
  bp->sect_per_cluster = (__u8)((curformat->clustsize / SECTSIZE) & 0xff);
  bp->media_id = (__u8)(curformat->diskid);

  /*
   * Set the number of sectors occupied by IO.SYS. This is needed to load
   * the whole IO.SYS which is necessary for some DOS versions like OpenDOS.
   */
  if (io_sys != NULL) {
	struct priv_boot *pbp;
	unsigned int sects;

	pbp = (struct priv_boot *)boot_data;
	sects = (io_sys->size + SECTSIZE - 1) / SECTSIZE;
	if (sects > MAX_IOSYS)
		sects = MAX_IOSYS;
	assign(pbp->io_sys_sects, htot(sects));
  }

  /*
   * Write the floppy boot sector, both FATs and the directory trees and
   * files.
   */
  if (verbose > 0)
	printf("Writing temporary ramdisk image file\n");
  if (boot_data_size != SECTSIZE) {
	fprintf(stderr, "%s: invalid floppy boot sector\n", progname);
	exit(EXIT_DOS_BOOTSECT);
  }
  (void)nbwrite(boot_data, boot_data_size, rdfile);
  (void)nbwrite(fat_buf, curformat->fatsects * SECTSIZE, rdfile);
  (void)nbwrite(fat_buf, curformat->fatsects * SECTSIZE, rdfile);
  writedir(rdfile, root_dir, NULL);

  /*
   * Rewind output file and return its file handle.
   */
  lseek(rdfile, 0L, 0);
  return(rdfile);
}

