/*
  $Id: lio.c,v 1.9 1996/08/17 13:47:47 luik Exp $

  lio.c - line input/output routines for omirrd.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: lio.c,v 1.9 1996/08/17 13:47:47 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <stdlib.h>
#include <string.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include "lio.h"


LioContext lioAlloc(void)
{
    LioContext context;
    if ((context = calloc(sizeof(LioContextRec), 1))) {
    }
    return context;
}

void lioFree(LioContext context)
{
    free(context);
}

LioContext lioAddReadCallback(LioContext context, int fd,
			      LioCallbackFunc callback, void *data)
{
    LioCallback cb;

    if (fd >= FD_SETSIZE || (cb = &context->readcbs[fd])->callback)
	return NULL /* XXX ERROR */;

    cb->callback = callback;
    cb->data = data;
    cb->buf_size = LIO_BUF_SIZE;
    cb->buf = malloc(cb->buf_size); /* XXX ERROR */
    cb->buf_start = 0;
    cb->buf_end = 0;
    return context;
}

LioContext lioRemoveReadCallback(LioContext context, int fd)
{
    LioCallback cb;

    if (fd >= FD_SETSIZE)
	return NULL /* XXX ERROR */;

    cb = &context->readcbs[fd];
    cb->callback = NULL;
    cb->data = NULL;
    if (cb->buf)
	free (cb->buf);
    cb->buf = NULL;
    return context;
}


/* lioMpReadFunc - read callback function for mp module. Called if
   data is available from fd, splits data into lines and calls
   associated lio read callback. Returns the number of bytes read or
   -1 on error. Note that EOF is treated as a error condition
   returning -1 with errno == 0.  */
int lioMpReadFunc(int fd, void *data)
{
    LioContext context = (LioContext) data;
    LioCallback cb;
    char *line_end;		/* pointer to end of line (NUL terminated) */
    int result;

    if (fd >= FD_SETSIZE || (cb = &context->readcbs[fd])->callback == NULL)
	return (errno = EBADF, -1);

    /* read into buf, split into lines at NUL bytes */
    result = read(fd, cb->buf + cb->buf_end, cb->buf_size - cb->buf_end);
    if (result > 0) {
	cb->buf_end += result;
	while ((line_end = memchr(cb->buf + cb->buf_start, '\0',
				  cb->buf_end - cb->buf_start))) {
	    (*cb->callback)(cb->buf + cb->buf_start, cb->data);
	    cb->buf_start = line_end - cb->buf + 1;
	}

	memmove(cb->buf, cb->buf + cb->buf_start, cb->buf_end - cb->buf_start);
	cb->buf_end -= cb->buf_start;
	cb->buf_start -= cb->buf_start;	/* cb->buf_start = 0 */

	/* If buffer is still more than 80% full, a long line needs to
           be processed, therefore increase buffer size.  */
	if (cb->buf_end > cb->buf_size / 10 * 8) {
	    cb->buf_size *= 2;
	    cb->buf = realloc(cb->buf, cb->buf_size); /* XXX ERROR */
	}

	/* If buffer is nearly empty (less than 10% used) and its size
           is greater or equal than 4 * LIO_BUF_SIZE, shrink buffer
           again.  */
	if ((cb->buf_end < cb->buf_size / 10
	     && cb->buf_size >= LIO_BUF_SIZE * 4)) {
	    cb->buf_size /= 2;
	    cb->buf = realloc(cb->buf, cb->buf_size); /* XXX ERROR */
	}
    }
    else if (result == 0) {	/* handle end-of-file */
	if (cb->buf_start != cb->buf_end) { /* flush pending data */
	    cb->buf[cb->buf_end] = '\0';
	    (*cb->callback)(cb->buf + cb->buf_start, cb->data);
	}
	cb->buf_start = cb->buf_end = 0;
	result = -1;		/* treat end-of-file as error */
	errno = 0;
    }
    else if (result == -1 && errno == EINTR) {
	result = 0;		/* interrupted system call is not an error */
    }

    return (result);
}

