/*
 * linux_vars.c --
 *
 * This file contains the implementation of Linux specific SNMP variables.
 *
 * Copyright (c) 1996-1998
 *
 * Juergen Schoenwaelder	University of Twente, The Netherlands
 *				TU Braunschweig, Germany
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

#include "mib_module.h"

#ifdef HAVE_LINUX

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>


struct variable linux_variables[] = {
    {LINUXCPU, STRING, RONLY, var_linux, 1, {1}},
    {LINUXBOGO, GAUGE, RONLY, var_linux, 1, {2}},
    {LINUXLOAD1, INTEGER, RONLY, var_linux, 1, {3}},
    {LINUXLOAD5, INTEGER, RONLY, var_linux, 1, {4}},
    {LINUXLOAD15, INTEGER, RONLY, var_linux, 1, {5}}
};

static oid linux_base [] = { 1, 3, 6, 1, 4, 1, 1575, 1, 5, 2 };

void
linux_init()
{
    mib_register(linux_base, 10, (struct variable *)linux_variables,
		 sizeof(linux_variables)/sizeof(*linux_variables),
		 sizeof(*linux_variables));
}

/*
 * Get information about the CPUs from the proc filesystem.
 */

static void
get_cpu_info(cpu, bogo)
    char cpu[];
    long *bogo;
{
    char line[256], key[256], val[256];
    FILE *in;

    *bogo = 0, cpu[0] = 0;
    in = fopen("/proc/cpuinfo", "r");
    if (! in) return;

    while (fgets(line, sizeof(line), in)) {
	if (2 == sscanf(line, "%[^:]: %[^\n]", key, val)) {
	  /* the 2.0.x and 2.1.x /proc/cpuinfo format differs; this 
	     should give a useable string anyway: */
	    if (strncmp(key, "cpu\t\t", 5) == 0
		|| strncmp(key, "model", 5) == 0
		|| strncmp(key, "vendor_id", 9) == 0) {
		if (strlen(cpu) + strlen(val) + 1 < 255) {
		  if (cpu[0]) { strcat(cpu, " "); }
		    strcat(cpu, val);
		}
	    } else if (strncmp(key, "bogomips", 8) == 0) {
		*bogo = atoi(val);
	    }
	}
    }
    fclose(in);
}

/*
 * Get the load averages for 1, 5 and 15 minutes from the 
 * /proc filesystem.
 */

static void
get_load_avg(avg1, avg5, avg15)
    unsigned *avg1;
    unsigned *avg5;
    unsigned *avg15;
{
    float f1, f5, f15;
    FILE *in;

    *avg1 = 0, *avg5 = 0, *avg15 = 0;
    in = fopen("/proc/loadavg", "r");
    if (! in) return;

    if (3 == fscanf(in, "%f %f %f", &f1, &f5, &f15)) {
	*avg1  = (int) (f1 * 100);
	*avg5  = (int) (f5 * 100);
	*avg15 = (int) (f15 * 100);
    }
    fclose(in);
}

/*
 * Entry for the linux mib:
 */

u_char *
var_linux(vp, name, length, exact, var_len, write_method)
     struct variable *vp;    /* IN - ptr to variable entry that points here */
     oid     *name;	     /* IN/OUT - input name req, output name found */
     int     *length;	     /* IN/OUT - length of input and output oid's */
     int     exact;	     /* IN - TRUE if an exact match was requested. */
     int     *var_len;	     /* OUT - length of var or 0 if function ret. */
     int     (**write_method)();   /* OUT - ptr to func to set var, else 0 */
{
    oid newname[MAX_NAME_LEN];
    int result;
    static char cpu[256];
    static long bogo = 0;
    static unsigned avg1, avg5, avg15;

    /* nothing writable provided: */
    *write_method = 0;

    /* default return type: */
    *var_len = sizeof(long);

    bcopy((char *)vp->name, (char *)newname, vp->namelen * sizeof(oid));
    newname[vp->namelen] = 0;
    result = compare(name, *length, newname, (int)vp->namelen + 1);
    if ((exact && (result != 0)) || (!exact && (result >= 0)))
	return NULL;
    bcopy((char *)newname, (char *)name, (vp->namelen + 1) * sizeof(oid));
    *length = vp->namelen + 1;
    
    if (! bogo) {
	get_cpu_info(cpu, &bogo);
    }

    switch (vp->magic) {
    case LINUXCPU:
	*var_len = strlen(cpu);
	return cpu;
	break;
    case LINUXBOGO:
	return (u_char *) &bogo;
	break;
    case LINUXLOAD1:
	get_load_avg(&avg1, &avg5, &avg15);
	return (u_char *) &avg1;
	break;
    case LINUXLOAD5:
	get_load_avg(&avg1, &avg5, &avg15);
	return (u_char *) &avg5;
	break;
    case LINUXLOAD15:
	get_load_avg(&avg1, &avg5, &avg15);
	return (u_char *) &avg15;
	break;
    }

    return NULL;
}

#endif
