/*
Copyright (c) 1998 Peter Zelezny.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "xchat.h"
#include <pwd.h>
#include <fcntl.h>
#include "serverlist.h"

extern struct xchatprefs prefs;
extern struct serverentry firstserverentry;

extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern struct session *new_session(struct server *serv, char *channel);
extern struct server *new_server(void);
extern void gui_change_nick(struct server *serv, char *newnick);
extern void check_prefs_dir(void);
extern char *get_xdir(void);


GtkWidget *slwin = 0;
GtkWidget *sllist;
GtkWidget *entry_server;
GtkWidget *entry_channel;
GtkWidget *entry_port;
GtkWidget *entry_comment;
GtkWidget *entry_password;
long selected = -1;

struct serverentry *new_serverentry(void);
gint close_server_list(struct session *);
void add_defaults(void);
void add_all_server_entries(void);


void load_serverentrys(void)
{
   int fh; 
   char file[84];
   
   sprintf(file, "%s/serverlist", get_xdir());
   fh = open(file, O_RDONLY);
   if(fh != -1)
   {  
      struct serverentry serv;
      struct serverentry *last = 0;
      
      while(read(fh, &serv, sizeof(struct serverentry)))
      {
	 struct serverentry *newsv = new_serverentry();
	 if(newsv)
	 {
	    memcpy(newsv, &serv, sizeof(struct serverentry));
	    newsv->next = 0;
	    if(last) last->next = newsv;
	    last = newsv;
	 }
      }
      close(fh);
   } else
     add_defaults();
}

void save_serverentrys(void)
{
   FILE *fh;
   char file[84];

   check_prefs_dir();
   
   sprintf(file, "%s/serverlist", get_xdir());
   fh = fopen(file, "w");
   if(fh)
   {  
      struct serverentry *serv = &firstserverentry;
      do
      {
	 if(serv->used) fwrite(serv, 1, sizeof(struct serverentry), fh);
	 serv = serv->next;
      } while(serv);
      fclose(fh);
   }
}

void free_serverentrys(void)
{
   struct serverentry *next;
   struct serverentry *serv = firstserverentry.next;
   while(serv)
   {
      next = serv->next;
      free(serv);
      serv = next;
   }
}

struct serverentry *find_server_entry(char *server, char *chan)
{
   struct serverentry *serv = &firstserverentry;
   while(serv)
   {
      if(serv->used)
      {
	 if(!strcasecmp(server, serv->server))
	 {
	    if(!strcasecmp(chan, serv->channel)) return serv;
	 }
      }
      serv = serv->next;
   }
   return 0;
}

struct serverentry *new_serverentry(void)
{
   struct serverentry *serv = &firstserverentry;
   struct serverentry *prev;
   while(1)
   {
      if(serv->used == 0)
      {
	 serv->used = 1;
	 return(serv);
      }
      prev = serv;
      serv = serv->next;
      if(!serv)
      {
	 serv = malloc(sizeof(struct serverentry));
	 if(serv)
	 {
	    memset(serv, 0, sizeof(struct serverentry));
	    prev->next = serv;
	 } else
	   return 0;
      }
   }
}

void add_server_entry(struct serverentry *serv)
{
   char tbuf[16];
   gchar *new[1][6];
   
   sprintf(tbuf, "%ld", serv->port);

   new[0][0] = serv->channel;
   new[0][1] = serv->server;
   new[0][2] = tbuf;
   new[0][3] = serv->password;
   new[0][4] = serv->comment;
   
   gtk_clist_append((GtkCList*)sllist, new[0]);
}

void connect_clicked(GtkWidget *wid, struct session *sess)
{
   static char s[128];
   char *server, *portstr, *chan, *pass;
   int port;
   if(selected != -1)
   {
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);      
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 2, &portstr);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 3, &pass); 
      port = atoi(portstr);
      strcpy(s, server);
      strcpy(sess->willjoinchannel, chan);
      close_server_list(sess);
      if(!sess)
      {
	 struct server *serv = new_server();
	 if(serv) sess = new_session(serv, 0);
      }
      strcpy(sess->server->password, pass);
      if(strcmp(sess->server->nick, prefs.nick1))
      {
	 gui_change_nick(sess->server, prefs.nick1);
      }
      
      connect_server(sess, s, port, FALSE);
   }
}

void connect_new_clicked(GtkWidget *wid, struct session *sess)
{
   //struct session *osess = sess;
   static char s[128], c[128];
   char *server, *portstr, *chan;
   if(selected != -1)
   {
      struct server *serv;
      int port;
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 2, &portstr);
      port = atoi(portstr);
      strcpy(c, chan);
      strcpy(s, server);
      serv = new_server();
      if(serv)
      {
	 close_server_list(sess);
	 sess = new_session(serv, 0);
	 strcpy(sess->willjoinchannel, c);
	 strcpy(sess->server->nick, prefs.nick1);
	 //memcpy(&sess->prefs, &osess->prefs, sizeof(struct sessionprefs));
	 //memcpy(&sess->server->prefs, &osess->server->prefs, sizeof(struct serverprefs));
	 connect_server(sess, s, port, FALSE);
      }
   }
}

void delete_server_clicked(GtkWidget *wid, struct session *sess)
{
   if(selected != -1)
   {
      char *server, *chan;
      struct serverentry *serv;
      
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);      
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 gtk_clist_remove((GtkCList*)sllist, selected);
	 serv->used = 0;
      }
   }
}

void new_server_clicked(GtkWidget *wid, struct session *sess)
{
   struct serverentry *serv = new_serverentry();
   strcpy(serv->channel, "#new");
   strcpy(serv->server, "*new*");
   serv->port = 6667;
   add_server_entry(serv);
}

void sl_row_unselected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   selected = -1;
   gtk_entry_set_text(GTK_ENTRY(entry_server), "");
   gtk_entry_set_text(GTK_ENTRY(entry_channel), "");
   gtk_entry_set_text(GTK_ENTRY(entry_port), "");
   gtk_entry_set_text(GTK_ENTRY(entry_password), "");
   gtk_entry_set_text(GTK_ENTRY(entry_comment), "");
}

void sl_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even, gpointer none)
{
   gchar *server, *channel, *port, *comment, *password;
   selected = row;
 
   gtk_clist_get_text(GTK_CLIST(sllist), row, 0, &channel);
   gtk_clist_get_text(GTK_CLIST(sllist), row, 1, &server);
   gtk_clist_get_text(GTK_CLIST(sllist), row, 2, &port);
   gtk_clist_get_text(GTK_CLIST(sllist), row, 3, &password);
   gtk_clist_get_text(GTK_CLIST(sllist), row, 4, &comment);
   
   gtk_entry_set_text(GTK_ENTRY(entry_channel), channel);  
   gtk_entry_set_text(GTK_ENTRY(entry_server), server);
   gtk_entry_set_text(GTK_ENTRY(entry_port), port);
   gtk_entry_set_text(GTK_ENTRY(entry_password), password);
   gtk_entry_set_text(GTK_ENTRY(entry_comment), comment);
}

void move_up_clicked(GtkWidget *igad, struct session *sess)
{
   if(selected != -1 && selected > 0)
   {
      char *server, *chan;
      struct serverentry *serv1, *serv2;
      
      gtk_clist_get_text(GTK_CLIST(sllist), selected-1, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected-1, 1, &server);
      if(!server || !server[0]) return;
      serv1 = find_server_entry(server, chan);
 
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      if(!server || !server[0]) return;
      serv2 = find_server_entry(server, chan);
      
      if(serv1 && serv2)
      {
	 struct serverentry *next1 = serv1->next;
	 struct serverentry *next2 = serv2->next;
	 struct serverentry t;
	 memcpy(&t, serv1, sizeof(struct serverentry));
	 memcpy(serv1, serv2, sizeof(struct serverentry));
	 memcpy(serv2, &t, sizeof(struct serverentry));
	 serv1->next = next1;
	 serv2->next = next2;
	 gtk_clist_clear((GtkCList*)sllist);
	 add_all_server_entries();
	 selected--;
	 gtk_clist_select_row((GtkCList*)sllist, selected, 0);
      }
   }
}

void move_dn_clicked(GtkWidget *igad, struct session *sess)
{
   if(selected != -1)
   {
      char *server, *chan;
      struct serverentry *serv1, *serv2;

      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);   
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      if(!server || !server[0]) return;
      serv1 = find_server_entry(server, chan);

      gtk_clist_get_text(GTK_CLIST(sllist), selected+1, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected+1, 1, &server);
      if(!server || !server[0]) return;
      serv2 = find_server_entry(server, chan);

      if(serv1 && serv2)
      {
	 struct serverentry *next1 = serv1->next;
	 struct serverentry *next2 = serv2->next;
	 struct serverentry t;
	 memcpy(&t, serv1, sizeof(struct serverentry));
	 memcpy(serv1, serv2, sizeof(struct serverentry));
	 memcpy(serv2, &t, sizeof(struct serverentry));
	 serv1->next = next1;
	 serv2->next = next2;
	 gtk_clist_clear((GtkCList*)sllist);
	 add_all_server_entries();
	 selected++;
         gtk_clist_select_row((GtkCList*)sllist, selected, 0);
      }
   } 
}

void skip_motd(GtkWidget *igad, gpointer serv)
{
   if(GTK_TOGGLE_BUTTON(igad) -> active)
      prefs.flags |= (1<<PREFS_SKIPMOTD);
   else
      prefs.flags &= ~(1<<PREFS_SKIPMOTD);
}

void handle_server(GtkWidget *igad, gpointer none)
{
   if(selected != -1)
   {
      char *server, *chan;
      struct serverentry *serv;

      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 server = gtk_entry_get_text((GtkEntry *)igad);
	 gtk_clist_set_text((GtkCList*)sllist, selected, 1, server);
	 strcpy(serv->server, server);
      }
   }
}

void handle_channel(GtkWidget *igad, gpointer none)
{
   if(selected != -1)
   {
      char *server, *chan;
      struct serverentry *serv;

      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 chan = gtk_entry_get_text((GtkEntry *)igad);
	 gtk_clist_set_text((GtkCList*)sllist, selected, 0, chan);
	 strcpy(serv->channel, chan);
      }
   }
}

void handle_comment(GtkWidget *igad, gpointer none)
{
   if(selected != -1)
   {
      char *server, *comment, *chan;
      struct serverentry *serv;

      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 comment = gtk_entry_get_text((GtkEntry *)igad);
	 gtk_clist_set_text((GtkCList*)sllist, selected, 4, comment);
	 strcpy(serv->comment, comment);
      }
   }
}

void handle_password(GtkWidget *igad, gpointer none)
{
   if(selected != -1)
   {
      char *server, *chan, *pw;
      struct serverentry *serv;
     
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 pw = gtk_entry_get_text((GtkEntry *)igad);
	 gtk_clist_set_text((GtkCList*)sllist, selected, 3, pw);
	 strcpy(serv->password, pw);
      }
   }
}

void handle_port(GtkWidget *igad, gpointer none)
{
   if(selected != -1)
   {
      char *server, *port, *chan;
      struct serverentry *serv;
      
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 0, &chan);      
      gtk_clist_get_text(GTK_CLIST(sllist), selected, 1, &server);
      serv = find_server_entry(server, chan);
      if(serv)
      {
	 char tbuf[16];
	 long p;
	 port = gtk_entry_get_text((GtkEntry *)igad);
	 p = atol(port);
	 sprintf(tbuf, "%ld", p);
	 if(p < 1)
	 {
	    p = 6667;
	    gtk_clist_set_text((GtkCList*)sllist, selected, 2, "6667");
	    gtk_entry_set_text((GtkEntry *)entry_port, "6667");
	 } else {
	   gtk_clist_set_text((GtkCList*)sllist, selected, 2, tbuf);
	   gtk_entry_set_text((GtkEntry *)entry_port, tbuf);
	 }
	 serv->port = p;
      }
   }
}

GtkWidget *nick1gad;
GtkWidget *nick2gad;
GtkWidget *nick3gad;
GtkWidget *realnamegad;
GtkWidget *usernamegad;

void add_all_server_entries(void)
{
      struct serverentry *serv = &firstserverentry;
      do
      {
	 if(serv->used) add_server_entry(serv);
	 serv = serv->next;
      } while(serv);
}

gint close_server_list(struct session *sess)
{
   strcpy(prefs.nick1, gtk_entry_get_text((GtkEntry*)nick1gad));
   strcpy(prefs.nick2, gtk_entry_get_text((GtkEntry*)nick2gad));
   strcpy(prefs.nick3, gtk_entry_get_text((GtkEntry*)nick3gad));
   strcpy(prefs.username, gtk_entry_get_text((GtkEntry*)usernamegad));
   strcpy(prefs.realname, gtk_entry_get_text((GtkEntry*)realnamegad));
   gtk_widget_destroy(slwin);
   slwin = 0;
   save_serverentrys();
   return FALSE;
}

void open_server_list(GtkWidget *widd, struct session *sess)
{
   static gchar *titles[] = {"Channel", "Server", "Port", "Password", "Comment"};
   GtkWidget *vbox, *box, *label, *input, *wid;

   if(slwin) close_server_list(sess);

   slwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(slwin), "X-Chat: Server List");
   gtk_signal_connect(GTK_OBJECT(slwin), "delete_event",
                       GTK_SIGNAL_FUNC(close_server_list), sess);
   gtk_window_set_policy(GTK_WINDOW(slwin), TRUE, TRUE, FALSE);
   
   vbox = gtk_vbox_new(0,2);
   gtk_container_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(slwin), vbox);
   gtk_widget_show(vbox);
   
   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   label = gtk_label_new("Nicknames:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   nick1gad = gtk_entry_new_with_max_length(63);
   gtk_entry_set_text(GTK_ENTRY(nick1gad), prefs.nick1);
   //gtk_signal_connect(GTK_OBJECT(nick1gad), "key_press_event",
//	               GTK_SIGNAL_FUNC(handle_nick1_key), 0);
   gtk_container_add(GTK_CONTAINER(box), nick1gad);
   gtk_widget_show(nick1gad);

   nick2gad = gtk_entry_new_with_max_length(63);
   gtk_entry_set_text(GTK_ENTRY(nick2gad), prefs.nick2);
   //gtk_signal_connect(GTK_OBJECT(nick2gad), "key_press_event",
   //                          GTK_SIGNAL_FUNC(handle_nick2_key), 0);
   gtk_container_add(GTK_CONTAINER(box), nick2gad);
   gtk_widget_show(nick2gad);
   
   nick3gad = gtk_entry_new_with_max_length(63);
   gtk_entry_set_text(GTK_ENTRY(nick3gad), prefs.nick3);
   //gtk_signal_connect(GTK_OBJECT(nick3gad), "key_press_event",
   //                          GTK_SIGNAL_FUNC(handle_nick3_key), 0);
   gtk_container_add(GTK_CONTAINER(box), nick3gad);
   gtk_widget_show(nick3gad);
   
   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);
   
   label = gtk_label_new("Realname:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   realnamegad = gtk_entry_new_with_max_length(63);
   //gtk_signal_connect(GTK_OBJECT(realnamegad), "key_press_event",
   //                          GTK_SIGNAL_FUNC(handle_realname_key), 0);
   gtk_entry_set_text(GTK_ENTRY(realnamegad), prefs.realname);
   gtk_container_add(GTK_CONTAINER(box), realnamegad);
   gtk_widget_show(realnamegad);
   
   label = gtk_label_new("Username:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   usernamegad = gtk_entry_new_with_max_length(63);
   //gtk_signal_connect(GTK_OBJECT(usernamegad), "key_press_event",
   //                          GTK_SIGNAL_FUNC(handle_username_key), 0);
   gtk_entry_set_text(GTK_ENTRY(usernamegad), prefs.username);
   gtk_container_add(GTK_CONTAINER(box), usernamegad);
   gtk_widget_show(usernamegad);

   wid = gtk_toggle_button_new_with_label("Skip MOTD?");
   gtk_signal_connect(GTK_OBJECT(wid), "toggled",
		      GTK_SIGNAL_FUNC(skip_motd), 0);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(wid),
			       (prefs.flags&(1<<PREFS_SKIPMOTD)));
   gtk_widget_show(wid);
   
   wid = gtk_clist_new_with_titles(5, titles);
   gtk_widget_set_usize(wid, -1, 200);
   gtk_container_add(GTK_CONTAINER(vbox), wid);
   gtk_clist_set_column_width(GTK_CLIST(wid), 0, 90);
   gtk_clist_set_column_width(GTK_CLIST(wid), 1, 165);
   gtk_clist_set_column_width(GTK_CLIST(wid), 2, 40);
   gtk_clist_set_column_width(GTK_CLIST(wid), 3, 60);
   gtk_clist_set_column_width(GTK_CLIST(wid), 4, 50);
   gtk_widget_show(wid);
   sllist = wid;
   gtk_signal_connect(GTK_OBJECT(sllist),
		      "select_row", GTK_SIGNAL_FUNC(sl_row_selected), 0);
   gtk_signal_connect(GTK_OBJECT(sllist),
		      "unselect_row", GTK_SIGNAL_FUNC(sl_row_unselected), (gpointer)1);

   box = gtk_hbox_new(0,0);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   label = gtk_label_new("Server Name:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   input = gtk_entry_new_with_max_length(99);
   gtk_container_add(GTK_CONTAINER(box), input);
   gtk_widget_show(input);
   entry_server = input;
   gtk_signal_connect(GTK_OBJECT(entry_server), "activate",
		      GTK_SIGNAL_FUNC(handle_server), 0);
   
   label = gtk_label_new("Port:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   input = gtk_entry_new_with_max_length(14);
   gtk_widget_set_usize(input, 80, -1);
   gtk_container_add(GTK_CONTAINER(box), input);
   gtk_widget_show(input);
   entry_port = input;
   gtk_signal_connect(GTK_OBJECT(entry_port), "activate",
		      GTK_SIGNAL_FUNC(handle_port), 0);
   
   label = gtk_label_new("Password:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   input = gtk_entry_new_with_max_length(85);
   gtk_signal_connect(GTK_OBJECT(input), "activate",
		      GTK_SIGNAL_FUNC(handle_password), 0);
   gtk_widget_set_usize(input, 80, -1);
   gtk_container_add(GTK_CONTAINER(box), input);
   gtk_widget_show(input);
   entry_password = input;
   
   box = gtk_hbox_new(0,5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   label = gtk_label_new("Comment:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   input = gtk_entry_new_with_max_length(99);
   gtk_signal_connect(GTK_OBJECT(input), "activate",
		      GTK_SIGNAL_FUNC(handle_comment), 0);
   gtk_widget_set_usize(input, 160, -1);
   gtk_container_add(GTK_CONTAINER(box), input);
   gtk_widget_show(input);
   entry_comment = input;
   
   label = gtk_label_new("Channel:");
   gtk_container_add(GTK_CONTAINER(box), label);
   gtk_widget_show(label);
   
   input = gtk_entry_new_with_max_length(85);
   gtk_container_add(GTK_CONTAINER(box), input);
   gtk_widget_show(input);
   entry_channel = input;
   gtk_signal_connect(GTK_OBJECT(entry_channel), "activate",
		      GTK_SIGNAL_FUNC(handle_channel), 0);

   wid = gtk_hseparator_new();
   gtk_box_pack_start(GTK_BOX(vbox), wid, FALSE, FALSE, 8);
   gtk_widget_show(wid);
   
   box = gtk_hbox_new(0,5);
   gtk_container_border_width(GTK_CONTAINER(box), 5);
   gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, TRUE, 5);
   gtk_widget_show(box);

   if(sess)
   {
      wid = gtk_button_new_with_label("Connect");
      gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(connect_clicked), sess);
      gtk_container_add(GTK_CONTAINER(box), wid);
      gtk_widget_show(wid);
   }

   wid = gtk_button_new_with_label("Connect in New Window");
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(connect_new_clicked), sess);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_widget_show(wid);
   
   wid = gtk_button_new_with_label("New Server");
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(new_server_clicked), sess);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_widget_show(wid);
   
   wid = gtk_button_new_with_label("Delete Server");
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(delete_server_clicked), sess);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_widget_show(wid);
   
   wid = gtk_button_new_with_label("Move Up");
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(move_up_clicked), sess);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_widget_show(wid);
   
   wid = gtk_button_new_with_label("Move Dn");
   gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		      GTK_SIGNAL_FUNC(move_dn_clicked), sess);
   gtk_container_add(GTK_CONTAINER(box), wid);
   gtk_widget_show(wid);
   
   add_all_server_entries();
   
   gtk_widget_show(slwin);
}

struct defaultserv dserv[] = 
{
     {"#EliteCafe", "mansfield.pa.us.pulsarnet.net", "PulsarNet",7000},
     {"#EliteCafe", "osaka.jp.pulsarnet.net", "PulsarNet",6667},  
     
     {"", "irc.stealth.net", "IRCNet",6667},
     {"", "irc.funet.fi", "IRCNet",6667},
     
     {"", "irc.dal.net", "DALNet",6667},
     {"", "liberty.nj.dal.net", "DALNet", 6667}, 
     
     {"", "irc.efnet.net", "EFNet",6667},
     {"", "irc.ais.net", "EFNet",6667},    
     
     {"", "sprynet.us.galaxynet.org", "GalaxyNet",6667},
     {"", "atlanta.ga.us.galaxynet.org", "GalaxyNet",6667}, 
     
     {"", "us.undernet.org", "UnderNet",6667},
     {"", "atlanta.ga.us.undernet.org", "UnderNet",6667},
     
     {0,0,0,0}
};

void add_defaults(void)
{
   struct serverentry *serv;
   int i = 0;
   while(1)
   {
      if(dserv[i].channel == 0) return;
      
      serv = new_serverentry();
      if(serv)
      {
	 strcpy(serv->channel, dserv[i].channel);
	 strcpy(serv->server, dserv[i].server);
	 serv->password[0] = 0;
	 strcpy(serv->comment, dserv[i].comment);
	 serv->port = dserv[i].port;	 
	 //add_server_entry(serv);
      }
      i++;
   }
}
