/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: CDD2600.cc,v $
 * Revision 1.12  1998/10/24 14:30:40  mueller
 * Changed prototype of 'readDiskToc()'.
 *
 * Revision 1.11  1998/10/03 15:11:05  mueller
 * Moved basic writing methods to class 'CDD2600Base'.
 *
 * Revision 1.10  1998/09/27 19:17:46  mueller
 * Fixed 'disc-info' for CDD2000.
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 *
 * Revision 1.9  1998/09/22 19:15:13  mueller
 * Removed memory allocations during write process.
 *
 * Revision 1.8  1998/09/08 11:54:22  mueller
 * Extended disk info structure because CDD2000 does not support the
 * 'READ DISK INFO' command.
 *
 * Revision 1.7  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.6  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.5  1998/08/25 19:23:46  mueller
 * Added index extraction for 'read-toc'.
 * Added disk info code.
 *
 * Revision 1.4  1998/08/07 12:44:30  mueller
 * Changed group 1 mode select commands to group 0 mode select commands.
 * Implement pre-gap detection in 'readDiscToc()'.
 *
 */

static char rcsid[] = "$Id: CDD2600.cc,v 1.12 1998/10/24 14:30:40 mueller Exp $";

#include <config.h>

#include <string.h>
#include <assert.h>

#include "CDD2600.h"
#include "SubChannel.h"

#include "Toc.h"
#include "util.h"

CDD2600::CDD2600(ScsiIf *scsiIf, Toc *toc)
  : CdrDriver(scsiIf, toc), CDD2600Base(this)
{
  driverName_ = "CDD2600 - Version 1.1";
  
  leadInLength_ = leadOutLength_ = 0;
  speed_ = 2;
  simulate_ = 1;

  memset(&diskInfo_, 0, sizeof(DiskInfo));
}

CDD2600::~CDD2600()
{
}

// static constructor
CdrDriver *CDD2600::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new CDD2600(scsiIf, toc);
}

// sets speed
// return: 0: OK
//         1: illegal speed
int CDD2600::speed(int s)
{
  if (s >= 0 && s <= 2) {
    speed_ = s;
    return 0;
  }
  else {
    return 1;
  }
}

// loads ('unload' == 0) or ejects ('unload' == 1) tray
// return: 0: OK
//         1: scsi command failed

int CDD2600::loadUnload(int unload) const
{
  unsigned char cmd[10];

  memset(cmd, 0, 10);

  cmd[0] = 0xe7; // MEDIUM LOAD/UNLOAD
  if (unload) {
    cmd[8] |= 0x01;
  }
  
  if (sendCmd(cmd, 10, NULL, 0, NULL, 0) != 0) {
    message(-2, "Cannot load/unload medium.");
    return 1;
  }

  return 0;
}

// sets various audio play parameters, output channels are set to stereo mode
// and given volume
// immediate: 0: wait until audio play command finished
//            1: command finishs immediately after playback has started
// sotc:      0: play across track boundaries
//            1: stop playing at track boundaries
int CDD2600::modeSelectPlay(int immediate, int sotc, unsigned char volume)
{
  unsigned char mp[16];

  memset(mp, 0, 16);

  mp[0] = 0x0e; // PLAY page code
  mp[1] = 14; // parameter length
  if (immediate != 0) {
    mp[2] |= 0x04;
  }
  if (sotc != 0) {
    mp[2] |= 0x02;
  }
  mp[8]  = 1;
  mp[9]  = volume;
  mp[10] = 2;
  mp[11] = volume;

  if (setModePage(mp) != 0) {
    message(-2, "Cannot set play parameters.");
    return 1;
  }

  return 0;
}

int CDD2600::initDao()
{
  long n;
  blockLength_ = AUDIO_BLOCK_LEN;
  blocksPerWrite_ = scsiIf_->maxDataLen() / blockLength_;

  assert(blocksPerWrite_ > 0);

  if (modeSelectBlockSize(blockLength_) != 0 ||
      modeSelectSpeed(-1, speed_, simulate_, 1) != 0 ||
      modeSelectCatalog(toc_) != 0 ||
      readSessionInfo(&leadInLength_, &leadOutLength_, 1) != 0) {
    return 1;
  }

  // allocate buffer for write zeros
  n = blocksPerWrite_ * blockLength_;
  delete[] zeroBuffer_;
  zeroBuffer_ = new char[n];
  memset(zeroBuffer_, 0, n);

  return 0;
}

int CDD2600::startDao()
{
  long lba = -leadInLength_ - 150; // Value is not really important since the
                                   // LBA is not used by 'writeData'.

  if (writeSession(toc_, multiSession_) != 0) {
    return 1;
  }

  message(0, "Writing lead-in and gap...");

  // write lead-in
  if (writeZeros(lba, leadInLength_ + 150/*2 second gap*/) != 0) {
    flushCache();
    return 1;
  }

  message(0, "");

  return 0;
}

int CDD2600::finishDao()
{
  long lba = 0;

  message(0, "Writing lead-out...");

  // write lead-out
  if (writeZeros(lba, leadOutLength_) != 0) {
    flushCache();
    return 1;
  }

  message(0, "\nFlushing cache...");
  
  if (flushCache() != 0) {
    return 1;
  }

  message(0, "");

  delete[] zeroBuffer_, zeroBuffer_ = NULL;

  return 0;
}

// Writes data to target, the block length depends on the actual writing mode
// and is stored internally. 'len' is number of blocks to write.
// 'lba' specifies the next logical block address for writing and is updated
// by this function but not used for writing
// return: 0: OK
//         1: scsi command failed
int CDD2600::writeData(long &lba, const char *buf, long len)
{
  assert(blocksPerWrite_ > 0);
  assert(blockLength_ > 0);
  int nwritten = 0;
  int writeLen = 0;
  unsigned char cmd[10];

  memset(cmd, 0, 10);
  cmd[0] = 0x2a; // WRITE1
  
  while (len > 0) {
    writeLen = (len > blocksPerWrite_ ? blocksPerWrite_ : len);

    cmd[7] = writeLen >> 8;
    cmd[8] = writeLen & 0xff;

    if (sendCmd(cmd, 10, (unsigned char *)(buf + (nwritten * blockLength_)),
		writeLen * blockLength_, NULL, 0) != 0) {
      message(-2, "Write data failed.");
      return 1;
    }

    lba += writeLen;

    len -= writeLen;
    nwritten += writeLen;
  }
      
  return 0;
}

Toc *CDD2600::readDiskToc(const char *audioFilename)
{

  blockLength_ = AUDIO_BLOCK_LEN;
  if (modeSelectBlockSize(blockLength_) != 0) {
    return NULL;
  }

  modeSelectSpeed(2, -1, 1, 0);

  return CdrDriver::readDiskToc(audioFilename);
}

// tries to read catalog number from disk and adds it to 'toc'
// return: 1 if valid catalog number was found, else 0

int CDD2600::readCatalog(Toc *toc)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char catalog[14];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x02; // get media catalog number
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read sub channel data.");
    return 0;
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 13; i++) {
      catalog[i] = data[0x09 + i];
    }
    catalog[13] = 0;

    if (toc->catalog(catalog) == 0) {
      return 1;
    }
  }

  return 0;
}

int CDD2600::analyzeTrack(int trackNr, long startLba, long endLba,
			  Msf *index, int *indexCnt, long *pregap,
			  char *isrcCode, unsigned char *ctl)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  int i;

  int ret = CdrDriver::analyzeTrack(trackNr, startLba, endLba, index,
				    indexCnt, pregap, isrcCode, ctl);

  if (ret == 0) {
    // read ISRC code from sub channel
    memset(cmd, 0, 10);
    cmd[0] = 0x42; // READ SUB CHANNEL
    cmd[2] = 0x40; // get sub channel data
    cmd[3] = 0x03; // get ISRC code
    cmd[6] = trackNr;
    cmd[7] = dataLen >> 8;
    cmd[8] = dataLen;

    if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
      message(-1, "Cannot read ISRC code.");
    }
    else {
      if (data[0x08] & 0x80) {
	for (i = 0; i < 12; i++) {
	  isrcCode[i] = data[0x09 + i];
	}
	isrcCode[12] = 0;
      }
    }
  }

  return ret;
}

int CDD2600::getTrackIndex(long lba, int *trackNr, int *indexNr, 
			   unsigned char *ctl)
{
  long relPos;

  readBlock(lba);

  return readSubChannelData(trackNr, indexNr, &relPos, ctl);
}

int CDD2600::readSubChannelData(int *trackNr, int *indexNr, long *relPos,
				unsigned char *ctl)
{
  unsigned char cmd[10];
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x00; // get sub Q channel data
  cmd[6] = 0;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read sub Q channel data.");
    return 1;
  }

  *trackNr = data[6];
  *indexNr = data[7];
  *relPos = 0;
  *relPos |= data[0x0c] << 24;
  *relPos |= data[0x0d] << 16;
  *relPos |= data[0x0e] << 8;
  *relPos |= data[0x0f];

  if (ctl != NULL) {
    *ctl = data[5] & 0x0f;
  }

  return 0;
}

// reads a single block of length 'blockLength_' from given sector
// return: 0: OK
//         1: error occured
void CDD2600::readBlock(unsigned long sector)
{
  unsigned char cmd[10];
  unsigned long dataLen = 2 * blockLength_;
  unsigned char *data = new (unsigned char)[dataLen];


  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x28; // READ10
  cmd[2] = sector >> 24;
  cmd[3] = sector >> 16;
  cmd[4] = sector >> 8;
  cmd[5] = sector;
  cmd[7] = 0;
  cmd[8] = 2;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen) != 0) {
    message(-2, "Cannot read block - ignored.");
  }

  delete[] data;
}

// Retrieve disk information.
// return: DiskInfo structure or 'NULL' on error
DiskInfo *CDD2600::diskInfo()
{
  unsigned char cmd[10];
  unsigned long dataLen = 34;
  unsigned char data[34];

  memset(&diskInfo_, 0, sizeof(DiskInfo));

  if (readCapacity(&(diskInfo_.capacity), 0) == 0) {
    diskInfo_.valid.capacity = 1;
  }
  
  if (readSessionInfo(&leadInLength_, &leadOutLength_, 0) == 0) {
    diskInfo_.valid.manufacturerId = 1;
    
    // start time of lead-in
    diskInfo_.manufacturerId = Msf(450150 - leadInLength_ - 150 );
    diskInfo_.empty = 1; // this is for the CDD2000 which does not support
                         // READ DISK INFORMATION
  }
  else {
    diskInfo_.empty = 0; // this is for the CDD2000 which does not support
                         // READ DISK INFORMATION
  }
  diskInfo_.valid.empty = 1;

  memset(cmd, 0, 10);
  memset(data, 0, dataLen);

  cmd[0] = 0x51; // READ DISK INFORMATION
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  if (sendCmd(cmd, 10, NULL, 0, data, dataLen, 0) == 0) {
    diskInfo_.empty = (data[2] & 0x03) == 0 ? 1 : 0;
    diskInfo_.cdrw = (data[2] & 0x10) != 0 ? 1 : 0;
    diskInfo_.valid.cdrw = 1;
  }

  return &diskInfo_;
}
